#!/usr/bin/perl
#
# Copyright 2000, 2001 Tivano Software GmbH. This software is
# distributed under the terms of the GNU General Public License. See
# COPYING for additional information.
#
# This file contains helper functions for the cdrw-taper.

use Fcntl ':flock';

##
## Configurable settings. Change as needed to adapt to your system.
##

# Amanda config directory
$CONFIG_ROOT = "/etc/amanda";

# Directory where intermediate directories will be created
$DUMP_DIR = "/var/tmp/amanda-dumps";

# Uncomment if intermedia dirs should be deleted after burning
$DELETE_DIRS = 1;

# Uncomment if you want to allow writing on CD-R media (non-erasable)
$WRITE_NON_ERASABLE = 0;

# Regexp for CD-RW devices
$CD_DEVICE_RE = "/dev/s(cd|g)\\d+:?.*";

# Original taper binary
$TAPER_ORIG = "/usr/lib/amanda/taper.orig";

# Blocksize of output media in bytes (2k for CD-RW)
$BLOCKSIZE = 2048;

# Blocks per media (CDRW: 650 MB = 650 * 1024k / 2k = 332800)
# $MEDIABLOCKS = 332800;
# Blocks per media (DVD+RW: 4700 MB = 4700 * 1000 * 1000 / 2k = 2294921)
$MEDIABLOCKS = 2294921;

# Binaries for mount, umount, sendmail, mkisofs and cdrecord
$MOUNT    = "mount";
$UMOUNT   = "umount";
$CDRECORD = "cdrecord";
$MKISOFS  = "mkisofs";
$SENDMAIL = "/usr/lib/sendmail";

##
## No user editable settings below here
##

$VERSION = "0.3";

local $changer_gravity;
local $changer_slots;
local %changer_media;

# Read Amanda's config file and chdir to the config directory
sub readAmandaConfig {
    my $cfg = shift;

    open CONFIG, "$CONFIG_ROOT/$cfg/amanda.conf"
	or die "taper: Cannot open $CONFIG_ROOT/$cfg/amanda.conf";
    while ($line = <CONFIG>) {
	next if $line =~ /^\s*\#/;
	if ($line =~ /^(tapedev|rawtapedev|tapelist|tapecycle|labelstr|logdir|runtapes|tapetype|tpchanger)\s+(\"([^\"]*)\"|\S+)/i) {
	    my $var = $1;
	    my $value = $2;
	    $value =~ s/(^"|"$)//g;
	    $value =~ s/[\\\"]/\\$&/g;
	    eval "\$$var = '$value'";
	} elsif ($line =~ /^(mailto)\s+\"([^\"]*)\"/i) {
	    my $var = $1;
	    my $value = $2;
	    $value =~ s/[\\\"]/\\$&/g;
	    eval "\@$var = split(' ', '$value')";
	} elsif ($line =~ /^define tapetype\s+\"([^\"]*)\"/i) {
	}
    }
    close CONFIG;
 
    # Assign better names to some "re-used" (mis-used?) amanda config
    # variables...
    $CDRW_MOUNT_DIR = $tapedev;
    $CDRW_WRITE_DEVICE = $rawtapedev;
 
    # Make $logdir absolute if it isn't already
    $logdir = "$CONFIG_ROOT/$cfg/$logdir" if $logdir !~ /^\//;

    # CD to the config directory
    chdir "$CONFIG_ROOT/$cfg"
	or die "taper: cannot chdir to $CONFIG_ROOT/$cfg: $!";
 
    # read the "tape"-list (better: media list :-) The media list is
    # sorted in reverse datestamp order (newest entry first).
    open ML, $tapelist or die "taper: Cannot open $tapelist: $!";
    while (<ML>) {
	s/[\r\n]//g;
	push @medialist, $_;
    }
    close ML;
    @medialist = sort by_datestamp @medialist;

    # If a tape changer is specified without a complete path, we need to
    # append our own directory to $PATH. Then, we ask it for info.
    if (defined($tpchanger)) {
	# THIS IS A HACK
	$ENV{"PATH"} .= ":".$INC[$#INC];
	my $changerinfo = `$tpchanger -info`;
	if (($? & 0xff) || ($? >> 8)) {
	    die "taper: tape changer failed with exit status $?";
	}
	if ($changerinfo !~ /^(.+)\s+(\d+)\s+([01])\s+$/) {
	    die "taper: can't parse $tpchanger -info output: $changerinfo";
	} else {
	    $changer_slots = $2;
	    $changer_gravity = $3;
	}
    }

    @nonErasable = ();
}

# This returns a list of the labels of all disks currently in the available
# drives. If the tape changer cannot go backwards, this steps through the
# media until it finds a usable one.
sub check_disk {
    my @labels = ();

    if (defined($tpchanger)) {
	my $slot, $dev, $label;
	($slot, $dev) = &loadSlot("current");
	if (defined($slot)) {
	    $label = &getLabel($dev);
	    if (!$changer_gravity) {
	        # Cycle through all slots and take note of the media labels
		if ($label) { $changer_media{$label} = $slot; }
	        for (my $i = 1; $i < $changer_slots; $i++) {
		    ($slot, $dev) = &loadSlot("next");
		    if (defined($slot)) {
			$label = &getLabel($dev);
			if ($label) {
			    $changer_media{$label} = $slot;
			    push @labels, $label;
		        } else {
			    my $writeDev = $dev;
			    if ($writeDev =~ /:/) { $writeDev = $'; }
			    my $result = `$CDRECORD -toc dev=$writeDev 2>&1`;
			    if ($result !~ /cdrecord: Cannot read TOC header/s
				    && $WRITE_NON_ERASABLE) {
			        $error = `$CDRECORD -atip dev=$writeDev`;
			        if ($error !~ /Is erasable/) {
			            push @nonErasable, $slot;
				}
			    }
			}
		    }
		}
	    } else {
	        # Cycle through slots until we find a media label that is_usable
		my $i = 0;
		while ($label && !is_usable($label) && $i++ < $changer_slots) {
		    ($slot, $dev) = &loadSlot("next");
		    if (defined($slot)) {
			$label = &getLabel($dev);
			if ($label) {
			    $changer_media{$label} = $slot;
			    push @labels, $label;
		        } else {
			    my $writeDev = $dev;
			    if ($writeDev =~ /:/) { $writeDev = $'; }
			    my $result = `$CDRECORD -toc dev=$writeDev 2>&1`;
			    if ($result !~ /cdrecord: Cannot read TOC header/s
				    && $WRITE_NON_ERASABLE) {
			        $error = `$CDRECORD -atip dev=$writeDev`;
			        if ($error !~ /Is erasable/) {
			            push @nonErasable, "current";
				}
			    }
			}
		    } else {
			$label = 0;
		    }
		}
	    }
	}
    } else { # no tpchanger
	my $label = &getLabel($CDRW_WRITE_DEVICE);
	if ($label) {
	    push @labels, $label;
        } else {
	    my $writeDev = $dev;
	    if ($writeDev =~ /:/) { $writeDev = $'; }
	    my $result = `$CDRECORD -toc dev=$writeDev 2>&1`;
	    if ($result !~ /cdrecord: Cannot read TOC header/s
		    && $WRITE_NON_ERASABLE) {
	        $error = `$CDRECORD -atip dev=$writeDev`;
	        if ($error !~ /Is erasable/) {
		    push @nonErasable, "current";
		}
	    }
	}
    }
    return @labels;
}

# Executes a "tpchanger -slot " command and parses the response.
# Returns (slotname, device) on success, or (undef, undef) on error
sub loadSlot {
    my $slot = shift;
    my $dev;

    if (!defined($tpchanger) && $slot eq "current") {
	return ($slot, $CDRW_WRITE_DEVICE);
    }

    $res = `$tpchanger -slot $slot`;
    if (($? & 0xff) || ($? >> 8)) {
	log_add("ERROR", "changer failed to load slot $slot with code $? and response $res");
    } else {
	if ($res =~ /^(\S+)\s+(\S+)\s*$/) {
	    $slot = $1;
	    $dev = $2;
	} else {
	    log_add("ERROR", "failed to parse changer response $res");
	    $slot = undef;
	}
    }

    return ($slot, $dev);
}

# Load the media with the given label on some writer device. Returns the
# device name on success, false otherwise.
# If the specified tape changer is a gravity changer, the media in the
# "current" slot *must* be the one to load.
sub loadMedia {
    my $label = shift;
    my $dev = 0;
 
    if (defined($tpchanger)) {
	my $slot;
	if (!$changer_gravity) {
	    if (!exists($changer_media{$label})) {
		return 0;
	    }
	    $slot = $changer_media{$label};
	} else {
	    $slot = "current";
	}
	($slot, $dev) = &loadSlot($slot);
    } else { # no changer
	$dev = $CDRW_WRITE_DEVICE;
    }

    # Now we know the device. Check if the media has the correct label.
    if ($dev && &getLabel($dev) eq $label) {
	return $dev;
    } else {
	log_add("ERROR", "label on media in $dev is not $label");
    }

    return 0;
}

# Find an intermediate storage directory that can take a file of the given
# size. Returns undef if nothing can be found.
sub findMediaDir {
    my $wantedBlocks = shift;
    my $best = 0;

    foreach my $usedMedia (keys %usedMedia) {
	if ($usedMedia{$usedMedia}->{freeBlocks} >= $wantedBlocks &&
		(!$best || $usedMedia{$best}->{freeBlocks} > $usedMedia{$usedMedia}->{freeBlocks})) {
	    $best = $usedMedia;
	}
    }
    if ($best) { return $best; }

    # Nothing found. Any more disks available?
    if ($#usableMedia < 0) { return undef; }
    my $next = shift @usableMedia;
    if (-d "$DUMP_DIR/$next") {
	system("rm -r $DUMP_DIR/$next/*");
    } else {
	mkdir("$DUMP_DIR/$next", 0700);
    }

    # Write amanda label file
    if (!open(LABEL, ">$DUMP_DIR/$next/AMANDA_LABEL")) {
	return undef;
    }
    print LABEL "$next\n";
    close LABEL;

    $usedMedia{$next} = {"freeBlocks" => $MEDIABLOCKS};

    return $next;
}

# Copy a file from the holding disk into the dump disk.  Since we're
# writing to a medium with a file system, we don't really need the 32k
# header that amanda puts in front of each dump. Instead, we write the
# relevant information from that header to a file called
# $basename.info and the dump itself to a file called
# $basename.(dump|tar)[.gz] (depending on dump method and
# compression). $basename is the name of the file in the holding disk.
sub dump_file {
    my ($input_name, $targetDir) = @_;

    my ($basename) = ($input_name =~ /.*\/(.*)/);
    if (!open(IN, "$input_name")) {
	 return "cannot read $input_name: $!";
    }

    # Read the 32k amanda header.
    my $buffer, $info;
    $error = "cannot read amanda header from $input_name";
    if (read(IN, $info, 32*1024) == 32*1024) {
	# Remove trailing 0-bytes
	$info =~ s/\0*$//s;
	my ($compext, $dumpext) =
	    ($info =~ /AMANDA: FILE \d+ [-\w.]+ \S+ lev \d+ comp (\S+) program (\S+)/);
	if ($compext && $dumpext) {
	    # Don't append a compression extension if the dump is uncompressed
	    $compext = "" if $compext eq "N";
	    # Use '.tar' as dump extension if the dump program is "tar" or "gtar"
	    # Otherwise, use the program's base name.
	    $dumpext =~ s%.*/%%;
	    $dumpext = "tar" if $dumpext =~ /^g?tar$/;
	    $dumpext = ".$dumpext";

	    # Write the information to $basename.info
	    $error = "cannot write info file $targetDir/$basename.info";
	    if (open(OUT, ">$targetDir/$basename.info")) {
		print OUT "$info\n";
		close OUT;
	    }

	    # Now dump the actual file
	    # TODO: concatenate multiple chunks of the file
	    $error = "cannot write dump file $targetDir/$basename$dumpext$compext";
	    if (open(OUT, ">$targetDir/$basename$dumpext$compext")) {
		$error = 0;
		while (read(IN, $buffer, 32*1024) && !$error) {
		    if (!(print OUT $buffer)) {
			$error = "error while writing $targetDir/$basename$dumpext$compext";
		    }
		}
		close OUT;
	    }
	}
    }
    close IN;
    return $error ? "$error: $!" : 0;
}

# Returns the size of the output file in BLOCKs - on CDRW, blocks are 2k bytes
sub fileSize {
    my $filename = shift;
    return int(((-s $filename) + $BLOCKSIZE - 1) / $BLOCKSIZE);
}

# Compare two media list entries for sorting in reverse datestamp
# (newest entry first) order
sub by_datestamp {
  my ($ds_a) = ($a =~ /(\d+)/);
  my ($ds_b) = ($b =~ /(\d+)/);
  return $ds_b <=> $ds_a;
}

# Check if the disk with a given label may be overwritten
sub is_usable {
  my ($check_label) = @_;
  my $count = 0;
  my $datestamp, $label, $reuse_flag;
  foreach my $entry (@medialist) {
    ($datestamp, $label, $reuse_flag) = split /\s+/, $entry;
    $count++ if $reuse_flag eq "reuse";
    last if $label eq $check_label;
  }
  # Not in the tapelist? -> New tape, may be used
  return 1 if $label ne $check_label;

  # Tape is in the tapelist -> may be reused if $reuse allows it and
  # the tape is old enough, i.e. $count >= $tapecycle or $datestamp == 0
  return 0 if $reuse_flag ne "reuse";
  return $datestamp == 0 || $count >= $tapecycle;
}

# This generates a list of media labels to be used in this run
sub findUsableMedia {
    my @media = ();
    my %available = ();
    my $label;
    my %tmp;

    foreach $label (@availableMedia) {
	$tmp{$label} = $available{$label} = is_usable($label);
    }

    # Find those available media that are not in the medialist
    # (i. e. new media), and use these
    foreach $label (@medialist) {
	($datestamp, $lbl, $reuse_flag) = split /\s+/, $label;
	delete $tmp{$lbl};
    }
    foreach $label (keys %tmp) {
	if ($#media < $runtapes - 1 && $tmp{$label}) {
	    push @media, $label;
	    delete $available{$label};
	}
    }

    # Append available to @media in order of appearance in reverse medialist
    # (i. e. oldest first)
    for (my $i = $#medialist; $i >= 0 && $#media < $runtapes - 1; $i--) {
	($datestamp, $label, $reuse_flag) = split /\s+/, $medialist[$i];
	if ($available{$label}) { push @media, $label; }
    }

    # Append remaining usable labels to @media in order of appearance in
    # reverse medialist (i. e. oldest first)
    for (my $i = $#medialist; $i >= 0 && $#media < $runtapes - 1; $i--) {
	($datestamp, $label, $reuse_flag) = split /\s+/, $medialist[$i];
	if (!exists($available{$label}) && is_usable($label)) {
	    push @media, $label;
	}
    }

    return @media;
}

# Actually burn an intermediate dir to the media with the given label.
# If $label is 0 we'll try to write on non-erasable media in some device.
# Returns false on success, otherwise an error message.
# For gravity changers, the media with the given label must be in the
# "current" slot. After burning, the changer will advance to the next slot.
sub burnDir {
    my ($dir, $label) = @_;
    my $dev;
    my $blank = " blank=fast";

    if ($label) {
        $dev = &loadMedia($label);
    } elsif ($#nonErasable >= 0) {
	my $slot = shift @nonErasable;
	($slot, $dev) = &loadSlot($slot);
	$blank = "";
    } else {
	return "No non-erasable media left!";
    }

    if (!$dev) {
	return "Cannot load media with label $label!";
    }

    if ($dev !~ /^(.*?):/) {
	return "Cannot extract cdrecord device specification from $dev!";
    }
    my $mountDev = $1; # for growisofs
    $dev = $';

    # Determine media type. This is pretty much a hack.
    my $atip = `$CDRECORD -atip dev=$dev`;
    # No disk -> shouldn't happen at this point
    if (!($atip =~ /ATIP start of lead in:/s)) {
	return "Failed to determine media type!?";
    }
    $atip =~ $&.$';
    my $MEDIA = "DVD+R";
    if ($atip =~ /Is not erasable/s) {
        $MEDIA = "CDR";
    } elsif ($atip =~ /Is erasable/s) {
        $MEDIA = "CDRW";
    } else {
	# We don't really care if it's DVD+R or +RW at this point...
	;
    }

    my $command = "$MKISOFS -J -R -pad -quiet $dir | $CDRECORD dev=$dev -data$blank -";
    if ($MEDIA =~ /^DVD\+R/) {
	$command = "growisofs -use-the-force-luke -Z $mountDev -J -R -pad -quiet $dir";
    }

    # Include the full mkisofs/cdrecord/growisofs output in error messages,
    # ignore the (quite verbose:-) output if there were no errors
    my $result = `($command) 2>&1`;
    my $status = $? >> 8;
    
    if (defined($tpchanger) && $changer_gravity) {
	&loadSlot("next");
    }

    return $status ? "'$command' finished with exit status $status\n\nLog:\n$result" : 0;
}

# Send an error message to every mail address in @mailto and exit. This is
# used to report errors that happen while burning the CD-RW
# (i.e. after the "QUIT" command) and can't be reported in a normal
# way.
sub error {
  my ($msg) = @_;
  # Report to STDERR anyway, maybe the driver is still logging.
  print STDERR "taper: ERROR: $msg\n";

  # Send the mail
  open MAIL, "| $SENDMAIL -t" or die "taper: Cannot send mail: $!";
  print MAIL "Subject: AMANDA CDRW-TAPER ERROR\n" or die "taper: Cannot send mail: $!";
  my $username = getpwuid($<);
  print MAIL "From: $username\n";
  foreach my $name (@mailto) {
    print MAIL "To: $name\n" or die "taper: Cannot send mail: $!";
  }
  print MAIL "\n" or die "taper: Cannot send mail: $!";
  print MAIL "$msg\n" or die "taper: Cannot send mail: $!";
  close MAIL or die "taper: Cannot send mail: $!";

  # Send the final acknowledge to the driver
  print "QUITTING";
  exit 1;
}

# Append an entry to the amanda logfile. If the logfile doesn't exist,
# log to STDERR
sub log_add {
  my ($type, $message) = @_;
 
    if (! -r "$logdir/log") {
	print STDERR "$type taper $message\n";
	return;
    }

  # Open and lock the log file
  open LOG, ">>$logdir/log"
    or die "taper: cannot open log file $logdir/log: $!";
  flock LOG, LOCK_EX
    or die "taper: cannot lock the log file $logdir/log: $!";
  seek(LOG, 0, 2);

  # write the entry
  print LOG "$type taper $message\n";
    
  # unlock and close the file
  flock LOG, LOCK_UN
    or die "taper: cannot unlock the log file $logdir/log: $!";
  close LOG or die "taper: cannot close the log file $logdir/log: $!";
}

# Read the amanda label from the given device.
# Returns 0 on failure, the label otherwise.
sub getLabel {
    my $dev = shift;
    my $label = 0;
    my $writeDev = $dev;

    # Strip the "0,0,0" from the device name
    if ($dev =~ /:/) {
	$dev = $`;
	$writeDev = $';
    } else {
	log_add("ERROR", "device specification $dev doesn't include a mount device");
	return 0;
    }

    # Mount the disk. Requires a 'mount' program that allows mounting
    # of $dev on $CDRW_MOUNT_DIR as non-priviledged user by specifying
    # either the device or the mount point (might be Linux specific).
    my $error = `$MOUNT -r $dev 2>&1`;
    $error =~ s/\s+/ /gs;
    if ($error) {
	log_add("ERROR", "could not mount disk: $error");
    } else {
	# read and check the media label
	if (!open(LABEL, "<$CDRW_MOUNT_DIR/AMANDA_LABEL")) {
	    log_add("ERROR", "not an amanda disk");
	} else {
	    $label = <LABEL>;
	    close LABEL;
	    $label =~ s/[\r\n]+//sg;
	    if ($label !~ $labelstr) {
		log_add("ERROR", "label $label doesn't match '$labelstr'");
		$label = 0;
	    }
	}

	# Umount the disk.
	$error = `$UMOUNT $CDRW_MOUNT_DIR 2>&1`;
	$error =~ s/\s+/ /gs;
	if ($error) {
	    log_add("ERROR", "could not umount disk: $error");
	}
    }

    return $label;
}

1;

