/*
 * This is a Roxen Challenger(r) module.
 * Adbanner version 0.11, Copyright (C) 1999 Darryl Fuller.
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Library General Public License 
 * as published by the Free Software Foundation; 
 * either version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU Library General Public License for more details. 
 *
 * You should have received a copy of the GNU Library General Public License 
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

inherit "module";
inherit "roxenlib";
inherit "socket";

#include <module.h>
#include <roxen.h> 
#include <stat.h> 

import Stdio;

#define HITS		"Hits"
#define REFERRALS	"Referrals"
#define STATS_START_TIME	"Stats_date_from"
#define STATS_ROLL_TIME	"Stats_rolled_at"
#define COMPANY_NAME	"Company_Name"
#define LINK		"Link"
#define BAR_MAX_LEN	500.0

#define DATA_EXTENSION		".dta"
#define HISTORY_EXTENSION	".hst"

#define TRACE_ENTER(A,B) do{if(id->misc->trace_enter)id->misc->trace_enter( (A), (B) );}while(0)              
#define TRACE_LEAVE(A) do{if(id->misc->trace_leave)id->misc->trace_leave( (A) );}while(0)                                                                        

#define WEB_SEP		"/"
#ifdef __NT__                                                                   
#define FILE_SEP	"\\"
#else
#define FILE_SEP	"/"
#endif                                                                          


/*
 * Global variables and constants
 */
constant cvs_version = "$Id: adbanner.pike,v 1.23 2000/09/11 21:57:18 darryl Exp $";

constant thread_safe = 1;         
int spinner = 0;
string current_day = "00";
#ifdef THREADS
object entry_mutex = Thread.Mutex();
#endif

void create()                                                      
{        
	defvar("adminpassword", "secret", "Administrator password",
		TYPE_STRING,
		"Password to maintain the adbanner  database online. "
		"Note that this is a very LOW security password, so "
		"don't use your regular password for this one and change often :)");

	defvar("full_search_path", "NONE", "Full search path for data files",
		TYPE_DIR,
		"This is the path to the location where the module will "
		"save the data files (in the real filesystm)");

	defvar("location", "/adbanners/", "Mountpoint", TYPE_LOCATION,
		"The URL-prefix for the ad banners.");

	defvar("hitbarimage", "http://you.yourdomain.ca/images/hitbar.gif", "Hits Bar Image", TYPE_LOCATION,
		"The URL of the image to use for hit bars in "
		"bar-graphs for ad banner statistics.");

	defvar("refbarimage", "http://you.yourdomain.ca/images/refferalbar.gif", "Referral Bar Image", TYPE_LOCATION,
		"The URL of the image to use for referral bars in "
		"bar-graphs for ad banner statistics.");
}


array register_module()
{
	return ({ MODULE_LOCATION | MODULE_PARSER,"Ad banner module", 
		 ("Manages Ad Banners."
		  "<p>Provides the <b>&lt;adbanner&gt</b> tag<br>"
		  "and variants <b>&lt;adbanner subdir=\"foo/bar\"&gt;</b>, <br>"
		  "<b>&lt;adbanner cmd=\"admin\"&gt;</b>, <br> "
		  "<b>&lt;adbanner cmd=\"stats\" company=\"FooBar Ltd.\"&gt;</b>,and "
		  "<b>&lt;adbanner cmd=\"stats\" subdir=\"foo/bar\" company=\"FooBar Ltd.\"&gt;</b>"
 		  "<br>"
 		  "This module is released under GNU Library General Public License"
		 ),
		0,
		1,	// Allow only one copy per server.
		});    
}



/* 
 * put a string in quotes if
 *	1) it contains a space
 *	2) it is not already in quotes
 */
string quote( string in_str )
{
	string out_str;
	/*
	 * does it contain a "space" character
	 */
	if ( search( in_str, " " ) )
	{
		/*
		 * do we begin with a quote?
		 */
		if (in_str[ .. 0 ] != "\"") 	
		{
			out_str = "\"" + in_str;
		}
		else
		{
			out_str = in_str;
		}

		/*
		 * do we end with a quote?
		 */
		if (out_str[ (sizeof(out_str) - 1) .. ] != "\"")
		{
			out_str += "\"";
		}
	}
	else
	{
		/*
		 * don't need to quote
		 */
		out_str = in_str;
	}

	return out_str;
}


/*
 * Given a date in the string format returned by
 * http_date (date format as in rfc1123),
 * return the month string
 */
string month( string date )
{
	string ret="";

	if (date)
	{
		/*
		 * Format of date:
		 Sun, 06 Nov 1994 08:49:37 GMT
		 * Month is in character positions 9-12
		 */
		 ret = date[8..10];
	}

	return ret;
}

/*
 * Given a date in the string format returned by
 * http_date (date format as in rfc1123),
 * return the day string
 */
string day( string date )
{
	string ret="";

	if (date)
	{
		/*
		 * Format of date:
		 Sun, 06 Nov 1994 08:49:37 GMT
		 * Month is in character positions 8-10
		 * Day is in character positions 5-6
		 */
		 ret = date[5..6];
	}

	return ret;
}

/* 
 * Given a string that begins and ends in quotes,
 * return it without quotes.
 * otherwise return the string unmodified
 */
string unquote( string in_str )
{
	string out_str;

	if ( (in_str) &&
	     (in_str[ .. 0 ] == "\"") &&
	     (in_str[ (sizeof(in_str) - 1) .. ] == "\"") )

	{
		out_str = in_str[1 .. (sizeof(in_str) - 2) ];
	}
	else
	{
		/*
		 * don't need to quote
		 */
		out_str = in_str;
	}

	return out_str;
}

/*
 * Given a line from a stats file, parse out the number of hits
 * and return it.
 */
int parse_hits( string stats_line )
{
	int hits=0;
	string rest;

	if (stats_line)
	{
		int pos = search(stats_line, "Hits: ");
		if (pos > 0)
		{
			sscanf(stats_line[pos..],"Hits: %d %s", hits, rest);			
		}
	}

	return hits;
}

/*
 * Given a line from a stats file, parse out the number of referrals
 * and return it.
 */
int parse_referrals( string stats_line )
{
	int referrals=0;
	string rest;

	if (stats_line)
	{
		int pos = search(stats_line, "Referrals: ");
		if (pos > 0)
		{
			sscanf(stats_line[pos..],"Referrals: %d %s", referrals, rest);			
		}
	}

	return referrals;
}

/*
 * save_entry
 */
void save_entry( string file_name, mapping recs )
{
 	string name, value;
	object fd;
	fd = Stdio.File();

	if(fd->open(file_name,"wc"))
	{
		foreach (indices(recs), name)
		{
			value = recs[ name ];

			fd->write( name + ": " + value + "\n" );
		}
		/** padding off the end */
		fd->write( "                                             \n" 
		           "                                             \n"
		           "                                             \n"
		           "                                             \n"
		           "                                             \n"
		           "                                             \n"
		           "                                             \n"
		           "                                             \n" );

		fd->close();
	}
}

/*
 * load_entry 
 */
mapping load_entry( string file_name )
{
	mapping recs = ([]);
	object fd;
	string name, value;
	string file_contents, line;

	fd = Stdio.File();

	if (fd->open(file_name, "r"))
	{
		file_contents = fd->read();
		fd->close();

		foreach(file_contents/"\n",line)
		{
			if (sscanf(line, "%s: %s", name, value))
			{
				recs[ name ] = value;
			}
		}
	}

	return recs;
}

int is_image_file( string filename )
{
        int     retval = 0;

	string ext = extension(filename);

	if (ext == "gif" )
		retval = 1;
	else if (ext == "GIF")
		retval = 1;
	else if (ext == "jpg")
		retval = 1;
	else if (ext == "JPG")
		retval = 1;
	else if (ext == "jpeg")
		retval = 1;
	else if (ext == "JPEG")
		retval = 1;

        return (retval);

}

/*
 * get_entry
 */
mapping get_entry(string file_name, int increment_hits, int increment_referrals)
{
	mapping recs = ([]);
	int	temp = 0;
	string	value = "0";


	/*
	 * Lock mutex
	 */
#ifdef THREADS
	object key = entry_mutex?entry_mutex->lock():0;
#endif

	/*
	 * Read in records
	 */
	recs = load_entry( file_name );

	/*
	 * increment hits and/or referrals if specified
	 */
	if (increment_hits)
	{
		value = recs[ HITS ];
		temp = (int)value;
		temp = temp + 1;
		value = "" + temp;
		recs[ HITS ] = value;

		/*
		 * If we haven't recorded a date for the start of 
		 * recorded statistics, do so now.
		 */
		if (! recs[ STATS_START_TIME ] )
		{
			recs[ STATS_START_TIME ] = http_date( time() );
		}
		if (! recs[ STATS_ROLL_TIME ] )
		{
			recs[ STATS_ROLL_TIME ] = http_date( time() );
		}
	}

	if (increment_referrals)
	{
		value = recs[ REFERRALS ];
		temp = (int)value;
		temp = temp + 1;
		value = "" + temp;
		recs[ REFERRALS ] = value;

		/*
		 * If we haven't recorded a date for the start of 
		 * recorded statistics, do so now.
		 */
		if (! recs[ STATS_START_TIME ] )
		{
			recs[ STATS_START_TIME ] = http_date( time() );
		}
		if (! recs[ STATS_ROLL_TIME ] )
		{
			recs[ STATS_ROLL_TIME ] = http_date( time() );
		}
	}

	/*
	 * write out records again if needed
	 */
	if ((increment_hits) || (increment_referrals))
	{
		save_entry( file_name, recs );
	}

	/*
	 * Unlock mutex
	 */
#ifdef THREADS
	destruct( key );
#endif

	/*
	 * return the records
	 */
	return( recs );
}

/*
 * append_stats_history
 */
void append_stats_history(string file_name, mapping entry)
{
	int	temp = 0;
	string	value = "0";


	/*
	 * Lock mutex
	 */
#ifdef THREADS
	object key = entry_mutex?entry_mutex->lock():0;
#endif

 	string name, value;
	object fd;
	fd = Stdio.File();

	if(fd->open(file_name,"wac"))
	{
		fd->write(
			  http_date( time() ) +
			  " Hits: " +
			  entry[ HITS ] + 
			  " Referrals: " +
			  entry[ REFERRALS ] + 
			  " (since: " +
			  entry[ STATS_START_TIME ] + 
			  " )" +
			  "\n"
			  );
		fd->close();
	}
	/*
	 * Unlock mutex
	 */
#ifdef THREADS
	destruct( key );
#endif

}

mixed ad_redirect( mapping variables )
{
	mixed res = 0;
	string ad = variables[ "ad" ];
	string subdir = variables[ "subdir" ];
	string fulldir = query("full_search_path");
	string file_subdir = subdir;
	if ( ( WEB_SEP != FILE_SEP ) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	if (file_subdir)
	{
		fulldir += FILE_SEP + file_subdir;
	}

	/*
	 * get data record for file (increment referrals)
	 */
	string ext = "." + extension( ad );
	string datafile = (ad - ext) + DATA_EXTENSION;
	mapping entry = get_entry( fulldir + FILE_SEP + datafile, 0, 1);

	string url = entry[ LINK ];

	if (url)
	{
		/*
		 * Spit out a redirect to the correct URL
		 */
        	res =  http_redirect(url, 0);     
	}

	return( res );
}

string render_history_file( string fulldir, string ad )
{
	string res="";

	string ext = "." + extension( ad );

	object fd;
	// fd = Stdio.File();
	fd = Stdio.FILE();
	string file_contents;
	string line;

	string historyfile = ((fulldir + FILE_SEP + ad) - ext) + HISTORY_EXTENSION;

	if (fd->open(historyfile, "r"))
	{
		res+="<p> daily statistics";

		/*
		file_contents = fd->read();
		res+= file_contents;
		*/


		int pos = 0;
		float max = 0;
		mapping hits = ([]); 
		mapping referrals = ([]); 
		mapping lines = ([]);
		
		lines[pos] = fd->gets();
		while (lines[pos])
		{
			hits[pos] = parse_hits(lines[pos]);
			referrals[pos] = parse_referrals(lines[pos]);

			if ((pos == 0) || 
			    (hits[pos] < hits[pos - 1]) || 
			    (referrals[pos] < referrals[pos - 1]))
			{
				if (hits[pos] > max)
				{
					max = hits[pos];
				}
				if (referrals[pos] > max)
				{
					max = referrals[pos];
				}
				if (referrals[pos] > max)
				{
					max = referrals[pos];
				}
			}
			else if (hits[pos] - hits[pos - 1] > max)
			{
				max = hits[pos] - hits[pos - 1];
			}
			else if (referrals[pos] - referrals[pos - 1] > max)
			{
				max = referrals[pos] - referrals[pos - 1];
			}

			++ pos;
			lines[pos] = fd->gets();
		}

		fd->close();

		pos = 0;
		string mon = "foo";

		res += "<table>\n";
		while(lines[pos])
		{
			float daily_hits = 0;
			float daily_referrals = 0;
			float width = 0;
			float scale = BAR_MAX_LEN / max;

			if (mon != month(lines[pos]))
			{
				mon = month(lines[pos]);
				res += "<tr><td><b>" + mon + "</b></td></tr>\n";
			}
			
			if ((pos == 0) || 
			    (hits[pos] < hits[pos - 1]) || 
			    (referrals[pos] < referrals[pos - 1]))
			{
				daily_hits = hits[pos];
				daily_referrals = referrals[pos];
			}
			else
			{
				daily_hits = hits[pos] - hits[pos - 1];
				daily_referrals = referrals[pos] - referrals[pos - 1];
			}

			res+="<tr>\n";
			res+="<td>" + day(lines[pos]) + "</td>";
			res += "<td> Hits </td>";
			res += "<td>" + daily_hits + "</td>";

			width = (scale * daily_hits) + 1.0; 

			res += "<td>";
			res += "<IMG SRC=\"" + query("hitbarimage") + "\" ";
			res += "width=\"" + width  + "\" ";
			res += "height=\"15\">";
			res += "</td>";

			res+="</tr>\n";
			res+="<tr>\n";
			res+="<td></td><td> Referrals </td>\n";

			width = (scale * daily_referrals) + 1.0; 

			res+= "<td>" + daily_referrals + "</td>";
			res += "<td>";
			res += "<IMG SRC=\"" + query("refbarimage") + "\" ";
			res += "width=\"" + width + "\" ";
			res += "height=\"15\">";
			res += "</td>";

			res+="</tr>\n";

			++ pos;
		}
		res += "</table>\n";
	}

	return ( res );
}

string render_ad( string subdir, string ad, int increment_hits, int show_stats, string key_match, string value_match)
{
	string res="";
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;

	/*
	 * get data record for file (increment hits if called for)
	 */
	string ext = "." + extension( ad );
	string datafile = (ad - ext) + DATA_EXTENSION;
	mapping entry = get_entry( fulldir + FILE_SEP + datafile, increment_hits, 0);

	if ((key_match == "") || (key_match == 0) ||
	    (value_match == "") || (value_match == 0) ||
	    (entry[ key_match ] == value_match ))
	{
		/*
		 * Formulate response
		 */
		res+="<!-- start adbanner and data -->\n";

		string path = simplify_path( query("location") + WEB_SEP + subdir + WEB_SEP + ad );


		res+="<A HREF=\"" +
			simplify_path(query("location") + "/redirect") +
			"?action=reference" + 
			"&id=" + spinner +
			"&ad=" + ad;
			
		if (subdir != "")
		{
			res +="&subdir=" + subdir;
		}
		res+= "\" target=\"NEW\">";

		res+="<img border=0";
		if (entry[ COMPANY_NAME ])
		{
			res+=" alt=" + entry[ COMPANY_NAME ];
		}
		res+=" src=\"" + path + "\">";

		res+="</A>\n";

		if (! show_stats)
		{
			/*
			 * To prevent page caching in the browser from interfering with
			 * web banner rotation too much (as in, they always see the same
			 * banner on a given page), put an expiry on the page... ten
			 * minutes from now.  
			 * We only do this if not "showing stats" because if we
			 * were, we'd probably be rendering on the admin page.
			 */
			int t=time() + 60 * 10;
			res += "<META HTTP-EQUIV=\"Expires\" CONTENT=\"" + http_date(t) + "\">\n";

			res+="<!-- \n";
		}
		else
			res+="<pre>";

		res+= COMPANY_NAME + " " + entry[ COMPANY_NAME ] + "\n";

		res+= HITS + " " + entry[ HITS ]+ "\n";

		res+= REFERRALS + " " + entry[ REFERRALS ] + "\n";

		res+= STATS_START_TIME + " " + entry[ STATS_START_TIME ] + "\n";


		if (! show_stats)
			res+= "-->\n";
		else
			res+="</pre>";

		/*
		 * show old stats
		 */
		if (show_stats)
		{
			res += render_history_file(fulldir, ad);
		}

		res+="<!-- end adbanner and data -->\n";
	}

	return ( res );
}

/*
 * Render form for adding a new ad 
 */
string newad_form( string subdir, string url )
{
	string res = "";

	res += "<h2> create a new adbanner in ";

	if (subdir == "")
	{
		res += "the root directory";
	}
	else
	{
		res += subdir;
	}

	res += "</h2>\n";

	/*
	 * the new ad form.
	 */
	res += "<form ENCTYPE=\"multipart/form-data\" ";
	res += "action=\"" + url + "\" method=POST>\n";
	res += "<input type=hidden name=id value=" + spinner + " >\n";
	res += "<input type=hidden name=subdir value=" + subdir + " >\n";
	res += "<input type=hidden name=action value=newad>\n";
	res += "Admin Password: ";
	res += "<input type=password name=password ><br>\n";
	res += "Company Name: ";
	res += "<input name=" + COMPANY_NAME + " size=40><br>\n";
	res += "Link (url): ";
	res += "<input name=" + LINK + " size=40><br>\n";
	res += "Adbanner (gif or jpeg): ";
	res += "<input type=file name=adbanner><br>\n";
	res += "<input type=submit value=\"Create new adbanner!\" ><br>\n";
	res += "</form>\n";


	return res;
}

/*
 * Render form for adding a subdirectory 
 */
string mkdir_form( string subdir, string url )
{
	string res = "";

	res += "<h2> Add a new subdirectory to ";

	if (subdir == "")
	{
		res += "the root directory";
	}
	else
	{
		res += subdir;
	}

	res += "</h2>\n";

	/*
	 * the new subdirectory form.
	 */
	res += "<form action=\"" + url + "\" method=POST>\n";
	res += "<input type=hidden name=id value=" + spinner + " >\n";
	res += "<input type=hidden name=subdir value=" + subdir + " >\n";
	res += "<input type=hidden name=action value=mkdir>\n";
	res += "Admin Password: ";
	res += "<input type=password name=password ><br>\n";
	res += "New Subdirectory: ";
	res += "<input name=newsubdir ><br>\n";
	res += "<input type=submit value=\"Create new Subdirectory!\" ><br>\n";
	res += "</form>\n";


	return res;
}

string change_ad_form( string subdir, string ad, string url )
{
	string res = "";
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	res += "<h3> Change data for " + ad + "</h3>\n";

	/* 
	 * Read data for the ad
	 */
	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;

	/*
	 * get data record for file (do not increment hits)
	 */
	string ext = "." + extension( ad );
	string datafile = (ad - ext) + DATA_EXTENSION;
	mapping entry = get_entry( fulldir + FILE_SEP + datafile, 0, 0);

	/*
	 * change ad form.
	 */
	res += "<form action=\"" + url + "\" method=POST>\n";
	res += "<input type=hidden name=action value=change_ad>\n";
	res += "<input type=hidden name=id value=" + spinner + " >\n";
	res += "<input type=hidden name=subdir value=" + subdir + " >\n";
	res += "<input type=hidden name=adbanner value=" + ad + " >\n";
	res += "Admin Password: ";
	res += "<input type=password name=password ><br>\n";
	res += "Company Name: ";
	res += "<input name=" + COMPANY_NAME + " value= \"" + unquote(entry[ COMPANY_NAME ])+ "\" size=40><br>\n";
	res += "Link (url): ";
	res += "<input name=" + LINK + " value=\"" + entry[ LINK ] + "\" size=40><br>\n";
	res += "<input type=submit value=\"Change adbanner data!\" ><br>\n";
	res += "</form>\n";


	return res;
}

string delete_ad_form( string subdir, string ad, string url )
{
	string res = "";

	res += "<h3> Delete ad " + ad + "</h3>\n";

	/*
	 * the delete ad form.
	 */
	res += "<form action=\"" + url + "\" method=POST>\n";
	res += "<input type=hidden name=id value=" + spinner + " >\n";
	res += "<input type=hidden name=subdir value=" + subdir + " >\n";
	res += "<input type=hidden name=action value=delete_ad>\n";
	res += "<input type=hidden name=adbanner value=" + ad + " >\n";
	res += "Admin Password: ";
	res += "<input type=password name=password ><br>\n";
	res += "<input type=submit value=\"Delete ad!\" ><br>\n";
	res += "</form>\n";

	return res;
}

string reset_adstats_form( string subdir, string ad, string url )
{
	string res = "";

	res += "<h3> Reset Stats for " + ad + "</h3>\n";

	/*
	 * the reset ad statistics form.
	 */
	res += "<form action=\"" + url + "\" method=POST>\n";
	res += "<input type=hidden name=id value=" + spinner + " >\n";
	res += "<input type=hidden name=subdir value=" + subdir + " >\n";
	res += "<input type=hidden name=action value=reset_adstats>\n";
	res += "<input type=hidden name=adbanner value=" + ad + " >\n";
	res += "Admin Password: ";
	res += "<input type=password name=password ><br>\n";
	res += "<input type=submit value=\"Reset Stats!\" ><br>\n";
	res += "</form>\n";

	return res;
}

string admin_form( string subdir, string url )
{
	string res = "";
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	/*
	 * Do Heading
	 */
	if ( subdir == "" )
	{
		/*
		 * Expire this page so we don't cache admin page.
		 */
		res += "<META HTTP-EQUIV=\"Expires\" CONTENT=\"Tue, 04 Dec 1999 21:29:02 GMT\">\n";

		/*
		 * Heading for directory
		 */
		res += "<hr size=10>\n";
		res += "<h1>Adbanners in / (root directory)</h1>\n";
	}
	else
	{
		/*
		 * Heading for directory
		 */
		res += "<hr size=10>\n";
		res += "<h1>Adbanners in " + subdir + " sub-directory</h1>";
	}

	/*
	 * Generate listing of directory
	 */
	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;
	array listing = get_dir( fulldir );

	if (listing)
	{

		/*
		 * Do ads in this directory
		 */

		/*
		 * filter out entries in listings that are not image files.
		 * (ad banners are image files).
		 */
		array filtered = Array.filter( listing, is_image_file );
		int size = sizeof(filtered);
		for (int i = 0; i < size; i=i+1)
		{
			/*
			 * do forms for each ad
			 */
			res += render_ad( subdir, filtered[ i ], 0, 1, "", "");

			res += delete_ad_form( subdir, filtered[ i ], url );

			res += change_ad_form( subdir, filtered[ i ], url );

			res += reset_adstats_form( subdir, filtered[ i ], url );

			res += "<hr>\n";
		}

		/*
		 * Do form for adding a new ad
		 */
		res += newad_form( subdir, url );

		/*
		 * Do form for adding a subdirectory
		 */
		res += mkdir_form( subdir, url );

		/*
		 * Do subdirectories
		 */
		size = sizeof( listing );
		for (int i = 0; i < size; i=i+1)
		{
			array test = get_dir( fulldir + FILE_SEP + listing[i] );	
			if (test)
			{
				res += admin_form( subdir + WEB_SEP + listing[i], url );
			}
		}
	}

	return res;
}

string render_stats( string subdir, string company, string url, int recurse )
{
	string res = "";
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	/*
	 * check null case
	 */
   	if (subdir == 0)
	{
		subdir = "";
	}

	/*
	 * check null case
	 */
   	if (company == 0)
	{
		company = "";
	}

	/*
	 * Do Heading
	 */
	if ( subdir == "" )
	{
		/*
		 * Expire this page so we don't cache admin page.
		 */
		res += "<META HTTP-EQUIV=\"Expires\" CONTENT=\"Tue, 04 Dec 1999 21:29:02 GMT\">\n";

		/*
		 * Heading for directory
		 */
		res += "<hr size=10>\n";
		res += "<h1>" + company+ " Adbanners in / (root directory)</h1>\n";
	}
	else
	{
		/*
		 * Heading for directory
		 */
		res += "<hr size=10>\n";
		res += "<h1>" + company + " Adbanners in " + subdir + " sub-directory</h1>";
	}

	/*
	 * Generate listing of directory
	 */
	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;
	array listing = get_dir( fulldir );

	if (listing)
	{

		/*
		 * Do ads in this directory
		 */

		/*
		 * filter out entries in listings that are not image files.
		 * (ad banners are image files).
		 */
		array filtered = Array.filter( listing, is_image_file );
		int size = sizeof(filtered);
		for (int i = 0; i < size; i=i+1)
		{
			/*
			 * render each ad
			 */
			res += render_ad( subdir, filtered[ i ], 0, 1, COMPANY_NAME, company);

			res += "<hr>\n";
		}

		if (recurse != 0)
		{
			/*
			 * Do subdirectories
			 */
			size = sizeof( listing );
			for (int i = 0; i < size; i=i+1)
			{
				array test = get_dir( fulldir + FILE_SEP + listing[i] );	
				if (test)
				{
					res += render_stats( subdir + WEB_SEP + listing[i], company, url, 1 );
				}
			}
		}
	}

	return res;
}

void roll_stats( string subdir )
{
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	/*
	 * Generate listing of directory
	 */
	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;
	array listing = get_dir( fulldir );

	if (listing)
	{

		/*
		 * check ads in this directory for roll
		 */

		/*
		 * filter out entries in listings that are not image files.
		 * (ad banners are image files).
		 */
		array filtered = Array.filter( listing, is_image_file );
		int size = sizeof(filtered);
		for (int i = 0; i < size; i=i+1)
		{
			/*
			 * get data record for file
			 */
			string ext = "." + extension( filtered[ i ] );
			string datafile = (filtered[ i ] - ext) + DATA_EXTENSION;

			/*
			 * Lock mutex
			 */
#ifdef THREADS
			object key = entry_mutex?entry_mutex->lock():0;
#endif
			mapping entry = load_entry( fulldir + FILE_SEP + datafile);


			/*
			 * check the day against the current day
			 */
			if (current_day != day(entry[ STATS_ROLL_TIME ]))
			{
				/*
				 * Write old stats to history file
				 */
				string ext = "." + extension( datafile );
				string historyfile = fulldir + FILE_SEP + (datafile - ext) + HISTORY_EXTENSION;
				append_stats_history(historyfile, entry);

				/*
				 * reset the ad roll time
				 */
				entry[ STATS_ROLL_TIME ] = http_date( time() );


				/*
				 * Save the new data
				 */
				save_entry( fulldir + FILE_SEP + datafile, entry );

			}

			/*
			 * Unlock mutex
			 */
#ifdef THREADS
			destruct( key );
#endif
		}

		/*
		 * Do subdirectories
		 */
		size = sizeof( listing );
		for (int i = 0; i < size; i=i+1)
		{
			array test = get_dir( fulldir + FILE_SEP + listing[i] );	
			if (test)
			{
				roll_stats( subdir + FILE_SEP + listing[i] );
			}
		}
	}

}

string adbanner_subdir( string subdir, string url )
{
	string res = "";

	if (! subdir)
	{
		subdir = "";
	}
	string file_subdir = subdir;
	if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
	{
		file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
	}

	/*
	 * Generate listing of directory
	 */
	string fulldir = query("full_search_path") + FILE_SEP + file_subdir;
	array listing = get_dir( fulldir );

	if (listing)
	{

		/*
		 * filter out entries in listings that are not image files.
		 * (ad banners are image files).
		 */
		array filtered = Array.filter( listing, is_image_file );

		int size = sizeof( filtered );

		if ( size > 0 ) 
		{
			/*
			 * use spinner + randome to pick one file in listing
			 */
			int pos = (spinner + random(size)) % size;

			/*
			 * Render the ad
			 */
			res += render_ad( subdir, filtered[ pos ], 1, 0, "", "");
		}

	}

	return res;
}

/*
 * Handle the results of the change  ad form.
 * Checks admin password, then tries to change
 * the specified adbanner's data 
 */
string change_ad_results( mapping variables )
{
	string res = "";

	/*
	 * check password
	 */
	if ( variables[ "password" ] != query( "adminpassword" ) )
	{
		res += "<h1>ERROR: Incorrect admin password entered</h1>\n";
	}
	else
	{

		/*
		 * Create path up to ad
		 */
		string fulldir = query("full_search_path");

		if (variables[ "subdir" ])
		{
			string file_subdir = variables[ "subdir" ];
			if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
			{
				file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
			}
			fulldir += FILE_SEP + file_subdir;
		}

		/*
		 * ad banner gif full path and filename
		 */
		string file_name = fulldir + FILE_SEP + variables [ "adbanner" ];

		/*
		 * ad banner data file full path and filename
		 */
		string ext = "." + extension( file_name );
		string datafile = (file_name - ext) + DATA_EXTENSION;

		/*
		 * Read the old ad data
		 */
		res += "<h1>Changeing ad data in " + datafile + "</h1> \n";
		mapping entry = get_entry(datafile, 0, 0);

		/*
		 * change the ad data
		 */
		if ( ( variables[ LINK ] ) &&
		     ( sizeof(variables[ LINK ]) ) )
		{
			entry[ LINK ] = variables[ LINK ];
		}
		if ( (variables[ COMPANY_NAME ] ) &&
		     ( sizeof(variables[ COMPANY_NAME ]) ) )
		{
			entry[ COMPANY_NAME ] = quote(variables[ COMPANY_NAME ]);
		}

		/*
		 * Save the new data
		 */
		save_entry( datafile, entry );
	}

	return res;
}


/*
 * Handle the results of the delete ad form.
 * Checks admin password, then tries to set
 * the ad statistics to zero
 */
string reset_adstats_results( mapping variables )
{
	string res = "";

	/*
	 * check password
	 */
	if ( variables[ "password" ] != query( "adminpassword" ) )
	{
		res += "<h1>ERROR: Incorrect admin password entered</h1>\n";
	}
	else
	{

		/*
		 * Create path up to ad
		 */
		string fulldir = query("full_search_path");

		if (variables[ "subdir" ])
		{
			string file_subdir = variables[ "subdir" ];
			if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
			{
				file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
			}
			fulldir += FILE_SEP + file_subdir;
		}

		/*
		 * ad banner gif full path and filename
		 */
		string file_name = fulldir + FILE_SEP + variables [ "adbanner" ];

		/*
		 * ad banner data file full path and filename
		 */
		string ext = "." + extension( file_name );
		string datafile = (file_name - ext) + DATA_EXTENSION;

		/*
		 * Read the old ad data
		 */
		res += "<h1>Resetting stats for  " + file_name + "</h1> \n";
		mapping entry = get_entry(datafile, 0, 0);

		/*
		 * Write old stats to history file
		 */
		string ext = "." + extension( file_name );
		string historyfile = (file_name - ext) + HISTORY_EXTENSION;
		append_stats_history(historyfile, entry);

		/*
		 * reset the ad statistics
		 */
		m_delete( entry, HITS );
		m_delete( entry, REFERRALS );
		entry[ STATS_START_TIME ] = http_date( time() );

		/*
		 * Save the new data
		 */
		save_entry( datafile, entry );
	}

	return res;
}


/*
 * Handle the results of the delete ad form.
 * Checks admin password, then tries to delete
 * the specified adbanner 
 */
string delete_ad_results( mapping variables )
{
	string res = "";

	/*
	 * check password
	 */
	if ( variables[ "password" ] != query( "adminpassword" ) )
	{
		res += "<h1>ERROR: Incorrect admin password entered</h1>\n";
	}
	else
	{

		/*
		 * Create path up to ad
		 */
		string fulldir = query("full_search_path");

		if (variables[ "subdir" ])
		{
			string file_subdir = variables[ "subdir" ];
			if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
			{
				file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
			}
			fulldir += FILE_SEP + file_subdir;
		}

		/*
		 * ad banner gif full path and filename
		 */
		string file_name = fulldir + FILE_SEP + variables [ "adbanner" ];

		/*
		 * ad banner data file full path and filename
		 */
		string ext = "." + extension( file_name );
		string datafile = (file_name - ext) + DATA_EXTENSION;

		/*
		 * delete the files
		 */
		if (! rm(file_name))
		{
			res += "<h1>ERROR: could not remove";
			res += file_name + "</h1>\n";
		}
		else
		{
			rm( datafile );
			res += "<h1>REMOVED ";
			res += file_name + "</h1>\n";
		}
	}

	return res;
}

/*
 * Handle the results of the mkdir form.
 * Checks admin password, then tries to create
 * the requested subdirectory in the 
 * adbanner "filesystem"
 */
string mkdir_results( mapping variables )
{
	string res = "";

	/*
	 * check password
	 */
	if ( variables[ "password" ] != query( "adminpassword" ) )
	{
		res += "<h1>ERROR: Incorrect admin password entered</h1>\n";
	}
	else
	{

		/*
		 * Create path up to new directory
		 */
		string fulldir = query("full_search_path");

		if (variables[ "subdir" ])
		{
			string file_subdir = variables[ "subdir" ];
			if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
			{
				file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
			}
			fulldir += FILE_SEP + file_subdir;
		}

		/*
		 * ensure that a new subdirectory was specified
		 */
		if ( (! variables[ "newsubdir" ]) || 
		     ( variables[ "newsubdir" ] == "" ) )
		{
			res += "<h1>ERROR: No new subdirectory specified</h1>\n";
		}
		else
		{
			/*
			 * try to create the directory
			 */
			string newdir = simplify_path(fulldir + FILE_SEP + variables[ "newsubdir" ]);

			if (mkdir( newdir ))
			{
				res += "<h1>Created " + 
					newdir +
					"</h1>";
			}
			else
			{
			
				res += "<h1>ERROR: Failed to create " + 
					newdir +
					"</h1>";
			}
		}

	}


	return( res );
}
/*
 * Handle the results of the newad form.
 * Checks admin password, then tries to create
 * the new adbanner.
 */
string newad_results( mapping variables )
{
	string res = "";
	object fd;
	fd = Stdio.File();

	/*
	 * check password
	 */
	if ( variables[ "password" ] != query( "adminpassword" ) )
	{
		res += "<h1>ERROR: Incorrect admin password entered</h1>\n";
	}
	else
	{

		/*
		 * Check that required variables are there...
		 */
		if ( (! variables [ LINK ] ) ||
		     (! sizeof(variables [ LINK ]))  ||
		     (! variables [ "adbanner" ]) ||
		     (! variables [ "adbanner.filename"] ) ) 
		{
			res += "<h1>ERROR: Missing required field</h1>\n";
		}
		else if (! is_image_file( variables [ "adbanner.filename"] ) )
		{
			res += "<h1>ERROR: Adbanner is not an image file</h1>\n";
		}
		else
		{
			/*
			 * create entry for mapping
			 */
			mapping entry = ([]);

			entry [ LINK ] = variables[ LINK ];
			if ((variables[ COMPANY_NAME ]) &&
			    (sizeof(variables[ COMPANY_NAME ]) > 0))
			{
				entry [ COMPANY_NAME ] = quote(variables[ COMPANY_NAME ]);
			}

			/*
			 * write the ad file
			 */
			string subdir = variables[ "subdir" ];
			string file_subdir = subdir;
			if ( (WEB_SEP != FILE_SEP) && (file_subdir) )
			{
				file_subdir = replace( file_subdir, WEB_SEP, FILE_SEP );
			}
			string fulldir = query("full_search_path");

			if (file_subdir)
			{
				fulldir += FILE_SEP + file_subdir;
			}

			string file_name = fulldir + FILE_SEP + variables [ "adbanner.filename"];

			if(! fd->open(file_name,"wct"))
			{
				res += "<h1>ERROR: Could not open file: ";
				res += file_name;
				res += "</h1>\n";
				
			}
			else
			{

				/*
				 * write the imag file
				 */
				fd->write( variables[ "adbanner" ] );
				fd->close();

				res += "<h1> Created " + file_name + "</h1>\n";

				/*
				 * write the entry data.
				 */
				string ext = "." + extension( file_name );
				string datafile = (file_name - ext) + DATA_EXTENSION;

				save_entry( datafile, entry );

				res += "<p> datafile: " + datafile + "\n";
			}

		}

	}


	return( res );
}

string tag_adbanner(string tag_name, mapping arguments, object req_id, object file, mapping defines) 
{
	string res="";

	// Check to see if we need to try rolling the stats
	if (day(http_date(time())) != current_day)
	{
		current_day = day(http_date(time()));
		roll_stats("");
	}

	/*
	 * tick over the spinner
	 */
	++ spinner;

	if (req_id[ "variables" ])
	{
		/*
		 * We have some sort of query coming in
		 */
		switch( req_id["variables"]["action"] )
		{
			case "mkdir":
				/*
				 * Check password and make directory
				 */
				res+= mkdir_results( req_id["variables"] );
				break;
			case "newad":
				/*
				 * Check tha password and make new adbanner
				 */
				res+= newad_results( req_id["variables"] );
				break;
			case "delete_ad":
				/*
				 * Check tha password and delete the ad
				 */
				res+= delete_ad_results( req_id["variables"] );
				break;
			case "change_ad":
				/*
				 * Check tha password and change the ad
				 */
				res+= change_ad_results( req_id["variables"] );
				break;
			case "reset_adstats":
				/*
				 * Check tha password and set ad stats to reset
				 */
				res+= reset_adstats_results( req_id["variables"] );
				break;
			default:
				/* 
				 * Do nothing, will fall through with
				 * res = "", and be handled by
				 * normal adbanner stuff
				 */
				break;
		}

	}

	/*
	 * if some sort of response has not been already formulated, just
	 * spit out the ad banner.
	 */
	if (res == "")
	{
		if ( arguments->cmd == "admin" )
		{
			/*
			 * output the admin form for adbanner
			 */
			res+= admin_form( "", req_id["not_query"] );
		}
		else if ( arguments->cmd == "stats" )
		{
			if ((arguments->subdir == 0) || (arguments->subdir == ""))
			{
				/*
				 * output stats for adbanner in all directories
				 */
				res+= render_stats( "", arguments->company, req_id["not_query"], 1 );
			}
			else
			{
				/*
				 * output stats for adbanner in a single subdirectory
				 */
				res+= render_stats( arguments->subdir, arguments->company, req_id["not_query"], 0 );
			}

		}
		else
		{
			/*
			 * Handle an ad banner + subdir
			 */
			res+= adbanner_subdir( arguments->subdir, req_id["not_query"] );
		}
	}

	/* 
	*********************** DEBUG stuff ***************************

	foreach(indices(req_id), string idd)
	      res += "<b>"+idd+"</b> : "+sprintf("%O", req_id[idd])+"<hr>";

	*********************** DEBUG stuff ***************************
	*/


	return res;
}

/*
 * Callback for parsing tags
 */
mapping query_tag_callers() 
{ 
	return ([ "adbanner":tag_adbanner, ]); 
}

/*
 * Callback for querying location of virtual filesystem.
 * Returns which position in the virtual file system the module should have.
 */
string query_location()
{
	return query( "location" );
}

/*
 * Returns an array with filenames if there is a directory matching dir_name.
 * Otherwise it returns zero.  Since directory listings are not allowed,
 * return zero always.
 */
array find_dir(string dir_name, object request_id)
{
	return 0;
}

/*
 * Returns the result or zero if there is a file matching file_name.
 */
mixed stat_file( mixed f, mixed id )                                            
{
  array fs = 0;

  if (is_image_file( f ))
  {
	string filename = query("full_search_path") + FILE_SEP + f;

  	fs = file_stat(filename);  // No security currently in this function 
  }

  return( fs );
}

/*
 * Return what the file file_name really is called, e.g. if the module test
 * looks for its files in /usr/www/foo/, and the file bar is requested, it
 * should return /usr/www/foo/bar.
 */
string real_file( mixed f, mixed id )                                           
{                                                                               
  // This filesystem might be inherited by other filesystem, therefore 'this'
  if(this->stat_file( f, id ))                                                  
    return ( simplify_path(query("full_search_path") + FILE_SEP + f) );     
}  

/*
 * Returns an open file object of the class /precompiled/file if there is 
 * a file matching file_name that this module handles, or a response 
 * mapping (usually generated with the http_* functions)
 *
 * the file "redirect" is a special file that does an adbanner redirect.
 */
mixed find_file( string f, object id )                                          
{                                                                               
  TRACE_ENTER("find_file(\""+f+"\")", 0);                                       
                                                                                
  object o;                                                                     
  int size = 0; 
  string tmp;                                                                   
  string oldf = f; 
  int redirect = 0;

  // Is this a redirect special file?
  if ((f == "redirect") || (f == "/redirect"))
  {
  	redirect = 1;
  }
  else
  {
	  f = query("full_search_path") + FILE_SEP + f;         
#ifdef __NT__                                                                   
	  if(f[-1]=='/') f = f[..strlen(f)-2];        
#endif                                                                          
	  size = Stdio.file_size( f );     
  }

  switch(id->method)                                                            
  {                                                                             
  case "GET":                                                                   
  case "HEAD":                                                                  
  case "POST":                                                                  
                                                                                
    switch(-size)                                                               
    {                                                                           
    case 1:                                                                     
    case 3:                                                                     
    case 4:                                                                     
      TRACE_LEAVE("No file");                                                   
      return 0; // Is no-file
                                                                                
    case 2:                                                                     
      TRACE_LEAVE("Is directory");                                              
      return -1; // Is dir 
                                  
    default:                                                                    
      if (redirect)
      {
      	return ad_redirect( id->variables );
      }

      if(f[ -1 ] == '/') // Trying to access file with '/' appended
      {                                                                         
        // Do not try redirect on top level directory 
        if(sizeof(id->not_query) < 2)                                           
          return 0;                                                             
        //redirects++;  
        TRACE_LEAVE("Redirecting to \"" +                                       
                    id->not_query[..sizeof(id->not_query)-2] +                  
                    "\"");                                                      
        return http_redirect(id->not_query[..sizeof(id->not_query)-2], id);     
      }                                                                         
                                                                                
      
      // if(!id->misc->internal_get && QUERY(.files)
      //    && (tmp = (id->not_query/"/")[-1])  
      //    && tmp[0] == '.') {     
      //   TRACE_LEAVE("Is .-file"); 
      //   return 0;   
      // } 
      

      TRACE_ENTER("Opening file \"" + f + "\"", 0);                             
      o = open( f, "r" );                                                       
 
      if( (!o) || (! is_image_file(f)) )
      {                                                                         
        //errors++;                        
        report_error("Open of " + f + " failed. Permission denied.\n");         
                                                                                
        TRACE_LEAVE("");                                                        
        TRACE_LEAVE("Permission denied.");                                      
        return http_low_answer(403, "<h2>File exists, but access forbidden "    
                               "by user</h2>");                                 
      } 

      id->realfile = f;                                                         
      TRACE_LEAVE("");                                                          
      // accesses++;                                                               
      TRACE_LEAVE("Normal return");                                             
      return o;                                                                 
    }                                                                           
    break; 

  case "MKDIR":
    TRACE_LEAVE("MKDIR: Permission denied");                                  
    return http_auth_required("foo","<h1>Permission to 'MKDIR' denied</h1>");
    break;

  case "PUT":                                                                   
    id->misc->error_code = 405;                                               
    TRACE_LEAVE("PUT disallowed");                                            
    return 0; 
    break;

   case "CHMOD":                                                                
    // Change permission of a file.                                             
    id->misc->error_code = 405;                                               
    TRACE_LEAVE("CHMOD disallowed (since PUT is disallowed)");                
    return 0; 
    break;

   case "MV":                                                                   
    id->misc->error_code = 405;                                               
    TRACE_LEAVE("MV disallowed (since PUT is disallowed)");                   
    return 0;  
    break;

  case "DELETE":                                                                
    id->misc->error_code = 405;                                               
    TRACE_LEAVE("DELETE: Disabled");                                          
    return 0;                                                                 
    break;
  default:                                                                      
    TRACE_LEAVE("Not supported");                                               
    return 0;                                                                   
  }
  report_error("Not reached..\n");                                              
  TRACE_LEAVE("Not reached");                                                   
  return 0;                                                                     
} 

