// combfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "combfilter.h"
#include "delay.h"
#include "valuerequester.h"
#include "request.h"
#include "data.h"

class CombFilterRequester : public ValueRequester<double> {
	friend class CombFilter;
protected:
	CombFilterRequester(CombFilter *);
	redefined void configureRequest(class Request *);
private:
	CombFilter* client;
};

CombFilterRequester::CombFilterRequester(CombFilter* c)
	: ValueRequester<double>("Comb-filter Selected Region:",
	                "Gain Factor:",
	                c->gain), client(c) {}
void
CombFilterRequester::configureRequest(Request* request) {
	request->appendValue("Fundamental Frequency/Loop Time:",
	                    &client->delayTime,
			    NonNegativeNumbers);
	request->appendValue("Ring Time (seconds):", &client->ringTime,
	                    NonNegativeNumbers);
	request->appendChoice("Filter Value Mode:",
			     "|Fundamental Frequency|Delay Loop Time|",
			     &client->filterMode);
	ValueRequester<double>::configureRequest(request);
}

//********

CombFilter::CombFilter(Data* input, Data *output, double delaytime,
		double ringtime) : SimpleInputOutputFunction(input, output),
		myDelay(nil), delayTime(delaytime),
		ringTime(ringtime), gain(1) {
	initialize();
}

CombFilter::CombFilter(Data* input, Data *output)
	: SimpleInputOutputFunction(input, output), myDelay(nil),
	  delayTime(1000.0), ringTime(0.10), gain(1), filterMode(Frequency) {}

CombFilter::~CombFilter() {
	delete myDelay;
}

void
CombFilter::clear() {
	myDelay->restoreState();
}

Requester *
CombFilter::createRequester() {
	return new CombFilterRequester(this);
}

void
CombFilter::initialize() {
	if(filterMode == Frequency)
		delayTime = 1.0/min(delayTime, sampRate()/2.0);
	int size = iround(sampRate() * delayTime);
	myDelay = new Delay(target(), size);
	feedBack = pow(.001, (delayTime/ringTime));
	Super::initialize();
}

void
CombFilter::restoreState() {
	Super::restoreState();
	clear();
}

boolean
CombFilter::areCompatibleLengths(int oldlen, int newlen) {
	return oldlen >= newlen;
}

int
CombFilter::newLength(int oldLength) {
	return iroundUp(oldLength + ringTime * sampRate());
}

int
CombFilter::doProcessing() {
	BUG("CombFilter::doProcessing()");
	int status = true;
	int flushing = needToFlush();
	while((!inQueueEmpty() || flushing) && status == true) {
		status = processInput(flushing);
	}
	moreOutputIf(true);		// let this run until output eof
	return status;
}

int
CombFilter::doSimpleProcess(double *buffer, int len) {
	for (int n = 0; n < len; ++n) {
		double output = myDelay->getPast();
		myDelay->push(buffer[n] + output * feedBack);
		myDelay->increment();
		buffer[n] = output * gain;
	}
	return len;
}
