// scale.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/shape.h>
#include <InterViews/message.h>
#include <InterViews/painter.h>
#include <InterViews/scene.h>
#include <InterViews/font.h>
#include <InterViews/canvas.h>
#include <InterViews/event.h>
#include <InterViews/box.h>
#include <InterViews/glue.h>
#include <InterViews/perspective.h>
#include "localdefs.h"
#include "scale.h"
#include "format.h"
#include "vmessage.h"
#include "controller.h"

extern "C" {
#include <math.h>
}
#ifdef NeXT
#include <m68k/math.h>
#elif defined(__ppc__)
#include <math.h>
#else
#include <values.h>
#endif

char *
ScaleMarks::Tick::setString(char *str) {
	delete [] string;
	return string = newstr(str);
}

ScaleMarks::TickList::~TickList() { 
	while(currentsize > 0) delete list[--currentsize]; 
	delete list;
}

// *************

const int ScaleMarks::defaultLabelPad = 2;
const int ScaleMarks::defaultBorderWidth = 0;

ScaleMarks::ScaleMarks(const Range &r, Scale::NumberDisplay nd, Controller *c)
		: scaleRange(r), _numberDisplay(nd),
		  displayIntegers(nd == Scale::AsInteger), controller(c) {
	Init();
}

ScaleMarks::~ScaleMarks() {
	setViewPerspective(nil);
	Resource::unref(shown);
	delete tickList;
}

void
ScaleMarks::Init() {
	tickList = nil;
	view = nil;
	shown = new Perspective;
	setVertMargins(defaultBorderWidth, defaultBorderWidth);
	setHorizMargins(defaultBorderWidth, defaultBorderWidth);
	topSpacer = 0;
	bottomSpacer = 0;
	rangeChanged = true;
}

// These are redefined InterViews virtual functions.

void
ScaleMarks::Resize() {
	doAdjust();
}

void
ScaleMarks::Update() {
	if(viewChanged()) {
		newRange();
		if(view != nil) *shown = *view;
		BorderedArea::Update();
	}
}

void
ScaleMarks::Handle(Event &e)
{
	if(e.meta_is_down())
		return;		// ignore all window manager events	
	// these are the only events we are interested in
	if(e.leftmouse || e.rightmouse || e.middlemouse || e.type() == Event::key)
		controller->handleEvent(e);	// send it back to top
}

// new public functions

void
ScaleMarks::setRange(const Range& range) {
	Range newRange = range;		// copy to avoid const
	newRange.check();
	if(newRange != scaleRange) {
		scaleRange = newRange; 
		rangeChanged = true;
	}
}

void
ScaleMarks::setSpacers(int top, int bottom) {
	if(topSpacer != top || bottomSpacer != bottom) {
		topSpacer = top;
		bottomSpacer = bottom;
		BorderedArea::Update();
	}
}

// internal functions

void
ScaleMarks::doAdjust() {
	Range currentRange = getCurrentRange();
	currentRange.limit(-MAXDOUBLE, MAXDOUBLE);	// protect against NaN
	double minval = currentRange.min();
	double range, primeIncr;
	const int minTicks = 2;
	const int maxTicks = 11;
	int ztick;
	if(currentRange.includesZero()) {
		range = currentRange.absoluteMax();
		ztick = 1;	/* origin counts as tick mark */
	}
	else	{		/* scale all pos or neg */
		range = currentRange.spread();
		ztick = 0;
	}
	int zeroRange = false;
	/* find max power of ten less than range on one side of origin */
	if (range > 0.0)
	    primeIncr = lowerPowerOfTen(range);
	else {
	    primeIncr = 1.0;
	    ztick = 1;	// add extra tick at end
	    zeroRange = true;
	}
	if(displayIntegers)
		primeIncr = max(1.0, primeIncr);	// must be >= 1
	double totalrange = currentRange.spread();
	int nPrimeTicks, nSecTicks = 10;
	// make sure there is no more than 11 and at least 2 prime ticks
	// other than the origin 
	while((nPrimeTicks = int(totalrange/primeIncr) + ztick) > maxTicks) {
		primeIncr *= 2; 		// 0, 10, 20 -> 0, 20, 40,
	}
	while((nPrimeTicks = int(totalrange/primeIncr) + ztick) <= minTicks) {
		if(displayIntegers && primeIncr == 1.0)
			break;				// cannot subdivide!
		else if(nSecTicks == 10) {
			primeIncr /= 2; 	// 10, 20, 30 -> 10, 15, 20, 25
			nSecTicks = 2;
		}
		else {
			primeIncr /= 5;		// 10, 15, 20 -> 10, 11, 12, 13
			nSecTicks = 10;
		}
	}
	int primeTicksOnly = displayIntegers && (primeIncr < 5.0);
	nPrimeTicks = max(1, int(totalrange/primeIncr) + ztick);	// recompute
        double tickIncrement, tickdist, baseval, currentTickLoc;
        double basePrimeVal;
	int ticknumber;
		
	if (!zeroRange) {
	    nSecTicks = (scaleLength()/nPrimeTicks < 31) ? 2 : nSecTicks;
	    tickIncrement = primeIncr / nSecTicks;
	    totalTicks = int(totalrange/tickIncrement) + ztick;
	    // actual fractional distance between smallest ticks
	    tickdist = scaleLength()/(totalrange/tickIncrement);
	    // bottom-most tick value (whether or not labeled)
	    baseval = roundUpToNearest(minval, tickIncrement);
	    currentTickLoc = fabs((minval-baseval)/tickIncrement) * tickdist;
	    currentTickLoc += baseLocation();	// offset by borderwidth
	    basePrimeVal = roundUpToNearest(minval, primeIncr);
	    ticknumber = -int(fabs(minval - basePrimeVal)/tickIncrement);
	}
	// special case for zoom-in such that start of screen and end of screen
	// are all the same frame
	else {
	    nSecTicks = 1;
	    tickIncrement = 0;
	    totalTicks = 2;
	    // ticks set at opposite ends of scale
	    tickdist = scaleLength();
	    // bottom-most tick value (whether or not labeled)
	    baseval = minval;
	    currentTickLoc = baseLocation();
	    basePrimeVal = baseval;
	    ticknumber = 0;
	}
	
	const int shortTickLength = 2, medTickLength = 4, longTickLength = 6;

	// add all ticks into list
	delete tickList;
	tickList = new TickList(totalTicks);
	for(int i = 0; i < totalTicks; i++) {
		int loc = int(currentTickLoc);
		int len = shortTickLength;
		char *string = NULL;
		if(!(ticknumber % nSecTicks)) { // every 2 or 10 ticks
			len = longTickLength;
			double curVal = baseval + (i * tickIncrement);
			double incr = tickIncrement*nSecTicks;
			string = displayIntegers ? iFormat(curVal, incr) :
					 (_numberDisplay == Scale::AsFloat) ?
						kFormat(curVal, incr) : tFormat(curVal, incr);
		}
		else if(nSecTicks == 10 && !(ticknumber % 5))
			len = medTickLength;
		if(!primeTicksOnly || len == longTickLength)
			tickList->addTick(loc, len, string);
		currentTickLoc += tickdist;
		ticknumber++;
	}
}

int
ScaleMarks::textMajorDimension(char *string) {
    return getTextMajorDimension(output->GetFont(), string) + labelPad;
}

int
ScaleMarks::textMinorDimension(char *string) {
    return getTextMinorDimension(output->GetFont(), string) + labelPad;
}

void
ScaleMarks::setViewPerspective(Perspective *np) {
	// this does what Init() does in other view classes
	if(np != view) {
		if(view != nil)
			view->Detach(this);
		view = np;
		if(view != nil) {
			view->Attach(this);
			*shown = *view;
		}
	}
}

// *************

VScaleMarks::VScaleMarks(const Range &r,
						 Scale::NumberDisplay nd,
						 Scale::Position,
						 Controller *c)
	: ScaleMarks(r, nd, c)
{ 
	Init();
}

void
VScaleMarks::Init() {
	SetClassName("VerticalScale");
}

void
VScaleMarks::Reconfig() {
	const char *a = GetAttribute("padding");
	if (a != nil) labelPad = atoi(a);
	else labelPad = defaultLabelPad;
	a = GetAttribute("borderWidth");
	int border = (a != nil) ? atoi(a) : defaultBorderWidth;
	// always leave room for labels on ends
	int textPad = textMajorDimension("0e+00")/2;
	setVertMargins(border + textPad, border + textPad);
	boolean allowShrink = AttributeIsSet("allowShrink");
	shape->height = textMajorDimension("0.00e+00") * 12 
		+ topBorder() + bottomBorder() + topSpacer + bottomSpacer;
	shape->width = textMinorDimension("0.00e+00") + 20;
	shape->Rigid(0, 0, (allowShrink ? shape->height/2 : 0), vfil);
}

void
VScaleMarks::Redraw(Coord l, Coord b, Coord r, Coord t) {
	output->Clip(canvas, l, b, r, t);
	output->ClearRect(canvas, l, b, r, t);
	drawTicks(b, t);
	output->Line(canvas, xmax, baseLocation(), xmax, 
		scaleLength() + baseLocation());
	output->NoClip();
}

void
VScaleMarks::drawTicks(Coord begin, Coord end) {
	int averageOffset = textMajorDimension("+000.00e+00")/2;
	int bottom = begin - averageOffset, top = end + averageOffset;
	for(int i = 0; i < tickList->currentSize(); i++) {
		Tick *currentTick = tickList->get(i);
		int loc = currentTick->loc;
		if(loc < bottom || loc > top)
			continue; // skip unneeded redraws
		int len = currentTick->length;
		char *string = currentTick->string;
		output->Line(canvas, xmax - len, loc, xmax, loc);
		if(string) {
			int textOffset = textMajorDimension(string)/2;
			output->Text(canvas, string, labelPad,
				loc - textOffset);
		}
	}
}

Range
VScaleMarks::getCurrentRange() {
	register Perspective *v = view;
	Range newrange = scaleRange;
	if(v != nil) {
		double factor = v->curheight/double(v->height);
		newrange.scaleBy(factor).check();
	}
	return newrange;
}

int
VScaleMarks::getTextMajorDimension(const Font *f, char *) {
	return f->Height();
}

int
VScaleMarks::getTextMinorDimension(const Font *f, char *string) {
	return f->Width(string);
}

int
VScaleMarks::scaleLength() {
	return ymax + 1 - topBorder() - bottomBorder() - topSpacer - bottomSpacer;
}

void
VScaleMarks::newRange() {
	if(view != nil) {
		double factor = double(view->height) / shown->height;
		scaleRange *= factor;
	}
}

int
VScaleMarks::viewChanged() {
	register Perspective *v = view;
	int status = 0;
	// only converned with changes in the y (height) dimension here
	if(v != nil && (v->cury != shown->cury
			|| v->curheight != shown->curheight
			|| v->height != shown->height)) {
		status = 1;
	}
	if(rangeChanged) {
		status = 1;
		rangeChanged = false;	// reset
	}
	return status;
}

int
VScaleMarks::baseLocation() {
	return BorderedArea::bottomBorder() + bottomSpacer;
}

// *************

HScaleMarks::HScaleMarks(const Range &r,
						 Scale::NumberDisplay nd,
						 Scale::Position pos,
						 Controller *c)
	: ScaleMarks(r, nd, c), isAbove(pos == Scale::IsAbove)
{
	Init();
}

void
HScaleMarks::Init() {
	SetClassName("HorizontalScale");
}

void
HScaleMarks::Reconfig() {
	const char *a = GetAttribute("padding");
	if (a != nil) labelPad = atoi(a);
	else labelPad = defaultLabelPad;
	a = GetAttribute("borderWidth");
	int border = (a != nil) ? atoi(a) : defaultBorderWidth;
	// always leave room for labels on ends
	int textPad = textMajorDimension("0e+00")/2;
	setHorizMargins(border + textPad, border + textPad);
	boolean allowShrink = AttributeIsSet("allowShrink");
	shape->height = textMinorDimension("0.00e+00") + 8;
	shape->width = textMajorDimension("0.00e+00") * 12 
		+ topBorder() + bottomBorder() + topSpacer + bottomSpacer;
	shape->Rigid((allowShrink ? shape->width/2 : 0), hfil, 0, 0);
}

void
HScaleMarks::Redraw(Coord l, Coord b, Coord r, Coord t) {
	output->Clip(canvas, l, b, r, t);
	output->ClearRect(canvas, l, b, r, t);
	drawTicks(l, r);
	int edge = isAbove ? 0 : ymax;
	output->Line(canvas, baseLocation(), edge,
		scaleLength() + baseLocation(), edge);
	output->NoClip();
}

void
HScaleMarks::drawTicks(Coord begin, Coord end) {
	int averageOffset = textMajorDimension("+000.00e+00")/2;
	int textSpacer = textMinorDimension("+000.00e+00") + labelPad;
	int textVLoc = isAbove ? ymax - textSpacer : textSpacer;
	int left = begin - averageOffset, right = end + averageOffset;
	int tickBase = isAbove ? 0 : ymax;
	for(int i = 0; i < tickList->currentSize(); i++) {
		Tick *currentTick = tickList->get(i);
		int loc = currentTick->loc;
		if(loc < left || loc > right)
			continue; // skip unneeded redraws
		int len = currentTick->length;
		char *string = currentTick->string;
		output->Line(canvas, loc, tickBase,
			loc, isAbove ? len : tickBase - len);
		if(string) {
			int textOffset = textMajorDimension(string)/2;
			output->Text(canvas, string, loc - textOffset, textVLoc);
		}
	}
}

Range
HScaleMarks::getCurrentRange() {
	register Perspective *v = view;
	Range newrange = scaleRange;
	if(v != nil) {
		double rmin = newrange.min();
		double rmax = newrange.max();
		double spread = displayIntegers ? newrange.size() : newrange.spread();
		double fstart = (v->curx - v->x0) / double(v->width);
		double fwidth = v->curwidth/double(v->width);
		rmin = rmin + (fstart * spread);
		rmax = rmin + (fwidth * spread);
		if(displayIntegers) {
			rmin = iround(rmin);
			rmax = iround(rmax);
		}
		newrange.set(rmin, rmax);
	}
	return newrange;
}

int
HScaleMarks::getTextMajorDimension(const Font *f, char *string) {
	return f->Width(string);
}

int
HScaleMarks::getTextMinorDimension(const Font *f, char *) {
	return f->Height();
}

int
HScaleMarks::scaleLength() {
	return xmax + 1 - topBorder() - bottomBorder() - topSpacer - bottomSpacer;
}

void
HScaleMarks::newRange() {
	if(view != nil) {
		double factor = double(view->width) / shown->width;
		scaleRange *= factor;
	}
}

int
HScaleMarks::viewChanged() {
	register Perspective *v = view;
	int status = 0;
	// only converned with changes in the x (width) dimension here
	if(v != nil && (v->curx != shown->curx
			|| v->curwidth != shown->curwidth
			|| v->width != shown->width)) {
		status = 1;
	}
	if(rangeChanged) {
		status = 1;
		rangeChanged = false;	// reset
	}
	return status;
}

int
HScaleMarks::topBorder() { return leftEdge(); }

int
HScaleMarks::bottomBorder() { return margin()->right; }

int
HScaleMarks::baseLocation() { return leftEdge() + topSpacer; }

// ********

Scale::Scale()
	: scalemarks(nil), placeHolder(nil), label(nil), scaleShown(true), pad(0) {
}

Scale::~Scale() {
	Resource::unref(
		scaleShown ? (Resource *) placeHolder : (Resource *) scalemarks
	);
}

void
Scale::Update() {
	MonoScene::Update();
	label->Update();
	scalemarks->Update();
}

void
Scale::Reconfig() {
	MonoScene::Reconfig();
}

void
Scale::setLabel(const char *name) {
	label->setText(name);
}
 
const char *
Scale::getLabel() const {
	return label->getText();
}

void
Scale::setNumberDisplay(Scale::NumberDisplay nd) {
	 scalemarks->setNumberDisplay(nd);
}

void
Scale::setRange(const Range &range) {
	scalemarks->setRange(range);
}

Range
Scale::getRange() {
	return scalemarks->getRange();
}

Scale::NumberDisplay
Scale::numberDisplay() { return scalemarks->numberDisplay(); }

void
Scale::setSpacers(int top, int bottom) {
	scalemarks->setSpacers(top, bottom);
}

void
Scale::setViewPerspective(Perspective *np) {
	scalemarks->setViewPerspective(np);
}

void
Scale::showScale(boolean show) {
	if(show != scaleShown) {
		if(show) {
			if(placeHolder != nil)
				interior->Remove(placeHolder);
			interior->Insert(scalemarks);
		}
		else {
			interior->Remove(scalemarks);
			interior->Insert(placeHolder = getPlaceHolder());
		}
		interior->Change(scalemarks);
		scaleShown = show;
	}
}

int
Scale::Height() {
	Canvas *c = GetCanvas();
	return (c != nil) ? c->Height() : 0;
}

int
Scale::Width() {
	Canvas *c = GetCanvas();
	return (c != nil) ? c->Width() : 0;
}

// **********

VScale::VScale(const char *name,
			   const Range& range,
			   Controller *c,
			   NumberDisplay nd,
			   Position pos) {
	Init(name, range, nd, pos, c);
}

void
VScale::Init(const char *name, const Range& range,
		NumberDisplay nd, Position pos, Controller *c) {
	SetClassName("VScale");
	label = new VMessage("VerticalScaleLabel", "MMMMMMMM", Center, pad);
	label->setText(name);
	scalemarks = new VScaleMarks(range, nd, pos,c);
	Insert(interior = new VBox(label,scalemarks));		
}

void
VScale::Reconfig() {
	Shape* s = label->GetShape();
	s->Rigid();
	Scale::Reconfig();
	s = scalemarks->GetShape();
	s->width = shape->width;	// set width of marks to width of container
}

int 
VScale::topMargin() {
	Canvas *c = label->GetCanvas();
	int labelheight = (c != nil) ? c->Height() : 0;
	return scalemarks->topBorder() + labelheight;
}

int
VScale::bottomMargin() { return scalemarks->bottomBorder(); }

Interactor *
VScale::getPlaceHolder() {
	return (placeHolder == nil) ? new VGlue(3,0,vfil) : placeHolder;
}

// **********

HScale::HScale(const char *name, const Range& range, Controller *c,
		NumberDisplay nd, Position pos)
{
	Init(name, range, nd, pos,c);
}

void
HScale::Init(const char *name,
			 const Range& range,
			 NumberDisplay nd, 
			 Position pos,
			 Controller *c)
{
	SetClassName("HScale");
	label = new VMessage("HorizontalScaleLabel", "0000000000000000",
		Center, pad);
	label->setText(name);
	scalemarks = new HScaleMarks(range, nd, pos, c);
	Insert(interior = new VBox(
		new HBox(
			new HGlue,
			label,
			new HGlue
		),
		scalemarks)
	);
}

int 
HScale::topMargin() {
	return scalemarks->topBorder();
}

int 
HScale::bottomMargin() {
	return scalemarks->bottomBorder();
}

Interactor *
HScale::getPlaceHolder() {
	return (placeHolder == nil) ? new HGlue(3,0,hfil) : placeHolder;
}

