/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree_getnext.c,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

#include "dtree.h"
#include "dtree_private.h"

/*
 *    dtree_get()
 *    Return a pointer associated with Index or NULL if none.
 *
 *    Quaternary(4) Flower efficiencys:  Note: 1/3 = 1/4 + 1/16 + 1/64...  
 *
 *    Brown(1)  Flower = 1/(1*V+1*I+1+1/3) = 30%
 *    Red(2)    Flower = 2/(2*V+2*I+1+1/3) = 38%
 *    Orange(3) Flower = 3/(3*V+3*I+1+1/3) = 41%
 *    Yellow(4) Flower = 4/(4*V+4*I+1+1/3) = 43%
 *
 *    Therefore the average will be about 37% or .7 pointer average
 *    I.E. 1/(1I + 1V + .7P) = 37%.
 */
void **                                 /* Returned pointer to Value */
dtree_get
        (
         Pvoid_t        Pdt,            /* Root Pointer to 1st Branch in tree */
         ulong_t        Index           /* Index or Key to lookup */
        )
{
    Slot_t          Slot;          /* The Slot in current Branch */
    ulong_t         Digit;     /* Offset into current Branch pointing to Slot */
    ulong_t         SftIdx;        /* Shifted Index to just relevant bits */
    ulong_t         Level;         /* Current Level in the tree */
    PSlot_t         PSlot;         /* Pointer to next Branch */

    Level   = cDT_ROOT_LEVEL;       /* Start at level 16/32 */
    if (Pdt == (Pvoid_t)NULL) goto NullExit;  /* array not allocated */

    SftIdx  = Index;                /* For clarity - no code generated */
    Slot    = (Slot_t)Pdt;          /* For clarity - no code generated */

    /* 5 (4 clocks) PA-RISC 1.1 Instructions per Level */
    while(DT_IS_BRANCH_PTR(Slot))
    {
        /* Get next decoded Digit */
        Digit = DT_SFTIDX_TO_DIGIT(SftIdx,0);

        /* Shift out decoded Index bits */
        DT_SFTIDX_DIGITS(SftIdx, 0);

        /* Check if reached Leaf */
        if (--Level == 0)
        {
            LevelsDecended += cDT_ROOT_LEVEL-1L;
            /* Leaf - with no flower */
            return((void **)((PSlot_t)Slot + Digit));
        }

        /* Get Next Level Slot */
        Slot = *((PSlot_t)Slot + Digit);
    }

    /* Slot   : Is the Flower pointer
     * SftIdx : Is the Index to decode at this level
     */

    /* Search the Flower for a matching Index */
    if (Slot != cDT_NULL)
    {
        ulong_t ii = 0L;

        /* Get the Flower Size */
        ulong_t FlowerSize = DT_SLOTID_TO_FLWR_SIZE(Slot);

        /* Change Flower pointer to a normal pointer. */
        PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

        /* Search Flower for an Index match */
        do
        {
            if (SftIdx == *(PSlot + ii))
            {
                /* Return pointer to "Value" Slot */
                LevelsDecended += cDT_ROOT_LEVEL-Level;

                return((void **)(PSlot + ii + FlowerSize));
            }
        } while(++ii < FlowerSize);
    }

NullExit:
    LevelsDecended += cDT_ROOT_LEVEL-Level;
    return((void **)NULL); /* Requested Index not found */
}


/*
 *************************************************************************
 *                      Dtree - Neighbor routinesh                        
 *************************************************************************
 */

/*
 * dtree_first()
 * Return a pointer associated with first Index or NULL if none.
 */
PPvoid_t                         /* Pointer to "Value" if found, else NULL */
dtree_first(
            Pvoid_t  Pdt,        /* Root Pointer to 1st Branch in tree */
            ulong_t *PIndex      /* IN ignore; OUT return value if any */
           )
{
    PPvoid_t PPvalue;            /* value slot to return */

    if ((PPvalue = dtree_get(Pdt,  *PIndex)) == (PPvoid_t) NULL)
    {
         PPvalue = dtree_next(Pdt, PIndex);
    }

    /* Along with the Value pointer */
    return(PPvalue);

} /* dtree_first() */


#ifdef LATER
PPvoid_t
dtree_last(
           Pvoid_t  Pdt,        /* tree to search */
           ulong_t *PIndex      /* IN ignore; OUT return value if any */
          )
{
    Pvoid_t PPvalue;            /* value slot to return */

    if ((PPvalue = dtree_get(Pdt, *PIndex)) == (PPvoid_t) NULL)
    {
         PPvalue = dtree_prev(Pdt, PIndex);
    }

    /* Along with the Value pointer */
    return(PPvalue);

} /* dtree_last() */
#endif /* LATER */

/*
 * Pointer stack struct used below.
 */
typedef struct _POINTER_STACK
{
        Slot_t Slot;
        ulong_t Digit;
} ps_t, *Pps_t;

/*
 * dtree_next()
 * Return a pointer associated with next Index or NULL if none.
 */
PPvoid_t                      /* Pointer to "Value" if found, else NULL */
dtree_next(
           Pvoid_t  Pdt,      /* Root Pointer to 1st Branch in tree */
           ulong_t *PIndex    /* Pointer to Index or Key to begin search */
                              /* And modified to "found" Index */
          )
{
    Slot_t   Slot;             /* The Slot in current Branch */
    ulong_t  Digit;            /* Offset into current Branch pointing to Slot */
    ulong_t  SftIdx;           /* Shifted Index to just relevant bits */
    ulong_t  Level;            /* Current Level in the tree */
    ulong_t  SltHIdx;          /* Pointer Stack Index number */
    ulong_t  IdxAdj;           /* Index Adjusted */
    ps_t     Stack[cDT_ROOT_LEVEL];  /* Pointer Stack */

    /* Check if the array has been allocated */
    if (Pdt == (PPvoid_t)NULL) goto NullReturn;

    /* Get the requested Index + 1 */
    IdxAdj = SftIdx = *PIndex + 1L;

    /* Check if rolled over */
    if (SftIdx == 0L) goto NullReturn;

    SltHIdx = 0L;            /* Pointer Stack */
    Level = cDT_ROOT_LEVEL;  /* Level in tree */
    Slot = (Slot_t)Pdt;      /* For clarity - no code generated */

    /* Needed into this loop
     *  Slot
     *  SftIdx
     *  SltHIdx
     *  Level
     */
    for (;;)   /* Inf loop */
    {
        /* Follow the pointer chain until a non-pointer or Leaf is reached */
        while (DT_IS_BRANCH_PTR(Slot))
        {
            /* Extract the index into the next branch */
            Digit = DT_SFTIDX_TO_DIGIT(SftIdx, 0);

            /* Shift out decoded Index bits */
            DT_SFTIDX_DIGITS(SftIdx, 0);

            /* In case we have to backup in the tree later */
            Stack[SltHIdx].Slot = Slot; 
            Stack[SltHIdx].Digit = Digit;
            SltHIdx++;

            /* Then we found it in a Leaf */
            if (--Level == 0)  
            { 
                *PIndex = IdxAdj;
                return((PPvoid_t)((PSlot_t)Slot + Digit)); 
            }

            /* Go to the next level in the tree */
            Slot = *((PSlot_t)Slot + Digit);
        }

        /* At this point we know it is Flower Pointer, Narrow Pointer or NULL */

        if (Slot != cDT_NULL)
        {
            PSlot_t PSlot;
            ulong_t FlowerSize, ii;

            /* Change Flower pointer to a normal pointer. */
            PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

            /* Get the Flower Size */
            FlowerSize = DT_SLOTID_TO_FLWR_SIZE(Slot);

            /* Search Flower for an >= Index */
            ii = 0L;
            do
            {
                if (SftIdx <= *(PSlot + ii))
                {
                    ulong_t IdxRet; /* Index Returned */

                    /* This fixes the problem of shift of 32==0. */
                    if (Level == cDT_ROOT_LEVEL) { IdxAdj = 0L; }

                    /* Optimizer problems, so use this one -- Adjust 
                     * to only the decoded digits.
                     */
                    IdxAdj &= -1L << (Level * cDT_BITS_PER_DIGIT);

                    /* Note: if Level == 16, this is a shift of zero. */
                    IdxRet = *(PSlot + ii) >> 
                              (cDT_BITS_PER_SLOT-(Level * cDT_BITS_PER_DIGIT));

                    /* Return found index to caller Hi & Lo digits */
                    *PIndex = IdxRet | IdxAdj;

                    /* Return pointer to "Value" Slot */
                    return((PPvoid_t)(PSlot + ii + FlowerSize));
                }

            } while(++ii < FlowerSize);
        }

        /* Exhausted this branch, so backup the tree */
        do
        {
            /* If at top of tree, none found */
            if (SltHIdx-- == 0L) goto NullReturn;

            Slot = Stack[SltHIdx].Slot;
            Level++;
            Digit = Stack[SltHIdx].Digit + 1;
            SftIdx = Digit << (cDT_BITS_PER_SLOT-cDT_BITS_PER_DIGIT);

        } while (SftIdx == 0L); /* if zero, up another more level */

        /* Clean out the lower bits and increment */
        /* IdxAdj +=  1L << ((Level-1L) * cDT_BITS_PER_DIGIT); */
        /* IdxAdj &= -1L << ((Level-1L) * cDT_BITS_PER_DIGIT); */

        /* Same algoithm as above, but a little faster */
        IdxAdj |= (1L << ((Level-1L) * cDT_BITS_PER_DIGIT)) -1L;
        IdxAdj++;

    } /* Inf loop: No exit here */

NullReturn:
    return((PPvoid_t)NULL);

} /* dtree_next() */

