/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree_test.c,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

/* This program will measure the performance of the Trie */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "dtree.h"
#include "dtree_private.h" 

/* Needed for timing routines */
#include <sys/time.h>

#define STARTTm   gettimeofday(&TBeg, NULL)
#define ENDTm     gettimeofday(&TEnd, NULL)

#define DeltaUSec \
    ((double)(TEnd.tv_sec * 1000000.0 + TEnd.tv_usec) \
    - (TBeg.tv_sec * 1000000.0 + TBeg.tv_usec))

/* For brief naming of the dtree and cast typing returns */
#define DG(PName, Index) ((ulong_t *)DTG((PName), (Index)))
#define DI(PName, Index) ((ulong_t *)DTI((PName), (Index)))
#define DN(PName, Index) ((ulong_t *)DTN((PName), (Index)))
#define DF(PName, Index) ((ulong_t *)DTF((PName), (Index)))

#define INITN  0xc1fce3d1  /* Initial Random Number */

ulong_t Backups;
struct timeval TBeg, TEnd;

/* a dtree lives here */
void *Dt = NULL;

/* For storing Random number associated with Index */
typedef struct _ARRAY *PA_t;
typedef struct _ARRAY
{
    ulong_t Randm;
    ulong_t Index;

} A_t;

ulong_t 
Swizzle(ulong_t Index)
{
    ulong_t NewIndex, ii;

    for (NewIndex = Index, ii = 0L; 
         ii < ((cDT_BITS_PER_SLOT/cDT_BITS_PER_DIGIT)-1L); 
         ii++
        )
    {
        Index <<= cDT_BITS_PER_DIGIT;
        NewIndex >>= cDT_BITS_PER_DIGIT;
        NewIndex |= Index & 0xc0000000;
    }
    return(NewIndex);
}


void
PrintBranches(ulong_t NumIndexes)
{
    ulong_t Elements, TotalSlots, Slots;
    long ii;
    char *color;

    /* printf(" Elements      Orange         Red            Brown        "
     *        " Yellow        Branches    TotSlots   %%Eff\n\n");
     */

    /* Calculate Total Slots used in dtree */
    Elements = TotalSlots = 0L;
    for (ii = 0; ii <cDT_ROOT_LEVEL; ii++)
    {
        if (ii < cDT_MAX_LEVEL)
        {
            Slots = DT_SLOTID_TO_BRANCHSLOTS(ii) * 
                    gDtreeStats.ss_AlcBranchCnt[ii];
        }
        else
        {
            /* Calculate Slots per flower */
            Slots = ((ii & 3L) + 1L);

            /* Any Mpy by number of them */
            Slots *= gDtreeStats.ss_AlcBranchCnt[ii] * 2L;
        }
        TotalSlots += Slots;
    }
    printf("Total %lu-Indexes(%0.2f%%), Total %lu-Slots, "
           "Average TreeDepth = %5.3f\n\n", 
           TotalIndexes,
           (double)TotalIndexes * 100.0/(double)TotalSlots, 
           TotalSlots, 
           (double)LevelsDecended/(double)TotalIndexes);

    Elements = 0L;
    for (ii = cDT_ROOT_LEVEL-1; ii >= 0 ; ii--)
    {
            color = NULL;
            if (ii == cDT_YL_FLWR_ID) color = "Ylw";
            if (ii == cDT_BR_FLWR_ID) color = "Brn";
            if (ii == cDT_RD_FLWR_ID) color = "Red";
            if (ii == cDT_OR_FLWR_ID) color = "Orn";

            if (ii < cDT_MAX_LEVEL)
            {
                Slots = DT_SLOTID_TO_BRANCHSLOTS(ii) * 
                        gDtreeStats.ss_AlcBranchCnt[ii];
                if (Slots) color = "Branch";
            }
            else
            {
                /* Calculate Slots per flower */
                Slots = ((ii & 3L) + 1L);

                /* Any Mpy by number of them */
                Slots *= gDtreeStats.ss_AlcBranchCnt[ii];

                /* Accumulate Indexes */
                Elements += Slots;
                Slots *= 2L;   /* Twice Slots per Index */
            }
            if (color != NULL)
            {
                ulong_t Count = gDtreeStats.ss_AlcBranchCnt[ii];

                if (ii == 0)
                {
                    Slots -= gDtreeStats.ss_Leaves*4L;
                    Count -= gDtreeStats.ss_Leaves;
                }
                printf("%lu-%s(%3.1f%%)  ", Count, color, 
                       (double)Slots * 100.0/(double)TotalSlots);
            }
    }
    Elements   += gDtreeStats.ss_Leaves*4L;

    printf("%lu-%s(%0.2f%%)\n\n",gDtreeStats.ss_Leaves, "Leaf", 
           (double)gDtreeStats.ss_Leaves * 400.0/(double)TotalSlots);

    if (Elements != NumIndexes)
    {
        printf("\nOOPS -- number of Indexes %lu should be == %lu\n",
               Elements, NumIndexes);
    }

    printf("Total Free Flowers = ");
    for (ii = cDT_ROOT_LEVEL-1; ii >= 0 ; ii--)
    {
        color = NULL;
        if (ii == cDT_YL_FLWR_ID) color = "Ylw";
        if (ii == cDT_BR_FLWR_ID) color = "Brn";
        if (ii == cDT_RD_FLWR_ID) color = "Red";
        if (ii == cDT_OR_FLWR_ID) color = "Orn";
        if (ii < cDT_MAX_LEVEL)
        {
            Slots = DT_SLOTID_TO_BRANCHSLOTS(ii) * 
                    (double)gDtreeStats.ss_FreeBranchCnt[ii];
            if (Slots) color = "Branch";
        }
        else
        {
            /* Calculate Slots per flower */
            Slots = ((ii & 3L) + 1L);

            /* Any Mpy by number of them */
            Slots *= gDtreeStats.ss_FreeBranchCnt[ii] * 2L;
        }
        if (color != NULL)
        {
            printf("%lu-%s(%0.2f%%)  ", 
                   Slots,
                   color,
                   (double)Slots * 100.0/(double)TotalSlots);
        }
    }
    printf("\n\n");

} /* PrintBranches() */

static unsigned int
Random(unsigned int Seed)
{
    if ((int)Seed < 0)  { Seed += Seed; Seed ^= 16611; }
    else                { Seed += Seed; } return(Seed);
    return(Seed);
}

static int
NumbComp(const void *PA, const void *PB) /* for qsort() */
{ 
    ulong_t A = ((PA_t)PA)->Randm;
    ulong_t B = ((PA_t)PB)->Randm;
    if (A > B) return(1);
    if (A == B) return(0);
    return(-1);
}

/*
 * main() 
 *
 * Arg1 is the number of elements to store in the arrays.
 */
int
main(int argc, char *argv[])
{
    ulong_t TstIndex, ii, jj, *PHpBeg, *PHpEnd;
    PA_t    PTst;
    ulong_t Elements= 1000000;     /* Default = 1M */
    ulong_t NSkip, SwizzFlg;       /* Default = Random */
    ulong_t Initial = 0;
    int Dtii;
    ulong_t *PValue;

    NSkip = SwizzFlg = 0;

    if (argc > 1)
        Elements = strtoul(argv[1], NULL, 0); /* Size of Linear Array */

    if (argc <= 1 || Elements == 0L)
    {
        printf("\n%s <#Indexes> <StepSize(0=Random)> <StartingIndex> "
               "<SwizzFlag>\n", argv[0]);
    }
    if (argc > 2) NSkip = strtoul(argv[2], NULL, 0);
    if (argc > 3) Initial = strtoul(argv[3], NULL, 0);
    if (argc > 4) SwizzFlg = 1L;

    printf("\n%s %ld %lu 0x%lx %lu\n\n", 
           argv[0], Elements, NSkip, Initial, SwizzFlg);

    if (Elements == 0L) exit(1);

    printf("%d bit version with %ld Levels in Tree\n", 
           (int)sizeof(long)*8, cDT_ROOT_LEVEL);

    /* Get Linear Array Pairs, Beginning and End address */
    PTst = (PA_t)sbrk((int)(Elements * sizeof(A_t)));  

    printf("Store %ld number pairs in a Linear Array\n", Elements);

    /* ---------------------For later lookups in sequential order */
    for (Dtii = INITN, ii = 0L; ii < Elements; ii++)
    {
        if (SwizzFlg)
        {
            Dtii = Random(Dtii);
            PTst[ii].Randm = Swizzle(Dtii);

            /* Not used in Random mode (NSkip == 0) */
            PTst[ii].Index = Swizzle(Initial + (ii * NSkip));
        }
        else
        {
            PTst[ii].Randm = Dtii = Random(Dtii);

            /* Not used in Random mode (NSkip == 0) */
            PTst[ii].Index = Initial + (ii * NSkip);
        }
    }
    PHpBeg = (ulong_t *)sbrk((int)0);  // Get Start address of RAM

    /* --------------------Now store same set of numbers in the dtree */

    if (NSkip)
    {
        printf("Store %lu sequential indexes in dtree, "
               "starting with %ld, stepping %ld\n", 
               Elements, Initial, NSkip);

        for (Dtii = Initial, STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DI(Dt, PTst[ii].Index);
            if (PValue == NULL) 
            {
                fprintf(stderr,"ii = %lu, TotalSbrkMem = %lu, sbrk(0) = 0x%lx ",
                        ii, 
                        gDtreeStats.ss_TotalSbrkMem, 
                        (unsigned long)sbrk((int)0));
                perror("Errno is");
                exit(1);
            }
            *PValue = PTst[ii].Index;

            /* PrintBranches(ii+1); */  /* Watch the memory grow */

        } ENDTm;
    }
    else
    {

        printf("Store %lu random indexes in dtree\n", Elements);

        for (Dtii = Initial, STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DI(Dt, PTst[ii].Randm);
            if (PValue == NULL) 
            {
                fprintf(stderr,"ii = %lu, TotalSbrkMem = %lu, sbrk(0) = 0x%lx ",
                        ii, 
                        gDtreeStats.ss_TotalSbrkMem, 
                        (unsigned long)sbrk(0));
                perror("Errno is");
                exit(1);
            }
            *PValue = PTst[ii].Randm;
        } ENDTm;
    }
    PHpEnd = (ulong_t *)sbrk((int)0);  /* Get Ending address of mem */

    printf("dtree_ins  = %7.3f uSec/Index, %5.2f%% efficient "
           "with %ld numbers stored\n", 
            DeltaUSec/Elements, 
            Elements*100.0/(double)(PHpEnd - PHpBeg), 
            Elements);

    /* ---------------------Now lookup in inserted order */

    /* Initialize the Tree Depth counter in dtree_get */
    LevelsDecended = 0L;

    if (NSkip)
    {
        for (jj = 0, STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DG(Dt, PTst[ii].Index);
            if (*PValue != PTst[ii].Index)
            {
                printf("OOPS @ ii=%ld dtree_get failed 0x%lx "
                       "Should be 0x%lx\n", ii, *PValue, PTst[ii].Index);
            }

        } ENDTm;
    }
    else
    {
        for (jj = 0, STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DG(Dt, PTst[ii].Randm);

            if (*PValue != PTst[ii].Randm)
            {
                printf("OOPS @ ii=%ld dtree_get failed "
                       "0x%lx Should be 0x%lx\n", ii, *PValue, PTst[ii].Randm);
            }
        } ENDTm;
    }
    printf("dtree_get  = %7.3f uSec/Index-access by INSERTED order, "
           "%4.2f TreeDepth/Index\n", 
           DeltaUSec/(double)Elements,
           ((double)LevelsDecended/(double)Elements));

    if (NSkip)
    {
        for (STARTTm, ii = 0L, PValue = DF(Dt, TstIndex); 
             PValue != NULL;
/*>--->*/    PValue = DN(Dt, TstIndex), ii++)
        {
            if (*PValue != TstIndex)
            {
                printf("OOPS @ ii=%ld dtree_next failed "
                       "'Index' = 0x%lx != 'Value' = 0x%lx\n",
                       ii, TstIndex, *PValue);
            }
        } ENDTm;
        printf("dtree_next = %7.3f uSec/Index-access by INCREASING "
               "order, ii = %ld, High Value = %lu\n", 
               DeltaUSec/(double)Elements, ii, TstIndex);
    }

    /* ---------------------Now sort Random numbers for sequental order */
    STARTTm;
    qsort((void *)PTst, (size_t)Elements, (size_t)sizeof(A_t), NumbComp);
    ENDTm;
    printf("qsort()    = %7.3f uSec/Index\n", DeltaUSec/Elements);

    /* ---------------------Now lookup numbers in sorted order */
    LevelsDecended = 0L;
    if (NSkip)
    {
        for (STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DG(Dt, PTst[ii].Index);

            if (*PValue != PTst[ii].Index)
            {
                printf("OOPS @ ii=%ld dtree_get failed 0x%lx "
                       "Should be 0x%lx\n", ii, *PValue, PTst[ii].Index);
            }
        } ENDTm;
        printf("dtree_get  = %7.3f uSec/Index-access by RANDOM order\n\n", 
               DeltaUSec/Elements);
    }
    else
    {
        for (STARTTm, ii = 0; ii < Elements; ii++)
        {
/*>--->*/   PValue = DG(Dt, PTst[ii].Randm);

            if (*PValue != PTst[ii].Randm)
            {
                printf("OOPS @ ii=%ld dtree_get failed 0x%lx "
                       "Should be 0x%lx\n", ii, *PValue, PTst[ii].Randm);
            }
        } ENDTm;
        printf("dtree_get  = %7.3f uSec/Index-access by SORTED order\n", 
        DeltaUSec/Elements);
    }

    if (NSkip == 0)
    {
        /* Get the 1st Index in the array */
        for (STARTTm, ii = 0L, TstIndex=0, PValue = DF(Dt, TstIndex); 
             ii < Elements; 
/*>--->*/    PValue = DN(Dt, TstIndex), ii++)
        {
#ifdef DEBUG
            printf("--Called with =");
            PrintIndex4(TstIndex);
            printf("\n");
            printf("--Should be =");
            PrintIndex4(PTst[ii].Randm);
            printf("\n");

            printf("--Return with =");
            PrintIndex4(*PValue);
            printf("\n");
#endif /* DEBUG */
            if (*PValue != PTst[ii].Randm)
            {
                printf("OOPS @ ii=%ld dtree_next failed 'Value' = 0x%lx Should be 0x%lx\n", ii, *PValue, PTst[ii].Randm);
            }
            if (TstIndex != PTst[ii].Randm)
            {
                printf("OOPS @ ii=%ld dtree_next failed 'Index' = 0x%lx Should be 0x%lx\n", ii, TstIndex, PTst[ii].Randm);
            }
        
        } ENDTm;
        printf("dtree_next = %7.3f uSec/Index-access by INCREASING order\n\n", 
        DeltaUSec/Elements);
    }

    PrintBranches(Elements);
    if (Backups) printf("%lu Backups\n", Backups);
    printf("\n");

    return 0;
} /* main() */

