/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: linux_model.c,v 1.34 2004/01/09 20:29:28 type2 Exp $ */

/*
 *******************************************************************************
 *
 *                            PROSPECT PROJECT
 *                 Linux Operational Process Model and State
 *
 *******************************************************************************
 */
#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: linux_model.c,v 1.34 2004/01/09 20:29:28 type2 Exp $";
#endif

/*
 * System header files
 */
#include <string.h>
#include <signal.h>
#include <sched.h>
#include <ctype.h>
#include <sys/times.h>
#include <sys/mman.h>

/*
 * Prospect Header Files
 */
#include "../dtree/dtree.h"
#include "prospect.h"
#include "linux_module.h"
#include "f_sym_name.h"
#include "bfile.h"

/*
 * Global model data.
 */
double        gFirstTrace_ms=0.0;    /* Time stamp of first trace seen */
double        gPrevTrace_ms=0.0;     /* Time stamp of previous trace seen */
double        gCurrentTrace_ms=0.0;  /* Time stamp of the current trace */

proc_list_t   gProclist;             /* The process list */
kernel_struct_t gK;                  /* The kernel place */

sigset_t gSigblockmask;           /* Signals we handle and want to blk */
int      gCurrcpu=0;              /* Current cpu for the buffer */
 
clock_t gBeforepoint=0;           /* Point of child exec */
struct tms gBeforetimes={0,};     /* Usr/sys time before child exec */
clock_t gAfterpoint=0;            /* Point of child exit */
struct tms gAftertimes={0,};      /* Usr/sys times after child exit */

int gDoGettimeofday=TRUE;           /* Use gettimeofday for timestamps */
unsigned long gPstats[PRO_MAX_STATS]={0,};  /* Prospect stats counters */

unsigned long gKernelVmOffset=0;  /* Virtual memory offset for kernel space */

/*
 * void putproc_by_pid(pid_t pid, process_t *p)
 *
 * Put process into trie by pid.
 */
void 
putproc_by_pid(pid_t pid, process_t *p)
{
    void **P;

    /* access the tree */
    P = DTI(gProclist.pl_procs, pid);
    if (P==NULL) {
        /* bad news */
        mBUG("putproc_by_pid(): DTI null, pid(%d) procs(%d) errno(%d): %s",
                pid, gProclist.pl_numProcs, errno, strerror(errno));
        prospect_exit(1);
    }
    if (*P == NULL) gProclist.pl_numProcs++;
    *P = p;

    /* update the cache - write through, eh */
    getproc_by_pid_cached(pid, 1);

} /* putproc_by_pid(pid_t pid, process_t *p) */

/*
 * process_t *getproc_by_pid_cached(pid_t pid, int refresh)
 *
 * Get process from trie by pid. Cached version.
 */
process_t* 
getproc_by_pid_cached(pid_t pid, int refresh)
{
    register unsigned int  ii;
    register int   copen=0;
    register unsigned long newi;
    static   unsigned long cacheline[8]={0,0,0,0,0,0,0,0};
    static   int  full=0;
    void **P;
#ifdef PERFTEST
    static unsigned long hits=0, misses=0, replaced=0, 
                         neg_one=0, notf=0, refreshed=0, refreshits=0;

    if (pid == -1) neg_one++;

    if (refresh) refreshed++;

    if (pid == 0 && refresh == 42)
    {
        ferr("PERFTEST: getproc_by_pid_cached(),\n");
        ferr_nt("\tHits      : %lu\n", hits);
        ferr_nt("\tMisses    : %lu\n", misses);
        ferr_nt("\tReplaced  : %lu\n", replaced);
        ferr_nt("\tNegatives : %d\n", neg_one);
        ferr_nt("\tNot found : %d\n", notf);
        ferr_nt("\tRefreshed : %d\n", refreshed);
        ferr_nt("\tRefreshits: %d\n", refreshits);
        ferr_nt("\tCacheline : [ ");
        for (ii=0; ii<4; ii++)
            ferr_nt("%ld ", cacheline[ii]);
        ferr_nt("\n\t             ");
        for (ii=4; ii<8; ii++)
            ferr_nt("0x%lx ", cacheline[ii]);
        ferr_nt(" ]\n");

        /* 42 is magic */
        return NULL;
    }
#endif

    /*
     * If this is a refresh request, update the cache
     * entry for this tid and pass it back for good measure.
     */
    if (refresh)
    {
        P = DTG(gProclist.pl_procs, pid);
        if (!P)
        {
#ifdef PERFTEST
            notf++;
#endif
        }
        else
        {
            /* check if this is cached and update */
            for (ii=0; ii<4; ii++)
            {
                if (cacheline[ii] == pid) 
                {
                    cacheline[ii+4] = (unsigned long) *P;
#ifdef PERFTEST
                    refreshits++;
#endif
                }
            }
        }
        /* note: if not in cache, maybe we should cache it? */
        return *P;
    }

    /*
     * Use a four item cache filling a cachline at
     * worst case (64-bit).  Use linear search.
     * The cacheline is 4 indices then 4 values.
     * Works for 32 bit too.
     */
    for (ii=0; ii<4; ii++)
    {
        if (cacheline[ii]==pid)
            if (cacheline[ii+4]) 
            {
#ifdef PERFTEST
                hits ++;
#endif
                return (process_t*)cacheline[ii+4];
            }
        /* remember first open slot */
        if (!full) 
            if (cacheline[ii+4] == (unsigned long) NULL) 
                copen=ii;
    }
#ifdef PERFTEST
    misses++;
#endif

    /* we missed, lookup and randomly replace */
    P = DTG(gProclist.pl_procs, pid);
    if (!P) 
    {
#ifdef PERFTEST
        notf++;
#endif
        return NULL;
    }

    if (!full)
    {
        /* cache is not full yet, so use open slots */
        newi = copen;
        /* last item in cacheline ? */
        if (copen == 3) full = TRUE;
    }
    else
    {
        /* Grab a random number to use for index; use current
         * TSC value for a quick random generator.
         */ 
        mTICKNOW(newi);
        newi &= 0x3LU;
    }

    /* replace the cached value at that index */
#ifdef PERFTEST
    if (cacheline[newi+4]) replaced++;
#endif
    cacheline[newi] = pid;
    cacheline[newi+4] = (unsigned long) *P;

    return *P;

} /* getproc_by_pid_cached() */

/*
 * int alloc_proc(pid_t pid);
 *
 * Allocate a new process struct.
 * Take a quick look in /proc to see if it exists there 
 * and if so snarf up some stuff there too.
 */
process_t* 
alloc_proc(pid_t pid)
{
    process_t *p;

    mINFORM("In alloc_proc(%u)", pid);
    p = CALLOC(sizeof(process_t), 1);
    if (!p) {
        mBUG("Calloc failed, line %d, file %s", __LINE__, __FILE__);
        prospect_exit(1);
    }
    p->pr_myPid = pid;
    p->pr_is32bit = TRUE;  /* for now... */

    /* alloc vas head */
    p->pr_vasHead = CALLOC(sizeof(vh_t),1);
    if (p->pr_vasHead == NULL) {
        mBUG("Calloc failed, line %d, file %s", __LINE__, __FILE__);
        prospect_exit(1);
    }
    p->pr_vasHead->vh_pid = pid;
    p->pr_vasHead->vh_proc = p;
    p->pr_vasHead->vh_bck = NULL;
    p->pr_vasHead->vh_fwd = NULL;
    p->pr_vasHead->vh_entries = 0;

    p->pr_ucpu_log = (unsigned long*)CALLOC(gConf.numcpus, 
            sizeof(unsigned long));
    p->pr_kcpu_log = (unsigned long*)CALLOC(gConf.numcpus,
            sizeof(unsigned long));

    /* can't read /proc for this pid if tracing in, it may be gone */
    if (!mTRACEIN) read_proc_for_pid(p);
    return p;
} /* alloc_proc() */

/*
 * void alloc_k_prof(void)
 *
 * Allocate space for kernel profile in kernel struct.  This happens
 * at different times depening if this is a real run or from a trace.
 */
void
alloc_k_prof(void)
{
    if (gConf.numcpus && gK.k_prof.kp_tot_cpu_log==NULL) {
        gK.k_prof.kp_tot_cpu_log = 
            (unsigned long*)CALLOC(gConf.numcpus, sizeof(unsigned long));
        gK.k_prof.kp_intr_cpu_log = 
            (unsigned long*)CALLOC(gConf.numcpus, sizeof(unsigned long));
        gK.k_prof.kp_kthread_cpu_log = 
            (unsigned long*)CALLOC(gConf.numcpus, sizeof(unsigned long));
        gK.k_prof.kp_usr_cpu_log = 
            (unsigned long*)CALLOC(gConf.numcpus, sizeof(unsigned long));
    }
} /* alloc_k_prof() */

/*
 * void read_proc_for_pid(process_t *)
 *
 * Read couple of files in /proc for the passed
 * process and fill in the values.
 */
int
read_proc_for_pid(process_t *p)
{
    struct stat sbuf;
    FILE *strm;
    int len, d1, d4, d5, d6, d7, d8;
    char fname[32], buf[MAXPATHLEN], buf2[32], c3, *ctmp;
    unsigned long lu9;
    pid_t pid;

    mINFORM("In read_proc_for_pid(%p)", p);
    if (p==NULL) {
        mBUG("read_proc_for_pid(): passed null for p");
        return 1;
    }
    pid = p->pr_myPid;

    /* check to see if a /proc entry exists */
    sprintf(fname,"/proc/%u", pid);
    if (!stat(fname, &sbuf)) {
        /* yohoo it's there */
        mINFORM(" entry exists in /proc, parsing...");
        sprintf(fname,"/proc/%u/exe", pid);
        len=readlink(fname, buf, MAXPATHLEN-1);
        if (len>0) {
            buf[len]='\0';
            p->pr_path=strdup(buf);
            mINFORM(" path: %s", buf);
        }
        else {
            if (p->pr_path) FREE(p->pr_path);
            p->pr_path=NULL;
            mINFORM(" couldn't read %s link, errno(%d): %s", 
                    fname, errno, strerror(errno));
            /* could be a kernel thread, maybe not */
        }
        sprintf(fname,"/proc/%u/stat", pid);
        if ((strm=fopen(fname, "r")) == NULL) {
            mINFORM("  error opening %s, errno(%d):%s", fname, errno, 
                    strerror(errno));
            return 1;
        }
        if (fgets(buf, MAXPATHLEN-1, strm) == NULL) {
            mINFORM(" error reading file %s", fname);
            fclose(strm);
            return 1;
        }
        if ((len=sscanf(buf, "%d (%s", &d1, buf2)) != 2) {
            mINFORM(" error fscaning %s, len=%d", fname, len);
            fclose(strm);
            return 1;
        }
        buf2[strlen(buf2)-1] = '\0';  /* chop off ')' */
        ctmp = strrchr(buf, ')');
        if ((len=sscanf(ctmp+2, "%c %d %d %d %d %d %lu",
                        &c3, &d4, &d5, &d6, &d7, &d8, &lu9)) != 7) {
            buf[strlen(buf)-1]='\0';
            mINFORM(" error fscanning %s, numparsed=%d, buf(%s)", fname, len, 
                    buf);
            fclose(strm);
            return 1;
        }
        buf[strlen(buf)-1]='\0';
        mINFORM(" parsed stat: buf(%s)", buf); 
        fclose(strm);
        p->pr_myKname = strdup(buf2);
        p->pr_myParentPid = d4;
        p->pr_myGroupPid = d5;
        p->pr_mySid = d6;
        p->pr_flags = lu9;
        p->pr_parent = getproc_by_pid(p->pr_myParentPid);
        if(p->pr_myPid == gConf.my_pgrp) {
            gConf.my_pgrp = p->pr_myGroupPid;
        }
        if(gConf.bug_level)
            mINFORM(" pid(%d), kname(%s), ppid(%d), gpid(%d), sid(%d)," 
                    " flags(0x%x)",
                    p->pr_myPid, p->pr_myKname, p->pr_myParentPid,
                    p->pr_myGroupPid, p->pr_mySid, p->pr_flags);
    }
    else {
        mINFORM(" pid %u doesn't exist in /proc", pid);
    }

    return 0;
} /* read_proc_for_pid() */

/*
 * void read_system_map(void)
 *
 * Read in the System.map file.  If specified on command line, then that's
 * our only choice.  Otherwise check in these places in this order:
 *
 *   1. /boot/System.map -> System.map-`uname -r`     for others
 *   2. /lib/modules/`uname -r`/build/System.map      for kernel builders
 *
 * If the -e switch was specified, check if an uncompressed kernel image 
 * was specified on the command line with -K/path, else use the default of 
 *
 *      /lib/modules/`uname -r`/build/vmlinux  
 *
 * If none of these work, then output a message that disassembled profiles
 * for the kernel will be unavailable.
 *
 * Next, check for System.map validity by cross checking symbol locations in
 * /proc/ksyms for the symbol defined in the header for CROSSCHECKSTR.
 *
 * REVISIT: 2.4.19preXX-AC3 removes this symbol from the export list.  Our 
 * assumption that every kernel will have a system call table is no longer
 * valid, it seems.
 *
 * Don't pay attention to the versioning extras in ksyms.
 *
 * Once we've got a good System.map, read it in and fill in the kernel symbol
 * table.  Return 1 on error, 0 on success.
 */
int
read_system_map(void)
{
    struct stat sbuf;
    FILE *fd;
    char fname[256], buf[512], *ctmp;
    unsigned long crosscheck;
    unsigned int numsyms, numdups, strtblsize;
    char *strTbl;
    int textstarted, in_gate_section, gate_done; 
    syms_key_t *symkey;
    region_t *ktxt;
#ifdef __ia64__
    Elf64_Sym *symTbl, **symTblPtr, *PHtp_dup, **PHtp_glb;
#else
    Elf32_Sym *symTbl, **symTblPtr, *PHtp_dup, **PHtp_glb;
#endif

    mINFORM("In read_system_map()");

    /* if name not specified on cmd line */
    gOp.ckroot = FALSE;
    if (gConf.system_map == NULL) {
        /* try first option */
        sprintf(fname,"/boot/System.map-%s", gConf.my_utsname.release);
        if (!stat(fname, &sbuf)) {
            mINFORM(" using: %s", fname);
            gConf.system_map = strdup(fname);
        }
        else {
            /* second option */
            mINFORM(" %s failed", fname);
            sprintf(fname,"/lib/modules/%s/build/System.map", 
                    gConf.my_utsname.release);
            if (!stat(fname, &sbuf)) {
                mINFORM(" using: %s", fname);
                gConf.system_map = strdup(fname);
            }
            else {
                mINFORM(" %s failed", fname);
                /* third option */
                ferr("Couldn't find /boot/System.map-`uname -r` nor "
                     "/lib/modules/`uname-r`/System.map\n");
                hint("Specify the System.map with the --system-map option\n");
                return 1;
            }
            gOp.ckroot = TRUE;
        }
    }
    else {
        gOp.ckroot = TRUE;
    }
    
    /* Check for uncompressed kernel image only if we're diassembling */
    if (gConf.flags.disassemble) {
        if (gConf.kernel_image == NULL) {
            /* image not specified on cmd line, check default place */
            sprintf(fname,"/lib/modules/%s/build/vmlinux", 
                    gConf.my_utsname.release);
            if (!stat(fname, &sbuf)) {
                mINFORM(" using: %s", fname);
                gConf.kernel_image = strdup(fname);
            }
            else {
                mINFORM(" %s failed", fname);
                /* third option */
                sprintf(buf,
                "Couldn't find %s\n"
                "  Disassembled profiles of the kernel will not be available.\n"
                "  Hint: specify this image with the --vmlinux option.\n", 
                        fname);
                ferr(buf);
                if (gConf.strm_scr != stdout) {
                    pscr("\n");
                    pscr(buf);
                }
            }
        }
    }

    /* Read in /proc/ksyms for module information and return the
     * check symbol for System.map validity check.
     */ 
    crosscheck=read_proc_ksyms();
    mINFORM("Back from read_proc_ksyms()");

    if (!crosscheck) {
        mINFORM(" can't find " CROSSCHECKSTR);
        mBUG("Couldn't find \"" CROSSCHECKSTR "\" in /proc/ksyms.");
        return 1;
    }

    /* Now read in system.map and build symbol table, check for above address 
     * in process to save time in good case.
     */
    if ((fd = fopen(gConf.system_map, "r"))==NULL) {
        mINFORM(" strange, can't open %s", gConf.system_map);
        perr("Couldn't open \"%s\"", gConf.system_map);
        return 1;
    }
    /* Create kernel text region. The above routine read_proc_ksyms inserted
     * all modules into a sorted list at k_vas head.  Insert region at the
     * head and use that for kernel text.
     */
    if (gK.k_vas.vh_fwd && gK.k_vas.vh_fwd->rd_kerntext == 1) {
        mBUG("gK.k_vas.vh_fwd->rd_kerntext already initialized.");
        return 1;
    }
    ktxt = CALLOC(sizeof(region_t), 1);
    if (!ktxt) {
        mBUG("Disaster: Calloc returns nill for ktxt.");
        prospect_exit(1);
    }
    if (gConf.kernel_image) {
        ktxt->rd_path = strdup(gConf.kernel_image);
        ktxt->rd_name = "vmlinux";
        gK.k_path = strdup(gConf.kernel_image);
    }
    else {
        ktxt->rd_path = strdup(gConf.system_map);
        ktxt->rd_name = MALLOC(strlen(gConf.system_map)+1);
        extract_basename(ktxt->rd_name, gConf.system_map);
        gK.k_path = strdup(gConf.system_map);
    }
    ktxt->rd_kerntext = TRUE;
    ktxt->rd_syminit = TRUE;
    /* If there were no modules, this is the first region */
    if (gK.k_vas.vh_entries == 0) {
        gK.k_vas.vh_fwd = ktxt;
        gK.k_vas.vh_bck = ktxt;
    }
#ifdef __ia64__
    /* insert ia64 kernel text at tail */
    ktxt->rd_fwd = (region_t*)&gK.k_vas;
    ktxt->rd_bck = gK.k_vas.vh_bck;
    ktxt->rd_bck->rd_fwd = ktxt;
    gK.k_vas.vh_bck = ktxt;
#else
    /* insert ia32 kernel text at head */
    ktxt->rd_fwd = gK.k_vas.vh_fwd;
    ktxt->rd_bck = (region_t*) &gK.k_vas;
    ktxt->rd_fwd->rd_bck = ktxt;
    gK.k_vas.vh_fwd = ktxt;
#endif
    gK.k_vas.vh_entries++;

    /* initialize global kernel symbol table */
    symkey = &ktxt->rd_symkey;
    symkey->sk_fd = -1;
    if (gConf.kernel_image)
        symkey->sk_Path = strdup(gConf.kernel_image);
#ifdef __ia64__
    symkey->sk_isElf32 = FALSE;
#else
    symkey->sk_isElf32 = TRUE;
#endif

    /* Read in the system.map file and check it against /proc/ksyms.
     * Also, determince how much space we need for the string table,
     * the symbol table, and our pointer to symbols table.
     */
    numsyms = strtblsize = 0;
    textstarted = FALSE;
    in_gate_section = FALSE;
    gate_done = FALSE;
    do {
        char type;
        unsigned long vaddr;
        char symbol[512];
        int once=1;

        ctmp = fgets(buf, 512, fd);
        if (!ctmp) break;
        sscanf(buf, "%lx %c %s", &vaddr, &type, symbol);

        /* look for check value */
        if (once && !strcmp(symbol, CROSSCHECKSTR)) {
            if (vaddr != crosscheck) {
                mINFORM(" oh-oh, System.map(%p) != crosscheck(%p)",
                        vaddr, crosscheck);
                ferr("%s inconsistent with /proc/ksyms\n", gConf.system_map);
                return 1;
            }
            else {
                mINFORM(" good, System.map(%p) == crosscheck(%p)",
                        vaddr, crosscheck);
            }
            once=0;
        }

        /* sort through available symbols and accumulate for string table */
        switch (type) {
            /* get absolute markers */
            case 'A':
                /* virtual offset for kernel */
#ifdef __ia64__
                if (!strcmp(symbol, "_stext")) {
#else
                if (!strcmp(symbol, "_text")) {
#endif
                    symkey->sk_exec_tmem = vaddr;
                    ktxt->rd_start = (char*)vaddr;
                    textstarted = TRUE;
#ifdef __ia64__
                    gKernelVmOffset = 0xa000000000000000L;
#else
                    gKernelVmOffset = (unsigned long) vaddr;
#endif
                    mINFORM(" set gKernelVmOffset to 0x%lx", gKernelVmOffset);
                }
                /* note that sk_exec_dmem is used as end of kernel */
                if (!strcmp(symbol, "_end")) {
                    symkey->sk_exec_dmem = vaddr;
                    ktxt->rd_end = (char*)vaddr;
                    ktxt->rd_length = ktxt->rd_end - ktxt->rd_start;
                }
                /* fall through */
                if (!textstarted) break;

            /* local and global text symbols */
            case 't':
            case 'T':
                numsyms++;
                strtblsize += (strlen(symbol) + 1); /* add null */
                break;

            /* ignore all others: data, bss, etc. */
#ifdef __ia64__
            /* For ia64, make note of start_gate->end_gate section for 
             * signal handling routintes.  These get remmapped to 0xa...
             * Note that, currently at least, __start_gate_section is the
             * same as ia64_sigtramp.  We use the ia64_sigtramp symbol 
             * since it makes more sense, this could change later.
             */
            case 'D':
            case 'd':
                if (gate_done) break;
                if (!strcmp(symbol, "__start_gate_section") 
                        || !strcmp(symbol, "ia64_sigtramp")) {
                    in_gate_section = TRUE;
                } 
                else 
                    if (!in_gate_section) break;

                if (!strcmp(symbol, "__stop_gate_section")) {
                    in_gate_section = FALSE;
                    gate_done = TRUE;
                }
                /* add symbol and size */
                numsyms++;
                strtblsize += (strlen(symbol) + 1); /* add null */
                break;
#endif
        }
    } while (ctmp);
    mINFORM(" system.map has %d symbols for a %d string table size",
            numsyms, strtblsize);

    /* Allocate memory for tables and set up struct */
    symkey->sk_strings = (char*) MALLOC(strtblsize);
#ifdef __ia64__
    symkey->sk_ht_tbl = MALLOC((numsyms)*sizeof(Elf64_Sym*));
    symkey->sk_lesyms = MALLOC((numsyms)*sizeof(Elf64_Sym));
#else
    symkey->sk_ht_tbl = MALLOC(numsyms*sizeof(Elf32_Sym*));
    symkey->sk_lesyms = MALLOC(numsyms*sizeof(Elf32_Sym));
#endif

    if (!symkey->sk_strings || !symkey->sk_ht_tbl
            || !symkey->sk_lesyms)
    {
        ferr("malloc error on kernel string and symbol table allocations\n"); 
        return 1;
    }

    symkey->sk_fd  = -1;
    if (gConf.kernel_image)
        symkey->sk_Path = gConf.kernel_image;
    else
        symkey->sk_Path = "Kernel_Image_Not_Found";
    mINFORM(" kernel symkey->sk_Path=<%s>", symkey->sk_Path);
    symkey->sk_Inum = 1; /* don't let f_sym_name_list mess us up */ 
    symkey->sk_Dev = 1;
    symkey->sk_ElfFlag = 1;
#ifdef __ia64__
    symkey->sk_isElf32 = 0;
#else
    symkey->sk_isElf32 = 1;
#endif
    symkey->sk_sym_count = numsyms;

    /* Now, rewind the the file and read again to fill in the string
     * table and symbol tables.  Could do this in one read I suppose...
     */
    rewind(fd);
#ifdef __ia64__
    symTbl = (Elf64_Sym*) symkey->sk_lesyms;
#else
    symTbl = (Elf32_Sym*) symkey->sk_lesyms;
#endif
    symTblPtr = symkey->sk_ht_tbl;
    strTbl = symkey->sk_strings;
    numsyms=0;
    in_gate_section = FALSE;
    do {
        char type;
        unsigned long vaddr;
        char symbol[512];

        ctmp = fgets(buf, 512, fd);
        if (!ctmp) break;
        sscanf(buf, "%lx %c %s", &vaddr, &type, symbol);
        /* add symbol to tables: note fallthrough-type switch */
        switch (type) {
#ifdef __ia64__
            case 'D':
            case 'd':
                /* ia64 gate section; note that since ia64_sigtramp and
                 * __start_gate_section are at the same address, the code later
                 * will mark the __start_gate_section as a dup of ia64_sigtramp
                 * and thus we'll have a nicer looking output for this symbol.
                 */
                if (!strcmp(symbol, "__start_gate_section")
                        || !strcmp(symbol, "ia64_sigtramp")) {
                    if (symbol[0] == '_') {
                        mINFORM(" found __start_gate_section at 0x%lx", vaddr);
                        gOp.addr_start_gate = vaddr;
                    }
                    in_gate_section = TRUE;
                } 

                /* if not in gate section, disregard all d|D symbols */
                if (!in_gate_section) 
                    break;
                else
                    mINFORM(" in gate, recording %s at 0x%lx", symbol, vaddr);

                if (!strcmp(symbol, "__stop_gate_section")) {
                    mINFORM(" found __stop_gate_section at 0x%lx", vaddr);
                    gOp.addr_stop_gate = vaddr;
                    in_gate_section = FALSE;
                }
                /* fall through if in gate section is designed. */
#endif

            /* get absolute markers local and global text symbols */
            case 'A':
                /* skip the stuff before text start */
                if (vaddr<symkey->sk_exec_tmem) break; 
                /* fall through designed */

            case 't':
            case 'T':
                /* copy name over to the string table and record index */
                if (symbol[0]) strcpy(strTbl, symbol);
                symTbl->st_name = (unsigned long)strTbl -
                                  (unsigned long)symkey->sk_strings;
                strTbl += (strlen(symbol)+1);  /* include null byte */

                /* fill in rest of symbol struct */
                symTbl->st_value = (unsigned long)vaddr - 
                                   (unsigned long)symkey->sk_exec_tmem; 

                /* Determine type of symbol: note promote all text
                 * symbols to global so they show up in the profiles.
                 */ 
                if (islower(type)) {
                    symTbl->st_info = STB_LOCAL << 4;
                    symTbl->st_other = FALSE; 
                }
                else {
                    symTbl->st_info = STB_GLOBAL << 4;
                    symTbl->st_other = FALSE;
                }

                if (type=='A')
                    symTbl->st_info |= STT_SECTION;
                else
                    symTbl->st_info |= STT_FUNC;

                /* index symbol into our hit table */
                *symTblPtr = symTbl;

                /* increment to next symbol */
                numsyms++;
                symTbl++;
                symTblPtr++;
                /* fall through designed */

            /* Look for important symbols to remember */
            case 'R':
#ifndef __ia64__
            case 'D':
#endif
                if (!strcmp(symbol, "sys_call_table")) {
                    gOp.addr_sys_call_table = vaddr;
                    mINFORM(" found addr_sys_call_table at 0x%lx", vaddr);
                    break;
                }
                if (!strcmp(symbol, "sys_mmap2")) {
                    gOp.addr_sys_mmap2 = vaddr;
                    mINFORM(" found addr_sys_mmap2 at 0x%lx", vaddr);
                    break;
                }
                break;
        }
    } while (ctmp);
    fclose(fd);
    mINFORM(" created %d symbol entries.", numsyms);

    /* now, mark dups and calculate offset to next global, we run
     * through the table backwards.
     */
    PHtp_glb = --symTblPtr;     /* last global symbol (inc 1 extra above) */
    (*symTblPtr)->st_size = 0;  /* offset to next global */
    PHtp_dup = NULL;
    numdups = 0;

#ifdef __ia64__
    while (--symTblPtr >= (Elf64_Sym**)symkey->sk_ht_tbl)
#else
    while (--symTblPtr >= (Elf32_Sym**)symkey->sk_ht_tbl)
#endif
    {
        /* Mark offset to next Global symbol */
        (*symTblPtr)->st_size = PHtp_glb - symTblPtr;

        /* Check if we have a new Global symbol */
        if ((*symTblPtr)->st_other == FALSE) 
            PHtp_glb = symTblPtr;

        /* Mark symbols of duplicate Value */
        if (PHtp_dup) {
            if (PHtp_dup->st_value == (*symTblPtr)->st_value)
            {
                PHtp_dup->st_shndx = TRUE;
                numdups++;
            } 
            else 
            {
                PHtp_dup->st_shndx = FALSE;
            }
        }
        PHtp_dup = *symTblPtr;
    }

    /* The last (first since backwards)  symbol can never be a dup */
    PHtp_dup->st_shndx = FALSE;

    mINFORM(" we have %d duplicate symbols.", numdups);

    /* if debuging modules, output table */
    if (gConf.flags.kmod_debug) {
        region_t *m;
        m = (region_t*) gK.k_vas.vh_fwd;
        if (m) {
            ferr("DBG: Module table (%u entries):\n", gK.k_vas.vh_entries);
            while (m != (region_t *) &(gK.k_vas)) {
                ferr_nt("  [%s] start=0x%lx end=0x%lx length=%u\n",
                     m->rd_name, m->rd_start, m->rd_end, m->rd_length);
                ferr_nt("    %s\n", m->rd_path);
                /* print out symbols if were above 2 */
                if (gConf.bug_level>2) {
                    int ii;
#ifdef __ia64__
                    elf_hit_tbl_64_t *sym;
#else
                    elf_hit_tbl_32_t *sym;
#endif
                    if (f_sym_name_list(&m->rd_symkey, m->rd_path)) {
                        ferr_nt("      Open failed this for region\n");
                    }
                    else {
                        sym = m->rd_symkey.sk_ht_tbl;
                        for (ii=0; ii<m->rd_symkey.sk_sym_count; ii++) {
                            ferr_nt("      ");
                            elf_print_sym_entry((void*)sym, &m->rd_symkey);
                            sym++;
                        }
                    }
                }
                m = m->rd_fwd;
            }
        }
        else {
            ferr_nt("  %s:%d gK.k_vas.vh_fwd==0!!!", __FILE__, __LINE__);
        }
    }

#ifdef __ia64__
    gOp.addr_gate_map_start = 0xa000000000000000L + gConf.page_size;
    gOp.addr_gate_map_end = gOp.addr_gate_map_start + 
                            (gOp.addr_stop_gate - gOp.addr_start_gate);
    mINFORM(" gate mapping, start=0x%lx  end=0x%lx", 
            gOp.addr_gate_map_start, gOp.addr_gate_map_end);
#endif
    
    return 0;
} /* read_system_map() */

/*
 * unsigned long read_proc_ksyms()
 *
 * Reads in the /proc/ksyms file and extracts paths and address
 * ranges of all loaded modules.  Returns address of check var 
 * for crosscheck of System.map.
 *
 * Note: for some reason the ext3 and jbd modules do not have
 * the modutils special symbols in /proc/ksyms and so are 
 * ignored.  Perhaps there are other modules that this happens to.
 * This seems to be an issue on Mandrake systems (probably Red Hat
 * too), Debian does not seem to exhibit this.
 */
unsigned long
read_proc_ksyms(void)
{
    unsigned long crosscheck=0, syscall=0;
    FILE *fd=NULL;
    char *ctmp=NULL, buf[256], *p1=NULL, *p2=NULL, *p3=NULL;
    char mname[256], *mpath=NULL;
    unsigned long mstart=0, mstop=0;
    unsigned int mlen=0;
    int cfound=0, once1=1, nummods=0;
    region_t *kr;

    mINFORM("In read_proc_ksyms()");

    if ((fd=fopen("/proc/ksyms", "r")) == NULL) {
        mINFORM(" can't open /proc/ksyms");
        perr("Can't open /proc/ksyms");
        return 0;
    }
    /* assume we do not have sys_call_table */
    gOp.has_sys_call_table = FALSE;

    /* read entire /proc/ksysms file */
    once1 = TRUE;
    cfound = FALSE;
    do {
        ctmp = fgets(buf, 256, fd);
        if (ctmp) {
            /* watch for sys_call_table */
            if (once1 && strstr(buf, " sys_call_table")) {
                gOp.has_sys_call_table = TRUE;
                sscanf(buf, "%lx", &syscall);
                mINFORM(" ksyms has sys_call_table at 0x%lx", syscall);
                once1=FALSE;
            }
            /* watch for crosscheck (sym starts with space) */
            if (!cfound && strstr(buf, " "CROSSCHECKSTR)) {
                sscanf(buf, "%lx", &crosscheck);
                mINFORM(" scanned " CROSSCHECKSTR " from ksyms = %p", 
                        crosscheck);
                cfound=1;
                continue;
            }
            /* watch for __insmod_ special syms */
            if ((p1=strstr(buf, "__insmod_"))) {
                /* set initial ptr for name */
                p1+=9;
                /* path symbol */
                if ((p2=strstr(buf, "_O/"))) {
                    strncpy(mname, p1, (p2-p1));
                    mname[p2-p1]='\0';
                    /* create get path */
                    p2+=2;
                    if (!(p3=strstr(p2, "_M"))) {
                        ferr("FYI: Problem in module scan for \"%s\".\n",mname);
                        continue;
                    }
                    mpath = MALLOC((p3-p2)+1);
                    strncpy(mpath,p2,(p3-p2));
                    mpath[p3-p2]='\0';
                    kr = get_kr_by_name(mname);
                    kr->rd_path = mpath;
                    if (gConf.flags.kmod_debug && gConf.bug_level>1)
                        mINFORM(" module[%s] path=%s", mname, mpath);
                    nummods++;
                    continue;
                }

                /* address start+range symbol */
                if ((p2=strstr(buf, "_S.text_L"))) {
                    strncpy(mname, p1, (p2-p1));
                    mname[p2-p1]='\0';
                    sscanf(buf, "%lx", &mstart);
                    p2+=9;
                    sscanf(p2, "%u", &mlen);
                    mstop = mstart + mlen;
                    kr = get_kr_by_name(mname);
                    kr->rd_start = (char*)mstart;
                    kr->rd_end = (char*)mstop;
                    kr->rd_length = mlen; 
                    if (gConf.flags.kmod_debug && gConf.bug_level>1)
                        mINFORM(" module[%s] start=0x%lx stop=0x%lx len=%u", 
                                mname, mstart, mstop, mlen);
                    nummods++;
                    continue;
                }
            }
        }
    } while (ctmp);
    fclose(fd);

    if (gOp.has_sys_call_table == FALSE)
        mINFORM(" sys_call_table not found in this system.map.");

    mINFORM(" %d modules detected and processed", nummods/2);
    if (nummods%2) {
        if (gConf.bug_level || gConf.flags.kmod_debug) {
            ferr("FYI: odd number of module syms in /proc/ksyms\n");
        }
        mINFORM(" odd number of module syms in /proc/ksyms.");
    }

    if (gConf.flags.kmod_debug && gConf.bug_level>1) {
        /* print out full module table */
        region_t *m;
        m = (region_t*) gK.k_vas.vh_fwd;
        if (m) {
            ferr("Module table (%u entries):\n", gK.k_vas.vh_entries);
            while (m != (region_t *) &(gK.k_vas)) {
                ferr_nt("  [%s] start=0x%lx end=0x%lx length=%u\n",
                     m->rd_name, m->rd_start, m->rd_end, m->rd_length);
                ferr_nt("       %s\n", m->rd_path);
                m = m->rd_fwd;
            }
        }
        else {
            ferr_nt("  %s:%d gK.k_vas.vh_fwd==0!!!\n", __FILE__, __LINE__);
        }
    }
        
    return crosscheck;
} /* read_proc_ksyms() */

/*
 * region_t *get_kr_by_name(char*)
 *
 * Finds module region in global kernel module region list by
 * name and returns pointer to it.  Creates entry if not found.
 */
region_t*
get_kr_by_name(char *name)
{
    region_t *k;

    if (gK.k_vas.vh_entries == 0) {
        /* first entry */
        k = CALLOC(sizeof(region_t), 1);
        if (name) k->rd_name = strdup(name);
        gK.k_vas.vh_fwd = k;
        gK.k_vas.vh_bck = k;
        k->rd_fwd = (region_t*) &gK.k_vas;
        k->rd_bck = (region_t*) &gK.k_vas;
        gK.k_vas.vh_entries++;
        return k;
    }

    /* linear search */
    k = gK.k_vas.vh_fwd;
    while (k != (region_t*) &(gK.k_vas)) {
        if (!strcmp(k->rd_name, name)) {
            /* found match */
            return k;
        }
        k = k->rd_fwd;
    }

    /* Not found so insert new one.  Note that we can't do a sorted
     * insert because in this function we don't know the value.  So, we
     * insert at the *head* - since on initial read of proc/ksyms the
     * modules arrive in reverse sorted order.
     */
    k = CALLOC(sizeof(region_t), 1);
    if (name) k->rd_name = strdup(name);

    k->rd_fwd = gK.k_vas.vh_fwd;
    k->rd_bck = (region_t*) &(gK.k_vas);
    k->rd_fwd->rd_bck = k;
    gK.k_vas.vh_fwd = k;

    gK.k_vas.vh_entries++;
    return k;
} /* get_kr_by_name() */

/*
 * unsigned int read_uint_from_proc(char*)
 *
 * Read an in from a /proc file and return it.
 * Exit on failure.
 */
unsigned int
read_uint_from_proc(char *filen)
{
    FILE *strm;
    unsigned int val;

    mINFORM("In read_uint_from_proc(%s)", filen);

    if ((strm = fopen(filen, "r")) == NULL) {
        mINFORM(" open failed of file(%s), errno(%d): %s", filen, errno,
                strerror(errno));
        perr("Open failed of %s", filen);
        prospect_exit(1);
    }

    if (fscanf(strm, "%u", &val) !=1) {
        mINFORM(" couldn't fscanf(), errno(%d): %s", errno, strerror(errno));
        perr("Couldn't read %f", filen);
        prospect_exit(1);
    }

    fclose(strm);

    mINFORM(" read: %u", val);
    return val;
} /* read_int_from_proc() */

/*
 * void write_uint_to_proc(char*, unsigned int val)
 *
 * Write an int to a /proc file.  Exit on failure.
 */
void
write_uint_to_proc(char *filen, unsigned int val)
{
    FILE *strm;

    mINFORM("In write_uint_to_proc(%s, %u)", filen, val);

    if ((strm = fopen(filen, "w")) == NULL) {
        mINFORM(" open failed of file(%s), errno(%d): %s", filen, errno,
                strerror(errno));
        perr("Open failed of %s", filen);
        prospect_exit(1);
    }

    fprintf(strm, "%u", val);
    fclose(strm);
} /* write_int_to_proc() */

/*
 * int make_realtime(int pri)
 *
 * Sets real time priority according to the
 * passed parameter.  The parmeter pri can be
 * 0<=pri<=99 with 0 being timeshare.
 * Also locks process pages into memory.
 */
int
make_realtime(int pri)
{
    struct sched_param param;

    mINFORM("make_realtime(%d):", pri);

    /* some init stuff and limit pri's */
    if (pri <= 0) pri = REALTIME_OFF;
    else if (pri > 99) pri = 99;


    /* check if we're turning off */
    if (pri == REALTIME_OFF)
    {
        mINFORM(" Unlocking memory");
        /* Unlock our memory - do we need to lock? */
        if (munlockall() < 0)
        {
            mINFORM("  munlockall failed, errno=%d", errno);
            perr("munlockall failed");
        }

        /* Get back to a normal priority:
         * start at max, let system decay us 
         */
        param.sched_priority = sched_get_priority_max(SCHED_OTHER); 
        mINFORM(" Turning off my realtime priority (pri=%d)", 
                      param.sched_priority);
        if (sched_setscheduler(0, SCHED_OTHER, &param) < 0)
        {
            mINFORM("  sched_setscheduler() failed, errno=%d", errno);
            perr("sched_setscheduler failed");
        }

        return REALTIME_OFF;
    }

    /* 
     * Else, we're becoming realtime 
     */
    mINFORM(" Locking me into memory");  /* do we need to? */
    if (mlockall(MCL_CURRENT | MCL_FUTURE) < 0) 
    {
        mINFORM("  mlockall failed, errno=%d", errno);
        perr("mlockall  failed");
    }

    mINFORM(" Setting realtime of %d, SCHED_RR", pri);
    param.sched_priority = pri;
    if (sched_setscheduler(0, SCHED_RR, &param) < 0)
    {
        mINFORM("  sched_setscheduler() failed, errno=%d", errno);
        perr("sched_setscheduler failed");
    }

    return param.sched_priority;
} /* make_realtime() */

/*
 * int try_fixup_module_path(region_t *r)
 *
 * Check if the basename of passed-in path is in the /lib/modules`uname
 * -r`/modules.dep file.  If it is, then substitute that path for the path
 *  passed in.  These are modules loaded from the initrd which have bad paths
 *  on the initrd.  Returns zero on success, one otherwise.
 */
int
try_fixup_module_path(region_t *r)
{
    char buf[MAXPATHLEN], *basename, *sep;
    FILE *strm;
    static int badmodulesdep=FALSE;

    mINFORM("In try_fixup_module_path(%s):", r->rd_path);

    if (badmodulesdep) {
        mINFORM(" bad modules dep file flag active.");
        return 1;
    }
    
    basename = (char*) MALLOC(strlen(r->rd_path)+1);
    if (!basename) return 1;
    extract_basename(basename, r->rd_path);
    strcat(basename, ":");
    mINFORM(" extracted basename: %s", basename);
    
    /* open modules dep and see if basename exits */
    sprintf(buf,"/lib/modules/%s/modules.dep", gConf.my_utsname.release);
    strm = fopen(buf, "r");
    if (strm==NULL) {
        badmodulesdep = TRUE;
        ferr("problem opening %s\n", buf);
        mINFORM("problem opening %s\n", buf);
        FREE(basename);
        return 1;
    }
    while (fgets(buf, MAXPATHLEN, strm)) {
        if (strstr(buf, basename)) {
            /* found it, dup it to path */
            sep = strchr(buf, ':');
            if (sep) {
                *sep = '\0';
                FREE(r->rd_path);
                r->rd_path = strdup(buf);
                mINFORM(" path adjusted to: %s", r->rd_path);
            }
            fclose(strm);
            FREE(basename);
            return 0;
        }
    }

    fclose(strm);
    FREE(basename);
    return 1;
} /* try_fixup_module_path() */

