package PSP::parent;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: parent.pm,v 1.2 2001/02/10 22:21:42 muaddie Exp $

use strict;

=head1 NAME

 PSP::parent - a composite pattern of a parent containing children.

=head1 SYNOPSIS

 #more to come

=head1 DESCRIPTION

more to come.

=cut

sub new {
  my ($proto,$children) = @_;
  $children ||= [];

  my $this = {};
  bless $this, ref($proto)||$proto;

  if (ref $proto) {
    $this->parent($proto);
    if ($proto->isa("PSP::parent")) {
      $proto->add_child($this);
    }
  }

  $this->{children}  = $children;
  $this->{child_index} = -1;
  $this->{child_seen} = -1;

  return $this;
}

=head2 children

 instance
 (PSP::parent @children) children ()
  or
 instance
 (PSP::parent \@children) children ()

DESCRIPTION:

Returns a list of the immediate children of this object.

=cut

# return the list of children
sub children {
  my ($this,$children) = @_;
  defined $children and @{$this->{children}} = @$children;
  return @{$this->{children}} if wantarray;
  return $this->{children};
}

=head2 add_child and del_child

 instance
 () add_child ( parent child )

 instance
 () del_child ( parent child )

DESCRIPTION:

add_child() appends its input node to the children array.  

del_child() will iterate through its list of children, and return/remove
the first child it finds which matches.

EXAMPLE:

  $thing->add_child($child);
  for $child ($thing->children()) {
    $child->operation();
  }

=cut

# add a child to this list of children.
sub add_child {
  my ($this,$child) = @_;
  $child and push @{$this->{children}}, $child;
  $child;
}
# delete a child from this list of children.
sub del_child {
  my ($this,$child) = @_;
  return $this->delete_children([$child]) if defined $child;
  return $this->delete_children();
}
sub delete_children {
  my ($this,$children) = @_;
  my (@to_delete);
  if ($children) {
    for my $child (@$children) {
      if (grep {$_ eq $child} @{$this->{children}}) {
	push @to_delete, $child;
	@{$this->{children}} = grep {$_ ne $child} @{$this->{children}};
      }
    }
  } else {
    @to_delete = @{$this->{children}};
    @{$this->{children}} = ();
  }

  for my $child (@to_delete) {
    $child->free();
  }

  return;
}
sub free {
  my ($this) = @_;
  #print "deleting parent: $this->{parent}\n" if $this->{parent};
  undef $this->{parent};
  $this->delete_children();
}

=head1 find_child and matches

 instance
 (parent $child) find_child ( string pattern )

 instance
 (boolean $does_match) matches ( string pattern )

DESCRIPTION:

=cut

sub find_child {
  my ($this,$pattern) = @_;
  for my $child ($this->children()) {
    return $child if $child->matches($pattern);
  }
  return;
}
# abstract virtual method.
sub matches {
  die "method matches() must be overridden in derived class.";
}

=head2 reset_children

 instance
 () reset_children ()

DESCRIPTION:

Resets the object such that the first call to next_child() will return the 
first child.

=head2 first_child

 instance
 (PSP::parent) first_child ()

DESCRIPTION:

Returns the first immediate child of the error object, determined by
the internal list refernced by C<$this->children_order()>.

=head2 next_child

 instance
 (PSP::parent) next_child ()

DESCRIPTION:

Returns the next immediate child of this error object. This is
determined by incrimenting the index associated with the internal list
referenced by C<$this->children_order()>.

=head2 curr_child

 instance
 (PSP::parent) curr_child ()

DESCRIPTION:

Returns the current child being referenced. Note that if neither
C<next_child()> nor C<fist_child()> have been called, then this will
return nothing as there is no current child.

=cut

sub reset_children {
  my ($this) = @_;
  undef $this->{all_children_seen};
  $this->{curr_child_index} = -1;
}
sub first_child {
  my ($this) = @_;
  $this->reset_children();
  return $this->next_child();
}
sub next_child {
  my ($this) = @_;
  $this->{all_children_seen} and return;
  $this->{cur_child_index}++;
  return $this->curr_child();
}
sub curr_child {
  my ($this) = @_;
  if (my $child = $this->{children}->[$this->{cur_child_index}]) {
    $child->{child_seen}++;
    return $child;
  }
  $this->{all_children_seen}++;
  return;
}

=head2 depth_reset

 instance
 () depth_reset ()

DESCRIPTION:

=head2 depth_first_child

 instance
 (PSP::parent) depth_first_child ()

DESCRIPTION:

=head2 depth_next

 instance
 (PSP::parent) depth_next ()

DESCRIPTION:

This function searches the children of the current object
recursively. (Not just the current children, but grandchildren,
great-grandchildren, etc.) It will return an object if there are no
children or no touched children of that object. Once there are no
children below the current object, then the current object will be
returned. If the current object had been returned it will be
considered touched and a a depth_first_child call on that object will
result in a depth_first_child search on the parent object. If all
relations have been touched, then a the function will return undef.

For example, if Parent has children Child1, Child2 and Child3 and we
have called: 

 package foo;
 @ISA = qw(PSP::parent);
 sub new {
   my ($parent,$name) = @_;
   my $this = $parent->new();
   $this->{name} = $name;
   return $this;
 }

 my $parent = PSP::parent->new("Parent");
 $parent->add_child($parent->new("Child1"));
 my $child2 = $parent->add_child($parent->new("Child2"));
 $parent->add_child($parent->new("Child3"));

 $child2->add_child($child2->new("Child2-1");
 $child2->add_child($child2->new("Child2-2");

 for (my $obj = $parent->depth_first_child();
      $obj;
      $obj = $parent->depth_next()) {
   print "$obj->{name}\n";
 }
 print "All done!";

will result in:

 Parent
 Child1
 Child2
 Child2-1
 Child2-2
 Child3
 All done!

=cut

sub depth_reset_children {
  my ($this) = @_;
  $this->reset_children();
  for my $child ($this->children) {
    $child->depth_reset_children();
  }
}
sub depth_first_child {
  my ($this) = @_;
  $this->depth_reset_children();
  return $this->next_child();
}
sub depth_next_child {
  my ($this,$child) = @_;

  # start with the current child.
  $child ||= $this->curr_child() or return;

  # if we haven't seen it yet, return it.
  $child->{child_seen}++ or return $child;

  # next, recurse into it to find grand children.
  if (my $gchild = $child->depth_next()) {
    return $gchild;
  }

  # finally, return the next sibling.
  return $child->next_child();
} 

=head2 parent

 instance
 (PSP::parent) parent ()

DESCRIPTION:

Returns the parent of this error object. Note that the object returned
might have a more specific class.

=cut

sub parent {
  my ($this,$parent) = @_;
  defined $parent and $this->{parent} = $parent;
  return $this->{parent};
}

=head2 top_parent

 instance
 (PSP::parent) top_parent ()

DESCRIPTION:

Iterates from parent to parent until it finds a parent with no parent.
This is the top parent.

=cut

sub top_parent {
  my ($this,$parent) = @_;
  $parent ||= $this->{parent} or return;

  while ($parent->{parent}) {
    $parent = $parent->{parent};
  }

  return $parent;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

C<perl>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
