/*
 * File...........: qetharp26.c
 * Author(s)......: Thomas Spatzier <tspat@de.ibm.com>
 *                  (adaptations to 2.6 kernel)
 * originally written by: Frank Pavlic <pavlic@de.ibm.com>
 * 
 * Bugreports.to..: <Linux390@de.ibm.com>
 * (C) IBM Corporation 2001,2004
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include "qetharp26.h"

/*****************************************************
 *            Function implementation                *
 *****************************************************/

static inline void
qeth_hex_dump(unsigned char *buf, int len)
{
	int i;
	
	for (i = 0; i < len; i++) {
		if (i && !(i % 16))
			printf("\n");
		printf("%02x ", *(buf + i));
	}
	printf("\n");
}

static void
show_header() 
{
	printf("%-40.40s%-20.20s%-10.10s%-10.10s\n", 
	       "Address","HWaddress","HWType","Iface");
}

static
void show_entry5(__u8 ipaddr_type, __u8 *ip,
		 struct option_info *opin)
{
	char tmpbuff[32];
	struct in_addr inadr;
	struct hostent *machine=NULL;

	if (ipaddr_type == IP_VERSION_4) { 
		sprintf(tmpbuff,"%d.%d.%d.%d",ip[0],ip[1],ip[2],ip[3]);
		if (!opin->host_resolution) {
			if (inet_aton(tmpbuff,&inadr))
				machine = gethostbyaddr((char *)   
    					  &inadr,sizeof(inadr),AF_INET);
			else
				machine = gethostbyname(tmpbuff);
		}
		if (opin->compact_output==OPTION_INFO_COMPACT_OUTPUT) {
			printf("%s\n",tmpbuff);
		} else {
			if (machine)
				printf("%-40.40s",machine->h_name);
			else
				printf("%-40.40s",tmpbuff);
			printf("%-20.20s%-10.10s%-10.10s\n",
			       "","hiper",opin->dev_name);
		}
	}
}

static int
get_arp_from_hipersockets(struct qeth_arp_query_user_data *udata,
			  struct option_info *opin) 
{
	struct qeth_arp_qi_entry5 *entry;
	struct qeth_arp_qi_entry5_short *entry_s;
	int i;

	if (udata->mask_bits & QETH_QARP_STRIP_ENTRIES) {
		for (i = 0; i < udata->u.no_entries; i++) {
			entry_s = (struct qeth_arp_qi_entry5_short *)
				(((char *)udata) + 6 + i * sizeof(*entry_s));
			show_entry5(entry_s->ipaddr_type, entry_s->ipaddr,opin);
		}
	} else {
		for (i = 0; i < udata->u.no_entries; i++) {
			entry = (struct qeth_arp_qi_entry5 *)
				(((char *)udata) + 6 + i * sizeof(*entry));
			show_entry5(entry->ipaddr_type, entry->ipaddr, opin);
		}
	}
	return 0;
}

static
void show_entry7(__u8 ipaddr_type, __u8 *ip, __u8 *mac,
		 unsigned short flags, struct option_info *opin)
{
	char tmpbuff[32];
	struct in_addr inadr;
	struct hostent *machine=NULL;

	if (ipaddr_type == IP_VERSION_4) { 
		sprintf(tmpbuff,"%d.%d.%d.%d",ip[0],ip[1],ip[2],ip[3]);
		if (!opin->host_resolution) {
			if (inet_aton(tmpbuff,&inadr))
				machine = gethostbyaddr((char *)   
    					  &inadr,sizeof(inadr),AF_INET);
			else
				machine = gethostbyname(tmpbuff);
		}
		if (opin->compact_output==OPTION_INFO_COMPACT_OUTPUT) {
			printf("%s\n",tmpbuff);
		} else {
			if (machine)
				printf("%-40.40s",machine->h_name);
			else
				printf("%-40.40s",tmpbuff);
			sprintf(tmpbuff,"%02x:%02x:%02x:%02x:%02x:%02x",
				mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);
			printf("%-20.20s%-10.10s%-10.10s\n",
			       tmpbuff,
			       (flags==OSACARD_FLAGS)? "ether":
			       (flags==OSA_TR_FLAGS)? "tr":"n/a",
			       opin->dev_name);
		}
	}
}

static int
get_arp_from_osacard(struct qeth_arp_query_user_data *udata,
		     unsigned short flags, struct option_info *opin) 
{
	struct qeth_arp_qi_entry7 *entry;
	struct qeth_arp_qi_entry7_short *entry_s;
	int i;

	if (udata->mask_bits & QETH_QARP_STRIP_ENTRIES) {
		for (i = 0; i < udata->u.no_entries; i++){ 	
			entry_s = (struct qeth_arp_qi_entry7_short *)
				(((char *)udata) + 6 + i * sizeof(*entry_s));
			show_entry7(entry_s->ipaddr_type, entry_s->ipaddr,
				    entry_s->macaddr, flags, opin);
		}
	} else {
		for (i = 0; i < udata->u.no_entries; i++){ 	
			entry = (struct qeth_arp_qi_entry7 *)
				(((char *)udata) + 6 + i * sizeof(*entry));
			show_entry7(entry->ipaddr_type, entry->ipaddr,
				    entry->macaddr, flags, opin);
		}
	}
	return 0;
}

static int
qetharp_purge(struct option_info *opin)
{
	int sd;
 	struct ifreq ifr;

	if (!opin->dev_name) {
		printf("\nError: no interface specified!\n");
		return -1;
	}
		
	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("Socket failed: %m\n");
		return -1;
	}
	strcpy(ifr.ifr_name, opin->dev_name);
	if (ioctl(sd, SIOC_QETH_ARP_FLUSH_CACHE, &ifr) < 0) {
		close(sd);
		perror("\nUnsuccessful");
		return -1;
	}

	return 0;
}

static int
qetharp_add(struct option_info *opin)
{
	int sd;
 	struct ifreq ifr;
	struct qeth_arp_cache_entry arp_entry;
	unsigned int i1,i2,i3,i4,i5,i6,r;

	memset(&arp_entry, 0, sizeof(struct qeth_arp_cache_entry));
	if (!opin->dev_name) {
		printf("\nError: no interface specified!\n");
		return -1;
	}
	if (!opin->ip_addr) {
		printf("\nError: no ip address specified!\n");
		return -1;
	}
	r=sscanf(opin->ip_addr,"%u.%u.%u.%u",&i1,&i2,&i3,&i4);
	if ( (r!=4) || (i1>255) || (i2>255) || (i3>255) || (i4>255) ) {
		printf("\nError: invalid ip address specified!\n");
		return -1;
	}
	arp_entry.ipaddr[0]=i1;
	arp_entry.ipaddr[1]=i2;
	arp_entry.ipaddr[2]=i3;
	arp_entry.ipaddr[3]=i4;
	
	if (!opin->mac_addr) {
		printf("\nError: no MAC address specified!\n");
		return -1;
	}
	r=sscanf(opin->mac_addr,"%x:%x:%x:%x:%x:%x",&i1,&i2,&i3,&i4,&i5,&i6);
	if ( (r!=6) || (i1>255) || (i2>255) || (i3>255) || 
	     (i4>255) || (i5>255) || (i6>255) ) {
		printf("\nError: invalid MAC address specified!\n");
		return -1;
	}
	arp_entry.macaddr[0]=i1;
	arp_entry.macaddr[1]=i2;
	arp_entry.macaddr[2]=i3;
	arp_entry.macaddr[3]=i4;
	arp_entry.macaddr[4]=i5;
	arp_entry.macaddr[5]=i6;

	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("Socket failed: %m\n");
		return -1;
	}
	strcpy(ifr.ifr_name, opin->dev_name);
	ifr.ifr_ifru.ifru_data = (void*)&arp_entry;

	if (ioctl(sd, SIOC_QETH_ARP_ADD_ENTRY, &ifr) < 0) {
		close(sd);
		perror("\nUnsuccessful");
		return -1;
	}

	return 0;
}

static int
qetharp_delete(struct option_info *opin)
{
	int sd;
 	struct ifreq ifr;
	struct qeth_arp_cache_entry arp_entry;
	unsigned int i1,i2,i3,i4,r;

	memset(&arp_entry,0,sizeof(struct qeth_arp_cache_entry));
	if (!opin->dev_name) {
		printf("\nError: no interface specified!\n");
		return -1;
	}
	if (!opin->ip_addr) {
		printf("\nError: no ip address specified!\n");
		return -1;
	}
	r=sscanf(opin->ip_addr,"%u.%u.%u.%u",&i1,&i2,&i3,&i4);
	if ( (r!=4) || (i1>255) || (i2>255) || (i3>255) || (i4>255) ) {
		printf("\nError: invalid ip address specified!\n");
		return -1;
	}
	arp_entry.ipaddr[0]=i1;
	arp_entry.ipaddr[1]=i2;
	arp_entry.ipaddr[2]=i3;
	arp_entry.ipaddr[3]=i4;
	
	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("Socket failed: %m\n");
		return -1;
	}
	strcpy(ifr.ifr_name, opin->dev_name);
	ifr.ifr_ifru.ifru_data = (void*)&arp_entry;

	if (ioctl(sd, SIOC_QETH_ARP_REMOVE_ENTRY, &ifr) < 0) {
		close(sd);
		perror("\nUnsuccessful");
		return -1;
	}

	return 0;
}

static int
qetharp_query(struct option_info *opin)
{
	int sd;
 	struct ifreq ifr;
	struct qeth_arp_query_user_data *udata;
	int memsize,result;
	unsigned short mask_bits;

	if (!opin->dev_name) {
		printf("\nError: no interface specified!\n");
		return -1;
	}
		
	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("Socket failed: %m\n");
		return -1;
	}
	strcpy(ifr.ifr_name, opin->dev_name);
	memsize = QETH_QARP_USER_DATA_SIZE;
	udata = malloc(QETH_QARP_USER_DATA_SIZE);
	memcpy(&udata->u.data_len, &memsize, sizeof(int));
	udata->mask_bits = QETH_QARP_STRIP_ENTRIES;
	ifr.ifr_ifru.ifru_data = (char *) udata;
	if (ioctl(sd, SIOC_QETH_ARP_QUERY_INFO, &ifr) < 0) {
		close(sd);
		perror("\nUnsuccessful");
		return -1;
	}
	if (opin->compact_output!=OPTION_INFO_COMPACT_OUTPUT) {
		show_header();
	}
	mask_bits = udata->mask_bits & QETH_QARP_REQUEST_MASK;
	if (mask_bits == HIPERSOCKET_FLAGS) 
	        result = get_arp_from_hipersockets(udata, opin);
	else if (mask_bits == OSACARD_FLAGS)
		result = get_arp_from_osacard(udata, mask_bits, opin);
	else if (mask_bits == OSA_TR_FLAGS)
		result = get_arp_from_osacard(udata, mask_bits, opin);
	else {
		perror("\nReceived entries with invalid format");
		return -1;
	}
	free(udata);

	return result;
}

static void
qetharp_usage(void)
{
	printf("qetharp [-[nc]q interface]|[-p interface]|\n" \
	       "\t\t[-a interface -i ip-addr -m MAC-addr]|\n" \
	       "\t\t[-d interface -i ip-addr] [-h] [-v ]\n\n");
	printf("where:\n" \
	       "\tq: prints ARP entries found on the card\n" \
	       "\tn: in conjunction with the -q option it shows\n" \
	       "\t\tnumerical addresses instead of trying to\n" \
	       "\t\tresolve IP addresses to host names.\n" \
	       "\tc: in conjuction with the -q option it shows\n" \
	       "\t\tonly numerical addresses without any\n" \
	       "\t\tother information.\n" \
	       "\tp: flushes the ARP table of the card\n" \
	       "\ta: add static ARP entry\n" \
	       "\td: delete static ARP entry\n" \
	       "\tv: prints version information\n"
	       "\th: prints this usage information\n");
}

static int
qetharp_parse_info(struct option_info *opin)
{
	if ((opin->purge_flag+opin->query_flag+
	    opin->add_flag+opin->delete_flag)!=1) {
		printf("\nOnly use one of the options '-a', " \
		       "'-d', '-p' and 'q' at a time.\n");
		return -1;	
	}
	if (opin->purge_flag &&
	    (opin->query_flag || opin->host_resolution)) {
		printf("\nError in using '-p' option:\n" \
			"\tYou can not use '-p' option in conjunction with " \
			"'-q' or '-n'.\n");
		return -1;
	}
	if (opin->purge_flag) {
		return qetharp_purge(opin);
	}
	if ((opin->host_resolution) && 
	    !(opin->query_flag)) {
		printf("\nError in using '-n' option:\n" \
		       "\t'-q' option missing!\n");
		return -1;
	}
	if (opin->query_flag) {
		return qetharp_query(opin);
	}
	if (opin->add_flag) {
		if ((!opin->ip_flag)||(!opin->mac_flag)) {
			printf("\nError in using '-a' option:\n" \
			       "\t'-i' or '-m' option missing!\n");
			return -1;
		}
		return qetharp_add(opin);
	}
	if (opin->delete_flag) {
		if (!opin->ip_flag) {
			printf("\nError in using '-d' option:\n" \
			       "\t'-i' option missing!\n");
			return -1;
		}
		return qetharp_delete(opin);
	}
	return 0;
}

 
int main(int argc, char **argv) 
{
	
	int index,c,result;
	struct option_info info;

	opterr=0;
	result=0;


	memset(&info, 0, sizeof(info));
	while (1)
	{
		c = getopt_long(argc, argv,QETHARP_GETOPT_STRING,
				qetharp_options,&index);
		
		if (c==-1)
			break;

		switch (c) 
		{
		case '?':	
		case 'h':
		        qetharp_usage();
			exit(0);
		case 'v':
			printf("\nqetharp version 0.02\n");
			exit(0);
		case 'q':
			info.dev_name = optarg;
			info.query_flag =  OPTION_INFO_QUERY;
			break;
		case 'n':
			info.host_resolution =  OPTION_INFO_HOST_RESOLUTION;
			break;
		case 'p':
			info.dev_name = optarg;
			info.purge_flag = OPTION_INFO_PURGE;
			break;
		case 'c':
			info.compact_output = OPTION_INFO_COMPACT_OUTPUT;
			break;
		case 'a':
			info.dev_name = optarg;
			info.add_flag = OPTION_INFO_ADD;
			break;
		case 'd':
			info.dev_name = optarg;
			info.delete_flag = OPTION_INFO_DELETE;
			break;
		case 'i':
			info.ip_addr = optarg;
			info.ip_flag = OPTION_INFO_IP;
			break;
		case 'm':
			info.mac_addr = optarg;
			info.mac_flag = OPTION_INFO_MAC;
			break;
		default:
			qetharp_usage();
			exit(0);
		}
	}
	result = qetharp_parse_info(&info);
	return result;
}

