/*
 * s390-tools/zipl/src/procpart.c
 *   Scanner for the /proc/partitions file
 *
 * Copyright (C) 2003 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * Author(s): Peter Oberparleiter <Peter.Oberparleiter@de.ibm.com>
 */

#include "procpart.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>


static const char proc_partition_filename[] = "/proc/partitions";

struct part_file {
	char* buffer;
	off_t pos;
	size_t length;
};


#define INITIAL_FILE_BUFFER_SIZE	1024

/* Read file into buffer without querying its size (necessary for reading files
 * from /proc). Upon success, return zero and set BUFFER to point to the
 * file buffer and SIZE to contain the file size. Return non-zero otherwise. */
static int
read_file(const char* filename, char** buffer, size_t* size)
{
	FILE* file;
	char* data;
	char* new_data;
	size_t count;
	size_t current_size;
	int current;

	file = fopen(filename, "r");
	if (file == NULL)
		return -1;
	current_size = INITIAL_FILE_BUFFER_SIZE;
	count = 0;
	data = (char *) malloc(current_size);
	if (data == NULL) {
		fclose(file);
		return -1;
	}
	current = fgetc(file);
	while (current != EOF) {
		data[count++] = (char) current;
		if (count >= current_size) {
			new_data = (char *) malloc(current_size * 2);
			if (new_data == NULL) {
				free(data);
				fclose(file);
				return -1;
			}
			memcpy(new_data, data, current_size);
			free(data);
			data = new_data;
			current_size *= 2;
		}
		current = fgetc(file);
	}
	fclose(file);
	*buffer = data;
	*size = count;
	return 0;
}


/* Get the contents of /proc/partitions and fill in the respective fields of
 * FILE. Return 0 on success, non-zero otherwise. */
static int
get_part_file(struct part_file* file)
{
	int rc;

	rc = read_file(proc_partition_filename, &file->buffer, &file->length);
	file->pos = 0;
	return rc;
}


/* Free resources allocated for /proc/partitions file buffer identified by
 * FILE. */
static void
free_part_file(struct part_file* file)
{
	if (file->buffer != NULL) {
		free(file->buffer);
		file->buffer = NULL;
		file->pos = 0;
		file->length = 0;
	}
}


/* Return character at current FILE buffer position or EOF if at end of
 * file. */
static int
current_char(struct part_file* file)
{
	if (file->buffer != NULL)
		if (file->pos < (off_t) file->length)
			return file->buffer[file->pos];
	return EOF;
}


/* Advance the current file pointer of file buffer FILE until the current
 * character is no longer a whitespace. Return 0 if at least one whitespace
 * character was encountered, non-zero otherwise. */
static int
skip_whitespaces(struct part_file* file)
{
	int rc;

	rc = -1;
	while (isspace(current_char(file))) {
		rc = 0;
		file->pos++;
	}
	return rc;
}


/* Scan a positive integer number at the current position of file buffer FILE
 * and advance the position respectively. Upon success, return zero and set
 * NUMBER to contain the scanned number. Return non-zero otherwise. */
static int
scan_number(struct part_file* file, size_t* number)
{
	int rc;
	size_t old_number;

	*number=0;
	rc = -1;
	while (isdigit(current_char(file))) {
		rc = 0;
		old_number = *number;
		*number = *number * 10 + current_char(file) - '0';
		/* Check for overflow */
		if (old_number > *number) {
			rc = -1;
			break;
		}
		file->pos++;
	}
	return rc;
}


/* Scan a device node name at the current position of file buffer FILE and
 * advance the position respectively. Upon success, return zero and set
 * NAME to contain a copy of the scanned name. Return non-zero otherwise. */
static int
scan_name(struct part_file* file, char** name)
{
	off_t start_pos;

	start_pos = file->pos;
	while (!isspace(current_char(file)) &&
	       (current_char(file) != EOF))
			file->pos++;
	if (file->pos > start_pos) {
		*name = (char *) malloc(file->pos - start_pos + 1);
		if (*name == NULL)
			return -1;
		memcpy((void *) *name, (void *) &file->buffer[start_pos],
		       file->pos - start_pos);
		(*name)[file->pos - start_pos] = 0;
		return 0;
	} else
		return -1;
}


/* Advance the current file position to beginning of next line in file buffer
 * FILE or to end of file. */
static void
skip_line(struct part_file* file)
{
	while ((current_char(file) != '\n') && (current_char(file) != EOF))
		file->pos++;
	if (current_char(file) == '\n')
		file->pos++;
}


/* Return non-zero if the current file position of file buffer FILE is at the
 * end of file. Return zero otherwise. */
static int
eof(struct part_file* file)
{
	return file->pos >= (off_t) file->length;
}


/* Scan a line of the specified /proc/partitions FILE buffer and advance the
 * current file position pointer respectively. If the current line matches
 * the correct pattern, fill in the corresponding data into ENTRY and return 0.
 * Return non-zero otherwise. */
static int
scan_entry(struct part_file* file, struct procpart_entry* entry)
{
	int rc;
	size_t dev_major;
	size_t dev_minor;
	size_t blockcount;
	char* name;

	/* Scan for: (\s*)(\d+)(\s+)(\d+)(\s+)(\d+)(\s+)(\S+)(\.*)$ */
	skip_whitespaces(file);
	rc = scan_number(file, &dev_major);
	if (rc)
		return rc;
	rc = skip_whitespaces(file);
	if (rc)
		return rc;
	rc = scan_number(file, &dev_minor);
	if (rc)
		return rc;
	rc = skip_whitespaces(file);
	if (rc)
		return rc;
	rc = scan_number(file, &blockcount);
	if (rc)
		return rc;
	rc = skip_whitespaces(file);
	if (rc)
		return rc;
	rc = scan_name(file, &name);
	if (rc)
		return rc;
	skip_line(file);
	entry->device = makedev(dev_major, dev_minor);
	entry->blockcount = blockcount;
	entry->name = name;
	return 0;
}


/* Release resources associated with ENTRY. */
void
procpart_free_entry(struct procpart_entry* entry)
{
	if (entry->name != NULL) {
		free(entry->name);
		entry->name = NULL;
	}
}


/* Scan /proc/partitions for an entry matching DEVICE. When there is a match,
 * store entry data in ENTRY and return 0. Return non-zero otherwise. */
int
procpart_get_entry(dev_t device, struct procpart_entry* entry)
{
	struct part_file file;
	int rc;

	rc = get_part_file(&file);
	if (rc)
		return rc;
	rc = -1;
	while (!eof(&file)) {
		if (scan_entry(&file, entry) == 0) {
			if (entry->device == device) {
				rc = 0;
				break;
			}
			procpart_free_entry(entry);
		} else
			skip_line(&file);
	}
	free_part_file(&file);
	return rc;
}
