// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ellipseannulus.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

EllipseAnnulus::EllipseAnnulus(const EllipseAnnulus& a) : BaseEllipse(a) {}

EllipseAnnulus::EllipseAnnulus(FrameBase* p,
			       const Vector& ctr, const Vector& r, 
			       double a, 
			       const char* clr, int w, const char* f, 
			       const char* t, unsigned short prop, 
			       const char* c, const List<Tag>& tag)
  : BaseEllipse(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = 1;
  radii = new Vector[1];

  radii[0] = r;

  strcpy(type,"ellipse annulus");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

EllipseAnnulus::EllipseAnnulus(FrameBase* p, const Vector& ctr, 
			       const Vector& inner, const Vector& outer, 
			       int num,
			       double a, 
			       const char* clr, int w, const char* f,
			       const char* t, unsigned short prop,
			       const char* c, const List<Tag>& tag)
  : BaseEllipse(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = num+1;
  radii = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = ((outer-inner)/num)*i+inner;

  strcpy(type, "ellipse annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

EllipseAnnulus::EllipseAnnulus(FrameBase* p,
			       const Vector& ctr, double a, int an,
			       Vector* r,
			       const char* clr, int w, const char* f, 
			       const char* t, unsigned short prop,
			       const char* c, const List<Tag>& tag)
  : BaseEllipse(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = an;
  radii = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  strcpy(type, "ellipse annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void EllipseAnnulus::edit(const Vector& v, int h)
{
  Vector vv = v-center;
  Vector n = (vv * flip * Rotate(-angle));

  if (h<5) {
    // don't go thru the center
    if (vv[0]!=0 && vv[1]!=0) {
      Vector o = radii[annuli-1];
      for (int i=0; i<annuli; i++) {
	radii[i][0] *= fabs(n[0]/o[0]);
	radii[i][1] *= fabs(n[1]/o[1]);
      }
    }
  }
  else {
    // we must have some length
    double l = n.length();
    if (l>0)
      radii[h-5] *= l/radii[h-5][0];
  }

  updateBBox();
  doCallBack(&editCB);
}

void EllipseAnnulus::editEnd()
{
  sortRadii();

  updateBBox();
  doCallBack(&editCB);
}

void EllipseAnnulus::setRadii(const Vector& inner, const Vector& outer,
			      int num)
{
  annuli = num+1;
  if (radii)
    delete [] radii;
  radii = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = ((outer-inner)/num)*i+inner;
  sortRadii();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void EllipseAnnulus::setRadii(const Vector* r, int num)
{
  annuli = num;
  if (radii)
    delete [] radii;
  radii = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

// private

void EllipseAnnulus::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the radii have been sorted yet

  Vector max;
  for(int i=0; i<annuli; i++)
    if (max[0]<radii[i][0])
      max = radii[i];

  Vector r = max;
  if (properties & FIXED)
    r /= parent->getZoom();

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  handle[0] = Vector(-r[0],-r[1]) * m;
  handle[1] = Vector( r[0],-r[1]) * m;
  handle[2] = Vector( r[0], r[1]) * m;
  handle[3] = Vector(-r[0], r[1]) * m;

  // the rest are radii
  {
    for (int i=0; i<annuli; i++) {
      Vector rr = Vector(radii[i][0],0) * m;
      if (properties & FIXED)
	rr /= parent->getZoom();
      handle[i+4] = rr;
    }
  }
}

void EllipseAnnulus::sortRadii()
{
  for (int i=0; i<annuli; i++)
    for (int j=i+1; j<annuli; j++)
      if (radii[i][0]>radii[j][0]) {
	Vector d = radii[i];
	radii[i] = radii[j];
	radii[j] = d;
      }
}

int EllipseAnnulus::addRadii(const Vector& v)
{
  double l = (v * Translate(-center) * flip * Rotate(-angle)).length();
  if (l>0) {

    // new radii array
    Vector* old = radii;
    radii = new Vector[annuli+1];

    // copy old values
    for (int i=0; i<annuli; i++)
      radii[i] = old[i];

    // delete old
    if (old)
      delete [] old;

    // new radii on end
    radii[annuli] = radii[0] * (l/radii[0][0]);
    annuli++;

    numHandle = 4 + annuli;
    delete [] handle;
    handle = new Vector[numHandle];

    updateBBox();
  }

  // return handle number
  return annuli+4;
}

void EllipseAnnulus::deleteRadii(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (annuli>2 && hh<annuli) {
      // new radii array
      Vector* old = radii;
      radii = new Vector[annuli-1];

      // copy up to radii in question
      {
	for (int i=0; i<hh; i++)
	  radii[i] = old[i];
      }

      // copy remainder
      {
	for (int i=hh; i<annuli-1; i++)
	  radii[i] = old[i+1];
      }

      if (old)
	delete [] old;
      annuli--;

      numHandle = 4 + annuli;
      delete [] handle;
      handle = new Vector[numHandle];

      updateBBox();
      doCallBack(&editCB);
    }
  }
}

// list

void EllipseAnnulus::list(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << "ellipse(" << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<annuli; i++) {
	Vector r = ptr->mapLenFromRef(radii[i],sys);
	str << r[0] << ',' << r[1] << ',' ;
      }
      str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << "ellipse(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	    for (int i=0; i<annuli; i++) {
	      Vector r = ptr->mapLenFromRef(radii[i],sys,ARCSEC);
	      str << r[0] << "\"" << ',' << r[1] << "\"" << ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;

	    str << "ellipse(" << ra << ',' << dec << ',' ;
	    for (int i=0; i<annuli; i++) {
	      Vector r = ptr->mapLenFromRef(radii[i],sys,ARCSEC);
	      str << r[0] << "\""<< ',' << r[1] << "\""<< ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << "ellipse(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	for (int i=0; i<annuli; i++) {
	  Vector r = ptr->mapLenFromRef(radii[i],sys);
	  str << r[0] << ',' << r[1] << ',' ;
	}
	str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void EllipseAnnulus::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
			      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int i=0; i<annuli; i++) {
	listProsCoordSystem(str,sys,sky);
	str << "; ";

	Vector r = ptr->mapLenFromRef(radii[i],IMAGE);
	str << "ellipse " << setprecision(8) << v << r << radToDeg(angle);

	if (i!=0) {
	  Vector r1 = ptr->mapLenFromRef(radii[i-1],IMAGE);
	  str << " & !ellipse " << setprecision(8) << v << r1 
	      << radToDeg(angle);
	}

	str << delim;
      }
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  for (int i=0; i<annuli; i++) {
	    listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(radii[i],sys,ARCSEC);
	    str << "ellipse " << setprecision(8) << v[0] << "d " << v[1]<<"d " 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(radii[i-1],sys,ARCSEC);
	      str << " & !ellipse " << setprecision(8) 
		  << v[0] << "d " << v[1] << "d " 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    str << delim;
	  }
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;

	  for (int i=0; i<annuli; i++) {
	    listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(radii[i],sys,ARCSEC);
	    str << " ellipse " << ra << ' ' << dec << ' ' 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(radii[i-1],sys,ARCSEC);
	      str << " & !ellipse " << ra << ' ' << dec << ' ' 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    str << delim;
	  }
	}
	break;
      }
    }
    else
      str << "";
    break;
  }
}

void EllipseAnnulus::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  //  if (!(properties&INCLUDE))
  //    str << '-';

  for (int i=0; i<annuli; i++) {
    Vector v = ptr->mapFromRef(center,IMAGE);
    str << "ellipse(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
	<< radii[i][0] << ',' << radii[i][1] << ',' << radToDeg(angle) << ')';

    if (i!=0)
      str << " & !ellipse(" << setprecision(8) << v[0] << ',' << v[1] 
	  << ',' << radii[i-1][0] << ',' << radii[i-1][1] << ',' 
	  << radToDeg(angle) << ')';

    str << delim;
  }
}
