#charset "us-ascii"

/* Copyright 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - English (United States variant) implementation
 *   
 *   This defines the parts of the TADS 3 library that are specific to the
 *   English language as spoken (and written) in the United States.
 *   
 *   We have attempted to isolate here the parts of the library that are
 *   language-specific, so that translations to other languages or
 *   dialects can be created by replacing this module, without changing
 *   the rest of the library.
 *   
 *   In addition to this module, a separate set of US English messages are
 *   defined in the various msg_xxx.t modules.  Those modules define
 *   messages in English for different stylistic variations.  For a given
 *   game, the author must select one of the message modules - but only
 *   one, since they all define variations of the same messages.  To
 *   translate the library, a translator must create at least one module
 *   defining those messages as well; only one message module is required
 *   per language.
 *   
 *.                                  -----
 *   
 *   "Watch an immigrant struggling with a second language or a stroke
 *   patient with a first one, or deconstruct a snatch of baby talk, or
 *   try to program a computer to understand English, and ordinary speech
 *   begins to look different."
 *   
 *.         Stephen Pinker, "The Language Instinct"
 */

#include "tads.h"
#include "tok.h"
#include "adv3.h"
#include "en_us.h"
#include <vector.h>
#include <dict.h>
#include <gramprod.h>
#include <strcomp.h>


/* ------------------------------------------------------------------------ */
/*
 *   Simple yes/no confirmation.  The caller must display a prompt; we'll
 *   read a command line response, then return true if it's an affirmative
 *   response, nil if not.  
 */
yesOrNo()
{
    /* switch to no-command mode for the interactive input */
    "<.commandnone>";

    /* 
     *   Read a line of input.  Do not allow real-time event processing;
     *   this type of prompt is used in the middle of a command, so we
     *   don't want any interruptions.  Note that the caller must display
     *   any desired prompt, and since we don't allow interruptions, we
     *   won't need to redisplay the prompt, so we pass nil for the prompt
     *   callback.  
     */
    local str = inputManager.getInputLine(nil, nil);

    /* switch back to mid-command mode */
    "<.commandmid>";

    /* 
     *   If they answered with something starting with 'Y', it's
     *   affirmative, otherwise it's negative.  In reading the response,
     *   ignore any leading whitespace.  
     */
    return rexMatch('<space>*[yY]', str) != nil;
}

/* ------------------------------------------------------------------------ */
/*
 *   During start-up, install a case-insensitive truncating comparator in
 *   the main dictionary.
 */
PreinitObject
    execute()
    {
        /* create and set the main dictionary's comparator */
        G_dict.setComparator(new StringComparator(6, nil, []));
    }

    /* 
     *   Make sure we run BEFORE the main library preinitializer, so that
     *   we install the comparator in the dictionary before we add the
     *   vocabulary words to the dictionary.  This doesn't make any
     *   difference in terms of the correctness of the dictionary, since
     *   the dictionary will automatically rebuild itself whenever we
     *   install a new comparator, but it makes the preinitialization run
     *   a tiny bit faster by avoiding that rebuild step. 
     */
    execAfterMe = [adv3LibPreinit]
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific globals 
 */
languageGlobals: object
    /*
     *   The character to use to separate groups of digits in large
     *   numbers.  US English uses commas; most Europeans use periods.
     *   
     *   Note that this setting does not affect system-level BigNumber
     *   formatting, but this information can be passed when calling
     *   BigNumber formatting routines.  
     */
    digitGroupSeparator = ','

    /*
     *   The decimal point to display in floating-point numbers.  US
     *   English uses a period; most Europeans use a comma.
     *   
     *   Note that this setting doesn't affect system-level BigNumber
     *   formatting, but this information can be passed when calling
     *   BigNumber formatting routines.  
     */
    decimalPointCharacter = '.'
;


/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for ThingState. 
 */
modify ThingState
    /*
     *   Our state-specific tokens.  This is a list of words that can be
     *   used in noun phrases in parsed input to refer to this state, and
     *   ONLY to this state - that it, these words must not refer to any
     *   other states among the possible states for an object that can
     *   take on this state.
     *   
     *   If a noun phrase contains a word from this list, it means that
     *   the noun phrase CAN refer to an object in this state, and that
     *   the noun phrase CANNOT refer to an object in any other state that
     *   is possible for objects that can assume this state.  
     */
    stateTokens = []

    /*
     *   Match the name of an object in this state.  We'll check the token
     *   list for any words that apply only to *other* states the object
     *   can assume; if we find any, we'll reject the match, since the
     *   phrase must be referring to an object in a different state. 
     */
    matchName(obj, origTokens, adjustedTokens, states)
    {
        /* 
         *   scan our state list, looking for tokens that belong to other
         *   states 
         */
        foreach (local st in states)
        {
            /* 
             *   if this is our own state, skip it - we obviously want to
             *   allow words associated with our own state 
             */
            if (st == self)
                continue;

            /* 
             *   if we can find any tokens associated with this other
             *   state, then the noun phrase must be referring to some
             *   other object, because we're not in the state they're
             *   talking about 
             */
            if (st.findStateToken(adjustedTokens))
                return nil;
        }

        /* 
         *   we didn't find any words from other states, so we have no
         *   objection to matching this object 
         */
        return obj;
    }

    /*
     *   Check a token list for any tokens matching any of our
     *   state-specific words.  Returns true if we find any such words,
     *   nil if not.
     *   
     *   'toks' is the *adjusted* token list used in matchName().  
     */
    findStateToken(toks)
    {
        /* 
         *   Scan the token list for a match to any of our state-specific
         *   words.  Since we're using the adjusted token list, every
         *   other entry is a part of speech, so work through the list in
         *   pairs.  
         */
        for (local i = 1, local len = toks.length() ; i <= len ; i += 2)
        {
            /* 
             *   if this token matches any of our state tokens, indicate
             *   that we found a match 
             */
            if (stateTokens.indexWhich({x: x == toks[i]}) != nil)
                return true;
        }

        /* we didn't find a match */
        return nil;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for VocabObject. 
 */
modify VocabObject
    /* 
     *   The vocabulary initializer string for the object - this string
     *   can be initialized (most conveniently via a template) to a string
     *   of this format:
     *   
     *   'adj adj adj noun/noun/noun*plural plural plural'
     *   
     *   The noun part of the string can be a hyphen, '-', in which case
     *   it means that the string doesn't specify a noun or plural at all.
     *   This can be useful when nouns and plurals are all inherited from
     *   base classes, and only adjectives are to be specified.  (In fact,
     *   any word that consists of a single hyphen will be ignored, but
     *   this is generally only useful for the adjective-only case.)
     *   
     *   During preinitialization, we'll parse this string and generate
     *   dictionary entries and individual vocabulary properties for the
     *   parts of speech we find.  
     */
    vocabWords_ = ''

    /*
     *   On dynamic construction, initialize our vocabulary words and add
     *   them to the dictionary. 
     */
    construct()
    {
        /* initialize our vocabulary words from vocabWords_ */
        initializeVocab();

        /* add our vocabulary words to the dictionary */
        addToDictionary(&noun);
        addToDictionary(&adjective);
        addToDictionary(&plural);
        addToDictionary(&possessiveAdj);
        addToDictionary(&possessiveNoun);
    }

    /* add the words from a dictionary property to the global dictionary */
    addToDictionary(prop)
    {
        /* if we have any words defined, add them to the dictionary */
        if (self.(prop) != nil)
            G_dict.addWord(self, self.(prop), prop);
    }

    /* initialize the vocabulary from vocabWords_ */
    initializeVocab()
    {
        /* inherit vocabulary from this class and its superclasses */
        inheritVocab(self, new Vector(10));
    }

    /* 
     *   Inherit vocabulary from this class and its superclasses, adding
     *   the words to the given target object.  'target' is the object to
     *   which we add our vocabulary words, and 'done' is a vector of
     *   classes that have been visited so far.
     *   
     *   Since a class can be inherited more than once in an inheritance
     *   tree (for example, a class can have multiple superclasses, each
     *   of which have a common base class), we keep a vector of all of
     *   the classes we've visited.  If we're already in the vector, we'll
     *   skip adding vocabulary for this class or its superclasses, since
     *   we must have already traversed this branch of the tree from
     *   another subclass.  
     */
    inheritVocab(target, done)
    {
        /* 
         *   if we're in the list of classes handled already, don't bother
         *   visiting me again 
         */
        if (done.indexOf(self) != nil)
            return;

        /* add myself to the list of classes handled already */
        done.append(self);
        
        /* add words from our vocabWords_ to the target object */
        target.initializeVocabWith(vocabWords_);

        /* add vocabulary from each of our superclasses */
        foreach (local sc in getSuperclassList())
            sc.inheritVocab(target, done);
    }

    /* initialize our vocabulary from the given string */
    initializeVocabWith(str)
    {
        local sectPart;

        /* start off in the adjective section */
        sectPart = &adjective;

        /* scan the string until we run out of text */        
        while (str != '')
        {
            local len;
            local cur;

            /* find the next delimiter */
            len = rexMatch('<^space|star|/>*', str);

            /* if there's no match, use the whole rest of the string */
            if (len == nil)
                len = str.length();

            /* if there's anything before the delimiter, extract it */
            if (len != 0)
            {
                /* extract the part up to but not including the delimiter */
                cur = str.substr(1, len);

                /* 
                 *   if we're in the adjectives, and either this is the
                 *   last token or the next delimiter is not a space, this
                 *   is implicitly a noun 
                 */
                if (sectPart == &adjective
                    && (len == str.length()
                        || str.substr(len + 1, 1) != ' '))
                {
                    /* move to the noun section */
                    sectPart = &noun;
                }

                /* 
                 *   if the word isn't a single hyphen (in which case it's
                 *   a null word placeholder, not an actual vocabulary
                 *   word), add it to our own appropriate part-of-speech
                 *   property and to the dictionary
                 */
                if (cur != '-')
                {
                    /* add the word to our own list for this part of speech */
                    if (self.(sectPart) == nil)
                        self.(sectPart) = [cur];
                    else
                        self.(sectPart) += cur;
                    
                    /* add it to the dictionary */
                    G_dict.addWord(self, cur, sectPart);
                }
            }

            /* if we have a delimiter, see what we have */
            if (len + 1 < str.length())
            {
                /* check the delimiter */
                switch(str.substr(len + 1, 1))
                {
                case ' ':
                    /* stick with the current part */
                    break;
                    
                case '*':
                    /* start plurals */
                    sectPart = &plural;
                    break;
                    
                case '/':
                    /* start alternative nouns */
                    sectPart = &noun;
                    break;
                }

                /* remove the part up to and including the delimiter */
                str = str.substr(len + 2);
            }
            else
            {
                /* we've exhausted the string - we're done */
                break;
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific modifications for Thing.  This class contains the
 *   methods and properties of Thing that need to be replaced when the
 *   library is translated to another language.
 *   
 *   The properties and methods defined here should generally never be
 *   used by language-independent library code, because everything defined
 *   here is specific to English.  Translators are thus free to change the
 *   entire scheme defined here.  
 */
modify Thing
    /*
     *   Flag that this object's name is rendered as a plural (this
     *   applies to both a singular noun with plural usage, such as
     *   "pants" or "scissors," and an object used in the world model to
     *   represent a collection of real-world objects, such as "shrubs").  
     */
    isPlural = nil

    /*
     *   Flag that this is object's name is a "mass noun" - that is, a
     *   noun denoting a continuous (effectively infinitely divisible)
     *   substance or material, such as water, wood, or popcorn; and
     *   certain abstract concepts, such as knowledge or beauty.  Mass
     *   nouns are never rendered in the plural, and use different
     *   determiners than ordinary ("count") nouns: "some popcorn" vs "a
     *   kernel", for example.  
     */
    isMassNoun = nil

    /*
     *   Flags indicating that the object should be referred to with
     *   gendered pronouns (such as 'he' or 'she' rather than 'it').
     *   
     *   Note that these flags aren't mutually exclusive, so it's legal
     *   for the object to have both masculine and feminine usage.  This
     *   can be useful when creating collective objects that represent
     *   more than one individual, for example.  
     */
    isHim = nil
    isHer = nil

    /*
     *   Flag indicating that the object can be referred to with a neuter
     *   pronoun ('it').  By default, this is true if the object has
     *   neither masculine nor feminine gender, but it can be overridden
     *   so that an object has both gendered and ungendered usage.  This
     *   can be useful for collective objects, as well as for cases where
     *   gendered usage varies by speaker or situation, such as animals. 
     */
    isIt
    {
        /* by default, we're an 'it' if we're not a 'him' or a 'her' */
        return !(isHim || isHer);
    }

    /*
     *   Proper name flag.  If this is set, we will not add articles to
     *   the name when referring to the object.  
     */
    isProperName = nil

    /*
     *   The name of the object - this is a string giving the object's
     *   short description, for constructing sentences that refer to the
     *   object by name.  Each instance should override this to define the
     *   name of the object.  This string should not contain any articles;
     *   we use this string as the root to generate various forms of the
     *   object's name for use in different places in sentences.  
     */
    name = ''

    /*
     *   The name of the object, for the purposes of disambiguation
     *   prompts.  This should almost always be the object's ordinary
     *   name, so we return self.name by default.
     *   
     *   In rare cases, it might be desirable to override this.  In
     *   particular, if a game has two objects that are NOT defined as
     *   basic equivalents of one another (which means that the parser
     *   will always ask for disambiguation when the two are ambiguous
     *   with one another), but the two nonetheless have identical 'name'
     *   properties, this property should be overridden for one or both
     *   objects to give them different names.  This will ensure that we
     *   avoid asking questions of the form "which do you mean, the coin,
     *   or the coin?".  In most cases, non-equivalent objects will have
     *   distinct 'name' properties to begin with, so this is not usually
     *   an issue.  
     */
    disambigName = (name)
    theDisambigName = (theName)
    aDisambigName = (aName)
    countDisambigName(cnt) { return countName(cnt); }

    /*
     *   The name of the object, for the purposes of disambiguation prompts
     *   to disambiguation among this object and basic equivalents of this
     *   object (i.e., objects of the same class marked with
     *   isEquivalent=true).
     *   
     *   This is used in disambiguation prompts in place of the actual text
     *   typed by the user.  For example, suppose the user types ">take
     *   coin", then we ask for help disambiguating, and the player types
     *   ">gold".  This narrows things down to, say, three gold coins, but
     *   they're in different locations so we need to ask for further
     *   disambiguation.  Normally, we ask "which gold do you mean",
     *   because the player typed "gold" in the input.  Once we're down to
     *   equivalents, we don't have to rely on the input text any more,
     *   which is good because the input text could be fragmentary (as in
     *   our present example).  Since we have only equivalents, we can use
     *   the actual name of the objects (they're all the same, after all).
     *   This property gives the name we use.
     *   
     *   For English, this is simply the object's ordinary disambiguation
     *   name.  This property is separate from 'name' and 'disambigName'
     *   for the sake of languages that need to use an inflected form in
     *   this context.  
     */
    disambigEquivName = (disambigName)

    /*
     *   Single-item listing description.  This is used to display the
     *   item when it appears as a single (non-grouped) item in a list.
     *   By default, we just show the indefinite article description.  
     */
    listName = (aName)

    /*
     *   Return a string giving a count of the object ("five coins").
     *   'info' is a SenseInfo object giving the sense conditions under
     *   which the object is being described.  
     */
    countName(count)
    {
        /* if the count is one, use 'one' plus the singular name */
        if (count == 1)
            return 'one ' + name;

        /* 
         *   Get the number followed by a space - spell out numbers below
         *   100, but use numerals to denote larger numbers.  Append the
         *   plural name to the number and return the result.  
         */
        return spellIntBelowExt(count, 100, 0, DigitFormatGroupSep)
            + ' ' + pluralName;
    }

    /* 
     *   get a string with the appropriate pronoun for the object for the
     *   nominative case, objective case, possessive adjective, possessive
     *   noun 
     */
    itNom { return isPlural ? 'they' : 'it'; }
    itObj { return isPlural ? 'them' : 'it'; }
    itPossAdj { return isPlural ? 'their' : 'its'; }
    itPossNoun { return isPlural ? 'theirs' : 'its'; }

    /* get the object reflexive pronoun (itself, etc) */
    itReflexive { return isPlural ? 'themselves' : 'itself'; }

    /* demonstrative pronouns ('that' or 'those') */
    thatNom { return isPlural ? 'those' : 'that'; }
    thatObj { return isPlural ? 'those' : 'that'; }

    /*
     *   get a string with the appropriate pronoun for the object plus the
     *   correct conjugation of 'to be' 
     */
    itIs { return isPlural ? 'they are' : 'it is'; }

    /* get a pronoun plus a 'to be' contraction */
    itIsContraction { return isPlural ? 'they\'re' : 'it\'s'; }

    /*
     *   get a string with the appropriate pronoun for the object plus the
     *   correct conjugation of the given regular verb for the appropriate
     *   person 
     */
    itVerb(verb)
    {
        return itNom + ' ' + conjugateRegularVerb(verb);
    }

    /*
     *   Conjugate a regular verb in the present tense for our number.
     *   This is pretty easy: we add an 's' for the third person singular,
     *   and leave the verb unchanged for plural (it asks, they ask).  The
     *   only complication is that we must check some special cases to add
     *   the -s suffix: -y -> -ies (it carries), -o -> -oes (it goes).  
     */
    conjugateRegularVerb(verb)
    {
        /* check our number and person */
        if (isPlural)
        {
            /* we're plural, so simply use the base verb form ("they ask") */
            return verb;
        }
        else
        {
            /* 
             *   Third-person singular, so we must add the -s suffix.
             *   Check for special spelling cases:
             *   
             *   '-y' changes to '-ies'
             *   
             *   '-sh', '-ch', and '-o' endings add suffix '-es'
             */
            if (verb.endsWith('y'))
                return verb.substr(1, verb.length() - 1) + 'ies';
            else if (verb.endsWith('o')
                     || verb.endsWith('ch')
                     || verb.endsWith('sh'))
                return verb + 'es';
            else
                return verb + 's';
        }
    }

    /* 
     *   Get the name with a definite article ("the box").  If my name is
     *   a proper name, don't add an article, since a proper name has
     *   implicitly definite usage in English.
     *   
     *   This is in the nominative case, which makes no difference unless
     *   the name is a pronoun.  
     */
    theName { return (isProperName ? '' : 'the ') + name; }

    /* 
     *   theName in objective case.  In most cases, this is identical to
     *   the normal theName, so we use that by default.  This must be
     *   overridden if theName is a pronoun (which is usually only the
     *   case for player character actors; see our language-specific Actor
     *   modifications for information on that case).  
     */
    theNameObj { return theName; }

    /* 
     *   theName as a possessive adjective (Bob's book, your book) - we'll
     *   add an apostrophe-s suffix for singular-usage names (the book's),
     *   or just an apostrophe for plural-usage names (the shrubs').
     *   
     *   Note that there is some difference of opinion among experts about
     *   the proper usage for singular-usage words that end in 's'.  Some
     *   people like to us a bare apostrophe for any word that ends in 's'
     *   (so Chris -> Chris'); other people use apostrophe-s for singular
     *   words that end in an "s" sound and a bare apostrophe for words
     *   that end in an "s" that sounds like a "z" (so Charles Dickens ->
     *   Charles Dickens').  The usage we encode here is simpler: we add
     *   an apostrophe-s in all cases where our name has singular usage,
     *   and add a bare apostrophe when our name has plural usage.  
     */
    theNamePossAdj
    {
        return theName + (isPlural ? '\'' : '\'s');
    }

    /* 
     *   theName as a possessive noun (that is Bob's, that is yours) - by
     *   default, we'll just add an apostrophe-S (or just an apostrophe,
     *   if we have plural usage) to our name 
     */
    theNamePossNoun
    {
        return theName + (isPlural ? '\'' : '\'s');
    }

    /*
     *   theName with my nominal owner explicitly stated, if we have a
     *   nominal owner: "your backpack," "Bob's flashlight."  If we have
     *   no nominal owner, this is simply my theName.  
     */
    theNameWithOwner()
    {
        local owner;

        /* 
         *   if we have a nominal owner, show with our owner name;
         *   otherwise, just show our regular theName 
         */
        if ((owner = getNominalOwner()) != nil)
            return owner.theNamePossAdj + ' ' + name;
        else
            return theName;
    }

    /*
     *   Get a description of an object within this object, describing the
     *   object's location as this object.  By default, we'll append "in
     *   <theName>" to the given object name.  
     */
    childInName(childName)
        { return childInNameGen(childName, theName); }

    /* 
     *   Get a description of an object within this object, showing the
     *   owner of this object.  This is similar to childInName, but
     *   explicitly shows the owner of the containing object, if any: "the
     *   flashlight in bob's backpack". 
     */
    childInNameWithOwner(childName)
        { return childInNameGen(childName, theNameWithOwner); }

    /*
     *   Base routine for generating childInName and related names.  Takes
     *   the name to use for the child and the name to use for me, and
     *   combines them appropriately.  In most cases, only this routine
     *   needs to be overridden per subclass.  
     */
    childInNameGen(childName, myName)
        { return childName + ' in ' + myName; }

    /*
     *   Get my name (in various forms) with my owner or location
     *   explicitly specified.  If I'm immediately in my nominal owner,
     *   we'll show with the owner name ("bob's flashlight"); otherwise,
     *   we'll show with our immediate container ("the flashlight in the
     *   backpack").  These are used for the ownership/location
     *   distinguisher to list objects according to locations in
     *   disambiguation lists.
     *   
     *   We will presume that objects with proper names are never
     *   indistinguishable from other objects with proper names, so we
     *   won't worry about cases like "Bob's Bill".  This leaves us free
     *   to use appropriate articles in all cases.  
     */
    aNameOwnerLoc()
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil && isDirectlyIn(owner))
        {
            local ret;
            
            /* 
             *   we have an owner - show as "one of Bob's items" (or just
             *   "Bob's items" if this is a mass noun or a proper name) 
             */
            ret = owner.theNamePossAdj + ' ' + pluralName;
            if (!isMassNoun && !isPlural)
                ret = 'one of ' + ret;

            /* return the result */
            return ret;
        }
        else
        {
            /* we have no owner - show as "an item in the location" */
            return location.childInNameWithOwner(aName);
        }
    }
    theNameOwnerLoc()
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil && isDirectlyIn(owner))
        {
            /* we have an owner - show as "Bob's item" */
            return owner.theNamePossAdj + ' ' + name;
        }
        else
        {
            /* we have no owner - show as "the item in the location" */
            return location.childInNameWithOwner(theName);
        }
    }
    countNameOwnerLoc(cnt)
    {
        local owner;

        /* show in owner or location format, as appropriate */
        if ((owner = getNominalOwner()) != nil && isDirectlyIn(owner))
        {
            /* we have an owner - show as "Bob's five items" */
            return owner.theNamePossAdj + ' ' + countName(cnt);
        }
        else
        {
            /* we have no owner - show as "the five items in the location" */
            return location.childInNameWithOwner('the ' + countName(cnt));
        }
    }

    /* 
     *   The indefinite article to use; if this is nil, we'll try to
     *   figure it out by looking at the name. 
     */
    articleIndef = nil

    /* 
     *   Display the name with an indefinite article ("a box").  If an
     *   'articleIndef' property is defined to non-nil, we'll use that
     *   article; otherwise, we'll try to figure it out by looking at the
     *   name property.
     *   
     *   By default, we'll use the article "a" if the name starts with a
     *   consonant, or "an" if it starts with a vowel.
     *   
     *   If the name starts with a "y", we'll look at the second letter;
     *   if it's a consonant, we'll use "an", otherwise "a" (hence "an
     *   yttrium block" but "a yellow brick").
     *   
     *   If the object is marked as having plural usage, we will use
     *   "some" as the article ("some pants" or "some shrubs").
     *   
     *   Some objects will want to override the default behavior, because
     *   the lexical rules about when to use "a" and "an" are not without
     *   exception.  For example, silent-"h" words ("honor") are written
     *   with "an", and "h" words with a pronounced but weakly stressed
     *   initial "h" are sometimes used with "an" ("an historian").  Also,
     *   some 'y' words might not follow the generic 'y' rule.
     *   
     *   'U' words are especially likely not to follow any lexical rule -
     *   any 'u' word that sounds like it starts with 'y' should use 'a'
     *   rather than 'an', but there's no good way to figure that out just
     *   looking at the spelling (consider "a universal symbol" and "an
     *   unimportant word", or "a unanimous decision" and "an unassuming
     *   man").  We simply always use 'an' for a word starting with 'u',
     *   but this will have to be overridden when the 'u' sounds like 'y'.
     *   
     *   Note that this routine can be overridden entirely, but in most
     *   cases, an object can simply override the 'article' property to
     *   specify the article to use.  
     */
    aName
    {
        /* 
         *   The complete list of unaccented, accented, and ligaturized
         *   Latin vowels from the Unicode character set.  (The Unicode
         *   database doesn't classify characters as vowels or the like,
         *   so it seems the only way we can come up with this list is
         *   simply to enumerate the vowels.)
         *   
         *   These are all lower-case letters; all of these are either
         *   exclusively lower-case or have upper-case equivalents that
         *   map to these lower-case letters.
         *   
         *   (Note an implementation detail: the compiler will append all
         *   of these strings together at compile time, so we don't have
         *   to perform all of this concatenation work each time we
         *   execute this method.)
         *   
         *   Note that we consider any word starting with an '8' to start
         *   with a vowel, since 'eight' and 'eighty' both take 'an'.  
         */
        local vowels = '8aeiou\u00E0\u00E1\u00E2\u00E3\u00E4\u00E5\u00E6'
                       + '\u00E8\u00E9\u00EA\u00EB\u00EC\u00ED\u00EE\u00EF'
                       + '\u00F2\u00F3\u00F4\u00F5\u00F6\u00F8\u00F9\u00FA'
                       + '\u00FB\u00FC\u0101\u0103\u0105\u0113\u0115\u0117'
                       + '\u0119\u011B\u0129\u012B\u012D\u012F\u014D\u014F'
                       + '\u0151\u0169\u016B\u016D\u016F\u0171\u0173\u01A1'
                       + '\u01A3\u01B0\u01CE\u01D0\u01D2\u01D4\u01D6\u01D8'
                       + '\u01DA\u01DC\u01DF\u01E1\u01E3\u01EB\u01ED\u01FB'
                       + '\u01FD\u01FF\u0201\u0203\u0205\u0207\u0209\u020B'
                       + '\u020D\u020F\u0215\u0217\u0254\u025B\u0268\u0289'
                       + '\u1E01\u1E15\u1E17\u1E19\u1E1B\u1E1D\u1E2D\u1E2F'
                       + '\u1E4D\u1E4F\u1E51\u1E53\u1E73\u1E75\u1E77\u1E79'
                       + '\u1E7B\u1E9A\u1EA1\u1EA3\u1EA5\u1EA7\u1EA9\u1EAB'
                       + '\u1EAD\u1EAF\u1EB1\u1EB3\u1EB5\u1EB7\u1EB9\u1EBB'
                       + '\u1EBD\u1EBF\u1EC1\u1EC3\u1EC5\u1EC7\u1EC9\u1ECB'
                       + '\u1ECD\u1ECF\u1ED1\u1ED3\u1ED5\u1ED7\u1ED9\u1EDB'
                       + '\u1EDD\u1EDF\u1EE1\u1EE3\u1EE5\u1EE7\u1EE9\u1EEB'
                       + '\u1EED\u1EEF\u1EF1\uFF41\uFF4F\uFF55';

        /*
         *   A few upper-case vowels in unicode don't have lower-case
         *   mappings - consider them separately. 
         */
        local vowelsUpperOnly = '\u0130\u019f';

        /* 
         *   the various accented forms of the letter 'y' - these are all
         *   lower-case versions; the upper-case versions all map to these 
         */
        local ys = 'y\u00FD\u00FF\u0177\u01B4\u1E8F\u1E99\u1EF3'
                   + '\u1EF5\u1EF7\u1EF9\u24B4\uFF59';

        /* if it's a proper name, don't add an article at all */
        if (isProperName)
            return name;

        /* if there's an explicit indefinite article, use it */
        if (articleIndef != nil)
            return articleIndef + ' ' + name;

        /* if it's plural or a mass noun, use "some" as the article */
        if (isPlural || isMassNoun)
        {
            /* use "some" as the article */
            return 'some ' + name;
        }
        else
        {
            local firstChar;
            local firstCharLower;

            /* if it's empty, just use "a" */
            if (name == '')
                return 'a';

            /* get the first character of the name */
            firstChar = name.substr(1, 1);
            firstCharLower = firstChar.toLower();

            /* 
             *   look for the first character in the lower-case and
             *   upper-case-only vowel lists - if we find it, it takes
             *   'an' 
             */
            if (vowels.find(firstCharLower) != nil
                || vowelsUpperOnly.find(firstChar) != nil)
            {
                /* it starts with a vowel */
                return 'an ' + name;
            }
            else if (ys.find(firstCharLower) != nil)
            {
                local secondChar;

                /* get the second character, if there is one */
                secondChar = name.substr(2, 1);

                /* 
                 *   It starts with 'y' - if the second letter is a
                 *   consonant, assume the 'y' is a vowel sound, hence we
                 *   should use 'an'; otherwise assume the 'y' is a
                 *   dipthong 'ei' sound, which means we should use 'a'.
                 *   If there's no second character at all, or the second
                 *   character isn't alphabetic, use 'a' - "a Y" or "a
                 *   Y-connector".  
                 */
                if (secondChar == ''
                    || rexMatch(patIsAlpha, secondChar) == nil
                    || vowels.find(secondChar.toLower()) != nil
                    || vowelsUpperOnly.find(secondChar) != nil)
                {
                    /* 
                     *   it's just one character, or the second character
                     *   is non-alphabetic, or the second character is a
                     *   vowel - in any of these cases, use 'a' 
                     */
                    return 'a ' + name;
                }
                else
                {
                    /* the second character is a consonant - use 'an' */
                    return 'an ' + name;
                }
            }
            else if (rexMatch(patEleven, name) != nil)
            {
                /* it starts with 'eleven' in digit form, so it takes 'an' */
                return 'an ' + name;
            }
            else
            {
                /* it starts with a consonant */
                return 'a ' + name;
            }
        }
    }

    /* pre-compile some regular expressions for aName */
    patIsAlpha = static new RexPattern('<alpha>')
    patEleven = static new RexPattern('11(<^digit>|$)')

    /*
     *   Indefinite article name in objective case 
     */
    aNameObj { return aName; }
    
    /*
     *   Get the default plural name.  If the name ends in anything other
     *   than 'y', we'll add an 's'; otherwise we'll replace the 'y' with
     *   'ies'.  Some objects will have to override this to deal with
     *   irregular plurals ('child' -> 'children' and the like).
     *   
     *   This can only deal with simple adj-noun forms.  For more
     *   complicated forms, it must be overridden (e.g., "Attorney
     *   General" -> "Attorneys General").  
     */
    pluralName
    {
        local len;
        local lastChar;
        local lastPair;

        /* 
         *   if it's marked as having plural usage, just use the ordinary
         *   name, since it's already plural 
         */
        if (isPlural)
            return name;

        /* if there's no short description, return an empty string */
        len = name.length();
        if (len == 0)
            return '';

        /* if it's only one letter long, add an apostrophe-s */
        if (len == 1)
            return name + 's';

        /* get the last character of the name, and the last pair of chars */
        lastChar = name.substr(len, 1);
        lastPair = (len == 1 ? lastChar : name.substr(len - 1, 2));

        /*
         *   If the last letter is a capital letter, it must be an
         *   abbreviation of some kind; add an apostrophe-s.  If it's not
         *   a letter at all, it must be something other than an ordinary
         *   word, so add apostrophe-s in this case as well.  So, if the
         *   last character is anything except a lower-case letter, add
         *   apostrophe-s.  
         */
        if (rexMatch(patIsLower, lastChar) == nil)
            return name + '\'s';

        /* 
         *   if it ends in a non-vowel followed by 'y', change -y to -ies
         *   (if a vowel precedes a terminal 'y', we'll use the normal
         *   '-s' ending: "survey" -> "surveys", "essay" -> "essays",
         *   "day" -> "days") 
         */
        if (rexMatch(patVowelY, lastPair) != nil)
            return name.substr(1, len - 1) + 'ies';

        /* if it ends in s, x, z, or h, add -es */
        if ('sxzh'.find(lastChar) != nil)
            return name + 'es';

        /* for anything else, just add -s */
        return name + 's';
    }

    /* some pre-compiled patterns for pluralName */
    patIsLower = static new RexPattern('<lower>')
    patVowelY = static new RexPattern('[^aeoiu]y')

    /* get my name plus a being verb ("the box is") */
    nameIs { return theName + ' ' + verbToBe; }

    /* get my name plus a negative being verb ("the box isn't") */
    nameIsnt { return nameIs + 'n\'t'; }

    /* 
     *   My name with the given regular verb in agreement: if my name has
     *   singular usage, we'll add 's' to the verb, otherwise we won't.
     *   This can't be used with irregular verbs.  
     */
    nameVerb(verb) { return theName + ' ' + conjugateRegularVerb(verb); }

    /* being verb agreeing with this object as subject */
    verbToBe { return isPlural ? 'are' : 'is'; }

    /* past tense being verb agreeing with object as subject */
    verbWas { return isPlural ? 'were' : 'was'; }

    /* 'have' verb agreeing with this object as subject */
    verbToHave { return isPlural ? 'have' : 'has'; }

    /* 
     *   verb endings for regular '-s', '-es', and '-y/-ies' verbs,
     *   respectively, agreeing with this object as the subject 
     */
    verbEndingS { return isPlural ? '' : 's'; }
    verbEndingEs { return isPlural ? '' : 'es'; }
    verbEndingIes { return isPlural ? 'y' : 'ies'; }

    /* 
     *   Default preposition to use when an actor is in/on this object (as
     *   a nested location), and full prepositional phrase, with no
     *   article and with an indefinite article.  By default, we use 'in'
     *   as the preposition; subclasses can override as desired.  
     */
    actorInPrep = 'in'
    actorIntoPrep
    {
        if (actorInPrep is in ('in', 'on'))
            return actorInPrep + 'to';
        else
            return actorInPrep;
    }
    actorInName = (actorInPrep + ' ' + theNameObj)
    actorIntoName = (actorIntoPrep + ' ' + theNameObj)
    actorInAName = (actorInPrep + ' ' + aNameObj)

    /*
     *   For the most part, "strike" has the same meaning as "hit", so
     *   define this as a synonym for "attack" most objects.  There are a
     *   few English idioms where "strike" means something different, as
     *   in "strike match" or "strike tent."  
     */
    dobjFor(Strike) asDobjFor(Attack)
;

/* ------------------------------------------------------------------------ */
/*
 *   An object that uses the same name as another object.  This maps all
 *   of the properties involved in supplying the object's name, number,
 *   and other usage information from this object to a given target
 *   object, so that all messages involving this object use the same name
 *   as the target object.  This is a mix-in class that can be used with
 *   any other class.
 *   
 *   Note that we map only the *reported* name for the object.  We do NOT
 *   give this object any vocabulary from the other object.
 */
class NameAsOther: object
    /* the target object - we'll use the same name as this object */
    targetObj = nil

    /* map our naming and usage properties to the target object */
    isPlural = (targetObj.isPlural)
    isMassNoun = (targetObj.isMassNoun)
    isHim = (targetObj.isHim)
    isHer = (targetObj.isHer)
    isIt = (targetObj.isIt)
    isProperName = (targetObj.isProperName)
    name = (targetObj.name)

    /* map the derived name properties as well, in case any are overridden */
    disambigName = (targetObj.disambigName)
    theDisambigName = (targetObj.theDisambigName)
    aDisambigName = (targetObj.aDisambigName)
    countDisambigName(cnt) { return targetObj.countDisambigName(cnt); }
    disambigEquivName = (targetObj.disambigEquivName)
    listName = (targetObj.listName)
    countName(cnt) { return targetObj.countName(cnt); }

    /* map the pronoun properites, in case any are overridden */
    itNom = (targetObj.itNom)
    itObj = (targetObj.itObj)
    itPossAdj = (targetObj.itPossAdj)
    itPossNoun = (targetObj.itPossNoun)
    itReflexive = (targetObj.itReflexive)
    thatNom = (targetObj.thatNom)
    thatObj = (targetObj.thatObj)
    itIs = (targetObj.itIs)
    itIsContraction = (targetObj.itIsContraction)
    itVerb(verb) { return targetObj.itVerb(verb); }
    conjugateRegularVerb(verb)
        { return targetObj.conjugateRegularVerb(verb); }
    theName = (targetObj.theName)
    theNameObj = (targetObj.theNameObj)
    theNamePossAdj = (targetObj.theNamePossAdj)
    theNamePossNoun = (targetObj.theNamePossNoun)
    theNameWithOwner = (targetObj.theNameWithOwner)
    childInName(childName) { return targetObj.childInName(childName); }
    childInNameWithOwner(childName)
        { return targetObj.childInNameWithOwner(childName); }
    childInNameGen(childName, myName)
        { return targetObj.childInNameGen(childName, myName); }
    aNameOwnerLoc = (targetObj.aNameOwnerLoc)
    theNameOwnerLoc = (targetObj.theNameOwnerLoc)
    countNameOwnerLoc(cnt) { return targetObj.countNameOwnerLoc(cnt); }
    articleIndef = (targetObj.articleIndef)
    aName = (targetObj.aName)
    aNameObj = (targetObj.aNameObj)
    pluralName = (targetObj.pluralName)
    nameIs = (targetObj.nameIs)
    nameIsnt = (targetObj.nameIsnt)
    nameVerb(verb) { return targetObj.nameVerb(verb); }
    verbToBe = (targetObj.verbToBe)
    verbWas = (targetObj.verbWas)
    verbToHave = (targetObj.verbToHave)

    verbEndingS = (targetObj.verbEndingS)
    verbEndingEs = (targetObj.verbEndingEs)
    verbEndingIes = (targetObj.verbEndingIes)
    actorInPrep = (targetObj.actorInPrep)
    actorIntoPrep = (targetObj.actorIntoPrep)
    actorInName = (targetObj.actorInName)
    actorIntoName = (targetObj.actorIntoName)
    actorInAName = (targetObj.actorInAName)
;

/*
 *   Name as Parent - this is a special case of NameAsOther that uses the
 *   lexical parent of a nested object as the target object.  (The lexical
 *   parent is the enclosing object in a nested object definition; in
 *   other words, it's the object in which the nested object is embedded.)
 */
class NameAsParent: NameAsOther
    targetObj = (lexicalParent)
;


/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for Surface 
 */
modify Surface
    /* 
     *   objects contained in a Surface are described as being on the
     *   Surface
     */
    childInNameGen(childName, myName) { return childName + ' on ' + myName; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for Actor.
 *   
 *   An Actor has a "referral person" setting, which determines how we
 *   refer to the actor; this is almost exclusively for the use of the
 *   player character.  The typical convention is that we refer to the
 *   player character in the second person, but a game can override this
 *   on an actor-by-actor basis.  
 */
modify Actor
    /* by default, use my pronoun for my name */
    name = (itNom)

    /* 
     *   Pronoun selector.  This returns an index for selecting pronouns
     *   or other words based on number and gender, taking into account
     *   person, number, and gender.  The value returned is the sum of the
     *   following components:
     *   
     *   number/gender:
     *.  - singular neutral = 1
     *.  - singular masculine = 2
     *.  - singular feminine = 3
     *.  - plural = 4
     *   
     *   person:
     *.  - first person = 0
     *.  - second person = 4
     *.  - third person = 8
     *   
     *   The result can be used as a list selector as follows (1=first
     *   person, etc; s=singular, p=plural; n=neuter, m=masculine,
     *   f=feminine):
     *   
     *   [1/s/n, 1/s/m, 1/s/f, 1/p, 2/s/n, 2/s/m, 2/s/f, 2/p,
     *.  3/s/n, 3/s/m, 3/s/f, 3/p] 
     */
    pronounSelector
    {
        return ((referralPerson - FirstPerson)*4
                + (isPlural ? 4 : isHim ? 2 : isHer ? 3 : 1));
    }

    /* 
     *   get the verb form selector index for the person and number:
     *   
     *   [1/s, 2/s, 3/s, 1/p, 2/p, 3/p] 
     */
    conjugationSelector
    {
        return (referralPerson + (isPlural ? 3 : 0));
    }

    /* 
     *   get an appropriate pronoun for the object in the appropriate
     *   person for the nominative case, objective case, possessive
     *   adjective, possessive noun, and objective reflexive 
     */
    itNom
    {
        return ['I', 'I', 'I', 'we',
               'you', 'you', 'you', 'you',
               'it', 'he', 'she', 'they'][pronounSelector];
    }
    itObj
    {
        return ['me', 'me', 'me', 'us',
               'you', 'you', 'you', 'you',
               'it', 'him', 'her', 'them'][pronounSelector];
    }
    itPossAdj
    {
        return ['my', 'my', 'my', 'our',
               'your', 'your', 'your', 'your',
               'its', 'his', 'her', 'their'][pronounSelector];
    }
    itPossNoun
    {
        return ['mine', 'mine', 'mine', 'ours',
               'yours', 'yours', 'yours', 'yours',
               'its', 'his', 'hers', 'theirs'][pronounSelector];
    }
    itReflexive
    {
        return ['myself', 'myself', 'myself', 'ourselves',
               'yourself', 'yourself', 'yourself', 'yourselves',
               'itself', 'himself', 'herself', 'themselves'][pronounSelector];
    }

    /*
     *   Demonstrative pronoun, nominative case.  We'll use personal a
     *   personal pronoun if we have a gender or we're in the first or
     *   second person, otherwise we'll use 'that' or 'those' as we would
     *   for an inanimate object.  
     */
    thatNom
    {
        return ['I', 'I', 'I', 'we',
               'you', 'you', 'you', 'you',
               'that', 'he', 'she', 'those'][pronounSelector];
    }

    /* demonstrative pronoun, objective case */
    thatObj
    {
        return ['me', 'me', 'me', 'us',
               'you', 'you', 'you', 'you',
               'that', 'him', 'her', 'those'][pronounSelector];
    }

    /*
     *   get an appropriate pronoun for the object plus the correct
     *   conjugation of 'to be' for the appropriate person 
     */
    itIs
    {
        return itNom
            + [' am', ' are', ' is',
               ' are', ' are', ' are'][conjugationSelector];
    }

    /* get my pronoun with a being verb contraction ("the box's") */
    itIsContraction
    {
        return itNom + '\''
            + ['m', 're', 's', 're', 're', 're'][conjugationSelector];
    }
    
    /*
     *   get an appropriate pronoun for the object plus the correct
     *   conjugation of the given regular verb for the appropriate person 
     */
    itVerb(verb)
    {
        return itNom + ' ' + conjugateRegularVerb(verb);
    }

    /*
     *   Conjugate a regular verb in the present tense for our person and
     *   number.  This is pretty easy: we add an 's' for the third person
     *   singular, and leave the verb unchanged for every other case.  The
     *   only complication is that we must check some special cases to add
     *   the -s suffix: -y -> -ies, -o -> -oes.  
     */
    conjugateRegularVerb(verb)
    {
        /* 
         *   if we're in the third person, inherit the default handling;
         *   otherwise, use the base verb form regardless of number
         *   (regular verbs use the same conjugated forms for every case
         *   but third person singular: I ask, you ask, we ask, they ask).
         */
        if (referralPerson != ThirdPerson)
        {
            /* 
             *   we're not using the third-person, so the conjugation is
             *   the same as the base verb form 
             */
            return verb;
        }
        else
        {
            /* 
             *   we're using the third person, so inherit the base class
             *   handling, which conjugates third-person forms 
             */
            return inherited.conjugateRegularVerb(verb);
        }
    }

    /* 
     *   Get the name with a definite article ("the box").  If the
     *   narrator refers to us in the first or second person, use a
     *   pronoun rather than the short description.  
     */
    theName
    {
        return (referralPerson == ThirdPerson
                ? inherited.theName : ['I', 'you'][referralPerson]);
    }

    /* theName in objective case */
    theNameObj
    {
        return (referralPerson == ThirdPerson
                ? inherited.theNameObj : ['me', 'you'][referralPerson]);
    }

    /* theName as a possessive adjective */
    theNamePossAdj
    {
        return (referralPerson == ThirdPerson
                ? inherited.theNamePossAdj : ['my', 'your'][referralPerson]);
    }

    /* theName as a possessive noun */
    theNamePossNoun
    {
        return (referralPerson == ThirdPerson
                ? inherited.theNamePossNoun
                : ['mine', 'yours'][referralPerson]);
    }

    /*
     *   Get the name with an indefinite article.  Use the same rules of
     *   referral person as for definite articles. 
     */
    aName
    {
        return (referralPerson == ThirdPerson
                ? inherited.aName : ['I', 'you'][referralPerson]);
    }

    /* aName in objective case */
    aNameObj
    {
        return (referralPerson == ThirdPerson
                ? inherited.aNameObj : ['me', 'you'][referralPerson]);
    }

    /* being verb agreeing with this object as subject */
    verbToBe
    {
        return ['am', 'are', 'is', 'are', 'are', 'are'][conjugationSelector]; 
    }

    /* past tense being verb agreeing with this object as subject */
    verbWas
    {
        return ['was', 'were', 'was', 'were', 'were', 'were']
            [conjugationSelector]; 
    }

    /* 'have' verb agreeing with this object as subject */
    verbToHave
    {
        return ['have', 'have', 'has', 'have', 'have', 'have']
            [conjugationSelector];
    }

    /* 
     *   verb endings for regular '-s' and '-es' verbs, agreeing with this
     *   object as the subject 
     */
    verbEndingS
    {
        return ['', '', 's', '', '', ''][conjugationSelector];
    }
    verbEndingEs
    {
        return ['', '', 'es', '', '', ''][conjugationSelector];
    }
    verbEndingIes
    {
        return ['y', 'y', 'ies', 'y', 'y', 'y'][conjugationSelector];
    }

    /*
     *   Show my name for an arrival/departure message.  If we've been seen
     *   before by the player character, we'll show our definite name,
     *   otherwise our indefinite name.  
     */
    travelerName(arriving) { say (seenBy(gPlayerChar) ? theName : aName); }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific Vehicle changes 
 */
modify Vehicle
    /*
     *   Display the name of the traveler, for use in an arrival or
     *   departure message.
     */
    travelerName(arriving)
    {
        /* 
         *   By default, start with the indefinite name if we're arriving,
         *   or the definite name if we're leaving.
         *   
         *   If we're leaving, presumably they've seen us before, since we
         *   were already in the room to start with.  Since we've been
         *   seen before, the definite is appropriate.
         *   
         *   If we're arriving, even if we're not being seen for the first
         *   time, we haven't been seen yet in this place around this
         *   time, so the indefinite is appropriate.  
         */
        say(arriving ? aName : theName);

        /* show the list of actors aboard */
        aboardVehicleListerObj.showList(
            libGlobal.playerChar, nil, allContents(), 0, 0,
            libGlobal.playerChar.visibleInfoTable(), nil);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific PushTraveler changes 
 */
modify PushTraveler
    /*
     *   When an actor is pushing an object from one room to another, show
     *   its name with an additional clause indicating the object being
     *   moved along with us.  
     */
    travelerName(arriving)
    {
        "<<seenBy(gPlayerChar) ? theName : aName>>,
        pushing <<obj_.theNameObj>>,";
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   English-specific changes for various nested room types. 
 */
modify BasicChair
    /* by default, one sits *on* a chair */
    actorInPrep = 'on'
;

modify BasicPlatform
    /* by default, one stands *on* a platform */
    actorInPrep = 'on'
;

modify Booth
    /* by default, one is *in* a booth */
    actorInPrep = 'in'
;

/* ------------------------------------------------------------------------ */
/*
 *   Language modifications for Matchstick 
 */
modify Matchstick
    /* "strike match" means "light match" */
    dobjFor(Strike) asDobjFor(Burn)

    /* "light match" means "burn match" */
    dobjFor(Light) asDobjFor(Burn)
;

/* 
 *   Match state objects.  We show "lit" as the state for a lit match,
 *   nothing for an unlit match. 
 */
matchStateLit: ThingState 'lit'
    stateTokens = ['lit']
;
matchStateUnlit: ThingState
    stateTokens = ['unlit']
;



/* ------------------------------------------------------------------------ */
/*
 *   English-specific modifications for Room. 
 */
modify Room
    /* 
     *   For top-level rooms, describe an object as being in the room by
     *   describing it as being in the room's nominal drop destination,
     *   since that's where something nominally goes if it's dropped
     *   within the room.  
     */
    childInName(childName)
    {
        return getNominalDropDestination().childInName(childName);
    }
    childInNameWithOwner(childName)
    {
        return getNominalDropDestination().childInNameWithOwner(childName);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   English-specific modifications for the default room parts. 
 */

modify Floor
    childInNameGen(childName, myName) { return childName + ' on ' + myName; }
    actorInPrep = 'on'
;

modify defaultFloor
    noun = 'floor' 'ground'
    name = 'floor'
;

modify defaultGround
    noun = 'ground' 'floor'
    name = 'ground'
;

modify DefaultWall noun='wall' plural='walls' name='wall';
modify defaultCeiling noun='ceiling' name='ceiling';
modify defaultNorthWall adjective='n' 'north' name='north wall';
modify defaultSouthWall adjective='s' 'south' name='south wall';
modify defaultEastWall adjective='e' 'east' name='east wall';
modify defaultWestWall adjective='w' 'west' name='west wall';
modify defaultSky noun='sky' name='sky';


/* ------------------------------------------------------------------------ */
/*
 *   The English-specific modifications for directions. 
 */
modify Direction
    /* describe an actor arriving from this direction */
    sayArriving(actor)
    {
        /* show the generic arrival message */
        libMessages.sayArriving(actor);
    }

    /* describe an actor departing in this direction */
    sayDeparting(actor)
    {
        /* show the generic departure message */
        libMessages.sayDeparting(actor);
    }
;

/*
 *   The English-specific modifications for compass directions. 
 */
modify CompassDirection
    /* describe an actor arriving from this direction */
    sayArriving(actor)
    {
        /* show the generic compass direction description */
        libMessages.sayArrivingDir(actor, name);
    }

    /* describe an actor departing in this direction */
    sayDeparting(actor)
    {
        /* show the generic compass direction description */
        libMessages.sayDepartingDir(actor, name);
    }
;

/*
 *   The English-specific definitions for the compass direction objects.
 *   In addition to modifying the direction objects to define the name of
 *   the direction, we add a 'directionName' grammar rule.  
 */
#define DefineLangDir(root, dirNames, backPre) \
grammar directionName(root): dirNames: DirectionProd \
   dir = root##Direction \
; \
\
modify root##Direction \
   name = #@root \
   backToPrefix = backPre

DefineLangDir(north, 'north' | 'n', 'back to the');
DefineLangDir(south, 'south' | 's', 'back to the');
DefineLangDir(east, 'east' | 'e', 'back to the');
DefineLangDir(west, 'west' | 'w', 'back to the');
DefineLangDir(northeast, 'northeast' | 'ne', 'back to the');
DefineLangDir(northwest, 'northwest' | 'nw', 'back to the');
DefineLangDir(southeast, 'southeast' | 'se', 'back to the');
DefineLangDir(southwest, 'southwest' | 'sw', 'back to the');
DefineLangDir(up, 'up' | 'u', 'back');
DefineLangDir(down, 'down' | 'd', 'back');
DefineLangDir(in, 'in', 'back');
DefineLangDir(out, 'out', 'back');

/*
 *   The English-specific shipboard direction modifications.  Certain of
 *   the ship directions have no natural descriptions for arrival and/or
 *   departure; for example, there's no good way to say "arriving from
 *   fore."  Others don't fit any regular pattern: "he goes aft" rather
 *   than "he departs to aft."  As a result, these are a bit irregular
 *   compared to the compass directions and so are individually defined
 *   below.
 */

DefineLangDir(port, 'port' | 'p', 'back to')
    sayArriving(actor)
        { libMessages.sayArrivingShipDir(actor, 'the port direction'); }
    sayDeparting(actor) { libMessages.sayDepartingShipDir(actor, 'port'); }
;

DefineLangDir(starboard, 'starboard' | 'sb', 'back to')
    sayArriving(actor) { libMessages.sayArrivingShipDir(actor, 'starboard'); }
    sayDeparting(actor)
        { libMessages.sayDepartingShipDir(actor, 'starboard'); }
;

DefineLangDir(aft, 'aft' | 'a', 'back to')
    sayArriving(actor) { libMessages.sayArrivingShipDir(actor, 'aft'); }
    sayDeparting(actor) { libMessages.sayDepartingAft(actor); }
;

DefineLangDir(fore, 'fore' | 'forward' | 'f', 'back to')
    sayArriving(actor) { libMessages.sayArrivingShipDir(actor, 'forward'); }
    sayDeparting(actor) { libMessages.sayDepartingFore(actor); }
;

/* ------------------------------------------------------------------------ */
/*
 *   Set a pronoun antecedent to the given list of ResolveInfo objects.
 *   Pronoun handling is language-specific, so this implementation is part
 *   of the English library, not the generic library.
 *   
 *   If only one object is present, we'll set the object to be the
 *   antecedent of 'it', 'him', or 'her', according to the object's
 *   gender.  We'll also set the object as the single antecedent for
 *   'them'.
 *   
 *   If we have multiple objects present, we'll set the list to be the
 *   antecedent of 'them', and we'll forget about any antecedent for 'it'.
 *   
 *   Note that the input is a list of ResolveInfo objects, so we must pull
 *   out the underlying game objects when setting the antecedents.  
 */
setPronoun(lst)
{
    /* if the list is empty, ignore it */
    if (lst == [])
        return;

    /* 
     *   if we have multiple objects, the entire list is the antecedent
     *   for 'them'; otherwise, it's a singular antecedent which depends
     *   on its gender 
     */
    if (lst.length() > 1)
    {
        /* it's 'them' */
        setThem(lst.mapAll({x: x.obj_}));

        /* forget any 'it' */
        setIt(nil);
    }
    else
    {
        local obj;
        
        /*
         *   We have only one object, so set it as an antecedent according
         *   to its gender.
         *   
         *   Note that we'll set an object to be the antecedent for both
         *   'him' and 'her' if the object has both masculine and feminine
         *   usage.
         */

        /* get the game object from the ResolveInfo object */
        obj = lst[1].obj_;

        /* check for masculine usage */
        if (obj.isHim)
            setHim(obj);

        /* check for feminine usage */
        if (obj.isHer)
            setHer(obj);

        /* check for neuter usage */
        if (obj.isIt)
            setIt(obj);

        /* in any case, make it the antecedent for 'them' as well */
        setThem([obj]);
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Some helper routines for the library messages. 
 */
class MessageHelper: object
    /*
     *   Show a list of objects for a disambiguation query.  If
     *   'showIndefCounts' is true, we'll show the number of equivalent
     *   items for each equivalent item; otherwise, we'll just show an
     *   indefinite noun phrase for each equivalent item.  
     */
    askDisambigList(matchList, fullMatchList, showIndefCounts, dist)
    {
        /* show each item */
        for (local i = 1, local len = matchList.length() ; i <= len ; ++i)
        {
            local equivCnt;
            local obj;

            /* get the current object */
            obj = matchList[i].obj_;
            
            /*
             *   if this isn't the first, add a comma; if this is the
             *   last, add an "or" as well 
              */
            if (i == len)
                ", or ";
            else if (i != 1)
                ", ";

            /*
             *   Check to see if more than one equivalent of this item
             *   appears in the full list.  
             */
            for (equivCnt = 0, local j = 1,
                 local fullLen = fullMatchList.length() ; j <= fullLen ; ++j)
            {
                /* 
                 *   if this item is equivalent for the purposes of the
                 *   current distinguisher, count it 
                 */
                if (!dist.canDistinguish(obj, fullMatchList[j].obj_))
                {
                    /* it's equivalent - count it */
                    ++equivCnt;
                }
            }

            /* show this item with the appropriate article */
            if (equivCnt > 1)
            {
                /* 
                 *   we have multiple equivalents - show either with an
                 *   indefinite article or with a count, depending on the
                 *   flags the caller provided 
                 */
                if (showIndefCounts)
                {
                    /* a count is desired for each equivalent group */
                    say(dist.countName(obj, equivCnt));
                }
                else
                {
                    /* no counts desired - show with an indefinite article */
                    say(dist.aName(obj));
                }
            }
            else
            {
                /* there's only one - show with a definite article */
                say(dist.theName(obj));
            }
        }
    }

    /*
     *   Ask for a missing object - this is called when a command is
     *   completely missing a noun phrase for one of its object.  
     */
    askMissingObject(actor, action, which)
    {
        "<.parser>\^<<action.whatObj(which)>> do you want to
        <<action.verbInf(which)>>?<./parser> ";
    }

    /* 
     *   An object was missing - this is called under essentially the same
     *   circumstances as askMissingObject, but in cases where interactive
     *   resolution is impossible and we simply wish to report the problem
     *   and do not wish to ask for help.
     */
    missingObject(actor, action, which)
    {
        "<.parser>You must be more specific
        about <<action.whatObj(which)>> you want to
        <<action.verbInf(which)>>.<./parser> ";
    }

    /*
     *   Ask for a missing literal phrase. 
     */
    askMissingLiteral(actor, action, which)
    {
        /* use the standard missing-object message */
        askMissingObject(actor, action, which);
    }

    /*
     *   Show the message for a missing literal phrase 
     */
    missingLiteral(actor, action, which)
    {
        "<.parser>You must be more specific
        about <<action.whatObj(which)>> you want to
        <<action.verbInf(which, nil)>>.  Try, for example,
        <<action.verbInf(which, nil)>> <q>something</q>.<./parser> ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Distinguisher customizations for English.
 *   
 *   Each distinguisher must provide a method that gets the name of an
 *   item for a disamgiguation query.  Since these are inherently
 *   language-specific, they're defined here.  
 */
modify basicDistinguisher
    /*
     *   The basic distinguisher can tell apart objects that are not
     *   "basic equivalents" of one another.  Thus, we need make no
     *   distinctions when listing objects apart from showing their names. 
     */
    aName(obj) { return obj.aDisambigName; }
    theName(obj) { return obj.theDisambigName; }
    countName(obj, cnt) { return obj.countDisambigName(cnt); }
;

modify ownershipAndLocationDistinguisher
    aName(obj) { return obj.aNameOwnerLoc; }
    theName(obj) { return obj.theNameOwnerLoc; }
    countName(obj, cnt) { return obj.countNameOwnerLoc(cnt); }
;

modify litUnlitDistinguisher
    /*
     *   The lit/unlit distinguisher tells apart objects based on whether
     *   they're lit or unlit, so describe objects as lit or unlit
     *   explicitly.  
     */
    aName(obj) { return obj.aNameLit; }
    theName(obj) { return obj.theNameLit; }
    countName(obj, cnt) { return obj.pluralNameLit; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Enligh-specific light source modifications 
 */
modify LightSource
    /* provide lit/unlit names for litUnlitDistinguisher */
    aNameLit()
    {
        /* 
         *   if this is a mass noun or a plural name, just use the name
         *   with lit/unlit; otherwise, add "a"
         */
        if (isPlural || isMassNoun)
            return (isLit ? 'lit ' : 'unlit ') + name;
        else
            return (isLit ? 'a lit ' : 'an unlit ') + name;
    }
    theNameLit = ((isLit ? 'the lit ' : 'the unlit ') + name)
    pluralNameLit = ((isLit ? 'lit ' : 'unlit ') + pluralName)

    /* 
     *   Allow 'lit' and 'unlit' as adjectives - but even though we define
     *   these as our adjectives in the dictionary, we'll only accept the
     *   one appropriate for our current state, thanks to our state
     *   objects.  
     */
    adjective = 'lit' 'unlit'
;

/*
 *   Light source list states.  An illuminated light source shows its
 *   status as "providing light"; an unlit light source shows no extra
 *   status. 
 */
lightSourceStateOn: ThingState 'providing light'
    stateTokens = ['lit']
;
lightSourceStateOff: ThingState
    stateTokens = ['unlit']
;

/* ------------------------------------------------------------------------ */
/*
 *   Wearable states - a wearable item can be either worn or not worn.  
 */

/* "worn" */
wornState: ThingState 'being worn'
    /*
     *   In listings of worn items, don't bother mentioning our 'worn'
     *   status, as the entire list consists of items being worn - it
     *   would be redundant to point out that the items in a list of items
     *   being worn are being worn.  
     */
    wornName = nil
;

/* 
 *   "Unworn" state.  Don't bother mentioning the status of an unworn
 *   item, since this is the default for everything.  
 */
unwornState: ThingState;


/* ------------------------------------------------------------------------ */
/*
 *   The English-specific message builder.  
 */
langMessageBuilder: MessageBuilder
    /*
     *   The English message substitution parameter table.
     *   
     *   Note that we specify two additional elements for each table entry
     *   beyond the standard language-independent complement:
     *   
     *   info[4] = reflexive property - this is the property to invoke
     *   when the parameter is used reflexively (in other words, its
     *   target object is the same as the most recent target object used
     *   in the nominative case).  If this is nil, the parameter has no
     *   reflexive form.
     *   
     *   info[5] = true if this is a nominative usage, nil if not.  We use
     *   this to determine which target objects are used in the nominative
     *   case, so that we can remember those objects for subsequent
     *   reflexive usages.  
     */
    paramList_ =
    [
        /* parameters that imply the actor as the target object */
        ['you/he', &theName, 'actor', nil, true],
        ['you/she', &theName, 'actor', nil, true],
        ['you\'re/he\'s', &itIsContraction, 'actor', nil, true],
        ['you\'re/she\'s', &itIsContraction, 'actor', nil, true],
        ['you\'re', &itIsContraction, 'actor', nil, true],
        ['you/him', &theNameObj, 'actor', &itReflexive, nil],
        ['you/her', &theNameObj, 'actor', &itReflexive, nil],
        ['your/her', &theNamePossAdj, 'actor', nil, nil],
        ['your/his', &theNamePossAdj, 'actor', nil, nil],
        ['your', &theNamePossAdj, 'actor', nil, nil],
        ['yours/hers', &theNamePossNoun, 'actor', nil, nil],
        ['yours/his', &theNamePossNoun, 'actor', nil, nil],
        ['yours', &theNamePossNoun, 'actor', nil, nil],
        ['yourself/himself', &itReflexive, 'actor', nil, nil],
        ['yourself/herself', &itReflexive, 'actor', nil, nil],
        ['yourself', &itReflexive, 'actor', nil, nil],

        /* parameters that don't imply any target object */
        ['the/he', &theName, nil, nil, true],
        ['the/she', &theName, nil, nil, true],
        ['the/him', &theNameObj, nil, &itReflexive, nil],
        ['the/her', &theNameObj, nil, &itReflexive, nil],
        ['the\'s/her', &theNamePossAdj, nil, nil, nil],
        ['the\'s/hers', &theNamePossNoun, nil, nil, nil],
        ['s', &verbEndingS, nil, nil, true],
        ['es', &verbEndingEs, nil, nil, true],
        ['ies', &verbEndingIes, nil, nil, true],
        ['is', &verbToBe, nil, nil, true],
        ['are', &verbToBe, nil, nil, true],
        ['was', &verbWas, nil, nil, true],
        ['were', &verbWas, nil, nil, true],
        ['has', &verbToHave, nil, nil, true],
        ['have', &verbToHave, nil, nil, true],
        ['a/he', &aName, nil, nil, nil],
        ['an/he', &aName, nil, nil, true],
        ['a/she', &aName, nil, nil, true],
        ['an/she', &aName, nil, nil, true],
        ['a/him', &aNameObj, nil, &itReflexive, nil],
        ['an/him', &aNameObj, nil, &itReflexive, nil],
        ['a/her', &aNameObj, nil, &itReflexive, nil],
        ['an/her', &aNameObj, nil, &itReflexive, nil],
        ['it/he', &itNom, nil, nil, true],
        ['it/she', &itNom, nil, nil, true],
        ['it/him', &itObj, nil, &itReflexive, nil],
        ['it/her', &itObj, nil, &itReflexive, nil],

        /* 
         *   note that we don't have its/his, because that leaves
         *   ambiguous whether we want an adjective or noun form - so we
         *   only use the feminine pronouns with these, to make the
         *   meaning unambiguous 
         */
        ['its/her', &itPossAdj, nil, nil, nil],
        ['its/hers', &itPossNoun, nil, nil, nil],
        
        ['it\'s/he\'s', &itIsContraction, nil, nil, true],
        ['it\'s/she\'s', &itIsContraction, nil, nil, true],
        ['it\'s', &itIsContraction, nil, nil, true],
        ['that/he', &thatNom, nil, nil, true],
        ['that/she', &thatNom, nil, nil, true],
        ['that/him', &thatObj, nil, &itReflexive, nil],
        ['that/her', &thatObj, nil, &itReflexive, nil],
        ['itself', &itReflexive, nil, nil, nil],
        ['itself/himself', &itReflexive, nil, nil, nil],
        ['itself/herself', &itReflexive, nil, nil, nil],

        /* default preposition for standing in/on something */
        ['on', &actorInName, nil, nil, nil],
        ['in', &actorInName, nil, nil, nil],
        ['onto', &actorIntoName, nil, nil, nil],
        ['into', &actorIntoName, nil, nil, nil]
    ]

    /*
     *   The most recent target object used in the nominative case.  We
     *   note this so that we can supply reflexive mappings when the same
     *   object is re-used in the objective case.  This allows us to map
     *   things like "you can't take you" to the better-sounding "you
     *   can't take yourself".  
     */
    lastSubject_ = nil

    /*
     *   Get the target object property mapping.  If the target object is
     *   the same as the most recent subject object (i.e., the last object
     *   used in the nominative case), and this parameter has a reflexive
     *   form property, we'll return the reflexive form property.
     *   Otherwise, we'll return the standard property mapping.  
     */
    getTargetProp(targetObj, info)
    {
        local ret;
        
        /* 
         *   If this target object matches the last subject, and we have a
         *   reflexive rendering, return the property for the reflexive
         *   rendering.
         */
        if (targetObj == lastSubject_ && info[4] != nil)
        {
            /* use the reflexive rendering */
            ret = info[4];
        }
        else
        {
            /* no special handling; inherit the default handling */
            ret = inherited(targetObj, info);
        }

        /* if this is a nominative usage, note it as the last subject */
        if (info[5])
            lastSubject_ = targetObj;

        /* return the result */
        return ret;
    }

    /* end-of-sentence match pattern */
    patEndOfSentence = static new RexPattern('[.;:!?]<^alphanum>')

    /*
     *   Process result text. 
     */
    processResult(txt)
    {
        /* 
         *   If the text contains any sentence-ending punctuation, reset
         *   our internal memory of the subject of the sentence.  We
         *   consider the sentence to end with a period, semicolon, colon,
         *   question mark, or exclamation point followed by anything
         *   other than an alpha-numeric.  (We require the secondary
         *   character so that we don't confuse things like "3:00" or
         *   "7.5" to contain sentence-ending punctuation.)  
         */
        if (rexSearch(patEndOfSentence, txt) != nil)
        {
            /* 
             *   we have a sentence ending in this run of text, so any
             *   saved subject object will no longer apply after this text
             *   - forget our subject object
             */
            lastSubject_ = nil;
        }

        /* return the inherited processing */
        return inherited(txt);
    }

    /* some pre-compiled search patterns we use a lot */
    patIdObjSlashIdApostS = static new RexPattern(
        '(<^space>+)(<space>+<^space>+)\'s(/<^space>+)')
    patIdObjApostS = static new RexPattern(
        '(<^space>+)(<space>+<^space>+)\'s')

    /*
     *   Rewrite a parameter string for a language-specific syntax
     *   extension.
     *   
     *   For English, we'll handle the possessive apostrophe-s suffix
     *   specially, by allowing the apostrophe-s to be appended to the
     *   target object name.  If we find an apostrophe-s on the target
     *   object name, we'll move it to the preceding identifier name:
     *   
     *   the dobj's -> the's dobj
     *.  the dobj's/he -> the's dobj/he
     *.  he/the dobj's -> he/the's dobj 
     */
    langRewriteParam(paramStr)
    {
        /* look for "id obj's" and "id1 obj's/id2" */
        if (rexMatch(patIdObjSlashIdApostS, paramStr) != nil)
        {
            /* rewrite with the "'s" moved to the preceding parameter name */
            paramStr = rexGroup(1)[3] + '\'s'
                       + rexGroup(2)[3] + rexGroup(3)[3];
        }
        else if (rexMatch(patIdObjApostS, paramStr) != nil)
        {
            /* rewrite with the "'s" moved to the preceding parameter name */
            paramStr = rexGroup(1)[3] + '\'s' + rexGroup(2)[3];
        }

        /* return our (possibly modified) result */
        return paramStr;
    }
;


/* 
 *   ------------------------------------------------------------------------
 *   
 */
/*
 *   Spell out an integer number.
 *   
 *   This entire function is language-specific.  We have not attempted to
 *   break down the numeric conversion into smaller language-specific
 *   primitives, because we assume that the spelled-out form of numbers
 *   varies considerably from language to language and that it would be
 *   difficult to cover the full range of possibilities without simply
 *   making the entire function language-specific.
 *   
 *   Note that some of the spellIntXxx flags might not be meaningful in
 *   all languages, because most of the flags are by their very nature
 *   associated with language-specific idioms.  Translations are free to
 *   ignore flags that indicate variations with no local equivalent.  
 */
spellIntExt(val, flags)
{
    local str;
    local trailingSpace;
    local needAnd;
    local powers = [1000000000, ' billion ',
                    1000000,    ' million ',
                    1000,       ' thousand ',
                    100,        ' hundred '];

    /* start with an empty string */
    str = '';
    trailingSpace = nil;
    needAnd = nil;

    /* if it's zero, it's a special case */
    if (val == 0)
        return 'zero';

    /* 
     *   if the number is negative, note it in the string, and use the
     *   absolute value 
     */
    if (val < 0)
    {
        str = 'negative ';
        val = -val;
    }

    /* do each named power of ten */
    for (local i = 1 ; val >= 100 && i <= powers.length() ; i += 2)
    {
        /* 
         *   if we're in teen-hundreds mode, do the teen-hundreds - this only
         *   works for values from 1,100 to 9,999, since a number like 12,000
         *   doesn't work this way - 'one hundred twenty hundred' is no good 
         */
        if ((flags & SpellIntTeenHundreds) != 0
            && val >= 1100 && val < 10000)
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';
            
            /* spell it out as a number of hundreds */
            str += spellIntExt(val / 100, flags) + ' hundred ';

            /* take off the hundreds */
            val %= 100;

            /* note the trailing space */
            trailingSpace = true;

            /* we have something to put an 'and' after, if desired */
            needAnd = true;

            /* 
             *   whatever's left is below 100 now, so there's no need to
             *   keep scanning the big powers of ten
             */
            break;
        }

        /* if we have something in this power range, apply it */
        if (val >= powers[i])
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';

            /* add the number of multiples of this power and the power name */
            str += spellIntExt(val / powers[i], flags) + powers[i+1];

            /* take it out of the remaining value */
            val %= powers[i];

            /* 
             *   note that we have a trailing space in the string (all of
             *   the power-of-ten names have a trailing space, to make it
             *   easy to tack on the remainder of the value) 
             */
            trailingSpace = true;

            /* we have something to put an 'and' after, if one is desired */
            needAnd = true;
        }
    }

    /* 
     *   if we have anything left, and we have written something so far,
     *   and the caller wanted an 'and' before the tens part, add the
     *   'and' 
     */
    if ((flags & SpellIntAndTens) != 0
        && needAnd
        && val != 0)
    {
        /* add the 'and' */
        str += 'and ';
        trailingSpace = true;
    }

    /* do the tens */
    if (val >= 20)
    {
        /* anything above the teens is nice and regular */
        str += ['twenty', 'thirty', 'forty', 'fifty', 'sixty',
                'seventy', 'eighty', 'ninety'][val/10 - 1];
        val %= 10;

        /* if it's non-zero, we'll add the units, so add a hyphen */
        if (val != 0)
            str += '-';

        /* we no longer have a trailing space in the string */
        trailingSpace = nil;
    }
    else if (val >= 10)
    {
        /* we have a teen */
        str += ['ten', 'eleven', 'twelve', 'thirteen', 'fourteen',
                'fifteen', 'sixteen', 'seventeen', 'eighteen',
                'nineteen'][val - 9];

        /* we've finished with the number */
        val = 0;

        /* there's no trailing space */
        trailingSpace = nil;
    }

    /* if we have a units value, add it */
    if (val != 0)
    {
        /* add the units name */
        str += ['one', 'two', 'three', 'four', 'five',
                'six', 'seven', 'eight', 'nine'][val];

        /* we have no trailing space now */
        trailingSpace = nil;
    }

    /* if there's a trailing space, remove it */
    if (trailingSpace)
        str = str.substr(1, str.length() - 1);

    /* return the string */
    return str;
}


/* ------------------------------------------------------------------------ */
/*
 *   Additional token types for US English. 
 */

/* special "apostrophe-s" token */
enum token tokApostropheS;

/*
 *   Command tokenizer for US English.  Other language modules should
 *   provide their own tokenizers to allow for differences in punctuation
 *   and other lexical elements.  
 */
cmdTokenizer: Tokenizer
    rules_ = static
    [
        /* skip whitespace */
        [new RexPattern('<Space>+'), nil, &tokCvtSkip, nil],

        /* certain punctuation marks */
        ['[.,;:?!]', tokPunct, nil, nil],

        /*
         *   We have a special rule for spelled-out numbers from 21 to 99:
         *   when we see a 'tens' word followed by a hyphen followed by a
         *   digits word, we'll pull out the tens word, the hyphen, and
         *   the digits word as separate tokens.
         */
        [new RexPattern('<NoCase>(twenty|thirty|forty|fifty|sixty|'
                        + 'seventy|eighty|ninety)-'
                        + '(one|two|three|four|five|six|seven|eight|nine)'
                        + '(?!<AlphaNum>)'),
         tokWord, &tokCvtSpelledNumber, nil],


        /* 
         *   Initials.  We'll look for strings of three or two initials,
         *   set off by periods but without spaces.  We'll look for
         *   three-letter initials first ("G.H.W. Billfold"), then
         *   two-letter initials ("X.Y. Zed"), so that we find the longest
         *   sequence that's actually in the dictionary.  Note that we
         *   don't have a separate rule for individual initials, since
         *   we'll pick that up with the regular abbreviated word rule
         *   below.
         *   
         *   Some games could conceivably extend this to allow strings of
         *   initials of four letters or longer, but in practice people
         *   tend to elide the periods in longer sets of initials, so that
         *   the initials become an acronym, and thus would fit the
         *   ordinary word token rule.  
         */
        [new RexPattern('<alpha><period><alpha><period><alpha><period>'),
         tokWord, nil, &acceptAbbrTok],

        [new RexPattern('<alpha><period><alpha><period>'),
         tokWord, nil, &acceptAbbrTok],

        /*
         *   Abbbreviated word - this is a word that ends in a period,
         *   such as "Mr.".  This rule comes before the ordinary word rule
         *   because we will only consider the period to be part of the
         *   word (and not a separate token) if the entire string
         *   including the period is in the main vocabulary dictionary.
         */
        [new RexPattern('<Alpha|-><AlphaNum|-|squote>*<period>'),
         tokWord, nil, &acceptAbbrTok],

        /*
         *   A word ending in an apostrophe-s.
         *   
         *   By default, we parse this as two separate tokens: one for the
         *   word and one for the apostrophe-s.  However, if the string
         *   including the apostrophe-s appears explicitly in the
         *   dictionary, we'll keep the word together as a single token.  
         */
        [new RexPattern('<Alpha|-><AlphaNum|-|squote>*<squote>s'),
         tokWord, &tokCvtApostropheS, nil],

        /* 
         *   Words - note that we convert everything to lower-case.  A
         *   word must start with an alphabetic character or a dash, but
         *   can contain alphabetics, digits, hyphens, and apostrophes
         *   after that.  
         */
        [new RexPattern('<Alpha|-><AlphaNum|-|squote>*'), tokWord, nil, nil],

        /* an abbreviation word starting with a number */
        [new RexPattern('<Digit>(?=<AlphaNum|-|squote>*<Alpha|-|squote>)'
                        + '<AlphaNum|-|squote>*<period>'),
         tokWord, nil, &acceptAbbrTok],

        /*
         *   A word can start with a number, as long as there's something
         *   other than numbers in the string - if it's all numbers, we
         *   want to treat it as a numeric token. 
         */
        [new RexPattern('<Digit>(?=<AlphaNum|-|squote>*<Alpha|-|squote>)'
                        + '<AlphaNum|-|squote>*'), tokWord, nil, nil],

        /* strings with ASCII "straight" quotes */
        [new RexPattern('<min>([`\'"])(.*)%1(?!<AlphaNum>)'),
         tokString, nil, nil],

        /* some people like to use single quotes like `this' */
        [new RexPattern('<min>`(.*)\'(?!<AlphaNum>)'), tokString, nil, nil],

        /* strings with Latin-1 curly quotes (single and double) */
        [new RexPattern('<min>\u2018(.*)\u2019'), tokString, nil, nil],
        [new RexPattern('<min>\u201C(.*)\u201D'), tokString, nil, nil],

        /* 
         *   unterminated string - if we didn't just match a terminated
         *   string, but we have what looks like the start of a string,
         *   match to the end of the line 
         */
        [new RexPattern('([`\'"\u2018\u201C](.*)'), tokString, nil, nil],

        /* integer numbers */
        [new RexPattern('[0-9]+'), tokInt, nil, nil]
    ]

    /*
     *   Handle an apostrophe-s word.  We'll return this as two separate
     *   tokens: one for the word preceding the apostrophe-s, and one for
     *   the apostrophe-s itself.  
     */
    tokCvtApostropheS(txt, typ, toks)
    {
        local w;
        local s;
        
        /* 
         *   If the word appears in the dictionary with the apostrophe-s
         *   explicitly, then keep it together as a single token.
         *   
         *   Note that we ignore truncated matches; we want to match an
         *   apostrophe-s word only if it appears exactly as given.  We
         *   thus filter for results that do not have the 'truncation'
         *   flag set in the match results from the dictionary's
         *   StringComparator.  
         */
        if (G_dict.isWordDefined(txt,
            {result: (result & StrCompTrunc) == 0}))
        {
            /* 
             *   it's explicitly in the dictionary with the apostrophe-s -
             *   keep it together as a single token
             */
            toks.append([txt, typ, txt]);
        }
        else
        {
            /* 
             *   pull out the part up to but not including the apostrophe,
             *   and pull out the apostrophe-s part 
             */
            w = txt.substr(1, txt.length() - 2);
            s = txt.substr(txt.length() - 1);
            
            /* add the part before the apostrophe as the main token type */
            toks.append([w, typ, w]);
            
            /* add the apostrophe-s as a separate special token */
            toks.append([s, tokApostropheS, s]);
        }
    }

    /*
     *   Handle a spelled-out hyphenated number from 21 to 99.  We'll
     *   return this as three separate tokens: a word for the tens name, a
     *   word for the hyphen, and a word for the units name.  
     */
    tokCvtSpelledNumber(txt, typ, toks)
    {
        /* parse the number into its three parts with a regular expression */
        rexMatch(patAlphaDashAlpha, txt);

        /* add the part before the hyphen */
        toks.append([rexGroup(1)[3], typ, rexGroup(1)[3]]);

        /* add the hyphen */
        toks.append(['-', typ, '-']);

        /* add the part after the hyphen */
        toks.append([rexGroup(2)[3], typ, rexGroup(2)[3]]);
    }
    patAlphaDashAlpha = static new RexPattern('(<alpha>+)-(<alpha>+)')

    /*
     *   Check to see if we want to accept an abbreviated token - this is
     *   a token that ends in a period, which we use for abbreviated words
     *   like "Mr." or "Ave."  We'll accept the token only if it appears
     *   as given - including the period - in the dictionary.  Note that
     *   we ignore truncated matches, since the only way we'll accept a
     *   period in a word token is as the last character; there is thus no
     *   way that a token ending in a period could be a truncation of any
     *   longer valid token.  
     */
    acceptAbbrTok(txt)
    {
        /* look up the word, filtering out truncated results */
        return G_dict.isWordDefined(
            txt, {result: (result & StrCompTrunc) == 0});
    }

    /*
     *   Given a list of token strings, rebuild the original input string.
     *   We can't recover the exact input string, because the tokenization
     *   process throws away whitespace information, but we can at least
     *   come up with something that will display cleanly and produce the
     *   same results when run through the tokenizer. 
     */
    buildOrigText(toks)
    {
        local str;

        /* start with an empty string */
        str = '';

        /* concatenate each token in the list */
        for (local i = 1, local len = toks.length() ; i <= len ; ++i)
        {
            /* add the current token to the string */
            str += getTokOrig(toks[i]);

            /* 
             *   if this looks like a hyphenated number that we picked
             *   apart into two tokens, put it back together without
             *   spaces 
             */
            if (i + 2 <= len
                && rexMatch(patSpelledTens, getTokVal(toks[i]))
                && getTokVal(toks[i+1]) == '-'
                && rexMatch(patSpelledUnits, getTokVal(toks[i+2])))
            {
                /* 
                 *   it's a hyphenated number, all right - put the three
                 *   tokens back together without any intervening spaces,
                 *   so ['twenty', '-', 'one'] turns into 'twenty-one' 
                 */
                str += getTokOrig(toks[i+1]) + getTokOrig(toks[i+2]);

                /* skip ahead by the two extra tokens we're adding */
                i += 2;
            }
            else if (i + 1 <= len
                     && getTokType(toks[i]) == tokWord
                     && getTokType(toks[i+1]) == tokApostropheS)
            {
                /* 
                 *   it's a word followed by an apostrophe-s token - these
                 *   are appended together without any intervening spaces 
                 */
                str += getTokOrig(toks[i+1]);

                /* skip the extra token we added */
                ++i;
            }

            /* 
             *   if another token follows, and the next token isn't a
             *   punctuation mark, add a space before the next token 
             */
            if (i != len
                && !rexMatch(patPunct, getTokVal(toks[i+1])))
                str += ' ';
        }

        /* return the result string */
        return str;
    }

    /* some pre-compiled regular expressions */
    patSpelledTens = static new RexPattern(
        'twenty|thirty|forty|fifty|sixty|seventy|eighty|ninety')
    patSpelledUnits = static new RexPattern(
        'one|two|three|four|five|six|seven|eight|nine')
    patPunct = static new RexPattern('[.,;:?!]')
;


/* ------------------------------------------------------------------------ */
/*
 *   Grammar Rules 
 */

/*
 *   Command with explicit target actor.  When a command starts with an
 *   actor's name followed by a comma followed by a verb, we take it as
 *   being directed to the actor.
 */
grammar firstCommandPhrase(withActor):
    singleNoun->actor_ ',' commandPhrase->cmd_ : FirstCommandProdWithActor
;


/*
 *   Command-only conjunctions.  These words and groups of words can
 *   separate commands from one another, and can't be used to separate
 *   noun phrases in a noun list.  
 */
grammar commandOnlyConjunction(sentenceEnding):
    '.'
    | '!'
    : BasicProd

    /* these conjunctions end the sentence */
    isEndOfSentence() { return true; }
;

grammar commandOnlyConjunction(nonSentenceEnding):
    'then'
    | 'and' 'then'
    | ',' 'then'
    | ',' 'and' 'then'
    | ';'
    : BasicProd

    /* these conjunctions do not end a sentence */
    isEndOfSentence() { return nil; }
;


/*
 *   Command-or-noun conjunctions.  These words and groups of words can be
 *   used to separate commands from one another, and can also be used to
 *   separate noun phrases in a noun list.
 */
grammar commandOrNounConjunction(main):
    ','
    | 'and'
    | ',' 'and'
    : BasicProd

    /* these do not end a sentence */
    isEndOfSentence() { return nil; }
;

/*
 *   Noun conjunctions.  These words and groups of words can be used to
 *   separate noun phrases from one another.  Note that these do not need
 *   to be exclusive to noun phrases - these can occur as command
 *   conjunctions as well; this list is separated from
 *   commandOrNounConjunction in case there are conjunctions that can
 *   never be used as command conjunctions, since such conjunctions, which
 *   can appear here, would not appear in commandOrNounConjunctions. 
 */
grammar nounConjunction(main):
    ','
    | 'and'
    | ',' 'and'
    : BasicProd

    /* these conjunctions do not end a sentence */
    isEndOfSentence() { return nil; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun list: one or more noun phrases connected with conjunctions.
 *   This kind of noun list can end in a terminal noun phrase.  
 *   
 *   Note that a single noun phrase is a valid noun list, since a list can
 *   simply be a list of one.  The separate production nounMultiList can
 *   be used when multiple noun phrases are required.  
 */

/*
 *   a noun list can consist of a single terminal noun phrase 
 */
grammar nounList(terminal): terminalNounPhrase->np_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying noun phrase */
        return np_.resolveNouns(resolver, results);
    }
;

/* 
 *   a noun list can consist of a list of a single complete (non-terminal)
 *   noun phrase 
 */
grammar nounList(nonTerminal): completeNounPhrase->np_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying noun phrase */
        return np_.resolveNouns(resolver, results);
    }
;

/* 
 *   a noun list can consist of a list with two or more entries 
 */
grammar nounList(list): nounMultiList->lst_ : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying list */
        return lst_.resolveNouns(resolver, results);
    }
;

/*
 *   An empty noun list is one with no words at all.  This is matched when
 *   a command requires a noun list but the player doesn't include one;
 *   this construct has "badness" because we only want to match it when we
 *   have no choice.  
 */
grammar nounList(empty): [badness 100] : EmptyNounPhraseProd
    responseProd = nounList
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun Multi List: two or more noun phrases connected by conjunctions.
 *   This is almost the same as the basic nounList production, but this
 *   type of production requires at least two noun phrases, whereas the
 *   basic nounList production more generally defines its list as any
 *   number - including one - of noun phrases.  
 */

/*
 *   a multi list can consist of a noun multi- list plus a terminal noun
 *   phrase, separated by a conjunction 
 */
grammar nounMultiList(multi):
    nounMultiList->lst_ nounConjunction terminalNounPhrase->np_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return a list consisting of all of our objects */
        return np_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;

/*
 *   a multi list can consist of a non-terminal multi list 
 */
grammar nounMultiList(nonterminal): nonTerminalNounMultiList->lst_ 
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying list */
        return lst_.resolveNouns(resolver, results);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A non-terminal noun multi list is a noun list made up of at least two
 *   non-terminal noun phrases, connected by conjunctions.
 *   
 *   This is almost the same as the regular non-terminal noun list
 *   production, but this production requires two or more underlying noun
 *   phrases, whereas the basic non-terminal noun list matches any number
 *   of underlying phrases, including one.  
 */

/*
 *   a non-terminal multi-list can consist of a pair of complete noun
 *   phrases separated by a conjunction 
 */
grammar nonTerminalNounMultiList(pair):
    completeNounPhrase->np1_ nounConjunction completeNounPhrase->np2_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return the combination of the two underlying noun phrases */
        return np1_.resolveNouns(resolver, results)
            + np2_.resolveNouns(resolver, results);
    }
;

/*
 *   a non-terminal multi-list can consist of another non-terminal
 *   multi-list plus a complete noun phrase, connected by a conjunction
 */
grammar nonTerminalNounMultiList(multi):
    nonTerminalNounMultiList->lst_ nounConjunction completeNounPhrase->np_
    : NounListProd
    resolveNouns(resolver, results)
    {
        /* return the combination of the sublist and the noun phrase */
        return lst_.resolveNouns(resolver, results)
            + np_.resolveNouns(resolver, results);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   "Except" list.  This is a noun list that can contain anything that's
 *   in a regular noun list plus some things that only make sense as
 *   exceptions, such as possessive nouns (e.g., "mine").  
 */

grammar exceptList(single): exceptNounPhrase->np_ : ExceptListProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }
;

grammar exceptList(list):
    exceptNounPhrase->np_ nounConjunction exceptList->lst_
    : ExceptListProd
    resolveNouns(resolver, results)
    {
        /* return a list consisting of all of our objects */
        return np_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;

/*
 *   An "except" noun phrase is a normal "complete" noun phrase or a
 *   possessive noun phrase that doesn't explicitly qualify another phrase
 *   (for example, "all the coins but bob's" - the "bob's" is just a
 *   possessive noun phrase without another noun phrase attached, since it
 *   implicitly qualifies "the coins").  
 */
grammar exceptNounPhrase(singleComplete): completeNounPhraseWithoutAll->np_
    : ExceptListProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver, results);
    }
;
    
grammar exceptNounPhrase(singlePossessive): possessiveNounPhrase->poss_
    : ButPossessiveProd
;
    

/* ------------------------------------------------------------------------ */
/*
 *   A single noun is sometimes required where, structurally, a list is
 *   not allowed.  Single nouns should not be used to prohibit lists where
 *   there is no structural reason for the prohibition - these should be
 *   used only where it doesn't make sense to use a list structurally.  
 */
grammar singleNoun(normal):
    terminalNounPhrase->np_
    | completeNounPhrase->np_
    : SingleNounProd
;

/*
 *   An empty single noun is one with no words at all.  This is matched
 *   when a command requires a noun list but the player doesn't include
 *   one; this construct has "badness" because we only want to match it
 *   when we have no choice. 
 */
grammar singleNoun(empty): [badness 100] : EmptyNounPhraseProd
    responseProd = singleNoun
;

/*
 *   A user could attempt to use a noun list with more than one entry (a
 *   "multi list") where a single noun is required.  This is not a
 *   grammatical error, so we accept it grammatically; however, for
 *   disambiguation purposes we score it lower than a singleNoun
 *   production with only one noun phrase, and if we try to resolve it,
 *   we'll fail with an error.  
 */
grammar singleNoun(multiple): nounMultiList->np_ : SingleNounWithListProd
;


/* ------------------------------------------------------------------------ */
/*
 *   Prepositionally modified single noun phrases.  These can be used in
 *   indirect object responses, so allow for interactions like this:
 *   
 *   >unlock door
 *.  What do you want to unlock it with?
 *   
 *   >with the key
 *   
 *   The entire notion of prepositionally qualified noun phrases in
 *   interactive indirect object responses is specific to English, so this
 *   is implemented in the English module only.  However, the general
 *   notion of specialized responses to interactive indirect object
 *   queries is handled in the language-independent library in some cases,
 *   in such a way that the language-specific library can customize the
 *   behavior - see TIAction.askIobjResponseProd.  
 */
class PrepSingleNounProd: SingleNounProd
    resolveNouns(resolver, results)
    {
        return np_.resolveNouns(resolver,results);
    }
;

grammar inSingleNoun(main):
     singleNoun->np_ | ('in' | 'into' | 'in' 'to') singleNoun->np_
    : PrepSingleNounProd
;

grammar forSingleNoun(main):
   singleNoun->np_ | 'for' singleNoun->np_ : PrepSingleNounProd
;

grammar toSingleNoun(main):
   singleNoun->np_ | 'to' singleNoun->np_ : PrepSingleNounProd
;

grammar throughSingleNoun(main):
   singleNoun->np_ | ('through' | 'thru') singleNoun->np_
   : PrepSingleNounProd
;

grammar fromSingleNoun(main):
   singleNoun->np_ | 'from' singleNoun->np_ : PrepSingleNounProd
;

grammar onSingleNoun(main):
   singleNoun->np_ | 'on' singleNoun->np_ : PrepSingleNounProd
;

grammar withSingleNoun(main):
   singleNoun->np_ | 'with' singleNoun->np_ : PrepSingleNounProd
;

grammar atSingleNoun(main):
   singleNoun->np_ | 'at' singleNoun->np_ : PrepSingleNounProd
;

grammar outOfSingleNoun(main):
   singleNoun->np_ | 'out' 'of' singleNoun->np_ : PrepSingleNounProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Complete noun phrase - this is a fully-qualified noun phrase that
 *   cannot be modified with articles, quantifiers, or anything else.
 *   This is the highest-level individual noun phrase.
 */

grammar completeNounPhrase(main):
    completeNounPhraseWithAll->np_ | completeNounPhraseWithoutAll->np_
    : LayeredNounPhraseProd
;

/*
 *   Better than a miscellaneous word list, but less good than an empty
 *   list, is a pair of otherwise valid noun phrases connected by a
 *   preposition that's commonly used in command phrases.  This will match
 *   commands where the user has assumed a command with a prepositional
 *   structure that doesn't exist among the defined commands, but since we
 *   have badness, we'll be ignored for valid command syntax.  
 */
grammar completeNounPhrase(miscPrep):
    [badness 200] completeNounPhrase->np1_
        ('with' | 'into' | 'in' 'to' | 'through' | 'thru' | 'for'
         | 'onto' | 'on' 'to' | 'at' | 'under' | 'behind')
        completeNounPhrase->np2_
    : NounPhraseProd
    resolveNouns(resolver, results)
    {
        /* resolve the underlying noun phrases, for scoring purposes */
        np1_.resolveNouns(resolver, results);
        np2_.resolveNouns(resolver, results);

        /* note that we have an invalid prepositional phrase structure */
        results.noteBadPrep();

        /* return nothing */
        return [];
    }
;


/*
 *   A qualified noun phrase can, all by itself, be a full noun phrase 
 */
grammar completeNounPhraseWithoutAll(qualified): qualifiedNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   Pronoun rules.  A pronoun is a complete noun phrase; it does not
 *   allow further qualification.  
 */
grammar completeNounPhraseWithoutAll(it):   'it' : ItProd;
grammar completeNounPhraseWithoutAll(them): 'them' : ThemProd;
grammar completeNounPhraseWithoutAll(him):  'him' : HimProd;
grammar completeNounPhraseWithoutAll(her):  'her' : HerProd;

/*
 *   Reflexive second-person pronoun, for things like "bob, look at
 *   yourself" 
 */
grammar completeNounPhraseWithoutAll(yourself):
    'yourself' | 'yourselves' | 'you' : YouProd
;

/*
 *   First-person pronoun, for referring to the speaker: "bob, look at me" 
 */
grammar completeNounPhraseWithoutAll(me): 'me' | 'myself' : MeProd;

/*
 *   "All" and "all but".
 *   
 *   "All" is a "complete" noun phrase, because there's nothing else
 *   needed to make it a noun phrase.  We make this a special kind of
 *   complete noun phrase because 'all' is not acceptable as a complete
 *   noun phrase in some contexts where any of the other complete noun
 *   phrases are acceptable.
 *   
 *   "All but" is a "terminal" noun phrase - this is a special kind of
 *   complete noun phrase that cannot be followed by another noun phrase
 *   with "and".  "All but" is terminal because we want any and's that
 *   follow it to be part of the exception list, so that we interpret
 *   "take all but a and b" as excluding a and b, not as excluding a but
 *   then including b as a separate list.  
 */
grammar completeNounPhraseWithAll(main):
    'all' | 'everything'
    : EverythingProd
;

grammar terminalNounPhrase(allBut):
    ('all' | 'everything') ('but' | 'except' | 'except' 'for')
        exceptList->except_
    : EverythingButProd
;

/*
 *   Plural phrase with an exclusion list.  This is a terminal noun phrase
 *   because it ends in an exclusion list.  
 */
grammar terminalNounPhrase(pluralExcept):
    (qualifiedPluralNounPhrase->np_ | detPluralNounPhrase->np_)
    ('except' | 'except' 'for' | 'but' | 'but' 'not') exceptList->except_
    : ListButProd
;

/*
 *   Qualified singular with an exception 
 */
grammar terminalNounPhrase(anyBut):
    'any' nounPhrase->np_
    ('but' | 'except' | 'except' 'for' | 'but' 'not') exceptList->except_
    : IndefiniteNounButProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A qualified noun phrase is a noun phrase with an optional set of
 *   qualifiers: a definite or indefinite article, a quantifier, words such
 *   as 'any' and 'all', possessives, and locational specifiers ("the box
 *   on the table").
 *   
 *   Without qualification, a definite article is implicit, so we read
 *   "take box" as equivalent to "take the box."
 *   
 *   Grammar rule instantiations in language-specific modules should set
 *   property np_ to the underlying noun phrase match tree.  
 */

/*
 *   A qualified noun phrase can be either singular or plural.  The number
 *   is a feature of the overall phrase; the phrase might consist of
 *   subphrases of different numbers (for example, "bob's coins" is plural
 *   even though it contains a singular subphrase, "bob"; and "one of the
 *   coins" is singular, even though its subphrase "coins" is plural).  
 */
grammar qualifiedNounPhrase(main):
    qualifiedSingularNounPhrase->np_
    | qualifiedPluralNounPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Singular qualified noun phrase. 
 */

/* 
 *   A singular qualified noun phrase with an implicit or explicit definite
 *   article.  If there is no article, a definite article is implied (we
 *   interpret "take box" as though it were "take the box").  
 */
grammar qualifiedSingularNounPhrase(definite):
    ('the' | 'the' 'one' | ) indetSingularNounPhrase->np_
    : DefiniteNounProd
;

/*
 *   A singular qualified noun phrase with an explicit indefinite article. 
 */
grammar qualifiedSingularNounPhrase(indefinite):
    ('a' | 'an') indetSingularNounPhrase->np_
    : IndefiniteNounProd
;

/*
 *   A singular qualified noun phrase with an explicit arbitrary
 *   determiner.  
 */
grammar qualifiedSingularNounPhrase(arbitrary):
    ('any' | 'one' | 'any' 'one') indetSingularNounPhrase->np_
    : ArbitraryNounProd
;

/*
 *   A singular qualified noun phrase with a possessive adjective. 
 */
grammar qualifiedSingularNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetSingularNounPhrase->np_
    : PossessiveNounProd
;

/*
 *   A singular qualified noun phrase that arbitrarily selects from a
 *   plural set.  This is singular, even though the underlying noun phrase
 *   is plural, because we're explicitly selecting one item.  
 */
grammar qualifiedSingularNounPhrase(anyPlural):
    'any' 'of' explicitDetPluralNounPhrase->np_
    : ArbitraryNounProd
;

/*
 *   A singular object specified only by its containment, with a definite
 *   article.  
 */
grammar qualifiedSingularNounPhrase(theOneIn):
    'the' 'one' ('that' 'is' | 'that\'s' | ) ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : VagueContainerDefiniteNounPhraseProd

    /* 
     *   our main phrase is simply 'one' (so disambiguation prompts will
     *   read "which one do you mean...") 
     */
    mainPhraseText = 'one'
;

/*
 *   A singular object specified only by its containment, with an
 *   indefinite article. 
 */
grammar qualifiedSingularNounPhrase(anyOneIn):
    'anything' | 'one' ('that' 'is' | 'that\'s' | ) ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : VagueContainerIndefiniteNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   An "indeterminate" singular noun phrase is a noun phrase without any
 *   determiner.  A determiner is a phrase that specifies the phrase's
 *   number and indicates whether or not it refers to a specific object,
 *   and if so fixes which object it refers to; determiners include
 *   articles ("the", "a") and possessives.  
 */

/* an indetermine noun phrase can be a simple noun phrase */
grammar indetSingularNounPhrase(basic):
    nounPhrase->np_
    : LayeredNounPhraseProd
;

/* 
 *   An indetermine noun phrase can specify a location for the object(s).
 *   The location must be a singular noun phrase, but can itself be a
 *   fully qualified noun phrase (so it can have possessives, articles,
 *   and locational qualifiers of its own).
 *   
 *   Note that we take 'that are' even though the noun phrase is singular,
 *   because what we consider a singular noun phrase can have plural usage
 *   ("scissors", for example).  
 */
grammar indetSingularNounPhrase(locational):
    nounPhrase->np_
    ('that' 'is' | 'that\'s' | 'that' 'are' | ) ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : ContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Plural qualified noun phrase. 
 */

/* 
 *   A simple unqualified plural phrase with determiner.  Since this form
 *   of plural phrase doesn't have any additional syntax that makes it an
 *   unambiguous plural, we can only accept an actual plural for the
 *   underlying phrase here - we can't accept an adjective phrase.  
 */
grammar qualifiedPluralNounPhrase(determiner):
    detPluralOnlyNounPhrase->np_
    : LayeredNounPhraseProd
;

/* plural phrase qualified with a number and optional "any" */
grammar qualifiedPluralNounPhrase(anyNum):
    ('any' | ) numberPhrase->quant_ indetPluralNounPhrase->np_
    | ('any' | ) numberPhrase->quant_ 'of' explicitDetPluralNounPhrase->np_
    : QuantifiedPluralProd
;

/* plural phrase qualified with a number and "all" */
grammar qualifiedPluralNounPhrase(allNum):
    'all' numberPhrase->quant_ indetPluralNounPhrase->np_
    | 'all' numberPhrase->quant_ 'of' explicitDetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

/* plural phrase qualified with "both" */
grammar qualifiedPluralNounPhrase(both):
    'both' detPluralNounPhrase->np_
    | 'both' 'of' explicitDetPluralNounPhrase->np_
    : BothPluralProd
;

/* plural phrase qualified with "all" */
grammar qualifiedPluralNounPhrase(all):
    'all' detPluralNounPhrase->np_
    | 'all' 'of' explicitDetPluralNounPhrase->np_
    : AllPluralProd
;

/* vague plural phrase with location specified */
grammar qualifiedPluralNounPhrase(theOnesIn):
    ('the' 'ones' ('that' 'are' | )
     | ('everything' | 'all') ('that' 'is' | 'that\'s' | ))
    ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : AllInContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with a determiner.  The determiner can be
 *   explicit (such as an article or possessive) or it can implied (the
 *   implied determiner is the definite article, so "take boxes" is
 *   understood as "take the boxes").  
 */
grammar detPluralNounPhrase(main):
    indetPluralNounPhrase->np_ | explicitDetPluralNounPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A determiner plural phrase with an explicit underlying plural (i.e.,
 *   excluding adjective phrases with no explicitly plural words).  
 */
grammar detPluralOnlyNounPhrase(main):
    implicitDetPluralOnlyNounPhrase->np_
    | explicitDetPluralOnlyNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   An implicit determiner plural phrase is an indeterminate plural phrase
 *   without any extra determiner - i.e., the determiner is implicit.
 *   We'll treat this the same way we do a plural explicitly determined
 *   with a definite article, since this is the most natural interpretation
 *   in English.
 *   
 *   (This might seem like a pointless extra layer in the grammar, but it's
 *   necessary for the resolution process to have a layer that explicitly
 *   declares the phrase to be determined, even though the determiner is
 *   implied in the structure.  This extra layer is important because it
 *   explicitly calls results.noteMatches(), which is needed for rankings
 *   and the like.)  
 */
grammar implicitDetPluralOnlyNounPhrase(main):
    indetPluralOnlyNounPhrase->np_
    : DefinitePluralProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with an explicit determiner. 
 */

/* a plural noun phrase with a definite article */
grammar explicitDetPluralNounPhrase(definite):
    'the' indetPluralNounPhrase->np_
    : DefinitePluralProd
;

/* a plural noun phrase with a definite article and a number */
grammar explicitDetPluralNounPhrase(definiteNumber):
    'the' numberPhrase->quant_ indetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

/* a plural noun phrase with a possessive */
grammar explicitDetPluralNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetPluralNounPhrase->np_
    : PossessivePluralProd
;

/* a plural noun phrase with a possessive and a number */
grammar explicitDetPluralNounPhrase(possessiveNumber):
    possessiveAdjPhrase->poss_ numberPhrase->quant_
    indetPluralNounPhrase->np_
    : ExactQuantifiedPossessivePluralProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A plural noun phrase with an explicit determiner and only an
 *   explicitly plural underlying phrase. 
 */
grammar explicitDetPluralOnlyNounPhrase(definite):
    'the' indetPluralOnlyNounPhrase->np_
    : AllPluralProd
;

grammar explicitDetPluralOnlyNounPhrase(definiteNumber):
    'the' numberPhrase->quant_ indetPluralNounPhrase->np_
    : ExactQuantifiedPluralProd
;

grammar explicitDetPluralOnlyNounPhrase(possessive):
    possessiveAdjPhrase->poss_ indetPluralOnlyNounPhrase->np_
    : PossessivePluralProd
;

grammar explicitDetPluralOnlyNounPhrase(possessiveNumber):
    possessiveAdjPhrase->poss_ numberPhrase->quant_
    indetPluralNounPhrase->np_
    : ExactQuantifiedPossessivePluralProd
;


/* ------------------------------------------------------------------------ */
/*
 *   An indeterminate plural noun phrase.
 *   
 *   For the basic indeterminate plural phrase, allow an adjective phrase
 *   anywhere a plural phrase is allowed; this makes possible the
 *   short-hand of omitting a plural word when the plural number is
 *   unambiguous from context.  
 */

/* a simple plural noun phrase */
grammar indetPluralNounPhrase(basic):
    pluralPhrase->np_ | adjPhrase->np_
    : LayeredNounPhraseProd
;

/* 
 *   A plural noun phrase with a locational qualifier.  Note that even
 *   though the overall phrase is plural (and so the main underlying noun
 *   phrase is plural), the location phrase itself must always be singular.
 */
grammar indetPluralNounPhrase(locational):
    (pluralPhrase->np_ | adjPhrase->np_)
    ('that' 'are' | ) ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : ContainerNounPhraseProd
;

/*
 *   An indetermine plural noun phrase with only explicit plural phrases. 
 */
grammar indetPluralOnlyNounPhrase(basic):
    pluralPhrase->np_
    : LayeredNounPhraseProd
;

grammar indetPluralOnlyNounPhrase(locational):
    pluralPhrase->np_
    ('that' 'are' | ) ('in' | 'on' | 'from')
    qualifiedSingularNounPhrase->cont_
    : ContainerNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Noun Phrase.  This is the basic noun phrase, which serves as a
 *   building block for complete noun phrases.  This type of noun phrase
 *   can be qualified with articles, quantifiers, and possessives, and can
 *   be used to construct possessives via the addition of "'s" at the end
 *   of the phrase.
 *   
 *   In most cases, custom noun phrase rules should be added to this
 *   production, as long as qualification (with numbers, articles, and
 *   possessives) is allowed.  For a custom noun phrase rule that cannot
 *   be qualified, a completeNounPhrase rule should be added instead.  
 */
grammar nounPhrase(main): compoundNounPhrase->np_
    : LayeredNounPhraseProd
;

/*
 *   Plural phrase.  This is the basic plural phrase, and corresponds to
 *   the basic nounPhrase for plural forms.
 */
grammar pluralPhrase(main): compoundPluralPhrase->np_
    : LayeredNounPhraseProd
;

/* ------------------------------------------------------------------------ */
/*
 *   Compound noun phrase.  This is one or more noun phrases connected
 *   with 'of', as in "piece of paper".  The part after the 'of' is
 *   another compound noun phrase.
 *   
 *   Note that this general rule does not allow the noun phrase after the
 *   'of' to be qualified with an article or number, except that we make
 *   an exception to allow a definite article.  Other cases ("a piece of
 *   four papers") do not generally make sense, so we won't attempt to
 *   support them; instead, games can add as special cases new nounPhrase
 *   rules for specific literal sequences where more complex grammar is
 *   necessary.  
 */
grammar compoundNounPhrase(simple): simpleNounPhrase->np_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        return np_.getVocabMatchList(resolver, results);
    }
    getAdjustedTokens()
    {
        return np_.getAdjustedTokens();
    }
;

grammar compoundNounPhrase(of):
    simpleNounPhrase->np1_ 'of'->of_ compoundNounPhrase->np2_
    | simpleNounPhrase->np1_ 'of'->of_ 'the'->the_ compoundNounPhrase->np2_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        local lst1;
        local lst2;

        /* resolve the two underlying lists */
        lst1 = np1_.getVocabMatchList(resolver, results);
        lst2 = np2_.getVocabMatchList(resolver, results);

        /* 
         *   the result is the intersection of the two lists, since we
         *   want the list of objects with all of the underlying
         *   vocabulary words 
         */
        return intersectNounLists(lst1, lst2);
    }
    getAdjustedTokens()
    {
        local ofLst;

        /* generate the 'of the' list from the original words */
        if (the_ == nil)
            ofLst = [of_, miscVocab];
        else
            ofLst = [of_, miscVocab, the_, miscVocab];

        /* return the full list */
        return np1_.getAdjustedTokens() + ofLst + np2_.getAdjustedTokens();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Compound plural phrase - same as a compound noun phrase, but
 *   involving a plural part before the 'of'. 
 */

/*
 *   just a single plural phrase 
 */
grammar compoundPluralPhrase(simple): simplePluralPhrase->np_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        return np_.getVocabMatchList(resolver, results);
    }
    getAdjustedTokens()
    {
        return np_.getAdjustedTokens();
    }
;

/*
 *   <plural-phrase> of <noun-phrase> 
 */
grammar compoundPluralPhrase(of):
    simplePluralPhrase->np1_ 'of'->of_ compoundNounPhrase->np2_
    | simplePluralPhrase->np1_ 'of'->of_ 'the'->the_ compoundNounPhrase->np2_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        local lst1;
        local lst2;

        /* resolve the two underlying lists */
        lst1 = np1_.getVocabMatchList(resolver, results);
        lst2 = np2_.getVocabMatchList(resolver, results);

        /* 
         *   the result is the intersection of the two lists, since we
         *   want the list of objects with all of the underlying
         *   vocabulary words 
         */
        return intersectNounLists(lst1, lst2);
    }
    getAdjustedTokens()
    {
        local ofLst;

        /* generate the 'of the' list from the original words */
        if (the_ == nil)
            ofLst = [of_, miscVocab];
        else
            ofLst = [of_, miscVocab, the_, miscVocab];

        /* return the full list */
        return np1_.getAdjustedTokens() + ofLst + np2_.getAdjustedTokens();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Simple noun phrase.  This is the most basic noun phrase, which is
 *   simply a noun, optionally preceded by one or more adjectives.  
 */

/*
 *   just a <noun> 
 */
grammar simpleNounPhrase(noun): noun->noun_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* return the list of objects in scope matching the noun */
        return getWordMatches(noun_, &noun, resolver, 0, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [noun_, &noun];
    }
;

/*
 *   <adjective> <simple-noun-phrase> (this allows any number of
 *   adjectives to be applied) 
 */
grammar simpleNounPhrase(adjNP): adjective->adj_ simpleNounPhrase->np_
    : NounPhraseWithVocab

    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* 
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying noun phrase 
         */
        return intersectWordMatches(adj_, &adjective, resolver,
                                    0, VocabTruncated,
                                    np_.getVocabMatchList(resolver, results));
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective] + np_.getAdjustedTokens();
    }
;

/*
 *   A simple noun phrase can also include a number before or after a
 *   noun.  The number can be spelled or written with numerals.
 *   
 *   The number in this case is equivalent to an adjective - it's not
 *   meant to quantify the rest of the noun phrase, but rather is simply
 *   an adjective-like specifier.  For example, an elevator's control
 *   panel might have a set of numbered buttons which we want to refer to
 *   as "button 1," "button 2," and so on.  It is frequently the case that
 *   numeric adjectives are equally at home before or after their noun:
 *   "push 3 button" or "push button 3".  In addition, we accept a number
 *   by itself as a lone adjective, as in "push 3".  
 */

/*
 *   just a numeric adjective (for things like "push 3")
 */
grammar simpleNounPhrase(number): numberPhrase->num_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        local numList;
        
        /* 
         *   get the list of objects matching the numeral form of the
         *   number as an adjective 
         */
        numList = getWordMatches(toString(num_.getval()), &adjective,
                                 resolver, 0, VocabTruncated);

        /* add the list of objects matching the special '#' wildcard */
        numList += getWordMatches('#', &adjective, resolver,
                                  0, VocabTruncated);

        /* return the combined lists */
        return numList;
    }
    getAdjustedTokens()
    {
        return [toString(num_.getval()), &adjective];
    }
;

/*
 *   <numeric-adjective> <noun> (for things like "board 44 bus")
 */
grammar simpleNounPhrase(numberAndNoun): numberPhrase->num_ noun->noun_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        local nounList;
        local numList;
        
        /* get the list of objects matching the rest of the noun phrase */
        nounList = getWordMatches(noun_, &noun, resolver, 0, VocabTruncated);

        /* 
         *   get the list of objects matching the numeral form of the
         *   number as an adjective 
         */
        numList = getWordMatches(toString(num_.getval()), &adjective,
                                 resolver, 0, VocabTruncated);

        /* add the list of objects matching the special '#' wildcard */
        numList += getWordMatches('#', &adjective, resolver,
                                  0, VocabTruncated);

        /* intersect the two lists and return the results */
        return intersectNounLists(nounList, numList);
    }
    getAdjustedTokens()
    {
        return [toString(num_.getval()), &adjective, noun_, &noun];
    }
;   

/*
 *   <noun> <numeric-adjective> (for things like "press button 3")
 */
grammar simpleNounPhrase(nounAndNumber): noun->noun_ numberPhrase->num_
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        local nounList;
        local numList;
        
        /* get the list of objects matching the rest of the noun phrase */
        nounList = getWordMatches(noun_, &noun, resolver, 0, VocabTruncated);

        /* 
         *   get the list of objects matching the numeral form of the
         *   number as an adjective 
         */
        numList = getWordMatches(toString(num_.getval()), &adjective,
                                 resolver, 0, VocabTruncated);

        /* add the list of objects matching the special '#' wildcard */
        numList += getWordMatches('#', &adjective, resolver,
                                  0, VocabTruncated);

        /* intersect the two lists and return the results */
        return intersectNounLists(nounList, numList);
    }
    getAdjustedTokens()
    {
        return [noun_, &noun, toString(num_.getval()), &adjective];
    }
;

/*
 *   A simple noun phrase can also end in an adjective, which allows
 *   players to refer to objects using only their unique adjectives rather
 *   than their full names, which is sometimes more convenient: just "take
 *   gold" rather than "take gold key."
 *   
 *   When a particular phrase can be interpreted as either ending in an
 *   adjective or ending in a noun, we will always take the noun-ending
 *   interpretation - in such cases, the adjective-ending interpretation
 *   is probably a weak binding.  For example, "take pizza" almost
 *   certainly refers to the pizza itself when "pizza" and "pizza box" are
 *   both present, but it can also refer just to the box when no pizza is
 *   present.
 *   
 *   Equivalent to a noun phrase ending in an adjective is a noun phrase
 *   ending with an adjective followed by "one," as in "the red one."  
 */
grammar simpleNounPhrase(adj): adjective->adj_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* note in the results that we end in an adjective */
        results.noteAdjEnding();

        /* generate a list of objects matching the adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              EndsWithAdj, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

grammar simpleNounPhrase(adjAndOne): adjective->adj_ 'one'
    : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* generate a list of objects matching the adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              EndsWithAdj, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

/*
 *   In the worst case, a simple noun phrase can be constructed from
 *   arbitrary words that don't appear in our dictionary.
 */
grammar simpleNounPhrase(misc):
    [badness 300] miscWordList->lst_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* note in the results that we have an arbitrary word list */
        results.noteMiscWordList();
        
        /* get the match list from the underlying list */
        return lst_.getVocabMatchList(resolver, results);
    }
    getAdjustedTokens()
    {
        return lst_.getAdjustedTokens();
    }
;

/*
 *   If the command has qualifiers but omits everything else, we can have
 *   an empty simple noun phrase.  
 */
grammar simpleNounPhrase(empty): [badness 200] : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* we have an empty noun phrase */
        return results.emptyNounPhrase(resolver);
    }
    getAdjustedTokens()
    {
        return [];
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Possessive phrase.  This is a noun phrase expressing ownership of
 *   another object.
 *   
 *   Note that all possessive phrases that can possibly be ambiguous must
 *   define getOrigMainText() to return the "main noun phrase" text.  In
 *   English, this means that we must omit any "'s" suffix.  This is
 *   needed only when the phrase can be ambiguous, so pronouns don't need
 *   it since they are inherently unambiguous.  
 */
grammar possessiveAdjPhrase(its): 'its' : ItsAdjProd;
grammar possessiveAdjPhrase(his): 'his' : HisAdjProd;
grammar possessiveAdjPhrase(her): 'her' : HerAdjProd;
grammar possessiveAdjPhrase(their): 'their' : TheirAdjProd;
grammar possessiveAdjPhrase(your): 'your' : YourAdjProd;
grammar possessiveAdjPhrase(my): 'my' : MyAdjProd;

grammar possessiveAdjPhrase(adj): possessiveAdj->adj_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* return the word matches */
        return getWordMatches(adj_, &possessiveAdj, resolver,
                              0, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &possessiveAdj];
    }
;

grammar possessiveAdjPhrase(npApostropheS):
    nounPhrase->np_ tokApostropheS->apost_ : LayeredNounPhraseProd

    /* get the original text without the "'s" suffix */
    getOrigMainText()
    {
        /* return just the basic noun phrase part */
        return np_.getOrigText();
    }
;

/*
 *   Possessive noun phrases.  These are similar to possessive phrases,
 *   but are stand-alone phrases that can act as nouns rather than as
 *   qualifiers for other noun phrases.  For example, for a first-person
 *   player character, "mine" would be a possessive noun phrase referring
 *   to an object owned by the player character.
 *   
 *   Note that many of the words used for possessive nouns are the same as
 *   for possessive adjectives - for example "his" is the same in either
 *   case, as are "'s" words.  However, we make the distinction internally
 *   because the two have different grammatical uses, and some of the
 *   words do differ ("her" vs "hers", for example).  
 */
grammar possessiveNounPhrase(its): 'its': ItsNounProd;
grammar possessiveNounPhrase(his): 'his': HisNounProd;
grammar possessiveNounPhrase(hers): 'hers': HersNounProd;
grammar possessiveNounPhrase(theirs): 'theirs': TheirsNounProd;
grammar possessiveNounPhrase(yours): 'yours' : YoursNounProd;
grammar possessiveNounPhrase(mine): 'mine' : MineNounProd;

grammar possessiveNounPhrase(noun):
    possessiveNoun->noun_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* return the word matches for our possessive noun */
        return getWordMatches(noun_, &possessiveNoun, resolver,
                              0, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [noun_, &possessiveNoun];
    }
;

grammar possessiveNounPhrase(npApostropheS):
    nounPhrase->np_ tokApostropheS->apost_ : LayeredNounPhraseProd

    /* get the original text without the "'s" suffix */
    getOrigMainText()
    {
        /* return just the basic noun phrase part */
        return np_.getOrigText();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Simple plural phrase.  This is the most basic plural phrase, which is
 *   simply a plural noun, optionally preceded by one or more adjectives.
 *   
 *   (English doesn't have any sort of adjective declension in number, so
 *   there's no need to distinguish between plural and singular
 *   adjectives; this equivalent rule in languages with adjective-noun
 *   agreement in number would use plural adjectives here as well as
 *   plural nouns.)  
 */
grammar simplePluralPhrase(plural): plural->plural_ : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* generate a list of objects matching the adjective */
        return getWordMatches(plural_, &plural, resolver, 0, PluralTruncated);
    }
    getAdjustedTokens()
    {
        return [plural_, &plural];
    }
;

grammar simplePluralPhrase(adj): adjective->adj_ simplePluralPhrase->np_ :
    NounPhraseWithVocab

    /* resolve my object list */
    getVocabMatchList(resolver, results)
    {
        /* 
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying noun phrase 
         */
        return intersectWordMatches(adj_, &adjective, resolver,
                                    0, VocabTruncated,
                                    np_.getVocabMatchList(resolver, results));
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective] + np_.getAdjustedTokens();
    }
;

/*
 *   A simple plural phrase can end with an adjective and "ones," as in
 *   "the red ones." 
 */
grammar simplePluralPhrase(adjAndOnes): adjective->adj_ 'ones'
    : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* generate a list of objects matching the adjective */
        return getWordMatches(adj_, &adjective, resolver,
                              EndsWithAdj, VocabTruncated);
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;    

/*
 *   If the command has qualifiers that require a plural, but omits
 *   everything else, we can have an empty simple noun phrase.  
 */
grammar simplePluralPhrase(empty): [badness 300] : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* we have an empty noun phrase */
        return results.emptyNounPhrase(resolver);
    }
    getAdjustedTokens()
    {
        return [];
    }
;

/*
 *   A simple plural phrase can match unknown words as a last resort. 
 */
grammar simplePluralPhrase(misc):
    [badness 600] miscWordList->lst_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* note in the results that we have an arbitrary word list */
        results.noteMiscWordList();
        
        /* get the match list from the underlying list */
        return lst_.getVocabMatchList(resolver, results);
    }
    getAdjustedTokens()
    {
        return lst_.getAdjustedTokens();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   An "adjective phrase" is a phrase made entirely of adjectives. 
 */
grammar adjPhrase(adj): adjective->adj_ : NounPhraseWithVocab
    getVocabMatchList(resolver, results)
    {
        /* note the adjective ending */
        results.noteAdjEnding();

        /* return the match list */
        return getWordMatches(adj_, &adjective, resolver,
                              EndsWithAdj, VocabTruncated);
    }

    getAdjustedTokens()
    {
        return [adj_, &adjective];
    }
;

grammar adjPhrase(adjAdj): adjective->adj_ adjPhrase->ap_
    : NounPhraseWithVocab
    /* generate a list of my resolved objects */
    getVocabMatchList(resolver, results)
    {
        /* 
         *   return the list of objects in scope matching our adjective
         *   plus the list from the underlying adjective phrase
         */
        return intersectWordMatches(adj_, &adjective, resolver,
                                    0, VocabTruncated,
                                    ap_.getVocabMatchList(resolver, results));
    }
    getAdjustedTokens()
    {
        return [adj_, &adjective] + ap_.getAdjustedTokens();
    }
;
    

/* ------------------------------------------------------------------------ */
/*
 *   A "topic" is a special type of noun phrase used in commands like "ask
 *   <actor> about <topic>."  We define a topic as simply an ordinary
 *   single-noun phrase.  We distinguish this in the grammar to allow
 *   games to add special syntax for these.  
 */
grammar topicPhrase(main): singleNoun->np_ : TopicProd
;

/* ------------------------------------------------------------------------ */
/*
 *   A "quoted string" phrase is a literal enclosed in single or double
 *   quotes.
 *   
 *   Note that this is a separate production from literalPhrase.  This
 *   production can be used when *only* a quoted string is allowed.  The
 *   literalPhrase production allows both quoted and unquoted text.  
 */
grammar quotedStringPhrase(main): tokString->str_ : LiteralProd
    /* 
     *   get my string, with the quotes trimmed off (so we return simply
     *   the contents of the string) 
     */
    getStringText()
    {
        local txt;
        local hasOpen;
        local hasClose;
        
        /* get the original text */
        txt = str_;

        /* presume we won't find open or close quotes */
        hasOpen = hasClose = nil;

        /* 
         *   Check for quotes.  We'll accept regular ASCII "straight"
         *   single or double quotes, as well as Latin-1 curly single or
         *   double quotes.  The curly quotes must be used in their normal
         
         */
        if (txt.startsWith('\'') || txt.startsWith('"'))
        {
            /* single or double quote - check for a matching close quote */
            hasOpen = true;
            hasClose = (txt.length() > 2 && txt.endsWith(txt.substr(1, 1)));
        }
        else if (txt.startsWith('`'))
        {
            /* single in-slanted quote - check for either type of close */
            hasOpen = true;
            hasClose = (txt.length() > 2
                        && (txt.endsWith('`') || txt.endsWith('\'')));
        }
        else if (txt.startsWith('\u201C'))
        {
            /* it's a curly double quote */
            hasOpen = true;
            hasClose = txt.endsWith('\u201D');
        }
        else if (txt.startsWith('\u2018'))
        {
            /* it's a curly single quote */
            hasOpen = true;
            hasClose = txt.endsWith('\u2019');
        }

        /* trim off the quotes */
        if (hasOpen)
        {
            if (hasClose)
                txt = txt.substr(2, txt.length() - 2);
            else
                txt = txt.substr(2);
        }

        /* return the modified text */
        return txt;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A "literal" is essentially any phrase.  This can include a quoted
 *   string, a number, or any set of word tokens.  
 */
grammar literalPhrase(string): quotedStringPhrase->str_ : LiteralProd
    getLiteralText(results, action, which)
    {
        /* get the text from our underlying quoted string */
        return str_.getStringText();
    }
    
    getTentativeLiteralText()
    {
        /* 
         *   our result will never change, so our tentative text is the
         *   same as our regular literal text 
         */
        return str_.getStringText();
    }

    resolveLiteral(results)
    {
        /* flag the literal text */
        results.noteLiteral(str_.getOrigText());
    }
;

grammar literalPhrase(miscList): miscWordList->misc_ : LiteralProd
    getLiteralText(results, action, which)
    {
        /* my literal text is simply my original text */
        return misc_.getOrigText();
    }

    getTentativeLiteralText()
    {
        /* our regular text is permanent, so simply use it now */
        return misc_.getOrigText();
    }

    resolveLiteral(results)
    {
        /* 
         *   note the length of our literal phrase - when we have a choice
         *   of interpretations, we prefer to choose shorter literal
         *   phrases, since this means that we'll have more of our tokens
         *   being fully interpreted rather than bunched into an
         *   uninterpreted literal 
         */
        results.noteLiteral(misc_.getOrigText());
    }
;

grammar literalPhrase(empty): [badness 100]: EmptyLiteralPhraseProd
    resolveLiteral(results) { }
;

/* ------------------------------------------------------------------------ */
/*
 *   A main disambiguation phrase consists of a disambiguation phrase,
 *   optionally terminated with a period.  
 */
grammar mainDisambigPhrase(main): 
    disambigPhrase->dp_
    | disambigPhrase->dp_ '.'
    : BasicProd
    resolveNouns(resolver, results)
    {
        return dp_.resolveNouns(resolver, results);
    }
    getResponseList() { return dp_.getResponseList(); }
;

/*
 *   Basic disambiguation production class 
 */
class DisambigProd: BasicProd
    /*
     *   Remove the "ambiguous" flags from a result list.  This can be
     *   used to mark the response to a disambiguation query as no longer
     *   ambiguous.  
     */
    removeAmbigFlags(lst)
    {
        /* remove the "unclear disambig" flag from each result item */
        foreach (local cur in lst)
            cur.flags_ &= ~UnclearDisambig;

        /* return the list */
        return lst;
    }
;

/*
 *   A "disambiguation phrase" is a phrase that answers a disambiguation
 *   question ("which book do you mean...").
 *   
 *   A disambiguation question can be answered with several types of
 *   syntax:
 *   
 *.  all/everything/all of them
 *.  both/both of them
 *.  any/any of them
 *.  <disambig list>
 *.  the <ordinal list> ones
 *.  the former/the latter
 *   
 *   Note that we assign non-zero badness to all of the ordinal
 *   interpretations, so that we will take an actual vocabulary
 *   interpretation instead of an ordinal interpretation whenever
 *   possible.  For example, if an object's name is actually "the third
 *   button," this will give us greater affinity for using "third" as an
 *   adjective than as an ordinal in our own list.  
 */
grammar disambigPhrase(all):
    'all' | 'everything' | 'all' 'of' 'them' : DisambigProd
    resolveNouns(resolver, results)
    {
        /* they want everything we proposed - return the whole list */
        return removeAmbigFlags(resolver.getAll());
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(both): 'both' | 'both' 'of' 'them' : DisambigProd
    resolveNouns(resolver, results)
    {
        /* 
         *   they want two items - return the whole list (if it has more
         *   than two items, we'll simply act as though they wanted all of
         *   them) 
         */
        return removeAmbigFlags(resolver.getAll());
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(any): 'any' | 'any' 'of' 'them' : DisambigProd
    resolveNouns(resolver, results)
    {
        local lst;
        
        /* they want any item - arbitrarily pick the first one */
        lst = resolver.matchList.sublist(1, 1);

        /* 
         *   add the "unclear disambiguation" flag to the item we picked,
         *   to indicate that the selection was arbitrary 
         */
        if (lst.length() > 0)
            lst[1].flags_ |= UnclearDisambig;

        /* return the result */
        return lst;
    }

    /* there's only me in the response list */
    getResponseList() { return [self]; }
;

grammar disambigPhrase(list): disambigList->lst_ : DisambigProd
    resolveNouns(resolver, results)
    {
        return removeAmbigFlags(lst_.resolveNouns(resolver, results));
    }
    
    /* there's only me in the response list */
    getResponseList() { return lst_.getResponseList(); }
;

grammar disambigPhrase(ordinalList):
    disambigOrdinalList->lst_ 'ones'
    | 'the' disambigOrdinalList->lst_ 'ones'
    : DisambigProd

    resolveNouns(resolver, results)
    {
        /* return the list with the ambiguity flags removed */
        return removeAmbigFlags(lst_.resolveNouns(resolver, results));
    }

    /* the response list consists of my single ordinal list item */
    getResponseList() { return [lst_]; }
;

/*
 *   A disambig list consists of one or more disambig list items,
 *   connected by noun phrase conjunctions.  
 */
grammar disambigList(single): disambigListItem->item_ : DisambigProd
    resolveNouns(resolver, results)
    {
        return item_.resolveNouns(resolver, results);
    }

    /* the response list consists of my single item */
    getResponseList() { return [item_]; }
;

grammar disambigList(list):
    disambigListItem->item_ commandOrNounConjunction disambigList->lst_
    : DisambigProd

    resolveNouns(resolver, results)
    {
        return item_.resolveNouns(resolver, results)
            + lst_.resolveNouns(resolver, results);
    }

    /* my response list consists of each of our list items */
    getResponseList() { return [item_] + lst_.getResponseList(); }
;

/*
 *   Base class for ordinal disambiguation items 
 */
class DisambigOrdProd: DisambigProd
    resolveNouns(resolver, results)
    {
        /* note the ordinal match */
        results.noteDisambigOrdinal();

        /* select the result by the ordinal */
        return selectByOrdinal(ord_, resolver, results);
    }

    selectByOrdinal(ordTok, resolver, results)
    {
        local idx;
        local matchList = resolver.ordinalMatchList;
        
        /* 
         *   look up the meaning of the ordinal word (note that we assume
         *   that each ordinalWord is unique, since we only create one of
         *   each) 
         */
        idx = G_dict.findWord(ordTok, &ordinalWord)[1].numval;

        /* 
         *   if it's the special value -1, it indicates that we should
         *   select the *last* item in the list 
         */
        if (idx == -1)
            idx = matchList.length();

        /* if it's outside the limits of the match list, it's an error */
        if (idx > matchList.length())
        {
            /* note the problem */
            results.noteOrdinalOutOfRange(ordTok);

            /* no results */
            return [];
        }

        /* return the selected item as a one-item list */
        return matchList.sublist(idx, 1);
    }
;

/*
 *   A disambig vocab production is the base class for disambiguation
 *   phrases that involve vocabulary words. 
 */
class DisambigVocabProd: DisambigProd
;

/*
 *   A disambig list item consists of:
 *   
 *.  first/second/etc
 *.  the first/second/etc
 *.  first one/second one/etc
 *.  the first one/the second one/etc
 *.  <compound noun phrase>
 *.  possessive
 */

grammar disambigListItem(ordinal):
    ordinalWord->ord_
    | ordinalWord->ord_ 'one'
    | 'the' ordinalWord->ord_
    | 'the' ordinalWord->ord_ 'one'
    : DisambigOrdProd
;

grammar disambigListItem(noun):
    qualifiedNounPhrase->np_
    | terminalNounPhrase->np_
    : DisambigVocabProd
    resolveNouns(resolver, results)
    {
        /* get the matches for the underlying noun phrase */
        return np_.resolveNouns(resolver, results);
    }
;

grammar disambigListItem(plural):
    pluralPhrase->np_
    : DisambigVocabProd
    resolveNouns(resolver, results)
    {
        local lst;
        
        /* 
         *   get the underlying match list; since we explicitly have a
         *   plural, the result doesn't need to be unique, so simply
         *   return everything we find 
         */
        lst = np_.resolveNouns(resolver, results);

        /* 
         *   if we didn't get anything, it's an error; otherwise, take
         *   everything, since we explicitly wanted a plural usage 
         */
        if (lst.length() == 0)
            results.noMatch(np_.getOrigText());

        /* return the list */
        return lst;
    }
;

grammar disambigListItem(possessive): possessiveNounPhrase->poss_
    : BasicPossessiveProd, DisambigProd
    resolveNouns(resolver, results)
    {
        local lst;

        /* select from the match list using the possessive phrase */
        lst = selectWithPossessive(resolver, results,
                                   resolver.matchList, resolver.matchText);

        /* 
         *   if the list has more than one entry, treat the result as
         *   still ambiguous - a simple possessive response to a
         *   disambiguation query is implicitly definite, so must select a
         *   single object 
         */
        if (lst != nil && lst.length() > 1)
            results.ambiguousNounPhrase(self, resolver.matchText, lst,
                                        resolver.fullMatchList, lst,
                                        1, resolver);

        /* 
         *   if we failed to resolve it, return an empty list; otherwise
         *   return the list 
         */
        return (lst == nil ? [] : lst);
    }
;

/*
 *   A disambig ordinal list consists of two or more ordinal words
 *   separated by noun phrase conjunctions.  Note that there is a minimum
 *   of two entries in the list.  
 */
grammar disambigOrdinalList(tail):
    ordinalWord->ord1_ ('and' | ',') ordinalWord->ord2_ : DisambigOrdProd
    resolveNouns(resolver, results)
    {
        /* note the pair of ordinal matches */
        results.noteDisambigOrdinal();
        results.noteDisambigOrdinal();
        
        /* combine the selections of our two ordinals */
        return selectByOrdinal(ord1_, resolver, results)
            + selectByOrdinal(ord2_, resolver, results);
    }
;
   
grammar disambigOrdinalList(head):
    ordinalWord->ord_ ('and' | ',') disambigOrdinalList->lst_
    : DisambigOrdProd
    resolveNouns(resolver, results)
    {
        /* note the ordinal match */
        results.noteDisambigOrdinal();
        
        /* combine the selections of our ordinal and the sublist */
        return selectByOrdinal(ord_, resolver, results)
            + lst_.resolveNouns(resolver, results);
    }
;
   

/* ------------------------------------------------------------------------ */
/*
 *   Ordinal words.  We define a limited set of these, since we only use
 *   them in a few special contexts where it would be unreasonable to need
 *   even as many as define here.  
 */
#define defOrdinal(str, val) object ordinalWord=#@str numval=val

defOrdinal(former, 1);
defOrdinal(first, 1);
defOrdinal(second, 2);
defOrdinal(third, 3);
defOrdinal(fourth, 4);
defOrdinal(fifth, 5);
defOrdinal(sixth, 6);
defOrdinal(seventh, 7);
defOrdinal(eighth, 8);
defOrdinal(ninth, 9);
defOrdinal(tenth, 10);
defOrdinal(eleventh, 11);
defOrdinal(twelfth, 12);
defOrdinal(thirteenth, 13);
defOrdinal(fourteenth, 14);
defOrdinal(fifteenth, 15);
defOrdinal(sixteenth, 16);
defOrdinal(seventeenth, 17);
defOrdinal(eighteenth, 18);
defOrdinal(nineteenth, 19);
defOrdinal(twenthieth, 20);
defOrdinal(1st, 1);
defOrdinal(2nd, 2);
defOrdinal(3rd, 3);
defOrdinal(4th, 4);
defOrdinal(5th, 5);
defOrdinal(6th, 6);
defOrdinal(7th, 7);
defOrdinal(8th, 8);
defOrdinal(9th, 9);
defOrdinal(10th, 10);
defOrdinal(11th, 11);
defOrdinal(12th, 12);
defOrdinal(13th, 13);
defOrdinal(14th, 14);
defOrdinal(15th, 15);
defOrdinal(16th, 16);
defOrdinal(17th, 17);
defOrdinal(18th, 18);
defOrdinal(19th, 19);
defOrdinal(20th, 20);

/* 
 *   the special 'last' ordinal - the value -1 is special to indicate the
 *   last item in a list 
 */
defOrdinal(last, -1);
defOrdinal(latter, -1);


/* ------------------------------------------------------------------------ */
/*
 *   A quantifier is simply a number, entered with numerals or spelled
 *   out. 
 */
grammar numberPhrase(digits): tokInt->num_ : BasicProd
    getval() { return toInteger(num_); }
;

grammar numberPhrase(spelled): spelledNumber->num_ : BasicProd
    getval() { return num_.getval(); }
;

/*
 *   Number literals.  We'll define a set of special objects for numbers:
 *   each object defines a number and a value for the number.  
 */
#define defDigit(num, val) object digitWord=#@num numval=val
#define defTeen(num, val)  object teenWord=#@num numval=val
#define defTens(num, val)  object tensWord=#@num numval=val

defDigit(one, 1);
defDigit(two, 2);
defDigit(three, 3);
defDigit(four, 4);
defDigit(five, 5);
defDigit(six, 6);
defDigit(seven, 7);
defDigit(eight, 8);
defDigit(nine, 9);
defTeen(ten, 10);
defTeen(eleven, 11);
defTeen(twelve, 12);
defTeen(thirteen, 13);
defTeen(fourteen, 14);
defTeen(fifteen, 15);
defTeen(sixteen, 16);
defTeen(seventeen, 17);
defTeen(eighteen, 18);
defTeen(nineteen, 19);
defTens(twenty, 20);
defTens(thirty, 30);
defTens(forty, 40);
defTens(fifty, 50);
defTens(sixty, 60);
defTens(seventy, 70);
defTens(eighty, 80);
defTens(ninety, 90);

grammar spelledSmallNumber(digit): digitWord->num_ : BasicProd
    getval()
    {
        /* 
         *   Look up the units word - there should be only one in the
         *   dictionary, since these are our special words.  Return the
         *   object's numeric value property 'numval', which gives the
         *   number for the name.  
         */
        return G_dict.findWord(num_, &digitWord)[1].numval;
    }
;

grammar spelledSmallNumber(teen): teenWord->num_ : BasicProd
    getval()
    {
        /* look up the dictionary word for the number */
        return G_dict.findWord(num_, &teenWord)[1].numval;
    }
;

grammar spelledSmallNumber(tens): tensWord->num_ : BasicProd
    getval()
    {
        /* look up the dictionary word for the number */
        return G_dict.findWord(num_, &tensWord)[1].numval;
    }
;

grammar spelledSmallNumber(tensAndUnits):
    tensWord->tens_ '-'->sep_ digitWord->units_
    | tensWord->tens_ digitWord->units_
    : BasicProd
    getval()
    {
        /* look up the words, and add up the values */
        return G_dict.findWord(tens_, &tensWord)[1].numval
            + G_dict.findWord(units_, &digitWord)[1].numval;
    }
;

grammar spelledSmallNumber(zero): 'zero' : BasicProd
    getval() { return 0; }
;

grammar spelledHundred(small): spelledSmallNumber->num_ : BasicProd
    getval() { return num_.getval(); }
;

grammar spelledHundred(hundreds): spelledSmallNumber->hun_ 'hundred'
    : BasicProd
    getval() { return hun_.getval() * 100; }
;

grammar spelledHundred(hundredsPlus):
    spelledSmallNumber->hun_ 'hundred' spelledSmallNumber->num_
    | spelledSmallNumber->hun_ 'hundred' 'and'->and_ spelledSmallNumber->num_
    : BasicProd
    getval() { return hun_.getval() * 100 + num_.getval(); }
;

grammar spelledHundred(aHundred): 'a' 'hundred' : BasicProd
    getval() { return 100; }
;

grammar spelledHundred(aHundredPlus):
    'a' 'hundred' 'and' spelledSmallNumber->num_
    : BasicProd
    getval() { return 100 + num_.getval(); }
;

grammar spelledThousand(thousands): spelledHundred->thou_ 'thousand'
    : BasicProd
    getval() { return thou_.getval() * 1000; }
;

grammar spelledThousand(thousandsPlus):
    spelledHundred->thou_ 'thousand' spelledHundred->num_
    : BasicProd
    getval() { return thou_.getval() * 1000 + num_.getval(); }
;

grammar spelledThousand(thousandsAndSmall):
    spelledHundred->thou_ 'thousand' 'and' spelledSmallNumber->num_
    : BasicProd
    getval() { return thou_.getval() * 1000 + num_.getval(); }
;

grammar spelledThousand(aThousand): 'a' 'thousand' : BasicProd
    getval() { return 1000; }
;

grammar spelledThousand(aThousandAndSmall):
    'a' 'thousand' 'and' spelledSmallNumber->num_
    : BasicProd
    getval() { return 1000 + num_.getval(); }
;

grammar spelledMillion(millions): spelledHundred->mil_ 'million': BasicProd
    getval() { return mil_.getval() * 1000000; }
;

grammar spelledMillion(millionsPlus):
    spelledHundred->mil_ 'million' spelledThousand->thou_
    : BasicProd
    getval() { return mil_.getval() * 1000000 + thou_.getval(); }
;

grammar spelledMillion(aMillion): 'a' 'million' : BasicProd
    getval() { return 1000000; }
;

grammar spelledMillion(aMillionAndSmall):
    'a' 'million' 'and' spelledSmallNumber->num_
    : BasicProd
    getval() { return 1000000 + num_.getval(); }
;

grammar spelledMillion(millionsAndSmall):
    spelledHundred->mil_ 'million' 'and' spelledSmallNumber->num_
    : BasicProd
    getval() { return mil_.getval() * 1000000 + num_.getval(); }
;

grammar spelledNumber(main):
    spelledHundred->num_
    | spelledThousand->num_
    | spelledMillion->num_
    : BasicProd
    getval() { return num_.getval(); }
;


/* ------------------------------------------------------------------------ */
/*
 *   "OOPS" command syntax 
 */
grammar oopsCommand(main):
    oopsPhrase->oops_ | oopsPhrase->oops_ '.' : BasicProd
    getNewTokens() { return oops_.getNewTokens(); }
;

grammar oopsPhrase(main):
    'oops' miscWordList->lst_
    | 'oops' ',' miscWordList->lst_
    | 'o' miscWordList->lst_
    | 'o' ',' miscWordList->lst_
    : BasicProd
    getNewTokens() { return lst_.getOrigTokenList(); }
;

/* ------------------------------------------------------------------------ */
/*
 *   finishGame options.  We provide descriptions and keywords for the
 *   option objects here, because these are inherently language-specific.
 *   
 *   Note that we provide hyperlinks for our descriptions when possible.
 *   When we're in plain text mode, we can't show links, so we'll instead
 *   show an alternate form with the single-letter response highlighted in
 *   the text.  We don't highlight the single-letter response in the
 *   hyperlinked version because (a) if the user wants a shortcut, they
 *   can simply click the hyperlink, and (b) most UI's that show
 *   hyperlinks show a distinctive appearance for the hyperlink itself, so
 *   adding even more highlighting within the hyperlink starts to look
 *   awfully busy.  
 */
modify finishOptionQuit
    desc = "<<aHrefAlt('quit', 'QUIT', '<b>Q</b>UIT', 'Leave the story')>>"
    responseKeyword = 'quit'
    responseChar = 'q'
;

modify finishOptionRestore
    desc = "<<aHrefAlt('restore', 'RESTORE', '<b>R</b>ESTORE',
            'Restore a saved position')>> a saved position"
    responseKeyword = 'restore'
    responseChar = 'r'
;

modify finishOptionRestart
    desc = "<<aHrefAlt('restart', 'RESTART', 'RE<b>S</b>TART',
            'Start the story over from the beginning')>> the story"
    responseKeyword = 'restart'
    responseChar = 's'
;

modify finishOptionUndo
    desc = "<<aHrefAlt('undo', 'UNDO', '<b>U</b>NDO',
            'Undo the last move')>> the last move"
    responseKeyword = 'undo'
    responseChar = 'u'
;

modify finishOptionCredits
    desc = "show the <<aHrefAlt('credits', 'CREDITS', '<b>C</b>REDITS',
            'Show credits')>>"
    responseKeyword = 'credits'
    responseChar = 'c'
;

modify finishOptionAmusing
    desc = "show some <<aHrefAlt('amusing', 'AMUSING', '<b>A</b>MUSING',
            'Show some amusing things to try')>>
            things to try"
    responseKeyword = 'amusing'
    responseChar = 'a'
;

/* ------------------------------------------------------------------------ */
/*
 *   Language-specific Action modifications.
 */
modify Action
    /*
     *   In the English grammar, all 'predicate' grammar definitions
     *   (which are usually made via the VerbRule macro) are associated
     *   with Action match tree objects; in fact, each 'predicate' grammar
     *   match tree is the specific Action subclass associated with the
     *   grammar for the predicate.  This means that the Action associated
     *   with a grammar match is simply the grammar match object itself.
     *   Hence, we can resolve the action for a 'predicate' match simply
     *   by returning the match itself: it is the Action as well as the
     *   grammar match.
     *   
     *   This approach ('grammar predicate' matches are based on Action
     *   subclasses) works well for languages like English that encode the
     *   role of each phrase in the word order of the sentence.
     *   
     *   Languages that encode phrase roles using case markers or other
     *   devices tend to be freer with word order.  As a result,
     *   'predicate' grammars for such languages should generally not
     *   attempt to capture all possible word orderings for a given
     *   action, but should instead take the complementary approach of
     *   capturing the possible overall sentence structures independently
     *   of verb phrases, and plug in a verb phrase as a component, just
     *   like noun phrases plug into the English grammar.  In these cases,
     *   the match objects will NOT be Action subclasses; the Action
     *   objects will instead be buried down deeper in the match tree.
     *   Hence, resolveAction() must be defined on whatever class is used
     *   to construct 'predicate' grammar matches, instead of on Action,
     *   since Action will not be a 'predicate' match.  
     */
    resolveAction(issuingActor, targetActor) { return self; }

    /* 
     *   Display the interrogative pronoun for a missing object in one of
     *   our object roles.  In most cases, this is simply "what", but for
     *   some actions, "whom" is more appropriate (for example, the direct
     *   object of "ask" is implicitly a person, so "whom" is most
     *   appropriate for this role).  
     */
    whatObj(which)
    {
        /* intransitive verbs have no objects, so there's nothing to show */
    }

    /* 
     *   Translate an interrogative word for whatObj.  If the word is
     *   'whom', translate to the library message for 'whom'; this allows
     *   authors to use 'who' rather than 'whom' as the objective form of
     *   'who', which sounds less stuffy to many people.  
     */
    whatTranslate(txt)
    {
        /* 
         *   if it's 'whom', translate to the library message for 'whom';
         *   otherwise, just show the word as is 
         */
        return (txt == 'whom' ? libMessages.whomPronoun : txt);
    }

    /*
     *   Return a string with the appropriate pronoun (objective form) for
     *   a list of object matches, with the given resolved cardinality.
     *   This list is a list of ResolveInfo objects.  
     */
    objListPronoun(objList)
    {
        local himCnt, herCnt, themCnt;
        local FirstPersonCnt, SecondPersonCnt;
        local resolvedNumber;

        /* if there's no object list at all, just use 'it' */
        if (objList == nil || objList == [])
            return 'it';
        
        /* note the number of objects in the resolved list */
        resolvedNumber = objList.length();

        /* 
         *   In the tentatively resolved object list, we might have hidden
         *   away ambiguous matches.  Expand those back into the list so
         *   we have the full list of in-scope matches.
         */
        foreach (local cur in objList)
        {
            /* 
             *   if this one has hidden ambiguous objects, add the hidden
             *   objects back into our list 
             */
            if (cur.extraObjects != nil)
                objList += cur.extraObjects;
        }
        
        /* 
         *   if the desired cardinality is plural and the object list has
         *   more than one object, simply say 'them' 
         */
        if (objList.length() > 1 && resolvedNumber > 1)
            return 'them';

        /* 
         *   singular cardinality - count masculine and feminine objects,
         *   and count the referral persons 
         */
        himCnt = herCnt = themCnt = 0;
        FirstPersonCnt = SecondPersonCnt = 0;
        foreach (local cur in objList)
        {
            /* if it's masculine, count it */
            if (cur.obj_.isHim)
                ++himCnt;

            /* if it's feminine, count it */
            if (cur.obj_.isHer)
                ++herCnt;

            /* if it has plural usage, count it */
            if (cur.obj_.isPlural)
                ++themCnt;

            /* if it's first person usage, count it */
            if (cur.obj_.referralPerson == FirstPerson)
                ++FirstPersonCnt;

            /* if it's second person usage, count it */
            if (cur.obj_.referralPerson == SecondPerson)
                ++SecondPersonCnt;
        }

        /* 
         *   if they all have plural usage, show "them"; if they're all of
         *   one gender, show "him" or "her" as appropriate; if they're
         *   all neuter, show "it"; otherwise, show "them" 
         */
        if (themCnt == objList.length())
            return 'them';
        else if (FirstPersonCnt == objList.length())
            return 'myself';
        else if (SecondPersonCnt == objList.length())
            return 'yourself';
        else if (himCnt == objList.length() && herCnt == 0)
            return 'him';
        else if (herCnt == objList.length() && himCnt == 0)
            return 'her';
        else if (herCnt == 0 && himCnt == 0)
            return 'it';
        else
            return 'them';
    }

    /*
     *   Announce a default object used with this action.
     *   
     *   'resolvedAllObjects' indicates where we are in the command
     *   processing: this is true if we've already resolved all of the
     *   other objects in the command, nil if not.  We use this
     *   information to get the phrasing right according to the situation.
     */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /* 
         *   the basic action class takes no objects, so there can be no
         *   default announcement 
         */
        return '';
    }

    /*
     *   Announce all defaulted objects in the action.  By default, we
     *   show nothing.  
     */
    announceAllDefaultObjects(allResolved) { }

    /*
     *   Show the base present-tense form of our verb phrase as part of an
     *   infinitive.  
     */
    verbInf(which)
    {
        /* 
         *   show our verb phrase, stripping out the participle
         *   information (given by the "/participle" string in the
         *   verbPhrase string)
         */
        rexMatch('(.*)/<alphanum|-|squote>+(.*)', verbPhrase);
        say(rexGroup(1)[3] + rexGroup(2)[3]);
    }

    /*
     *   Get a string describing the full action in present participle
     *   form, using the current command objects: "taking the watch",
     *   "putting the book on the shelf" 
     */
    getParticiplePhrase()
    {
        /* 
         *   the participle phrasing is the part after the "/" in the
         *   verbPhrase 
         */
        rexSearch('/(.*)', verbPhrase);
        return rexGroup(1)[3];
    }

    /*
     *   Verb flags - these are used to control certain aspects of verb
     *   formatting.  By default, we have no special flags.  
     */
    verbFlags = 0
;

/*
 *   English-specific additions for single-object verbs. 
 */
modify TAction
    /* show an interrogative word for an object of the action */
    whatObj(which)
    {
        /* 
         *   show the interrogative for our direct object - this is the
         *   word enclosed in parentheses in our verbPhrase string 
         */
        rexSearch('<lparen>(.*)<rparen>', verbPhrase);
        say(whatTranslate(rexGroup(1)[3]));
    }

    /* announce a default object used with this action */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        local prep;
        
        /* 
         *   get any direct object preposition - this is the part from the
         *   verb phrase (the first "word/word" pair) up to the
         *   parenthesized interrogative for the direct object 
         */
        rexSearch('<alphanum|-|squote>+/<alphanum>+ (.*)<lparen>',
                  verbPhrase);
        if (rexGroup(1)[2] != 0)
            prep = rexGroup(1)[3] + ' ';
        else
            prep = '';

        /* show the preposition (if any) and the object */
        return prep + obj.theNameObj;
    }

    /* announce all defaulted objects */
    announceAllDefaultObjects(allResolved)
    {
        /* announce a defaulted direct object if appropriate */
        maybeAnnounceDefaultObject(dobjList_, DirectObject, allResolved);
    }

    /* show the verb's basic infinitive form */
    verbInf(which)
    {
        /* 
         *   show the present-tense verb form (removing the participle
         *   part - the "/xxxing" part), but do not include any words for
         *   the object (the "(what)" representing the direct object) 
         */
        rexSearch('(.*)/<alphanum|-|squote>+(.*) <lparen>.*<rparen>(.*)',
                  verbPhrase);
        say(rexGroup(1)[3] + rexGroup(2)[3] + rexGroup(3)[3]);
    }

    /* describe the full action in present participle form */
    getParticiplePhrase()
    {
        local ret;
        
        /* 
         *   the participle form is the part from the "/" to the
         *   interrogative word in parentheses 
         */
        rexSearch('/(<^lparen>+)<lparen>', verbPhrase);
        ret = rexGroup(1)[3];

        /* add the direct object */
        ret += ' ' + getDobj().theNameObj;

        /* 
         *   if there's anything after the direct object interrogative in
         *   the verbPhrase, it's another preposition to show at the end
         *   of the participle phrase 
         */
        rexSearch('<rparen>(.*)', verbPhrase);
        if (rexGroup(1)[2] != 0)
            ret += ' ' + rexGroup(1)[3];

        /* return the complete phrase string */
        return ret;
    }
;

/*
 *   English-specific additions for two-object verbs. 
 */
modify TIAction
    /* show the interrogative for one of our objects */
    whatObj(which)
    {
        switch (which)
        {
        case DirectObject:
            /* 
             *   the direct object interrogative is the first word in
             *   parentheses in our verbPhrase string 
             */
            rexSearch('<lparen>(<^rparen>*)<rparen>', verbPhrase);
            break;

        case IndirectObject:
            /* 
             *   the indirect object interrogative is the second
             *   parenthesized word in our verbPhrase string 
             */
            rexSearch('<rparen>.*<lparen>(.*)<rparen>', verbPhrase);
            break;
        }

        /* show the group match */
        say(whatTranslate(rexGroup(1)[3]));
    }

    /* announce a default object used with this action */
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        local verb;
        local prep;

        /* presume we won't have a verb or preposition */
        verb = '';
        prep = '';

        /* 
         *   Check the full phrasing - if we're showing the direct object,
         *   but an indirect object was supplied, use the verb's
         *   participle form ("asking bob") in the default string, since
         *   we must clarify that we're not tagging the default string on
         *   to the command line.  Don't include the participle form if we
         *   don't know all the objects yet, since in this case we are in
         *   fact tagging the default string onto the command so far, as
         *   there's nothing else in the command to get in the way.  
         */
        if (whichObj == DirectObject
            && resolvedAllObjects)
        {
            /* extract the verb's participle form */
            rexSearch('/(<^lparen>+) <lparen>', verbPhrase);
            verb = rexGroup(1)[3] + ' ';
        }

        /* get the preposition to use, if any */
        switch(whichObj)
        {
        case DirectObject:
            /* 
             *   the preposition for the direct object is the one that
             *   appears in the verbPhrase between the first "word/word"
             *   pair and the first parenthesized interrogative 
             */
            rexSearch('/<alphanum|-|squote>+ (<^lparen>*)<lparen>',
                      verbPhrase);
            if (rexGroup(1)[2] != 0)
                prep = rexGroup(1)[3];
            break;

        case IndirectObject:
            /* 
             *   for the indirect object, the preposition is the part
             *   between the two parenthesized interrogatives 
             */
            rexSearch('<rparen> (<^lparen>+ )<lparen>', verbPhrase);
            prep = rexGroup(1)[3];
            break;
        }

        /* build and return the complete phrase */
        return verb + prep + obj.theNameObj;
    }

    /* announce all defaulted objects */
    announceAllDefaultObjects(allResolved)
    {
        /* announce a defaulted direct object if appropriate */
        maybeAnnounceDefaultObject(dobjList_, DirectObject, allResolved);

        /* announce a defaulted indirect object if appropriate */
        maybeAnnounceDefaultObject(iobjList_, IndirectObject, allResolved);
    }

    /* show the verb's basic infinitive form */
    verbInf(which)
    {
        local prep;
        
        /*
         *   Our verb phrase can one of four formats, depending on which
         *   object role we're asking about (the part in <angle brackets>
         *   is the part we're responsible for generating):
         *   
         *   asking for dobj: verb prep it ('what do you want to <open
         *   with it>?')
         *   
         *   asking for dobj, but suppressing the iobj part: verb it
         *   ('whom do you want to <ask>?')  
         *   
         *   asking for iobj: verb it prep ('what do you want to <open it
         *   with>?')  
         */
        
        /* we always start with the verb */
        rexMatch('(.*)/<alphanum|-|squote>+(<^lparen>*) <lparen>',
                 verbPhrase);
        say(rexGroup(1)[3] + rexGroup(2)[3]);

        /* 
         *   extract the prepositional phrase that comes between the
         *   direct and indirect objects - this is simply the part of the
         *   verbPhrase that's between the two parenthesized "(what)"
         *   pronouns representing the objects 
         */
        rexSearch('<rparen> (<^lparen|rparen>*) <lparen>', verbPhrase);
        prep = rexGroup(1)[3];

        /*
         *   If we have a two-object verb, and either we're asking for the
         *   indirect object or we're asking for the direct object and
         *   we're not suppressing the indirect object phrase in the
         *   query, show the object phrase.  
         */
        if (which == IndirectObject || !omitIobjInDobjQuery)
        {
            local lst;
            
            /* 
             *   Get the resolution list (or tentative resolution list)
             *   for the *other* object, since we want to show a pronoun
             *   representing the other object.  If we don't have a
             *   fully-resolved list for the other object, use the
             *   tentative resolution list, which we're guaranteed to have
             *   by the time we start resolving anything (and thus by the
             *   time we need to ask for objects).  
             */
            lst = (which == DirectObject ? iobjList_ : dobjList_);
            if (lst == nil || lst == [])
                lst = (which == DirectObject
                       ? tentativeIobj_ : tentativeDobj_);

            /* show the appropriate pronoun, if we have any objects */
            if (lst != nil && lst != [])
            {
                /* 
                 *   if we're asking for the direct object, the pronoun
                 *   we're about to show is for the indirect object, so
                 *   the prepositional phrase comes before it 
                 */
                if (which == DirectObject)
                    " <<prep>>";
                
                /* show the pronoun for the direct object list */
                " <<objListPronoun(lst)>>";

                /* 
                 *   if we're asking for the indirect object, the pronoun
                 *   we just showed is for the direct object, so the
                 *   prepositional phrase comes after it 
                 */
                if (which == IndirectObject)
                    " <<prep>>";
            }
        }
    }
        
    /* describe the full action in present participle form */
    getParticiplePhrase()
    {
        local ret;
        
        /* 
         *   the participle form is the part from the "/" to the
         *   interrogative word in parentheses 
         */
        rexSearch('/(<^lparen>+)<lparen>', verbPhrase);
        ret = rexGroup(1)[3];

        /* add the direct object */
        ret += ' ' + getDobj().theNameObj;

        /* if we have an indirect object to show, add it */
        if (getIobj() != nil)
        {
            /* 
             *   sadd anything between the direct and indirect object in
             *   the verbPhrase template 
             */
            rexSearch('<rparen> (.*)<lparen>', verbPhrase);
            if (rexGroup(1)[2] != 0)
                ret += ' ' + rexGroup(1)[3];

            /* add the indirect object */
            ret += ' ' + getIobj().theNameObj;
        }

        /* return the result phrase */
        return ret;
    }

    /*
     *   Flag: omit the indirect object in a query for a missing direct
     *   object.  For many verbs, if we already know the indirect object
     *   and we need to ask for the direct object, the query sounds best
     *   when it includes the indirect object: "what do you want to put in
     *   it?"  or "what do you want to take from it?".  This is the
     *   default phrasing.
     *   
     *   However, the corresponding query for some verbs sounds weird:
     *   "what do you want to dig in with it?" or "whom do you want to ask
     *   about it?".  For such actions, this property should be set to
     *   true to indicate that the indirect object should be omitted from
     *   the queries, which will change the phrasing to "what do you want
     *   to dig in", "whom do you want to ask", and so on.  
     */
    omitIobjInDobjQuery = nil
;

/*
 *   English-specific additions for verbs of a direct object and a literal
 *   phrase.  
 */
modify LiteralTAction
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /* use the same handling as for a regular two-object action */
        return delegated TIAction(obj, whichObj, resolvedAllObjects);
    }
    
    whatObj(which)
    {
        /* use the same handling we use for a regular two-object action */
        delegated TIAction(which);
    }

    verbInf(which)
    {
        /* 
         *   if we're asking for the indirect object, and the literal
         *   phrase is the direct object, refer to the literal as 'it' and
         *   show the indirect object preposition 
         */
        if (which == IndirectObject && whichLiteral == DirectObject)
        {
            local verb;
            local prep;
            
            /* extract the verb */
            rexSearch('(.*)/<alphanum|-|squote>+(<^lparen>*) <lparen>',
                      verbPhrase);
            verb = rexGroup(1)[3] + rexGroup(2)[3];

            /* extract the indirect object preposition */
            rexSearch('<rparen> (<^lparen|rparen>*) <lparen>', verbPhrase);
            prep = rexGroup(1)[3];

            /* show the complete phrase */
            "<<verb>> it <<prep>>";
        }
        else
        {
            /* use the same handling as for a regular two-object action */
            delegated TIAction(which);
        }
    }

    getParticiplePhrase()
    {
        /* use the same handling as for a regular two-object action */
        return delegated TIAction();
    }
;

/*
 *   English-specific additions for verbs with topic phrases. 
 */
modify TopicAction
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        /* use the same handling as for a regular two-object action */
        return delegated TIAction(obj, whichObj, resolvedAllObjects);
    }
    
    whatObj(which)
    {
        /* use the same handling we use for a regular two-object action */
        delegated TIAction(which);
    }

    verbInf(which)
    {
        /* use the same handling as for a regular two-object action */
        delegated TIAction(which);
    }

    getParticiplePhrase()
    {
        /* use the same handling as for a regular two-object action */
        return delegated TIAction();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Verbs.
 *   
 *   The actual body of each of our verbs is defined in the main
 *   language-independent part of the library.  We only define the
 *   language-specific grammar rules here.  
 */

VerbRule(Take)
    ('take' | 'pick' 'up' | 'get') dobjList
    | 'pick' dobjList 'up'
    : TakeAction
    verbPhrase = 'take/taking (what)'
;

VerbRule(TakeFrom)
    ('take' | 'get') dobjList
        ('from' | 'out' 'of' | 'off' | 'off' 'of') singleIobj
    | 'remove' dobjList 'from' singleIobj
    : TakeFromAction
    verbPhrase = 'take/taking (what) from (what)'
;

VerbRule(Drop)
    ('drop' | 'put' 'down') dobjList
    | 'put' dobjList 'down'
    : DropAction
    verbPhrase = 'drop/dropping (what)'
;

VerbRule(Examine)
    ('examine' | 'inspect' | 'x' | 'look' 'at' | 'l' 'at') dobjList
    : ExamineAction
    verbPhrase = 'examine/examining (what)'
;

VerbRule(Read)
    'read' dobjList
    : ReadAction
    verbPhrase = 'read/reading (what)'
;

VerbRule(LookIn)
    ('look' | 'l') ('in' | 'inside') dobjList
    : LookInAction
    verbPhrase = 'look/looking in (what)'
;

VerbRule(Search)
    'search' dobjList
    : SearchAction
    verbPhrase = 'search/searching (what)'
;

VerbRule(LookThrough)
    ('look' | 'l') ('through' | 'thru') dobjList
    : LookThroughAction
    verbPhrase = 'look/looking through (what)'
;

VerbRule(LookUnder)
    ('look' | 'l') 'under' dobjList
    : LookUnderAction
    verbPhrase = 'look/looking under (what)'
;

VerbRule(LookBehind)
    ('look' | 'l') 'behind' dobjList
    : LookBehindAction
    verbPhrase = 'look/looking behind (what)'
;

VerbRule(Feel)
    ('feel' | 'touch') dobjList
    : FeelAction
    verbPhrase = 'touch/touching (what)'
;

VerbRule(Taste)
    'taste' dobjList
    : TasteAction
    verbPhrase = 'taste/tasting (what)'
;

VerbRule(Smell)
    ('smell' | 'sniff') dobjList
    : SmellAction
    verbPhrase = 'smell/smelling (what)'
;

VerbRule(SmellImplicit)
    'smell' | 'sniff'
    : SmellImplicitAction
    verbPhrase = 'smell/smelling'
;

VerbRule(ListenTo)
    ('hear' | 'listen' 'to' ) dobjList
    : ListenToAction
    verbPhrase = 'listen/listing to (what)'
;

VerbRule(ListenImplicit)
    'listen' | 'hear'
    : ListenImplicitAction
    verbPhrase = 'listen/listening'
;

VerbRule(PutIn)
    ('put' | 'place') dobjList
        ('in' | 'into' | 'in' 'to' | 'inside' | 'inside' 'of') singleIobj
    : PutInAction
    verbPhrase = 'put/putting (what) in (what)'
    askIobjResponseProd = inSingleNoun
;

VerbRule(PutOn)
    ('put' | 'place' | 'drop') dobjList
        ('on' | 'onto' | 'on' 'to' | 'upon') singleIobj
    | 'put' dobjList 'down' 'on' singleIobj
    : PutOnAction
    verbPhrase = 'put/putting (what) on (what)'
    askIobjResponseProd = onSingleNoun
;

VerbRule(PutUnder)
    ('put' | 'place') dobjList 'under' singleIobj
    : PutUnderAction
    verbPhrase = 'put/putting (what) under (what)'
;

VerbRule(PutInWhat)
    ('put' | 'place') dobjList
    : PutInAction
    verbPhrase = 'put/putting (what) in (what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = inSingleNoun;
    }
;

VerbRule(Wear)
    ('wear' | 'don' | 'put' 'on') dobjList
    | 'put' dobjList 'on'
    : WearAction
    verbPhrase = 'wear/wearing (what)'
;

VerbRule(Doff)
    ('doff' | 'take' 'off' | 'remove') dobjList
    | 'take' dobjList 'off'
    : DoffAction
    verbPhrase = 'take/taking (what) off'
;

VerbRule(Kiss)
    'kiss' singleDobj
    : KissAction
    verbPhrase = 'kiss/kissing (whom)'
;

VerbRule(AskFor)
    'ask' singleDobj 'for' iobjList
    | 'ask' 'for' iobjList 'from' singleDobj
    : AskForAction
    verbPhrase = 'ask/asking (whom) for (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = forSingleNoun
;

VerbRule(AskWhomFor)
    'ask' 'for' iobjList
    : AskForAction
    verbPhrase = 'ask/asking (whom) for (what)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty indirect object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(AskAbout)
    'ask' singleDobj 'about' singleTopic
    : AskAboutAction
    verbPhrase = 'ask/asking (whom) about (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
;

VerbRule(AskAboutImplicit)
    'a' singleTopic
    : AskAboutAction
    verbPhrase = 'ask/asking (whom) about (what)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty indirect object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(AskAboutWhat)
    'ask' singleDobj
    : AskAboutAction
    verbPhrase = 'ask/asking (whom) about (what)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty indirect object phrase */
        topicMatch = new EmptyNounPhraseProd();
        topicMatch.responseProd = topicPhrase;
    }
;


VerbRule(TellAbout)
    'tell' singleDobj 'about' singleTopic
    : TellAboutAction
    verbPhrase = 'tell/telling (whom) about (what)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(TellAboutWhat)
    'tell' singleDobj
    : TellAboutAction
    verbPhrase = 'tell/telling (whom) about (what)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty indirect object phrase */
        topicMatch = new EmptyNounPhraseProd();
        topicMatch.responseProd = topicPhrase;
    }
;

VerbRule(TalkTo)
    ('talk' 'to' | 't') singleDobj
    : TalkToAction
    verbPhrase = 'talk/talking to (whom)'
    askDobjResponseProd = singleNoun
;

VerbRule(Hello)
    'hello' | 'hi' | 'greetings'
    : HelloAction
    verbPhrase = 'say/saying hello'
;

VerbRule(Yell)
    'yell' | 'scream' | 'shout' | 'holler'
    : YellAction
    verbPhrase = 'yell/yelling'
;

VerbRule(GiveTo)
    ('give' | 'offer') dobjList 'to' singleIobj
    | ('give' | 'offer') singleIobj dobjList
    : GiveToAction
    verbPhrase = 'give/giving (what) to (whom)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(GiveToWhom)
    ('give' | 'offer') dobjList
    : GiveToAction
    verbPhrase = 'give/giving (what) to (whom)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = toSingleNoun;
    }
;

VerbRule(ShowTo)
    'show' dobjList 'to' singleIobj
    | 'show' singleIobj dobjList
    : ShowToAction
    verbPhrase = 'show/showing (what) to (whom)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(ShowToWhom)
    'show' dobjList
    : ShowToAction
    verbPhrase = 'show/showing (what) to (whom)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = toSingleNoun;
    }
;

VerbRule(Throw)
    ('throw' | 'toss') dobjList
    : ThrowAction
    verbPhrase = 'throw/throwing (what)'
;

VerbRule(ThrowAt)
    ('throw' | 'toss') dobjList 'at' singleIobj
    : ThrowAtAction
    verbPhrase = 'throw/throwing (what) at (what)'
    askIobjResponseProd = atSingleNoun
;

VerbRule(ThrowTo)
    ('throw' | 'toss') dobjList 'to' singleIobj
    : ThrowToAction
    verbPhrase = 'throw/throwing (what) to (whom)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(Follow)
    'follow' singleDobj
    : FollowAction
    verbPhrase = 'follow/following (whom)'
    askDobjResponseProd = singleNoun
;

VerbRule(Attack)
    ('attack' | 'kill' | 'hit' | 'kick' | 'punch') singleDobj
    : AttackAction
    verbPhrase = 'attack/attacking (whom)'
    askDobjResponseProd = singleNoun
;

VerbRule(AttackWith)
    ('attack' | 'kill' | 'hit' | 'kick' | 'punch' | 'strike')
        singleDobj
        'with' singleIobj
    : AttackWithAction
    verbPhrase = 'attack/attacking (whom) with (what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(Inventory)
    'i' | 'inventory' | 'take' 'inventory'
    : InventoryAction
    verbPhrase = 'take/taking inventory'
;

VerbRule(InventoryTall)
    'i' 'tall' | 'inventory' 'tall'
    : InventoryTallAction
    verbPhrase = 'take/taking "tall" inventory'
;

VerbRule(InventoryWide)
    'i' 'wide' | 'inventory' 'wide'
    : InventoryWideAction
    verbPhrase = 'take/taking "wide" inventory'
;

VerbRule(Wait)
    'z' | 'wait'
    : WaitAction
    verbPhrase = 'wait/waiting'
;

VerbRule(Look)
    'look' | 'look' 'around' | 'l' | 'l' 'around'
    : LookAction
    verbPhrase = 'look/looking around'
;

VerbRule(Quit)
    'quit' | 'q'
    : QuitAction
    verbPhrase = 'quit/quitting'
;

VerbRule(Again)
    'again' | 'g'
    : AgainAction
    verbPhrase = 'repeat/repeating the last command'
;

VerbRule(Footnote)
    ('footnote' | 'note') singleNumber
    : FootnoteAction
    verbPhrase = 'show/showing a footnote'
;

VerbRule(FootnotesFull)
    'footnotes' 'full'
    : FootnotesFullAction
    verbPhrase = 'enable/enabling all footnotes'
;

VerbRule(FootnotesMedium)
    'footnotes' 'medium'
    : FootnotesMediumAction
    verbPhrase = 'enable/enabling new footnotes'
;

VerbRule(FootnotesOff)
    'footnotes' 'off'
    : FootnotesOffAction
    verbPhrase = 'hide/hiding footnotes'
;

VerbRule(FootnotesStatus)
    'footnotes'
    : FootnotesStatusAction
    verbPhrase = 'show/showing footnote status'
;

VerbRule(Verbose)
    'verbose'
    : VerboseAction
    verbPhrase = 'enter/entering VERBOSE mode'
;

VerbRule(Terse)
    'terse'
    : TerseAction
    verbPhrase = 'enter/entering TERSE mode'
;

VerbRule(Score)
    'score' | 'status'
    : ScoreAction
    verbPhrase = 'show/showing score'
;

VerbRule(FullScore)
    'full' 'score' | 'fullscore' | 'full'
    : FullScoreAction
    verbPhrase = 'show/showing full score'
;

VerbRule(Notify)
    'notify'
    : NotifyAction
    verbPhrase = 'show/showing notification status'
;

VerbRule(NotifyOn)
    'notify' 'on'
    : NotifyOnAction
    verbPhrase = 'turn/turning on score notification'
;

VerbRule(NotifyOff)
    'notify' 'off'
    : NotifyOffAction
    verbPhrase = 'turn/turning off score notification'
;

VerbRule(Save)
    'save'
    : SaveAction
    verbPhrase = 'save/saving'
;

VerbRule(SaveString)
    'save' quotedStringPhrase->fname_
    : SaveStringAction
    verbPhrase = 'save/saving'
;

VerbRule(Restore)
    'restore'
    : RestoreAction
    verbPhrase = 'restore/restoring'
;

VerbRule(RestoreString)
    'restore' quotedStringPhrase->fname_
    : RestoreStringAction
    verbPhrase = 'restore/restoring'
;

VerbRule(Restart)
    'restart'
    : RestartAction
    verbPhrase = 'restart/restarting'
;

VerbRule(Pause)
    'pause'
    : PauseAction
    verbPhrase = 'pause/pausing'
;

VerbRule(Undo)
    'undo'
    : UndoAction
    verbPhrase = 'undo/undoing'
;

VerbRule(Version)
    'version'
    : VersionAction
    verbPhrase = 'show/showing version'
;

VerbRule(Credits)
    'credits'
    : CreditsAction
    verbPhrase = 'show/showing credits'
;

VerbRule(About)
    'about'
    : AboutAction
    verbPhrase = 'show/showing story information'
;

VerbRule(Script)
    'script'
    : ScriptAction
    verbPhrase = 'start/starting scripting'
;

VerbRule(ScriptString)
    'script' quotedStringPhrase->fname_
    : ScriptStringAction
    verbPhrase = 'start/starting scripting'
;

VerbRule(Unscript)
    'unscript'
    : UnscriptAction
    verbPhrase = 'end/ending scripting'
;

VerbRule(Travel)
    'go' singleDir | singleDir
    : TravelAction
    verbPhrase = ('go/going ' + dirMatch.dir.name)
;

VerbRule(Port)
    'go' 'to' ('port' | 'p')
    : PortAction
    dirMatch: DirectionProd { dir = portDirection }
    verbPhrase = 'go/going to port'
;   

VerbRule(Starboard)
    'go' 'to' ('starboard' | 'sb')
    : StarboardAction
    dirMatch: DirectionProd { dir = starboardDirection }
    verbPhrase = 'go/going to starboard'
;

VerbRule(In)
    'enter'
    : InAction
    dirMatch: DirectionProd { dir = inDirection }
    verbPhrase = 'enter/entering'
;

VerbRule(Out)
    'exit'
    : OutAction
    dirMatch: DirectionProd { dir = outDirection }
    verbPhrase = 'exit/exiting'
;

VerbRule(GoThrough)
    ('walk' | 'go' ) ('through' | 'thru')
        singleDobj
    : GoThroughAction
    verbPhrase = 'go/going through (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Enter)
    ('enter' | ('walk' | 'go') ('in' | 'in' 'to' | 'into'))
    singleDobj
    : EnterAction
    verbPhrase = 'enter/entering (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(GoBack)
    'back' | 'go' 'back' | 'return'
    : GoBackAction
    verbPhrase = 'go/going back'
;

VerbRule(Dig)
    ('dig' | 'dig' 'in') singleDobj
    : DigAction
    verbPhrase = 'dig/digging in (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(DigWith)
    ('dig' | 'dig' 'in') singleDobj 'with' singleIobj
    : DigWithAction
    verbPhrase = 'dig/digging in (what) with (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(Jump)
    'jump'
    : JumpAction
    verbPhrase = 'jump/jumping'
;

VerbRule(JumpOffI)
    'jump' 'off'
    : JumpOffIAction
    verbPhrase = 'jump/jumping off'
;

VerbRule(JumpOff)
    'jump' 'off' singleDobj
    : JumpOffAction
    verbPhrase = 'jump/jumping off (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(JumpOver)
    ('jump' | 'jump' 'over') singleDobj
    : JumpOverAction
    verbPhrase = 'jump/jumping over (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Push)
    ('push' | 'press') dobjList
    : PushAction
    verbPhrase = 'push/pushing (what)'
;

VerbRule(Pull)
    'pull' dobjList
    : PullAction
    verbPhrase = 'pull/pulling (what)'
;

VerbRule(Move)
    'move' dobjList
    : MoveAction
    verbPhrase = 'move/moving (what)'
;

VerbRule(MoveTo)
    ('push' | 'move') dobjList ('to' | 'under') singleIobj
    : MoveToAction
    verbPhrase = 'move/moving (what) to (what)'
    askIobjResponseProd = toSingleNoun
    omitIobjInDobjQuery = true
;

VerbRule(MoveWith)
    'move' singleDobj 'with' singleIobj
    : MoveWithAction
    verbPhrase = 'move/moving (what) with (what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
    omitIobjInDobjQuery = true
;

VerbRule(Turn)
    ('turn' | 'twist' | 'rotate') dobjList
    : TurnAction
    verbPhrase = 'turn/turning (what)'
;

VerbRule(TurnWith)
    ('turn' | 'twist' | 'rotate') singleDobj 'with' singleIobj
    : TurnWithAction
    verbPhrase = 'turn/turning (what) with (what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(TurnTo)
    ('turn' | 'twist' | 'rotate') singleDobj
        'to' singleLiteral
    : TurnToAction
    verbPhrase = 'turn/turning (what) to (what)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(SetTo)
    'set' singleDobj 'to' singleLiteral
    : SetToAction
    verbPhrase = 'set/setting (what) to (what)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(TypeOn)
    'type' singleLiteral 'on' singleDobj
    : TypeOnAction
    verbPhrase = 'type/typing (what) on (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(TypeWhatOn)
    'type' singleLiteral
    : TypeOnAction
    verbPhrase = 'type/typing (what) on (what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = onSingleNoun;
    }
;

VerbRule(EnterOn)
    'enter' singleLiteral ('on' | 'in' | 'with') singleDobj
    : EnterOnAction
    verbPhrase = 'enter/entering (what) on (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Consult)
    'consult' singleDobj : ConsultAction
    verbPhrase = 'consult/consulting (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(ConsultAbout)
    'consult' singleDobj ('on' | 'about') singleTopic
    | 'search' singleDobj 'for' singleTopic
    | ('look' 'up' | 'look' 'for' | 'find' | 'search' 'for' | 'read' 'about')
         singleTopic 'in' singleDobj
    | 'look' singleTopic 'up' 'in' singleDobj
    : ConsultAboutAction
    verbPhrase = 'consult/consulting (what) about (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
;

VerbRule(Switch)
    'switch' dobjList
    : SwitchAction
    verbPhrase = 'switch/switching (what)'
;

VerbRule(Flip)
    'flip' dobjList
    : FlipAction
    verbPhrase = 'flip/flipping (what)'
;

VerbRule(TurnOn)
    ('activate' | ('turn' | 'switch') 'on') dobjList
    | ('turn' | 'switch') dobjList 'on'
    : TurnOnAction
    verbPhrase = 'turn/turning on (what)'
;

VerbRule(TurnOff)
    ('deactivate' | ('turn' | 'switch') 'off') dobjList
    | ('turn' | 'switch') dobjList 'off'
    : TurnOffAction
    verbPhrase = 'turn/turning off (what)'
;

VerbRule(Light)
    'light' dobjList
    : LightAction
    verbPhrase = 'light/lighting (what)'
;

DefineTAction(Strike);
VerbRule(Strike)
    'strike' dobjList
    : StrikeAction
    verbPhrase = 'strike/striking (what)'
;

VerbRule(Burn)
    ('burn' | 'ignite' | 'set' 'fire' 'to') dobjList
    : BurnAction
    verbPhrase = 'light/lighting (what)'
;

VerbRule(BurnWith)
    ('light' | 'burn' | 'ignite' | 'set' 'fire' 'to') singleDobj
        'with' singleIobj
    : BurnWithAction
    verbPhrase = 'light/lighting (what) with (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(Extinguish)
    ('extinguish' | 'douse' | 'put' 'out' | 'blow' 'out') dobjList
    | ('blow' | 'put') dobjList 'out'
    : ExtinguishAction
    verbPhrase = 'extinguish/extinguishing (what)'
;

VerbRule(Break)
    ('break' | 'ruin' | 'destroy' | 'wreck') dobjList
    : BreakAction
    verbPhrase = 'break/breaking (what)'
;

VerbRule(CutWithWhat)
    'cut' singleDobj
    : CutWithAction
    verbPhrase = 'cut/cutting (what) with (what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = withSingleNoun;
    }
;

VerbRule(CutWith)
    'cut' singleDobj 'with' singleIobj
    : CutWithAction
    verbPhrase = 'cut/cutting (what) with (what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(Eat)
    ('eat' | 'consume') dobjList
    : EatAction
    verbPhrase = 'eat/eating (what)'
;

VerbRule(Drink)
    ('drink' | 'quaff' | 'imbibe') dobjList
    : DrinkAction
    verbPhrase = 'drink/drinking (what)'
;

VerbRule(Pour)
    'pour' dobjList
    : PourAction
    verbPhrase = 'pour/pouring (what)'
;

VerbRule(PourInto)
    'pour' dobjList ('in' | 'into' | 'in' 'to') singleIobj
    : PourIntoAction
    verbPhrase = 'pour/pouring (what) into (what)'
    askIobjResponseProd = inSingleNoun
;

VerbRule(PourOnto)
    'pour' dobjList ('on' | 'onto' | 'on' 'to') singleIobj
    : PourOntoAction
    verbPhrase = 'pour/pouring (what) onto (what)'
    askIobjResponseProd = withSingleNoun
;

VerbRule(Climb)
    'climb' singleDobj
    : ClimbAction
    verbPhrase = 'climb/climbing (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(ClimbUp)
    ('climb' | 'go' | 'walk') 'up' singleDobj
    : ClimbUpAction
    verbPhrase = 'climb/climbing up (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(ClimbDown)
    ('climb' | 'go' | 'walk') 'down' singleDobj
    : ClimbDownAction
    verbPhrase = 'climb/climbing down (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Clean)
    'clean' dobjList
    : CleanAction
    verbPhrase = 'clean/cleaning (what)'
;

VerbRule(CleanWith)
    'clean' dobjList 'with' singleIobj
    : CleanWithAction
    verbPhrase = 'clean/cleaning (what) with (what)'
    askIobjResponseProd = withSingleNoun
    omitIobjInDobjQuery = true
;

VerbRule(AttachTo)
    ('attach' | 'connect') dobjList 'to' singleIobj
    : AttachToAction
    askIobjResponseProd = toSingleNoun
    verbPhrase = 'attach/attaching (what) to (what)'
;

VerbRule(AttachToWhat)
    ('attach' | 'connect') dobjList
    : AttachToAction
    verbPhrase = 'attach/attaching (what) to (what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = toSingleNoun;
    }
;

VerbRule(DetachFrom)
    ('detach' | 'disconnect') dobjList 'from' singleIobj
    : DetachFromAction
    verbPhrase = 'detach/detaching (what) from (what)'
    askIobjResponseProd = fromSingleNoun
;

VerbRule(Detach)
    ('detach' | 'disconnect') dobjList
    : DetachAction
    verbPhrase = 'detach/detaching (what)'
;

VerbRule(Open)
    'open' dobjList
    : OpenAction
    verbPhrase = 'open/opening (what)'
;

VerbRule(Close)
    'close' dobjList
    : CloseAction
    verbPhrase = 'close/closing (what)'
;

VerbRule(Lock)
    'lock' dobjList
    : LockAction
    verbPhrase = 'lock/locking (what)'
;

VerbRule(Unlock)
    'unlock' dobjList
    : UnlockAction
    verbPhrase = 'unlock/unlocking (what)'
;

VerbRule(LockWith)
    'lock' singleDobj 'with' singleIobj
    : LockWithAction
    verbPhrase = 'lock/locking (what) with (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(UnlockWith)
    'unlock' singleDobj 'with' singleIobj
    : UnlockWithAction
    verbPhrase = 'unlock/unlocking (what) with (what)'
    omitIobjInDobjQuery = true
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
;

VerbRule(SitOn)
    'sit' ( | 'on' | 'down' | 'in' | 'down' 'on' | 'down' 'in')
        singleDobj
    : SitOnAction
    verbPhrase = 'sit/sitting on (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(LieOn)
    'lie' ( | 'on' | 'down' | 'in' | 'down' 'on' | 'down' 'in')
        singleDobj
    : LieOnAction
    verbPhrase = 'lie/lying on (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(StandOn)
    ('stand' | 'climb') ('on' | 'in') singleDobj
    : StandOnAction
    verbPhrase = 'stand/standing on (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Stand)
    'stand' | 'stand' 'up' | 'get' 'up'
    : StandAction
    verbPhrase = 'stand/standing up'
;

VerbRule(GetOutOf)
    ('get' 'out' 'of' | 'climb' 'out' 'of' | 'leave' | 'exit') singleDobj
    : GetOutOfAction
    verbPhrase = 'get/getting out of (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(GetOffOf)
    'get' ('off' | 'off' 'of' | 'down' 'from') singleDobj
    : GetOffOfAction
    verbPhrase = 'get/getting off of (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(GetOut)
    'get' 'out'
    | 'get' 'off'
    | 'disembark'
    | 'climb' 'out'
    : GetOutAction
    verbPhrase = 'get/getting out'
;

VerbRule(Board)
    ('board'
     | (('get' | 'climb')
        ('in' | 'into' | 'in' 'to' | 'on' | 'onto' | 'on' 'to')))
        singleDobj
    : BoardAction
    verbPhrase = 'get/getting in (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Sleep)
    'sleep'
    : SleepAction
    verbPhrase = 'sleep/sleeping'
;

VerbRule(Fasten)
    ('fasten' | 'buckle' | 'buckle' 'up') dobjList
    : FastenAction
    verbPhrase = 'fasten/fastening (what)'
;

VerbRule(FastenTo)
    ('fasten' | 'buckle') dobjList 'to' singleIobj
    : FastenToAction
    verbPhrase = 'fasten/fastening (what) to (what)'
    askIobjResponseProd = toSingleNoun
;

VerbRule(Unfasten)
    ('unfasten' | 'unbuckle') dobjList
    : UnfastenAction
    verbPhrase = 'unfasten/unfastening (what)'
;

VerbRule(UnfastenFrom)
    ('unfasten' | 'unbuckle') dobjList 'from' singleIobj
    : UnfastenFromAction
    verbPhrase = 'unfasten/unfastening (what) from (what)'
    askIobjResponseProd = fromSingleNoun
;

VerbRule(PlugInto)
    'plug' dobjList ('in' | 'into' | 'in' 'to') singleIobj
    : PlugIntoAction
    verbPhrase = 'plug/plugging (what) into (what)'
    askIobjResponseProd = inSingleNoun
;

VerbRule(PlugIntoWhat)
    'plug' dobjList
    : PlugIntoAction
    verbPhrase = 'plug/plugging (what) into (what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = inSingleNoun;
    }
;

VerbRule(PlugIn)
    'plug' dobjList 'in'
    | 'plug' 'in' dobjList
    : PlugInAction
    verbPhrase = 'plug/plugging (what)'
;

VerbRule(UnplugFrom)
    'unplug' dobjList 'from' singleIobj
    : UnplugFromAction
    verbPhrase = 'unplug/unplugging (what) from (what)'
    askIobjResponseProd = fromSingleNoun
;

VerbRule(Unplug)
    'unplug' dobjList
    : UnplugAction
    verbPhrase = 'unplug/unplugging (what)'
;

VerbRule(Screw)
    'screw' dobjList
    : ScrewAction
    verbPhrase = 'screw/screwing (what)'
;

VerbRule(ScrewWith)
    'screw' dobjList 'with' singleIobj
    : ScrewWithAction
    verbPhrase = 'screw/screwing (what) with (what)'
    omitIobjInDobjQuery = true
    askIobjResponseProd = withSingleNoun
;

VerbRule(Unscrew)
    'unscrew' dobjList
    : UnscrewAction
    verbPhrase = 'unscrew/unscrewing (what)'
;

VerbRule(UnscrewWith)
    'unscrew' dobjList 'with' singleIobj
    : UnscrewWithAction
    verbPhrase = 'unscrew/unscrewing (what) with (what)'
    omitIobjInDobjQuery = true
    askIobjResponseProd = withSingleNoun
;

VerbRule(PushTravelDir)
    ('push' | 'pull' | 'drag' | 'move') dobjList singleDir
    : PushTravelDirAction
    verbPhrase = ('push/pushing (what) ' + dirMatch.dir.name)
;

VerbRule(PushTravelThrough)
    ('push' | 'pull' | 'drag' | 'move') dobjList
    ('through' | 'thru') singleIobj
    : PushTravelThroughAction
    verbPhrase = 'push/pushing (what) through (what)'
;

VerbRule(PushTravelEnter)
    ('push' | 'pull' | 'drag' | 'move') dobjList
    ('in' | 'into' | 'in' 'to') singleIobj
    : PushTravelEnterAction
    verbPhrase = 'push/pushing (what) into (what)'
;

VerbRule(PushTravelGetOutOf)
    ('push' | 'pull' | 'drag' | 'move') dobjList
    'out' 'of' singleIobj
    : PushTravelGetOutOfAction
    verbPhrase = 'push/pushing (what) out of (what)'
;


VerbRule(PushTravelClimbUp)
    ('push' | 'pull' | 'drag' | 'move') dobjList
    'up' singleIobj
    : PushTravelClimbUpAction
    verbPhrase = 'push/pushing (what) up (what)'
    omitIobjInDobjQuery = true
;

VerbRule(PushTravelClimbDown)
    ('push' | 'pull' | 'drag' | 'move') dobjList
    'down' singleIobj
    : PushTravelClimbDownAction
    verbPhrase = 'push/pushing down up (what)'
;

VerbRule(Exits)
    'exits'
    : ExitsAction
    verbPhrase = 'exits/showing exits'
;

VerbRule(ExitsOnOff)
    'exits' ('on'->on_ | 'off'->off_)
    : ExitsOnOffAction
    verbPhrase = 'turn/turning off exits display'
;


/* ------------------------------------------------------------------------ */
/*
 *   "debug" verb - special verb to break into the debugger.  We'll only
 *   compile this into the game if we're compiling a debug version to
 *   begin with, since a non-debug version can't be run under the
 *   debugger.  
 */
#ifdef __DEBUG

VerbRule(Debug)
    'debug'
    : DebugAction
    verbPhrase = 'debug/debugging'
;

#endif /* __DEBUG */

