#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library: Resolvers.
 *   
 *   This module defines the Resolver classes.  A Resolver is an abstract
 *   object that the parser uses to control the resolution of noun phrases
 *   to game objects.  Specialized Resolver subclasses allow noun phrases
 *   to be resolved differently according to their grammatical function in
 *   a command.  
 */

#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Basic object resolver.  An Action object creates an object resolver
 *   to mediate the process of resolving noun phrases to objects.
 *   
 *   A resolver encapsulates a set of object resolution rules.  In most
 *   cases, an action that takes only a direct object can be its own
 *   resolver, because it needs only one set of resolution rules; for this
 *   reason, this basic Resolver implementation is designed to work with
 *   the direct object.  Actions with multiple objects will need separate
 *   resolvers for each object, since they might want to use different
 *   rules for the different objects.  
 */
class Resolver: object
    construct(action, issuingActor, targetActor)
    {
        /* remember my action and actor objects */
        action_ = action;
        issuer_ = issuingActor;
        actor_ = targetActor;

        /* cache the scope list */
        cacheScopeList(targetActor);
    }

    /* 
     *   Reset the resolver - this can be called if we are to re-use the
     *   same resolver to resolve a list of noun phrases again.
     */
    resetResolver()
    {
        /* forget the equivalents we've resolved so far */
        equivs_ = nil;
    }

    /* get the action we're resolving */
    getAction() { return action_; }

    /* get the target actor */
    getTargetActor() { return actor_; }

    /*
     *   Match an object's name.  By default, we'll call the object's own
     *   matchName method with the given original and adjusted token
     *   lists.  Subclasses can override this to call different match
     *   methods (such as matchNameDisambig).  
     */
    matchName(obj, origTokens, adjustedTokens)
    {
        return obj.matchName(origTokens, adjustedTokens);
    }
    
    /*
     *   Cache the scope list for this object.  By default, we cache the
     *   default scope list for the actor.
     */
    cacheScopeList(actor)
    {
        /* cache the actor's default scope list */
        scope_ = actor.scopeList();
    }

    /*
     *   Get the resolver for qualifier phrases.  By default, this simply
     *   returns myself, since the resolver for qualifiers is in most
     *   contexts the same as the main resolver.
     *   
     *   This can be overridden in contexts where the qualifier resolver
     *   is different from the main resolver.  In general, when a
     *   sub-resolver narrows the scope for resolving a phrase, such as an
     *   exclusion list or a disambiguation response, we will want to
     *   resolve qualifiers in the context of the main resolution scope
     *   rather than the narrowed scope.  
     */
    getQualifierResolver() { return self; }

    /*
     *   Determine if an object is in scope for the purposes of object
     *   resolution.  By default, we'll simply check to see if the object
     *   is in our cached scope list.  Some actions might want to override
     *   this to provide alternative behavior; for example, for a command
     *   whose scope is the set of all objects, it would be much more
     *   efficient to dispense with caching a scope list entirely and
     *   simply return true here.  
     */
    objInScope(obj) { return scope_.indexOf(obj) != nil; }

    /*
     *   Resolve a pronoun antecedent, given a pronoun selector.  This
     *   returns a list of ResolveInfo objects, for use in object
     *   resolution.  
     */
    resolvePronounAntecedent(typ)
    {
        local lst;

        /* get the raw antecedent list */
        lst = getRawPronounAntecedent(typ);

        /* if there is no antecedent, return an empty list */
        if (lst == nil || lst == [])
            return [];

        /* if we got a single object, turn it into a list */
        if (dataType(lst) == TypeObject)
            lst = [lst];

        /* remove objects not in scope */
        lst = lst.intersect(scope_);
        
        /* 
         *   Return a list of ResolveInfo objects based on the antecedent
         *   values.  
         */
        return lst.mapAll({x: new ResolveInfo(x, 0)});
    }

    /*
     *   Get the "raw" pronoun antecedent list for a given pronoun
     *   selector.  This returns a list of objects matching the pronoun.
     *   The list is raw in that it is given as a list of game objects
     *   (not ResolveInfo objects), and it isn't filtered for scope.  
     */
    getRawPronounAntecedent(typ)
    {

        /* check for pronouns that are relative to the issuer or target */
        switch(typ)
        {
        case PronounMe:
            /*
             *   It's a first-person construction.  If the issuing actor
             *   is the player character, this refers to the player
             *   character only if the game refers to the player character
             *   in the second person (so, if the game calls the PC "you",
             *   the player calls the PC "me").  If the issuing actor
             *   isn't the player character, then a first-person pronoun
             *   refers to the command's issuer.  
             */
            if (issuer_.isPlayerChar
                && issuer_.referralPerson != SecondPerson
                && !libGlobal.allowYouMeMixing)
            {
                /* 
                 *   the issuer is the player, but the game doesn't call
                 *   the PC "you", so "me" has no meaning 
                 */
                return [];
            }
            else
            {
                /* "me" refers to the command's issuer */
                return [issuer_];
            }

        case PronounYou:
            /*
             *   It's a second-person construction.  If the target actor
             *   is the player character, this refers to the player
             *   character only if the game refers to the player character
             *   in the first person (so, if the game calls the PC "me",
             *   then the player calls the PC "you").  If the target actor
             *   isn't the player character, then a second-person pronoun
             *   refers to the target actor. 
             */
            if (actor_.isPlayerChar
                && actor_.referralPerson != FirstPerson
                && !libGlobal.allowYouMeMixing)
            {
                /* 
                 *   the target is the player character, but the game
                 *   doesn't call the PC "me", so "you" has no meaning in
                 *   this command 
                 */
                return [];
            }
            else
            {
                /* "you" refers to the command's target */
                return [actor_];
            }

        default:
            /* 
             *   it's not a relative pronoun, so ask the target actor for
             *   the antecedent based on recent commands 
             */
            return actor_.getPronounAntecedent(typ);
        }
    }

    /*
     *   Get the "all" list - this is the list of objects that we should
     *   use when the object of the command is the special word "all".
     *   We'll ask the action to resolve 'all' for the direct object,
     *   since we are by default a direct object resolver.
     */
    getAll()
    {
        local lst;
        
        /* ask the action to resolve 'all' for the direct object */
        lst = action_.getAllDobj(actor_, scope_);

        /* filter the result to remove things that don't belong */
        lst = filterAll(lst);

        /* 
         *   return the results as ResolveInfo objects, with the
         *   'MatchedAll' flag set for each one 
         */
        return lst.mapAll({x: new ResolveInfo(x, MatchedAll)});
    }

    /*
     *   Filter an 'all' list to remove things that don't belong.  We
     *   always remove the actor executing the command, as well as any
     *   objects explicitly marked as hidden from 'all' lists. 
     */
    filterAll(lst)
    {
        local result;

        /* set up a vector to hold the result */
        result = new Vector(lst.length());

        /* 
         *   run through the list and include elements that we don't want
         *   to exclude 
         */
        foreach (local cur in lst)
        {
            /* 
             *   if this item isn't the actor, and isn't marked for
             *   exclusion from 'all' lists in general, include it 
             */
            if (cur != actor_ && !cur.hideFromAll(getAction()))
                result.append(cur);
        }

        /* return the result as a list */
        return result.toList();
    }

    /*
     *   Get the list of potential default objects.  This is simply the
     *   basic 'all' list, not filtered for exclusion with hideFromAll.  
     */
    getAllDefaults()
    {
        local lst;
        
        /* ask the action to resolve 'all' for the direct object */
        lst = action_.getAllDobj(actor_, scope_);

        /* return the results as ResolveInfo objects */
        return lst.mapAll({x: new ResolveInfo(x, 0)});
    }

    /*
     *   Filter an ambiguous list of objects resolving to a noun phrase.
     *   If the objects in the list vary in the degree of suitability for
     *   the command, returns a list consisting only of the most suitable
     *   objects.  If the objects are all equally suitable - or equally
     *   unsuitable - the whole list should be returned unchanged.
     *   
     *   requiredNum is the number of objects required in the final list
     *   by the caller; if the result list is larger than this, the caller
     *   will consider the results ambiguous.
     *   
     *   This routine does NOT perform any interactive disambiguation, but
     *   is merely a first attempt at reducing the number of matching
     *   objects by removing the obviously unsuitable ones.
     *   
     *   For example, for an "open" command, if the list consists of one
     *   object that's open and one object that's currently closed, the
     *   result list should include only the closed one, since it is
     *   obvious that the one that's already open does not need to be
     *   opened again.  On the other hand, if the list consists only of
     *   open objects, they should all be returned, since they're all
     *   equally unsuitable.
     *   
     *   It is not necessary to reduce the list to a single entry; it is
     *   adequate merely to reduce the ambiguity by removing any items
     *   that are clearly less suitable than the survivors.  
     */
    filterAmbiguousNounPhrase(lst, requiredNum)
    {
        return withGlobals(
            {:action_.filterAmbiguousDobj(lst, requiredNum)});
    }

    /*
     *   Filter a list of ambiguous matches for a noun phrase to reduce
     *   each set of equivalent items to a single such item, if desired.
     *   If no equivalent reduction is desired for this type of resolver,
     *   this can simply return the original list.  
     */
    filterAmbiguousEquivalents(lst)
    {
        /* if we have only one item, there's obviously nothing redundant */
        if (lst.length() == 1)
            return lst;
        
        /* scan the list, looking for equivalents */
        for (local i = 1, local len = lst.length() ; i <= len ; ++i)
        {
            /* 
             *   if this item is marked as equivalent, check for others
             *   like it 
             */
            if (lst[i].obj_.isEquivalent)
            {
                /*
                 *   If this object is in our list of previously-used
                 *   equivalents, and we have more equivalents to this
                 *   object in our list, then omit this one, so that we
                 *   keep a different equivalent this time.  This way, if
                 *   we have a noun list such as "take coin and coin",
                 *   we'll return different equivalent items for each
                 *   equivalent noun phrase. 
                 */
                if (equivs_ != nil
                    && equivs_.indexOf(lst[i].obj_) != nil
                    && lst.lastIndexWhich(
                       {x: x.obj_.isVocabEquivalent(lst[i].obj_)}) > i)
                {
                    /* 
                     *   we've already returned this one, and we have
                     *   another equivalent later in the list that we can
                     *   use instead this time - remove this one from the
                     *   list 
                     */
                    lst = lst.removeElementAt(i);

                    /* adjust the our counters for the removal */
                    --len;
                    --i;
                }
                else
                {
                    /*
                     *   We've decided to keep this element, either
                     *   because we haven't already returned it as a match
                     *   for this noun phrase, or because it's the last
                     *   one of its kind.  Add it to the list of
                     *   equivalents we've previously returned. 
                     */
                    if (equivs_ == nil)
                        equivs_ = new Vector(10);
                    equivs_.append(lst[i].obj_);

                    /* 
                     *   check each object at a higher index to see if
                     *   it's equivalent to this one 
                     */
                    for (local j = i + 1 ; j <= len ; ++j)
                    {
                        /* check this object */
                        if (lst[i].obj_.isVocabEquivalent(lst[j].obj_))
                        {
                            /* they match - remove the other one */
                            lst = lst.removeElementAt(j);
                            
                            /* reduce the list length accordingly */
                            --len;
                            
                            /* back up our scanning index as well */
                            --j;
                        }
                    }
                }
            }
        }

        /* return the updated list */
        return lst;
    }

    /*
     *   Filter a plural phrase to reduce the set to the logical subset,
     *   if possible.  If there is no logical subset, simply return the
     *   original set.  
     */
    filterPluralPhrase(lst)
    {
        return withGlobals({:action_.filterPluralDobj(lst)});
    }

    /*
     *   Select a resolution for an indefinite noun phrase ("a coin"),
     *   given a list of possible matches.  The matches will be given to
     *   us sorted from most likely to least likely, as done by
     *   filterAmbiguousNounPhrase().
     *   
     *   By default, we simply select the first 'n' items from the list
     *   (which are the most likely matches), because in most contexts, an
     *   indefinite noun phrase means that we should arbitrarily select
     *   any matching object.  This can be overridden for contexts in
     *   which indefinite noun phrases must be handled differently.  
     */
    selectIndefinite(results, lst, requiredNumber)
    {
        /* 
         *   arbitrarily choose the first 'requiredNumber' item(s) from
         *   the list 
         */
        return lst.sublist(1, requiredNumber);
    }

    /*
     *   Get the default object or objects for this phrase.  Returns a
     *   list of ResolveInfo objects if a default is available, or nil if
     *   no default is available.  This routine does not interact with the
     *   user; it should merely determine if the command implies a default
     *   strongly enough to assume it without asking the user.  
     */
    getDefaultObject()
    {
        /* ask the action to provide a default direct object */
        return withGlobals({:action_.getDefaultDobj(self)});
    }

    /*
     *   Resolve a noun phrase involving unknown words, if possible.  If
     *   it is not possible to resolve such a phrase, return nil;
     *   otherwise, return a list of resolved objects.  This routine does
     *   not interact with the user - "oops" prompting is handled
     *   separately.
     *   
     *   'tokList' is the token list for the phrase, in the canonical
     *   format as returned from the tokenizer.  Each element of 'tokList'
     *   is a sublist representing one token.
     *   
     *   Note that this routine allows for specialized unknown word
     *   resolution separately from the more general matchName mechanism.
     *   The purpose of this method is to allow the specific type of
     *   resolver to deal with unknown words specially, rather than using
     *   the matchName mechanism.  This routine is called as a last
     *   resort, only after the matchName mechanism fails to find any
     *   matches.  
     */
    resolveUnknownNounPhrase(tokList)
    {
        /* by default, we can't resolve an unknown noun phrase */
        return nil;
    }

    /*
     *   Execute a callback function in the global context of our actor
     *   and action - we'll set gActor and gAction to our own stored actor
     *   and action values, then call the callback, then restore the old
     *   globals. 
     */
    withGlobals(func)
    {
        /* invoke the function with our action and actor in the globals */
        return withParserGlobals(actor_, action_, func);
    }

    /*
     *   Get an indication of which object we're resolving.  By default,
     *   we'll indicate direct object; this should be overridden for
     *   resolvers of indirect and other types of objects.  
     */
    whichObject = DirectObject

    /* the cached scope list */
    scope_ = []

    /* my action */
    action_ = nil

    /* the issuing actor */
    issuer_ = nil

    /* the target actor object */
    actor_ = nil

    /* 
     *   List of equivalent objects we've resolved so far.  We use this to
     *   try to return different equivalent objects when multiple noun
     *   phrases refer to the same set of equivalents. 
     */
    equivs_ = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Proxy Resolver - this is used to create resolvers that refer methods
 *   not otherwise overridden back to an underlying resolver 
 */
class ProxyResolver: object
    construct(origResolver)
    {
        /* remember my underlying resolver */
        self.origResolver = origResolver;
    }

    propNotDefined(prop, [args])
    {
        /* delegate the call to the original resolver */
        return origResolver.(prop)(args...);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Basic resolver for indirect objects 
 */
class IobjResolver: Resolver
    /* 
     *   we resolve indirect objects
     */
    whichObject = IndirectObject

    /* resolve 'all' for the indirect object */
    getAll()
    {
        local lst;
        
        /* ask the action to resolve 'all' for the indirect object */
        lst = action_.getAllIobj(actor_, scope_);

        /* filter the result to remove things that don't belong */
        lst = filterAll(lst);

        /* 
         *   return the results as ResolveInfo objects, with the
         *   'MatchedAll' flag set for each one 
         */
        return lst.mapAll({x: new ResolveInfo(x, MatchedAll)});
    }

    /* get all possible default objects */
    getAllDefaults()
    {
        local lst;
        
        /* ask the action to resolve 'all' for the indirect object */
        lst = action_.getAllIobj(actor_, scope_);

        /* return the results as ResolveInfo objects */
        return lst.mapAll({x: new ResolveInfo(x, 0)});
    }

    /* filter an ambiguous noun phrase */
    filterAmbiguousNounPhrase(lst, requiredNum)
    {
        return withGlobals(
            {:action_.filterAmbiguousIobj(lst, requiredNum)});
    }

    /*
     *   Filter a plural phrase to reduce the set to the logical subset,
     *   if possible.  If there is no logical subset, simply return the
     *   original set.  
     */
    filterPluralPhrase(lst)
    {
        return withGlobals({:action_.filterPluralIobj(lst)});
    }

    /*
     *   Get the default object or objects for this phrase.  Since we
     *   resolve indirect objects, we'll ask the action for a default
     *   indirect object.  
     */
    getDefaultObject()
    {
        /* ask the action to provide a default indirect object */
        return withGlobals({:action_.getDefaultIobj(self)});
    }

;

/* ------------------------------------------------------------------------ */
/*
 *   Actor Resolver.  We use this to resolve the actor to whom a command
 *   is directed: the actor must be in scope for the player character.  
 */
class ActorResolver: Resolver
    construct(issuingActor)
    {
        /* remember the issuing actor */
        actor_ = issuingActor;

        /* cache the scope list for the actor who issued the command */
        cacheScopeList(issuingActor);
    }

    /*
     *   Cache the scope list for this object.  By default, we cache the
     *   default scope list for the actor.
     */
    cacheScopeList(issuingActor)
    {
        /* cache the actor's default scope list */
        scope_ = issuingActor.scopeList();
    }

    /*
     *   Determine if an object is in scope for the purposes of object
     *   resolution.  By default, we'll simply check to see if the object
     *   is in our cached scope list.  Some actions might want to override
     *   this to provide alternative behavior; for example, for a command
     *   whose scope is the set of all objects, it would be much more
     *   efficient to dispense with caching a scope list entirely and
     *   simply return true here.  
     */
    objInScope(obj) { return scope_.indexOf(obj) != nil; }

    /*
     *   Get the "all" list - this is the list of objects that we should
     *   use when the object of the command is the special word "all".  By
     *   default, we'll return everything in scope.  
     */
    getAll()
    {
        /* we can't address 'all' */
        throw new ParseFailureException(&cannotAddressMultiple);
    }

    /* get the default object list */
    getAllDefaults()
    {
        /* there are no default actors */
        return [];
    }

    /*
     *   Filter an ambiguous list of objects.  We will filter according to
     *   which objects are most logical as targets of commands.  
     */
    filterAmbiguousNounPhrase(lst, requiredNum)
    {
        local likelyCnt;
        
        /*
         *   Run through the list and see how many objects are likely
         *   command targets.  
         */
        likelyCnt = 0;
        foreach (local cur in lst)
        {
            /* if it's a likely command target, count it */
            if (cur.obj_.isLikelyCommandTarget)
                ++likelyCnt;
        }

        /* 
         *   If some of the targets are likely and others aren't, and we
         *   have at least the required number of likely targets, keep
         *   only the likely ones.  If they're all likely or all unlikely,
         *   it doesn't help us because we still have no basis for
         *   choosing some over others; if removing unlikely ones would
         *   not give us enough to meet the minimum number required it
         *   also doesn't help, because we don't have a basis for
         *   selecting as many as are needed.  
         */
        if (likelyCnt != 0 && likelyCnt != lst.length()
            && likelyCnt >= requiredNum)
        {
            /* 
             *   we have a useful subset of likely ones - filter the list
             *   down to the likely subset 
             */
            lst = lst.subset({cur: cur.obj_.isLikelyCommandTarget});
        }

        /* return the result */
        return lst;
    }

    /*
     *   Filter a plural list 
     */
    filterPluralPhrase(lst)
    {
        /* 
         *   Use the same filtering that we use for ambiguous nouns.  This
         *   simply reduces the set to the likely command targets if any
         *   are likely command targets. 
         */
        return filterAmbiguousNounPhrase(lst, 1);
    }

    /* get a default object */
    getDefaultObject()
    {
        /* there is never a default for the target actor */
        return nil;
    }

    /* resolve a noun phrase involving unknown words */
    resolveUnknownNounPhrase(tokList)
    {
        /* we can't resolve an unknown noun phrase used as an actor target */
        return nil;
    }

    /* 
     *   Get a raw pronoun antecedent list.  Since we are resolving the
     *   target actor, pronouns are relative to the issuing actor.  
     */
    getRawPronounAntecedent(typ)
    {
        /* check for pronouns that are relative to the issuer */
        switch(typ)
        {
        case PronounMe:
            /*
             *   It's a first-person construction.  If the issuing actor
             *   is the player character, and the PC is in the second
             *   person, this refers to the player character (the game
             *   calls the PC "you", so the player calls the PC "me").  If
             *   the issuing actor is an NPC, this is unconditionally the
             *   PC.  
             */
            if (actor_.isPlayerChar && actor_.referralPerson != SecondPerson)
                return [];
            else
                return [actor_];

        case PronounYou:
            /*
             *   It's a second-person construction.  If the issuer is the
             *   player character, and the player character is in the
             *   first person, this refers to the player character (the
             *   game calls the PC "me", so the player calls the PC
             *   "you").  If the issuer isn't the player character, "you"
             *   has no meaning.  
             */
            if (!actor_.isPlayerChar || actor_.referralPerson != FirstPerson)
                return [];
            else
                return [actor_];

        default:
            /* 
             *   it's not a relative pronoun, so ask the issuing actor for
             *   the antecedent based on recent commands 
             */
            return actor_.getPronounAntecedent(typ);
        }
    }

    /* we resolve target actors */
    whichObject = ActorObject

    /* the cached scope list */
    scope_ = []
;

