// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        bsptree.h
//
// Purpose:     Interface to class BSPTree
//
// Created:      8 Feb 1996  Georg Meszaros
//
// Modified:    13 Jun 1996  Georg Meszaros 
//
//
// Description:
//
// Header file for class bsptree.
// 
//
//</file>


//<class>
//
// Name:       BSPTree
//
// Purpose:    
//
//
// Public Interface:
//
/*
   - BSPTree();
   - BSPTree(Face* face);
     > constructor which connects face into this BSP-Tree node
       no links in the tree node are set

   - ~BSPTree();
     > recursively deletes the front- and the back-tree of this node, 
       calling the destructor for all faces in this node afterwards

   - void connect(Face* face);
     > inserts face into the list of facepointers held in this node

   - void setParent(BSPTree* parent);

   - QvMFFace* faceList();
     > returns the facelist held by this node
   
   - void insert(Face* face);
     > inserts face into this node and splits/filters it down
       until it is at the right position

   - void buildSVBSPTree(SVBSPTree* svbsp_root, BSPMode mode); 
     > initiates the building of a shadow volume BSP-tree, 
       filters all polygons contained in the BSP-tree down the
       SVBSP-tree and extend the shadow volume by the visible parts

   - void draw();
     > draws the faces in this node whether it is backfaced or not

   - void draw(const point3D& position, BSPMode mode);
     > draws the faces in this node including backface-calculation and setting

   - void drawBackToFront(const point3D& position, BSPMode mode);
     > draws all nodes from this down in back to front order

   - void print();
     > prints all contained faces   

   - unsigned long nodeNumber() const;
     > gets total number of nodes in the BSP-tree  
  
   - unsigned long faceNumber() const;
     > gets total number of faces inserted (created) in the BSP-tree

   - void resetAttributes();
     > calls resetAttributes in any Face (to reset global "current drawing" attributes)   
   
   - void reset();
     > calls reset in any Face

*/
// Description:
//
// The class BSP-Tree is in charge for building a BSP-Tree, only being
// provided with faces - all clipping and split is hidden behinf this
// class in the class Clipping, which is only a container for methods that
// belong together.
// A Face is inserted into the root-node, all decisions on how to
// split and where to insert the newly created split faces are made in insert.
//
//
//</class>


#ifndef harmony_scene_bsptree_h
#define harmony_scene_bsptree_h


enum BSPMode { BSP_PLAIN, BSP_BACK_FACE_CULLING, BSP_SHADOW_VOLUME };  // gmes


class Face;  // polygon.h
#include <ge3d/vectors.h>
#include <vrml/QvMFFace.h>
// gmes-TODO: use class derived from QvPList instead


class SVBSPTree;


class BSPTree
{
  public:
    
    BSPTree();

    BSPTree(Face* face);

    ~BSPTree();

    void connect(Face* face);

    void setParent(BSPTree* parent);

    QvMFFace* faceList();
   
    void insert(Face* face);

    void buildSVBSPTree(SVBSPTree* svbsp_root, BSPMode mode);
    void draw();
    void draw(const point3D& position, BSPMode mode);
    void drawBackToFront(const point3D& position, BSPMode mode);
    void drawFrontToBack(const point3D& position, BSPMode mode);
 
    void print();
    unsigned long nodeNumber() const;
    unsigned long faceNumber() const;

    void resetAttributes();  
    void reset();
    
  private:
    static unsigned long node_number_;  // total number of nodes in the BSP-tree  
    static unsigned long face_number_;  // total number of faces inserted
   
    QvMFFace facelist_;     // pointer list to all faces(on the same plane) in a node 
    BSPTree* front_;        // BSP-tree that is completely infront of
                            // the plane, which is spanned by the faces in this node 
    BSPTree* back_;         // same like front_ but in the back
    BSPTree* parent_;       // a link to the parent node, nil for the root

}; // BSPTree


inline void BSPTree::setParent(BSPTree* parent) { parent_ = parent; }
inline QvMFFace* BSPTree::faceList() { return &facelist_; }
inline unsigned long BSPTree::nodeNumber() const { return node_number_; }
inline unsigned long BSPTree::faceNumber() const { return face_number_; }

#endif
