//<copyright>
//
// Copyright (c) 1993,94,95,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        main.C
//
// Purpose:     main program for VRweb (VRML 3D scene viewer)
//
// Created:      3 Jun 93   Michael Pichler
//
// Changed:     26 May 97   Michael Pichler
//
// $Id: main.C,v 1.28 1997/05/26 17:31:49 mpichler Exp $
//
//
// Description
// -----------
//
// VRweb (vrweb-*) for UNIX/X11; communication with Web browsers.
//
//</file>


char HyperGWhat[] = "@(#)[Hyper-G] [HGC-H] vrweb*\t1.3 [VRweb Scene Viewer] [Michael Pichler et al.]";



#include "scenewin.h"
#include "hg3dopt.h"
#include "geomobj.h"
#include "srcanch.h"
#include "stranslate.h"
#include "gecontext.h"
#include "urlserver.h"
#include "remote.h"
#include "compdate.h"

#include "imgview.h"
#include "txtview.h"

#include <hyperg/hyperg/message.h>
#include <hyperg/utils/verbose.h>
#include <hyperg/widgets/cursors.h>
#include <InterViews/enter-scope.h>

#include <IV-look/kit.h>
#include <InterViews/display.h>
#include <InterViews/event.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/window.h>

#include <iostream.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>



/*** SceneViewer ***/

class SceneViewer: public SceneWindow  // standalone scene viewer
{
  public:
    SceneViewer (Session* session)
    : SceneWindow (session, STranslate::VRWEBSCENEVIEWER)
    {
      urlserver_ = 0;
      opened_file_ = 0;
    }

    ~SceneViewer ()
    {
      delete urlserver_;
    }

    // Scene3D
    void requestURLnode (               // URL-request for either
      QvWWWInline* nodeI,               //   inline node (where attach inline data)
      QvTexture2* nodeT,                //   texture node (to read texture image)
      const char* url,                  //   requested URL (of data to be fetched)
      const char* docurl                //   URL of requesting document (for local URLs)
    );

    int writeOriginalScene (ostream& os);

    void showHelp (const char* file);
    void hold ();

    int followLink (const GeometricObject*, const SourceAnchor*);

    void loadTexture (Material* mat);  // load material from file

    // SceneWindow/Scene3D
    void clear ();

    URLServer* urlserverInstance ()
    {
      if (!urlserver_)
        urlserver_ = new URLServer (this);
      return urlserver_;
    }

    virtual void acceptRemoteRequests (Window* win, int on);
    virtual void propertyRemoteRequest (Window* win, const Event& e);

    void execHelper (int quit, const char* url = 0);

  private:
    URLServer* urlserver_;
    FILE* opened_file_;
};  // SceneViewer


int verbose = 0;  /* turn on/off DEBUG-messages */


void showCopyright ()
{
  cout << "--- " << STranslate::str (STranslate::AboutVERSIONvrweb) << " of " << compDate () << " ---" << '\n';
  cout << "Copyright (c) 1996,97 IICM, Graz University of Technology, Austria." << endl;  // flushes
}


void helponarguments ()
{
  cerr << endl;
  showCopyright ();
  cerr <<
    "\n"
    "Usage: vrweb... [Options] [FILE]\n"
    "\n"
    "vrweb-ogl: Open GL - vrweb-gl: Iris GL - vrweb-mesa: Mesa library\n"
    "specify \"-\" as FILE to read from std. input\n"
    "\n"
    "Options:\n"
    "  -h[elp]        print this help\n"
    "  -remote file   open file in a running VRweb instance\n"
    "  -getRemoteID   get remote window ID and exit\n"
  << commonViewerOptions
  << endl;
}


/*** remote calls ***/

const char* const RemoteVersionProperty = "_VRWEB_VERSION";
const char* const RemoteCommandProperty = "_VRWEB_COMMAND";
const char* const remoteVersion = "VRweb 1.2";  // version of remote call interface


void SceneViewer::acceptRemoteRequests (Window* win, int on)
{
  // window is bound, offer it for remote calls

  // cerr << "setting remote version to " << remoteVersion << endl;
  remoteSetVersion (win->display ()->rep (), win->rep (), on ? remoteVersion : (const char*) NULL);
}


void SceneViewer::propertyRemoteRequest (Window* win, const Event& e)
{
  RString command;

  // command syntax (VRweb internal property):
  // file=FILENAME\n
  // URL=URL\n
  // other fields ignored (extensible)

  if (!remoteReceiveCommand (win->display ()->rep (), win->rep (), e.rep (), command))
    return;  // some other property changed

  DEBUGNL ("recieved remote command: " << command);

  RString task, filename, url;
  const char* cmd = command;
  while (*cmd)  // parse command fields
  {
    const char* nlpos = strchr (cmd, '\n');
    if (!nlpos)
    { DEBUGNL ("remote command garbled; ignoring fields from: " << cmd);
      break;
    }
    if (!strncmp (cmd, "file=", 5))
    { cmd += 5;
      RString temp (cmd, nlpos - cmd);
      filename = temp;
    }
    else if (!strncmp (cmd, "URL=", 4))
    { cmd += 4;
      RString temp (cmd, nlpos - cmd);
      url = temp;
    }
    else if (!strncmp (cmd, "task=", 5))
    { cmd += 5;
      RString temp (cmd, nlpos - cmd);
      task = temp;
    }
    else
    { DEBUGNL ("unknown remote command field (ignored) at beginning of: " << cmd);
    } // for future extension

    cmd = nlpos + 1;
  }

  DEBUGNL ("filename: " << filename << "; URL: " << url << "; task: " << task);

  GEContext* gecon = gecontext ();

  // goto destination (#viewpoint) of current scene
  if (!strcmp (task.string (), "gotoViewpoint"))
  {
    const char* fispos = strrchr (url.string (), '#');
    if (fispos)
    {
      // parenturl unchanged (still in same scene)
      setTitle (url);
      RString vpname = url.gRight (fispos - url.string () + 1);
      DEBUGNL ("activating viewpoint " << vpname << " of the URL " << url);

      activateCamera (vpname);
      reset ();  // and redraw
    }
    return;
  } // goto viewpoint

  // default task: open remote file

  FILE* file = fopen (filename, "r");
  if (!file)
  {
    cerr << "VRweb. Error: remote file '" << filename << "' not found." << endl;
    // setTitle (STranslate::str (STranslate::ProgressREADY));  // old scene not destroyed
    return;
  }

  if (gecon)
    gecon->pushCursor (HgCursors::instance ()->hourglass ());

  if (readSceneFILE (file))
    setTitle (STranslate::str (STranslate::ProgressREADY));  // parse error or VRML 2.0
  else
  {
    if (url.length ())
    {
      currentURL (url);  // #dest handled after build
      setTitle (url);
    }
    else
      setTitle (filename);

    reset ();  // and redraw
  }

  if (opened_file_)
    fclose (opened_file_);

  // web browser erases temp file when helper app
  // (here: remote caller) terminates;
  // file is kept open to be able to save it, although it already
  // seems to have disappeared from the file system

  setCurrentFilename (filename);
  execHelper (0, url.string ());  // do not exit here in case of VRML 2.0

  opened_file_ = file;  // will be closed in clear; available for saving
  DEBUGNL ("file " << (void*) opened_file_ << " is kept open until scene is destroyed.");

  if (gecon)
    gecon->popCursor ();
} // propertyRemoteRequest



// fork/exec helper app for other, unsupported input formats (read: VRML 2.0)
// to be called after readSceneFile (or setCurrentFilename subsequent to readSceneFILE)
// quit: flag, whether to quit VRweb when calling helper app (i.e. exec only)

void SceneViewer::execHelper (int quit, const char* urlin)
{
  float version = unsupportedVersion ();
  if (!version)  // supported format
    return;

  const RString* curfilename = currentFilename ();
  if (!curfilename)
  {
    cerr << "VRweb: cannot pass piped input to VRML 2.0 helper application" << endl;
    return;
  }

  RString helper;
  WidgetKit& kit = *WidgetKit::instance ();
  kit.style ()->find_attribute ("VRML2_command", helper);

  if (!helper.length ())
  {
    cerr << "VRweb: no helper application for VRML 2.0 (VRML2_command) defined" << endl;
    return;
  }

  RString filename = curfilename->string ();
  RString url;
  if (urlin)
    url = urlin;
  else
    kit.style ()->find_attribute ("URL", url);
  url.subst ("\"", "%22", false);  // secure netscape remote call
  url.subst ("'", "%27", false);   // and mailcap handling of URL

  helper.subst ("%s", filename, 0);  // case sensitive
  helper.subst ("%u", url, 0);

  DEBUGNL ("going to read " << filename << " with helper app for VRML " << version);

  cerr << "VRweb: executing VRML 2.0 helper application:\n  " << helper << endl;

  int pid = !quit && fork ();

  if (pid < 0)
  {
    HgMessage::error ("fork to run VRML 2.0 helper failed");
  }
  else if (!pid)  // child (or exec/quit)
  {
    // let sh parse the command
    execl ("/bin/sh", "/bin/sh", "-c", helper.string (), 0);  // no return on success
    HgMessage::error ("could not execute VRML 2.0 helper");
    // keep viewer alive when exit failed (also when quit is set)
  }
  // else: continue in parent (after fork)

} // execHelper



/*** main ***/


int main (int argc, char* argv[])
{
#ifdef MEMLEAK
_do_not_compile_Interviews_programs_with_memleak_ _at_least_not_on_SGI_;
MemLeakBegin ();
#endif

/*
char* test = new char [20];  // test malloc-debug
test [20] = 'j';
delete test;
*/

  HgMessage::init ("VRweb Scene Viewer");
  // copyright notice supressed in case of remote call

  Session* session = new Session ("Harmony", argc, argv, hg3d_options, hg3d_props);
  WidgetKit& kit = *WidgetKit::instance ();
  Style* style = new Style ("Scene", session->style ());
  kit.style (style);
  // style name (X/resources): Harmony.Scene
  // always use kit.style -- session.style never changes!!!

  if (style->value_is_on ("commandlinehelp"))
  {
    helponarguments ();
    return 0;  // exit
  }

  if (style->value_is_on ("verbose"))
    verbose = 1;

  // get ID of running VRweb window and exit
  if (style->value_is_on ("getRemoteID"))
  {
    unsigned long id = remoteGetWindowID (session->default_display ()->rep ());
    DEBUGNL ("this is the ID of a remote VRweb window (0 iff not found):");
    cout << "0x" << hex << id << endl;
    return 0;
  }

  // let remote VRweb instance go to a specific camera (viewpoint)
  if (style->value_is_on ("remoteView"))
  {
    RString command ("task=gotoViewpoint\n");

    RString strval;
    if (style->find_attribute ("URL", strval))
    { command += "URL=";
      command += strval;
      command += "\n";
    }

    if (!strchr (strval.string (), '#'))
    {
      HgMessage::error ("must provide viewpoint URL (#camera) for remoteView command");
      return 1;
    }
    DEBUGNL ("passing viewpoint URL " << strval << " to remote instance");

    if (!remoteSendCommand (session->default_display ()->rep (), command))
    {
      HgMessage::error ("could not send remote viewpoint request");
      return 1;
    }
    return 0;
  }

  // handle remote calls (see propertyRemoteRequest above)

  RString filename;
  if (style->find_attribute ("remote", filename))
  {
    RString command ("file=");
    command += filename;
    command += "\n";

    RString strval;
    if (style->find_attribute ("URL", strval))
    { command += "URL=";
      command += strval;
      command += "\n";
    }

    if (remoteSendCommand (session->default_display ()->rep (), command))
    {
      DEBUGNL ("VRweb: " << command << "... handled remote");
      sleep (5);  // hack (ensure remote instance has opened file before exiting here)
      return 0;  // my work ends here
    }
    else
    { DEBUGNL ("VRweb remote: will open " << filename << " myself");
    }
  }

  showCopyright ();

  // create application window
  SceneViewer scene (session);  // the scene

  DEBUGNL ("VRweb: mapping application window");
  Window* appwin = scene.appwin ();
  appwin->map ();  // map application window
/*
  // care for immediate mapping (feedback on loading) -- does not work
  appwin->display ()->flush ();
  appwin->repair ();
  appwin->display ()->sync ();  // flush not sufficient
*/
/*
  appwin->repair ();
  appwin->display ()->flush ();
  Event e;  // read pending events to map and draw window -- does not work
  while (session->pending ())
  {
    session->read (e);
    e.handle ();
  }
*/

  // read initial scene

  if (argc == 2)  // IV-options are already removed from argument list
    if (argv [1][0] == '-' && !argv [1][1])  // argument "-": stdin
    { scene.readSceneFILE (stdin);
      scene.setTitle ("<stdin>");
    }
    else                                     // filename argument
    {
      if (scene.readSceneFile (argv [1]))
      {
        scene.setTitle (STranslate::str (STranslate::ProgressREADY));  // parse error or VRML 2.0
        scene.execHelper (1);  // quit on VRML 2.0
      }
      else
      {
        const char* url = scene.mostRecentURL ();
        scene.setTitle (*url ? url : argv [1]);
      }
    }
  else if (filename.length ())  // remote file w/o remote instance
  {
    if (scene.readSceneFile (filename.string ()))
    {
      scene.setTitle (STranslate::str (STranslate::ProgressREADY));  // parse error or VRML 2.0
      scene.execHelper (1);  // quit on VRML 2.0
    }
    else
    {
      const char* url = scene.mostRecentURL ();
      scene.setTitle (*url ? url : filename.string ());
    }
  }
  else  // no filename on command line
  {
    if (style->find_attribute ("initialDemo", filename))
    { scene.readSceneFile (filename.string ());  // should be VRML 1.0, don't fool me
      scene.setTitle (filename.string ());
    }
  }

  // run the session (dispatcher loop)
  // int retval = session->run_window (scene.appwin ());
  int retval = session->run ();  // window already mapped at the beginning

// if dbmalloc is active scan memory for traps
char* finaltest = new char;
delete finaltest;

#ifdef MEMLEAK
MemLeakEnd ();
#endif


  return retval;

} // main



/*** SceneViewer ***/


// clear

void SceneViewer::clear ()
{
  DEBUGNL ("SceneViewer::clear: clearing pending requests and scene data");
  if (urlserver_)
    urlserver_->clearAllRequests ();
  if (opened_file_)
  { DEBUGNL ("closing file " << (void*) opened_file_ << endl);
    fclose (opened_file_);
    opened_file_ = 0;
  }
  SceneWindow::clear ();
}


// requestURLnode
// request an inline URL (must be a VRML scene)
// asynchronous request (function will return immediately)
// file: URLs will always be read from file system
// otherwise they will be read from the Web if docurl is known

void SceneViewer::requestURLnode (QvWWWInline* nodeI, QvTexture2* nodeT, const char* url, const char* docurl)
{
  // to deactivate: Scene3D::requestURLnode (nodeI, nodeT, url, docurl);
  // ass.: docurl non-NULL
  if (!url || !*url)
    return;

  int local = localInlines ();
  int absurl = !strncmp (url, "http:", 5);
  int localparent = !strncmp (docurl, "file:", 5);

  if (local && absurl)
  { HgMessage::error ("cannot read http-URL from local file");
    return;
  }
  if (!absurl && (!*docurl || localparent))
  { // if trying to read relative URL without knowing parent URL
    local = 1;  // assume we should read from file instead
  }

  if (!strncmp (url, "file:", 5))
  { local = 1;  // file: URLs always read from local file
    url += 5;  // path after "file:" prefix
  }


  if (local)  // read inline/texture from file
  {
    RString filename;
    DEBUGNL ("reading file " << url << "; directory: " << currentPath () << "; parent URL: " << docurl);
    if (*url == '/')  // absolute path
      filename = url;
    else
    {
      const char* slpos = strrchr (docurl, '/');
      if (localparent && slpos)  // take directory form file: URL
      { // "file:DIR/junk" yields "DIR/"
        RString temp (docurl + 5, slpos - docurl - 4);
        filename = temp;
      }
      else  // directory file was loaded from
        filename = currentPath ();
      // directory with trailing '/', can append filename
      filename += url;
    }
    DEBUGNL ("absolute filename: " << filename);
    if (nodeI)
    {
      RString oldparent (mostRecentURL ());
      RString fileurl = "file:" + filename;
      currentURL (fileurl.string ());
      // cerr << "new parent URL set to " << fileurl << endl;
      readInlineVRMLFile (nodeI, filename.string ());
      // currentURL (oldparent);  // restore the old parent URL
      // (other parts of the scene graph not necessarily built yet) - no, not the case
    }
    if (nodeT)
      readTextureFile (nodeT, filename.string ());
  }
  else  // fetch from the web (asynchronously)
  {
    URLServer* urlserver = urlserverInstance ();
    urlserver->appendRequest (nodeI, nodeT, url, docurl);
    urlserver->handleRequests (nodeT ? 1 : 0);  // begin to handle inline request now
    // textures only requested when needed
  }

} // requestURLnode


// writeOriginalScene
// write original scene (either known by name or kept FILE* open)

int SceneViewer::writeOriginalScene (ostream& os)
{
  if (opened_file_)  // copy file still opened
  { DEBUGNL ("writeOriginalScene of file that has been kept open");
    rewind (opened_file_);
    return copyFile (opened_file_, os);
  }

  return Scene3D::writeOriginalScene (os);

} // writeOriginalScene


// showHelp
// now shared by Harmony version in scene3d.C
// will be moved here when Harmony has its own help system

void SceneViewer::showHelp (const char* file)
{
  GEContext* gecon = gecontext ();

  if (gecon)
    gecon->pushCursor (HgCursors::instance ()->hourglass ());
    
  SceneWindow::showHelp (file);

  if (gecon)
    gecon->popCursor ();
}


// hold this window

void SceneViewer::hold ()
{
  Window* win = appwin ();
  if (win->bound ())
    acceptRemoteRequests (win, 0);  // do no longer accept remote requests
}


// followLink (SDF)
// anachronism of early demo days - will be removed soon

// follow a link by loading another scene or starting another viewer
// name syntax: ".*(filename)T.*"  (.* ... any no. of any characters)
// where 'T' (type) is either T for text, I for image (TIFF) or 3 for 3D scenes
// returns 1 if new 3D scene must be displayed, zero otherwise
// group anchors are ignored in the standalone viewer

int SceneViewer::followLink (const GeometricObject* hitobj, const SourceAnchor*)
{
/*
  // group anchor tests
  long id = hitobj->groupAnchorId (groups);
  if (id)
    cerr << "followLink: would activate link with anchor id " << id << endl;
  else
  { cerr << "followLink: no link to activate" << endl;
    return 0;
  }
*/

  const char* name = hitobj->name ();

  const char* pos1 = strchr (name, '(');
  char* pos2 = (char*) strchr (name, ')');

  if (!pos1 || !pos2 || pos2-pos1 < 2)
  {
    cerr << "hg3d. syntax error in anchor name '" << name << "'." << endl;
    return '\0';
  }

  pos1++;  // pos1: filename
  *pos2 = '\0';
  char file [256];
  sprintf (file, "%s%s", currentPath (), pos1);

  *pos2 = ')';
  char type = pos2 [1];

  const char* ppos = strrchr (pos1, '.');
  const char* slpos = strchr (pos1, '/');
  int noext = !ppos || (ppos && slpos && ppos < slpos);  // no . or only in path

  switch (type)
  { case 'T':
      if (noext)  // default: .txt
        strcat (file, ".txt");
      showtext (file);
    break;

    case 'I':
      if (noext)  // default: .tif
        strcat (file, ".tif");
      showimage (file);
    break;

    case '3':
      if (noext)  // default: .sdf
        strcat (file, ".sdf");
      readSceneFile (file);
    return 1;

    default:
      cerr << "hg3d. unknown type '" << type << "' of anchor '" << name << "'." << endl;
  }

  return 0;

} // followLink
