// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1995,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        vrmlscene.h
//
// Purpose:     interface to 3D scene, VRML representation
//
// Created:     24 Apr 95   Michael Pichler
//
// Changed:      5 Apr 96   Georg Meszaros (BSP)
//
// Changed:     20 Feb 97   Alexander Nussbaumer (editing)
//
// Changed:     27 Feb 97   Michael Pichler
//
// $Id: vrmlscene.h,v 1.30 1997/04/30 10:39:12 mpichler Exp $
//
//</file>


#ifndef harmony_scene_vrmlscene_h
#define harmony_scene_vrmlscene_h

#include "arrays.h"
#include "scenedata.h"

#include <vrml/QvTransform.h>

const int undosteps = 10; // undo transformation

class QvGroup;
class QvInput;
class QvIndexedFaceSet;
class QvMaterial;
class QvNode;
class QvPerspectiveCamera;
class QvOrthographicCamera;
class QvSwitch;
class QvTransform;
class QvTexture2;
class QvTexture2Transform;
class QvWWWAnchor;
class MaterialDialog;
class TextureDialog;
class WWWAnchorDialog;
class StructureViewerDialog;
class TransformDialog;
class ViewpointDialog;
class GridDialog;
class FontChar;
class BSPTree;  // gmes

// enum BSPMode moved to bsptree.h


class VRMLScene: public SceneData
{ 
  public:
    VRMLScene (Scene3D* scene);
    ~VRMLScene ();

    // *** input ***
    int readInput (QvInput& in);
    const char* formatName () const  { return "VRML"; }
    int readInlineVRML (                // read inline VRML scene
      QvWWWInline* node, FILE*);        // returns nonzero on success
    void increaseNumFaces (unsigned num)
    { numfaces_ += num; }
    void increaseNumPrimitives ()  // text, cubes, cones, cylinders, spheres
    { numprimitives_++; }
    void hasTextures ()  { hastextures_ = 1; }

    // fonts
    enum { fnt_serif, fnt_sans, fnt_fixed,  // (base) font types
      fnt_bold = 3, fnt_italic = 6,  // only used inside getFontChars
      fnt_numdefined = 12  // no. of available fonts
    };
    FontChar* getFontChars (            // array of 256 character definitions or NULL
      int family,                       //   (base) font type
      int& bold, int& italic            //   bold and italic flags (in/out)
    );                                  //   out value: whether to apply linewidth/slanting

    // enum {  // rebuild flags (not yet used)
    //   bld_camera = 0x1  // default camera (only on initial processing)
    // };
    void rebuild (int extras);
    void colorRebuild ();               // recompute hilit colors
    // { rebuild (bld_colors); }        // (only when highlighting colors changed)
    void buildFont (QvNode* fontroot);  // build font info

    // *** output ***
    int writeData (ostream& os, int format);
    void printInfo (int all);           // print scene information
    unsigned long getNumFaces () const  // number of faces (polygons)
    { return numfaces_; }
    unsigned long getNumPrimitives () const  // no. of primitives
    { return numprimitives_; }
    unsigned long getNumBSPFaces () const  // gmes, 19960405
    { return num_bsp_faces_; }
    unsigned long getNumBSPNodes () const  // gmes, 19960405 
    { return num_bsp_nodes_; }
    int doesHaveTextures () const       // get texture flag
    { return hastextures_; }
    int supportsOutputformat (int format);

    // *** drawing ***
    void draw (int mode);

    // rendering mode: Z-Buffer or BSP tree
    enum { rnd_zbuffer, rnd_bsptree };
    static void renderMode (int mode)  { render_mode_ = mode; }
    static int renderMode ()  { return render_mode_; }
    // BSP modi see bsptree.h
    static void bspMode (int mode)  { bsp_mode_ = mode; }
    static int bspMode ()  { return bsp_mode_; }

    // *** picking ***
    virtual void* pickObject (
      const point3D& A,                 //   ray start
      const vector3D& b,                //   ray direction
      float tnear,                      //   near distance
      float tfar,                       //   far distance
      GeometricObject** gobj = 0,       //   GeometricObject hit (return)
      QvNode** node = 0,                //   node hit (return)
      QvWWWAnchor** anchor = 0,         //   anchor hit (return)
      point3D* hitpoint = 0,            //   optionally calculates hit point
      vector3D* normal = 0,             //   and face normal vector (normalized)
      const StringArray** groups = 0,   //   optionally determines groups hit
      IntArray** hitpath = 0,           //   optionally creates path to VRML node hit
      float* hittime = 0                //   optionally returns the hit time
    );
    void selectedPosition (             // get position of selected node
      RString& pos                      // (to be used for VRML anchor definition)
    );

    // *** find objects ***
    // anuss-TODO: description
    QvNode* findNode (QvGroup* parent, int nodetype, int recursive = false);
    QvNode* findNodeRecursive (QvNode* node, int nodetype);
    // QvGroup* findParent (QvNode* node = NULL);
    enum { getpar_noHyperGanchors = 0x1 };
    QvGroup* getParentByPath (          // get parent of node
      const IntArray* path,             //   defined by path
      int flags = 0                     //   any special treatment (getpar_*)
    ) const;                            //   returns NULL on any error
    QvNode* getNodeByName (             // get node and parent (wrlpass.C)
      const char* name,                 //   by node name (DEF label)
      QvGroup*& parent,                 //   parent of node found
      int& childindex                   //   index of node in parent's childlist
    );
    QvTexture2* findTexture (           // find texture node (wrlpass.C)
      const char* fnamefield            //   by file name field
    );

    // *** anchors ***
    void clearAnchors ();               // clear all Hyper-G anchors (wrlpass.C)
    void defineHyperGanchor (           // store Hyper-G anchor
      long id,                          //   anchor id
      const RString& aobj,              //   anchor object
      const char* posstr,               //   anchor position field
      const RString& title              //   anchor title (descr.)
    );
    void findTextureMaterial (          // find texture material
      const char* name,                 //   name (Hyper-G texture anchor)
      Material*&,
      QvTexture2*& texture              //   returns VRML texture node
    )  { texture = findTexture (name); }

    // *** camera ***
    Camera* getCamera () const  { return camera_; }
    void storeCamera ();                // store active camera
    void restoreCamera ();              // reset camera
    // scene defines its camera, name is registered in viewpoint list
    void hasCamera (QvPerspectiveCamera*, const char* name);
    void hasCamera (QvOrthographicCamera*, const char* name);

    void activateCamera (
      const char*,
      QvPerspectiveCamera*, QvOrthographicCamera*
    );
    void activatePCam (QvPerspectiveCamera*);   // set active camera
    void activateOCam (QvOrthographicCamera*);  // set active camera
    const QvPerspectiveCamera* activePCam () const
    { return activepcam_; }
    const QvOrthographicCamera* activeOCam () const
    { return activeocam_; }

    // *** light sources ***
    void hasLight ()   { haslight_ = 1; }   // scene contains light source
    int nextLight ()                    // get next useable light index
    { return ++numlights_; }
    int numLights () const              // return no. of used lights
    { return numlights_; }
    void deactivateLights (             // deactivate all light sources
      int n = 0                         //   except the first n ones
    );

    // confexify: rpersche, May 96
    void convexifyFaceSet (QvIndexedFaceSet*);
    // triangulate face set (triangulate.C)
    static void doConvexify (int flag)  { doconvexify_ = flag; }

    // autosmooth: pzemljic, May 96
    void autosmooth (QvIndexedFaceSet*, float crease_angle);
    // generate vertex normal vectors (asmooth.C)
    static void doAutosmooth (int flag)  { doautosmooth_ = flag; }

    /*** editing ***/ // anuss

    void startEditing ()
    { startedediting_ = 1; }
    int startedEditing () const
    { return startedediting_; }

    // object transformation
    int selTransformInsideGroup ()
    { return sel_transform_insidegroup_; }
    QvTransform* selectedTransform () const
    { return sel_transform_; }

    void translateSelected (const vector3D& trans);
    void rotateSelected (vector3D& axis, float angle);  // radians
    void scaleSelected (const vector3D& scale);
    void storeRotation ();

    // object selection
    void updateEditContext ();
    void unselectPropertyNodes ();
    int selectParent ();
    int selectChild ();
    int viewSelectedNode ();
    void viewAll ();
    int selectNext (int step);  // +1 or -1

    // undo
    void storeForUndo ();
    int undoManipulation ();
    void clearUndoStorage ();

    // insert, delete, cut, copy, paste
    QvNode* getCutCopyPastePath (IntArray& path);
    int deleteNode (const IntArray&);
    int insertNode (QvNode*, const IntArray&);
    int deleteSelectedNode ();
    int cutSelectedNode ();
    int copySelectedNode ();
    int pasteSelectedNode ();
    int findInsertPath (IntArray& path);
    int readInputInsert (QvInput& in);
    int insertVRMLPrimitive (int nodetype);
    int insertWWWInline (const char* filename);

    // constraints
    void pickTranslation (vector3D& trans);
    void pickScale (vector3D& scale);
    void constraints (int flag)
    { constraints_ = flag; }
    int constraints ()
    { return constraints_; }

    // edit dialogs
    void setWWWAnchorDialog (WWWAnchorDialog* dialog)
    { wwwanchordialog_  = dialog; }
    void setTextureDialog (TextureDialog* dialog)
    { texturedialog_ = dialog; }
    void setMaterialDialog (MaterialDialog* dialog)
    { materialdialog_ = dialog; }
    void setStructureViewerDialog (StructureViewerDialog* dialog)
    { structureviewerdialog_  = dialog; }
    void setTransformDialog (TransformDialog* dialog)
    { transformdialog_ = dialog; }
    void setGridDialog (GridDialog* dialog)
    { griddialog_  = dialog; }
    void setViewpointDialog (ViewpointDialog* dialog)
    { viewpointdialog_ = dialog; }

    void updateTransformDialog (int what = 0);
    void updateMaterialDialog ();
    void updateTextureDialog ();
    void updateWWWAnchorDialog ();
    void updateStructureViewerDialog ();
    void updateGridDialog ();

    int applyWWWAnchor (const char* url, const char* description);
    void deleteWWWAnchor ();
    int applyNodeSelection (int item);
    void applyNodeName (int whichitem, RString& name);

    int applyTexture (const char* filename, int wraps, int wrapt,
                      float transx, float transy, float rot, float scalex, float scaley);
    int applyMaterial (const float*, const float*, const float*, const float*, float, float);
    int applyTransform (const vector3D* translate, const vector3D* rotate, const float* angle, const vector3D* scale);
    int insertPropertyNode (QvNode* propnode);
    void setGrid (const vector3D& axis, const vector3D& position, float extent, float distance);

    void parseSceneGraph (QvGroup* group, IntArray* path);
    int getSelectedNodeByNumber (QvGroup* parent, IntArray* path, int& number);
    void getNameOfSelectedNode (RString& name);

    void applyViewpoint (
      int which, const point3D& position, const vector3D& axis, float angle, float viewangle, const char* name = 0
    );
    void switchtoViewpoint (int which);
    void updateViewpointDialog ();
    void updateViewpointValues (int which);
    void addViewpoint ();
    void deleteViewpoint (int which);
    void setViewpoint (int which);

    void setSelectedBoundingbox (const point3D& min, const point3D& max, const matrix4D lwmat, const matrix4D omat);

    void calculateObjectExtent ();
    void calculateHasExtent (QvGroup* node);  // resets the memver hasextent_ for every node in the scene graph
    void getObjectExtent (point3D& omin, point3D& omax)
    { omin = sel_omin_; omax = sel_omax_; }
    void getSceneExtent (point3D& min, point3D& max)
    { min = root_->omin_; max = root_->omax_; }

    void calculateCameraMatrix ();
    void getBoundingboxObject (point3D& min, point3D& max, matrix4D mat = 0) const;
    void getBoundingboxLocal (point3D& min, point3D& max, matrix4D mat = 0) const;
    void getBoundingboxWorld (point3D& min, point3D& max) const;

    int checkVRMLScene (QvGroup* root = 0);   // chcks if only one top separator exists
    void checkVRMLTree (QvGroup* node);
    void debugScenegraph ();
    void debugParse (QvGroup* group, IntArray* path);
    void printPath (IntArray* path);

  private:
    void drawVRML ();
    void buildBSPTree ();     // gmes
    void drawBSPTree ();      // gmes

    FontChar* loadFont (const char* filename, FontChar*&);

    // anuss: editing helpers
    // QvGroup* getParentByPath (const IntArray* patharr) const;
    int getSelectionByPath (const IntArray* path, QvGroup*& parent, QvNode*& node, int& whichnode, int& pathsize) const;
    QvNode* getSelectedNodeByPath (const IntArray* path);

    QvNode* searchPropertyNode (int nodetype, IntArray** propertypath = 0); 
    QvNode* searchGroupNode (int nodetype, IntArray** propertypath = 0); 
    int numberOfRelevantNodes (QvGroup* parent, int whichnode, int nodetype);  
    int multipleRelevantOfProperty (int nodetype, const IntArray* path = 0);

    void selectNodeByPath (IntArray* path);
    void selectTransform ();
    void searchTransform ();
    void makeTransformForSelected ();
    // void rebuild ();                    // rebuild data structure after changing some transformationsa

    void drawGrid ();
    int selectMaterial ();
    int selectWWWAnchor ();

    QvNode* root_;  // scene graph
    BSPTree* bsp_root_;  // BSP-Tree, gmes

    Camera* camera_;     // active camera
    Camera* bakcamera_;  // stored camera
    // active VRML camera
    QvPerspectiveCamera* activepcam_;
    QvOrthographicCamera* activeocam_;

    int matbuilt_;   // flag: trf matrices built (set on first drawing)
    int hascamera_;  // flag: camera definition contained in VRML scene
    int haslight_;   // flag: light source contained in VRML scene
    int numlights_;  // no. of currently activated lights

    unsigned long num_bsp_faces_;  // gmes, 19960405
    unsigned long num_bsp_nodes_;  // gmes, 19960405
    unsigned long numfaces_;
    unsigned long numprimitives_;
    int hastextures_;

    QvTransform* sel_transform_;  // anuss: selected transformation
    int sel_transform_insidegroup_;  // anuss: selected transformation is inside the selected groupnode
    int transformation_allowed_;
    float vrmlversion_;    // anuss: current VRML Version
    QvTexture2* sel_texture_;
    QvTexture2Transform* sel_texturetransform_;
    QvMaterial* sel_material_;
    QvWWWAnchor* sel_wwwanchor_;
    point3D sel_omin_, sel_omax_;  // object boundingbox
    int sel_istopseparator_;       // true if the selected node is the top separator of the scene graph

    vector3D sel_dragtranslate_, sel_dragrotate_, sel_dragscale_;
    float sel_dragangle_;

    matrix4D cameramat_, camerainvmat_;

    int constraints_;  // for object translation

    vector3D rotbasisaxis_; // remove!
    float rotbasisangle_; // remove!

    QvNode* paste_;
    int pastenr_;

    // undo
    QvTransform undotransform_[undosteps];
    QvTransform* undowhichtransform_[undosteps];
    IntArray undowhichpath_[undosteps];
    int undostart_, undonext_, undoempty_;

    QvSwitch* camswitch_; // for viewpoints
    int startedediting_;

    // edit dialogs
    MaterialDialog* materialdialog_;
    TextureDialog* texturedialog_;
    WWWAnchorDialog* wwwanchordialog_;
    StructureViewerDialog* structureviewerdialog_;
    TransformDialog* transformdialog_;
    ViewpointDialog* viewpointdialog_;
    GridDialog* griddialog_;

    vector3D grid_, grid_position_;
    float grid_extent_, grid_distance_;

    static int doconvexify_, doautosmooth_;

    // gmes: BSP flags/constants
    static int render_mode_;
    static int bsp_mode_;
    int quickwire_;
  public: // for setting in NavigationSettingsDialog
    static float frame_tolerance_;
  private:
    int frames_to_render_;
    static float area_margin_;
    static int first_x_polygons_;

    static FontChar* fontchars_ [fnt_numdefined];

  public:
    // red-green stereo colors
    static int leftred_, leftgreen_, leftblue_;
    static int rightred_, rightgreen_, rightblue_;

}; // VRMLScene


#endif
