/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "constants.h"
#include "osis.h"
#include "gwrappers.h"
#include "bible.h"
#include "books.h"


void osis_insert_empty_lines (WriteText * wt)
{
  wt->text ("\n");
  wt->text ("\n");
}


OsisRoot::OsisRoot (WriteText * wt, const ustring& name, const ustring& description)
// Add the root data for the osis file.
{
  mywt = wt;
  mywt->text ("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
  mywt->text ("\n");
  mywt->text ("<osis xmlns=\"http://www.bibletechnologies.net/2003/OSIS/namespace\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.bibletechnologies.net/2003/OSIS/namespace osisCore.2.1.xsd\">\n");
  mywt->text ("\n");
  mywt->text ("<osisText osisIDWork=\"" + name + "\" osisRefWork=\"defaultReferenceScheme\" xml:lang=\"en\">\n");
  mywt->text ("\n");
  mywt->text (" <header>\n");
  mywt->text ("  <work osisWork=\"" + name + "\">\n");
  mywt->text ("   <title>" + description + "</title>\n");
  mywt->text ("   <identifier type=\"OSIS\">Bible." + name + "</identifier>\n");
  mywt->text ("   <refSystem>Bible." + name + "</refSystem>\n");
  mywt->text ("  </work>\n");
  mywt->text ("  <work osisWork=\"defaultReferenceScheme\">\n");
  mywt->text ("   <refSystem>Bible."+ name + "</refSystem>\n");
  mywt->text ("  </work>\n");
  mywt->text (" </header>\n");
  osis_insert_empty_lines (mywt);
}


OsisRoot::~OsisRoot ()
{
  osis_insert_empty_lines (mywt);
  mywt->text ("</osisText>\n");
  mywt->text ("</osis>\n");
}


OsisBook::OsisBook (WriteText * wt, const ustring& english_book)
// Add the book data for the osis file.
{
  // Save variables.
  mywt = wt;
  // Get the abbreviation to be used.
  unsigned int index = books_english_to_id (english_book);
  book = books_id_to_osis (index);
  // Insert xml.  
  osis_insert_empty_lines (mywt);
  mywt->text ("  <div type=\"book\" osisID=\"" + book + "\">\n");
  osis_insert_empty_lines (mywt);
}


OsisBook::~OsisBook ()
{
  osis_insert_empty_lines (mywt);
  mywt->text ("  </div>\n");
  osis_insert_empty_lines (mywt);
}


OsisChapter::OsisChapter (WriteText * wt, const ustring& book, unsigned int chapter_in)
// Add the chapter data for the osis file.
{
  // Save variables.
  mywt = wt;
  chapter = chapter_in;
  osis_insert_empty_lines (mywt);
  mywt->text ("   <chapter osisID=\"" + book + "." + convert_to_string (chapter) + "\">\n");  
  osis_insert_empty_lines (mywt);
}


OsisChapter::~OsisChapter ()
{
  osis_insert_empty_lines (mywt);
  mywt->text ("   </chapter>");
  osis_insert_empty_lines (mywt);
}


OsisVerse::OsisVerse (WriteText * wt, const ustring& book, unsigned int chapter, const ustring& verse)
// Add the verse data for the osis file.
{
  // Save variables.
  mywt = wt;
  // SWORD does not support verses like 1a. Work around this.
  string verse_in (verse);
  string verse_out;
  // Remove any "a" from a verse, e.g. v 10a-11 becomes v 10-11.
  for (unsigned int i = 0; i < verse_in.length(); i++) {
    if (verse_in.substr (i, 1) != "a")
      verse_out.append (verse_in.substr (i, 1));
  }
  // Remove any "b" from the verse.
  verse_in = verse_out;
  verse_out.clear ();
  for (unsigned int i = 0; i < verse_in.length(); i++) {
    if (verse_in.substr (i, 1) != "b")
      verse_out.append (verse_in.substr (i, 1));
  }
  // Insert xml code. 
  wt->text ("    <verse osisID=\"");
  wt->text (book + "." + convert_to_string (chapter) + "." + verse_out);
  wt->text ("\">");
}


OsisVerse::~OsisVerse ()
{
  mywt->text ("</verse>\n");
  mywt->text ("\n");
}


OsisParagraph::OsisParagraph (WriteText * wt)
// Open a paragraph.
{
  // Save variables.
  mywt = wt;
  mywt->text ("<div type=\"section\">");
}


OsisParagraph::~OsisParagraph ()
// Close a paragraph.
{
  mywt->text ("</div>");
}
