/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "gwrappers.h"
#include "gtkwrappers.h"
#include "scripturechecksdisplayresultsbrowser.h"
#include "directories.h"
#include "htmlbrowser.h"
#include "xmlutils.h"


ScriptureChecksDisplayResultsBrowser::ScriptureChecksDisplayResultsBrowser (CheckDialogType displaywhat, const ustring& filename)
{
  // Set variables.
  displayusfm = (displaywhat == cdtMarkersCount);
  displaycharacters = (displaywhat == cdtCharactersCount);
  // Read the file.
  bool was_error = false;
  gchar *contents;
  gsize  length;
  GError *error;
  error = NULL;
  if (!g_file_get_contents (filename.c_str(), &contents, &length, &error)) {
    cerr << error->message << endl;
    g_error_free (error);
    was_error = true;
  }
  // Add opening lines for html code.
  lines.push_back ("<html>");
  lines.push_back ("<head>");
  lines.push_back ("<meta content=\"text/html; charset=UTF-8\" http-equiv=\"content-type\">");
  if (displayusfm)
    lines.push_back ("<title>Count USFMs</title>");
  if (displaycharacters)
    lines.push_back ("<title>Count Characters</title>");
  lines.push_back ("</head>");
  lines.push_back ("<body>");
  lines.push_back ("<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\">");
  lines.push_back ("<tbody>");
  lines.push_back ("<tr>");
  if (displayusfm)
    lines.push_back ("<td><b>Marker</b></td>");
  if (displaycharacters) {
    lines.push_back ("<td><b>Character</b></td>");
    lines.push_back ("<td><b>Decimal</b></td>");
    lines.push_back ("<td><b>Hexadecimal</b></td>");
  }
  lines.push_back ("<td style=\"text-align: right;\" valign=\"undefined\"><b>Count</b></td>");
  lines.push_back ("</tr>");
  // If length is (about) zero, don't parse it, because it does not have data,
  // and parsing an empty file gives a segmentation error.
  if (length < 10)
    was_error = true;
  // Set up parser.
  GMarkupParseContext *context = NULL;
  GMarkupParser parser = {
    start_element_handler,
    end_element_handler,
    text_handler,
    passthrough_handler,
    error_handler
  };
  // Parse xml file.
  if (!was_error) {
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), gpointer (this), NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error) {
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error)
    g_markup_parse_context_free (context);
  current_value.clear();
  // Display the results in the browser.
  lines.push_back ("</tbody>");
  lines.push_back ("</table>");
  lines.push_back ("<br>");
  lines.push_back ("</body>");
  lines.push_back ("</html>");
  ustring outputfilename;
  if (displayusfm)
    outputfilename = gw_build_filename (directories_get_temp (), "usfmcount.html");
  if (displaycharacters)
    outputfilename = gw_build_filename (directories_get_temp (), "charactercount.html");
  write_lines (outputfilename, lines);
  htmlbrowser (outputfilename);
}


ScriptureChecksDisplayResultsBrowser::~ScriptureChecksDisplayResultsBrowser ()
{
}


void ScriptureChecksDisplayResultsBrowser::start_element_handler 
                                  (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error)
{
  ((ScriptureChecksDisplayResultsBrowser *) user_data)->start_element_handler (element_name);
}


void ScriptureChecksDisplayResultsBrowser::end_element_handler
                                (GMarkupParseContext *context,
                                 const gchar         *element_name,
                                 gpointer             user_data,
                                 GError             **error)
{
  ((ScriptureChecksDisplayResultsBrowser *) user_data)->end_element_handler (element_name);
}


void ScriptureChecksDisplayResultsBrowser::text_handler
                         (GMarkupParseContext *context,
                          const gchar         *text,
                          gsize                text_len,
                          gpointer             user_data,
                          GError             **error)
{
  ((ScriptureChecksDisplayResultsBrowser *) user_data)->text_handler (text);
}



void ScriptureChecksDisplayResultsBrowser::passthrough_handler
                                   (GMarkupParseContext *context,
                                    const gchar         *passthrough_text,
                                    gsize                text_len,
                                    gpointer             user_data,
                                    GError             **error)
{
}


void ScriptureChecksDisplayResultsBrowser::error_handler
                                   (GMarkupParseContext *context,
                                    GError              *error,
                                    gpointer             user_data)
{
  cerr << error->message << endl;
}


void ScriptureChecksDisplayResultsBrowser::start_element_handler (const gchar *element_name)
// When we encounter a new element that starts data, this handler deals with that.
{
  ustring element (element_name);
  if (element_name == "message") {
    usfm.clear();
    count.clear();
    character.clear();
    decimal_entity.clear();
    hexadecimal_entity.clear();
  }
  current_value.clear();
}


void ScriptureChecksDisplayResultsBrowser::end_element_handler (const gchar *element_name)
/*
When we encounter an element that ends data, this handler deals with that.
*/
{
  ustring element (element_name);
  current_value = trim (current_value);
  if (element == "usfm") {
    usfm = current_value;
  } else if (element == "count") {
    count = current_value;
  } else if (element == "character") {
    character = current_value;
  } else if (element == "decimal-entity-reference") {
    decimal_entity = current_value;
  } else if (element == "hexadecimal-entity-reference") {
    hexadecimal_entity = current_value;
  } else if (element == "message") {
    lines.push_back ("<tr>");
    if (displayusfm)
      lines.push_back ("<td>" + usfm + "</td>");
    if (displaycharacters) {
      lines.push_back ("<td>" + character + "</td>");
      // Change certain characters to xml entities.
      vector <size_t> dummy;
      decimal_entity.insert (0, "&#");
      decimal_entity.append (";");
      xml_handle_entities (decimal_entity, dummy);
      lines.push_back ("<td>" + decimal_entity + "</td>");
      lines.push_back ("<td>" + hexadecimal_entity + "</td>");
    }
    lines.push_back ("<td style=\"text-align: right;\" valign=\"undefined\">" + count + "</td>");
    lines.push_back ("</tr>");
  }
  current_value.clear();
}


void ScriptureChecksDisplayResultsBrowser::text_handler (const gchar *text)
/*
When we encounter the text inside an element, this handler deals with that.
*/
{
  current_value.append (text);
}
