/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "synchronize.h"
#include "directories.h"
#include "gwrappers.h"
#include "utilities.h"
#include "projectutils.h"
#include "shell.h"


SynchronizeProjects::SynchronizeProjects (int dummy)
// Synchronizes the data of all projects with external USFM files.
{
  // Initialize variables.
  errors = false;

  // Get all projects.
  vector <ustring> projects = projects_get_all ();
  
  // Sync them all.
  for (unsigned int i = 0; i < projects.size(); i++) {
    SynchronizeProject sp (projects[i]);
    // Store errors and messages.
    if (sp.errors)
      errors = true;
  }
}


SynchronizeProject::SynchronizeProject (const ustring& project)
// Synchronizes the data of a project with external USFM files.
{
  // Initialize variables.
  errors = false;
  
  // If no project open, do nothing.
  if (project.empty())
    return;
  
  // Get all linked books, if there are any, and their parameters.
  ProjectLinksGet plg (project);

  // Go through all books to synchronize them.  
  for (unsigned int i = 0; i < plg.books.size(); i++) {
    SynchronizeBook sb (project, plg.books[i]);
    for (unsigned int i2 = 0; i2 < sb.messages.size(); i2++) {
      messages.push_back (sb.messages[i2]);
    }
    // Store result of the synchronize operation.
    if (sb.book_updated)
      books_updated.insert (plg.books[i]);
    if (sb.errors)
      errors = true;
  }
}


SynchronizeBook::SynchronizeBook (const ustring& project, const ustring& book)
{
  // Initialize variables.
  errors = false;
  book_updated = false;
  result = sbeOkay;

  // Get all the books that are linked, and their parameters.
  ProjectLinksGet plg (project);

  // Go through all books and synchronize the one we want.
  for (unsigned int i = 0; i < plg.books.size(); i++) {
    if (book != plg.books[i])
      continue;
    ustring message;
    message = "Project " + project + ", book " + book + ", linked to " + plg.filenames[i];
    gw_message (message);
    messages.push_back (message);
    // Is book offline?
    if (!g_file_test (plg.filenames[i].c_str(), G_FILE_TEST_IS_REGULAR)) {
      message = "- offline";
      gw_message (message);
      messages.push_back (message);
      result = sbeOffline;
      continue;
    }
    bool bibledit_updated = plg.updateds[i];
    bool file_updated = false;
    {
      unsigned int actual_time = file_get_modification_time (plg.filenames[i]);
      if (actual_time != plg.filetimes[i]) file_updated = true;
      unsigned int actual_size = file_get_size (plg.filenames[i]);
      if (actual_size != plg.filesizes[i]) file_updated = true;
    }      
    if (bibledit_updated) {
      if (file_updated) {
        message = "- cannot synchronize them because both contain changes";
        gw_message (message);
        messages.push_back (message);
        errors = true;
        result = sbeClash;
      } else {
        ustring command = "bibledit --export-book";
        command.append (" --project" + shell_quote_space (project));
        command.append (" --book" + shell_quote_space (book));
        command.append (" --file" + shell_quote_space (plg.filenames[i]));
        if (system (command.c_str()) == 0) {
          project_link_book (project, book, plg.filenames[i]);
          message = "- exported to file";
        } else {
          errors = true;
          message = "- cannot export to file";
        }
        result = sbeExport;
        gw_message (message);
        messages.push_back (message);
      }
    } else {
      if (file_updated) {
        ustring command = "bibledit --import-book";
        command.append (" --project" + shell_quote_space (project));
        command.append (" --file" + shell_quote_space (plg.filenames[i]));
        command.append (" --book" + shell_quote_space (book));
        if (system (command.c_str()) == 0) {
          book_updated = true;
          project_link_book (project, book, plg.filenames[i]);
          message = "- imported from file";
        } else {
          errors = true;
          message = "- cannot import from file";
        }
        result = sbeExport;
        gw_message (message);
        messages.push_back (message);
      } else {
        message = "- in sync";
        gw_message (message);
        messages.push_back (message);
      }
    }
  }
}
