/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include <exception>
#include <stdexcept>
#include <stdio.h>
#include <unistd.h>
#include <iostream>
#include <cstdlib>
#include <string>
#include <fstream>
#include <sstream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <config.h>
#include <vector>
#include <set>


using namespace std;


/*
This utilitiy helps in installing Bibledit on the OLPC.

It gathers all libraries that Bibledit, and its dependencies, use, and 
lists them out.

It will create a gzipped tarball that contains all libraries Bibledit needs.
Also Bibledit itself will be put in it.

If this tarball is unpacked in the OLPC, then Bibledit will work there.
*/


void read_dependencies (const string& program, set<string>& all_dependencies, vector<string>& new_dependencies)
{
  // Execute the command "ldd program" and collect its output.
  FILE *stream;
  string command = "ldd " + program;
  stream = popen (command.c_str (), "r");
  char buf[1024];
  while (fgets (buf, sizeof (buf), stream)) {
    string dependency = buf;
    // Remove newline at the end.
    dependency.erase (dependency.length() - 1, 1); 
    // Find path to library.
    size_t position = dependency.find ("/");
    if (position != string::npos) {
      dependency.erase (0, position);
      position = dependency.find (" ");
      dependency.erase (position, 100);
      // If libraries not already found, insert it and add it to the list of the ones still to check.
      if (all_dependencies.find (dependency) == all_dependencies.end()) {
        all_dependencies.insert (dependency);
        new_dependencies.push_back (dependency);
      }
    }
  }
  pclose (stream);
}


string unix_which (const string program)
// Return full path of program, e.g. "bibledit" would normally return "/usr/bin/bibledit".
{
  string path;
  FILE *stream;
  string command = "which " + program;
  stream = popen (command.c_str (), "r");
  char buf[1024];
  if (fgets (buf, sizeof (buf), stream)) {
    path = buf;
    path.erase (path.length() - 1, 1); 
    size_t position = path.find (" ");
    if (position != string::npos) {
      path.erase (0, position);
    }
  }
  pclose (stream);
  return path;
}


string read_symlink (const string& path)
{
  string links_to;
  FILE *stream;
  string command = "ls -l " + path;
  stream = popen (command.c_str (), "r");
  char buf[1024];
  if (fgets (buf, sizeof (buf), stream)) {
    string s = buf;
    // Remove newline at the end.
    s.erase (s.length() - 1, 1); 
    if (s.substr (0, 1) == "l") {
      size_t position;
      position = s.find ("->");
      if (position != string::npos) {
        s.erase (0, position + 3);
        position = path.find_last_of ("/");
        string directory = path.substr (0, ++position);
        links_to = directory + s;        
      }
    }
  }
  pclose (stream);
  return links_to;
}


int main(int argc, char *argv[])
{
  // Usage info.
  if (argc == 1) {
    cout << "Bibledit for OLPC Librarian " << PACKAGE_VERSION << endl;
    cout << "Produces a tarball with all libraries Bibledit depends on." << endl;
    cout << "Usage:" << endl;
    cout << "bibledit-olpc-libraries tarball" << endl;
    cout << "tarball is the name of the gzipped tarball that will be created, with all" << endl;
    cout << "libraries that need to be transferred to the OLPC image" << endl;
    return 0;
  }
  // Storage.
  vector<string> programs_libraries_to_process;
  set<string> all_dependencies;
  // Initially load all programs Bibledit uses in the list of dependencies to process.
  programs_libraries_to_process.push_back (unix_which ("bibledit"));
  programs_libraries_to_process.push_back (unix_which ("which"));
  programs_libraries_to_process.push_back (unix_which ("grep"));
  programs_libraries_to_process.push_back (unix_which ("chmod"));
  programs_libraries_to_process.push_back (unix_which ("test"));
  programs_libraries_to_process.push_back (unix_which ("tee"));
  programs_libraries_to_process.push_back (unix_which ("date"));
  programs_libraries_to_process.push_back (unix_which ("tail"));
  programs_libraries_to_process.push_back (unix_which ("gzip"));
  programs_libraries_to_process.push_back (unix_which ("gunzip"));
  programs_libraries_to_process.push_back (unix_which ("iconv"));
  programs_libraries_to_process.push_back (unix_which ("strings"));
  programs_libraries_to_process.push_back (unix_which ("killall"));
  programs_libraries_to_process.push_back (unix_which ("mkfifo"));
  programs_libraries_to_process.push_back (unix_which ("cat"));
  programs_libraries_to_process.push_back (unix_which ("sort"));
  programs_libraries_to_process.push_back (unix_which ("head"));
  for (unsigned int i = 0; i < programs_libraries_to_process.size(); i++) {
    all_dependencies.insert (programs_libraries_to_process[i]);  
  }
  // Go through all programs and libraries to check, and handle all their dependencies.
  unsigned int process_pointer = 0;
  do {
    read_dependencies (programs_libraries_to_process[process_pointer], all_dependencies, programs_libraries_to_process);
    process_pointer++;
  } while (process_pointer < programs_libraries_to_process.size());
  cout << "Bibledit depends on " << all_dependencies.size() << " programs and libraries" << endl;  
  // List the dependencies.
  vector <string> dependencies (all_dependencies.begin(), all_dependencies.end());
  for (unsigned int i = 0; i < dependencies.size(); i++) {
    cout << dependencies[i] << endl; 
  }
  // Create the list of files to include in the tarball.  
  vector <string> tarfiles;
  tarfiles.push_back ("/usr/share/bibledit/*");
  tarfiles.push_back ("/usr/bin/bibledit*");
  tarfiles.push_back ("/usr/lib/libsqlite*");
  // Create tarball and gzip it.
  string tarball = argv[1];
  tarball.append (".tar");
  unlink (tarball.c_str());
  string command = "tar -cP";
  for (unsigned int i = 0; i < tarfiles.size(); i++) {
    command.append (" " + tarfiles[i]);
  }
  command.append (" > " + tarball);
  system (command.c_str());
  string gzipped_tarball = tarball + ".gz";
  unlink (gzipped_tarball.c_str());
  command = "gzip " + tarball;
  system (command.c_str());  
  cout << "Created gzipped tarball " << gzipped_tarball << endl;
  return 0;
}
