/*
   bogosort - sorts or doesn't sort files or its standard input

   Copyright (C) 2000 Ulrik Haugen

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/
/* $Id: bogosort.c,v 1.2 2000/11/03 14:20:36 qha Exp $ */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/types.h>
#include "system.h"

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#else
# include "getopt.h"
#endif /* HAVE_GETOPT_H */

#ifdef HAVE_ERROR_H
# include <error.h>
#else
# include "error.h"
#endif /* HAVE_ERROR_H */

#include <stdio.h>

#include "getlines.h"
#include "sortedp.h"

char *xmalloc();
char *xrealloc();
char *xstrdup();

static void seed(void);
static int decode_switches(int argc, char **argv);
static void usage(int status);

/* The name the program was run with, stripped of any leading path. */
char *program_name;

char *oname = "stdout";		/* --output */
FILE *ofile = NULL;
int nosort = 0;			/* --nosort */
int verbose = 0;		/* --verbose */
int readrandom = 1;		/* --norandom */

int
main(int argc, char **argv)
{
    int nextarg;
    int lineno, sortedno;
    int linecount;
    int element;
    char *iname = NULL;		/* infiles */
    FILE *ifile = NULL;
    char *temp = NULL;
    char **lines = NULL;

    /* The name the program was run with */
    /* stripped of any leading path.     */
    program_name = strrchr(argv[0], '/');

    nextarg = decode_switches(argc, argv);

    /* seed rand! */
    seed();

    /* get input filenames and read the files into memory */
    if (nextarg < argc) {
	while (nextarg < argc) {
	    iname = argv[nextarg++];
	    ifile = fopen(iname, "r");
	    if (!ifile)
		error(EXIT_FAILURE, errno,
			_("cannot open %s for reading"),
			iname);

	    if (lines == (char **)NULL)
		lines = getlines(ifile);
	    else
		lines = getmorelines(ifile, lines);

	    if (fclose(ifile))
		error(0, errno,
			_("failed to close %s"),
			iname);
	}
    } else {
	lines = getlines(stdin);
    }

    /* sort the file */
    /* first count lines in lines */
    for (linecount = 0; lines[linecount] != NULL; linecount++) ;

    /*
     * if I wanted to sort the lines it would make a lot more sense to do the
     * check before the first iteration, but I don't.
     */
    /* then randomize the lines until they're sorted */
    do {
	for (sortedno = linecount - 1; sortedno > 0; sortedno--) {
	    element = (int) ((rand() * (float) linecount) / (RAND_MAX + 1.0));
	    temp = lines[element];
	    lines[element] = lines[sortedno];
	    lines[sortedno] = temp;
	}
	if (verbose) fprintf(stderr, ".");
    } while (!nosort && !sortedp(lines));
    if (verbose) fprintf(stderr, "\n");

    /* and print it */
    for (lineno = 0; lineno < linecount; lineno++)
	fputs(lines[lineno], ofile);

    /* clean up */
    for (lineno = 0; lineno < linecount; lineno++)
	free(lines[lineno]);
    free(lines);

    if (ofile != stdout)
	if (fclose(ofile))
	    error(0, errno,
		    _("failed to close %s"),
		    oname);

    if (error_message_count)
	exit (EXIT_FAILURE);
    exit (EXIT_SUCCESS);
}

static void
seed(void)
{
    char *randomname = NULL;
    FILE *randomfile = NULL;
    unsigned int seedbits;
    size_t readints;

#if defined(WANT__DEV_URANDOM)
    randomname = "/dev/urandom";
#elif defined(WANT__DEV_RANDOM)
    randomname = "/dev/random";
#else
    readrandom = 0;
#endif

    if (verbose) fprintf(stderr, _("seeding rand..."));
    if (readrandom && (randomfile = fopen(randomname, "r"))) {
	readints = fread(&seedbits, sizeof (int), 1, randomfile);
	if (readints != 1) {
	    if (feof(randomfile)) {
		if (verbose) putc('\n', stderr);
		error(0, errno,
			_("unexpected end of file reading %s,\n"
			    "seed may not be very random..."),
			randomname);
	    } else if (ferror(randomfile)) {
		if (verbose) putc('\n', stderr);
		error(0, errno,
			_("error reading %s,\n"
			    "seed may not be very random..."),
			randomname);
	    } else {
		if (verbose) putc('\n', stderr);
		error(0, errno,
			_("unexpected error reading %s,\n"
			    "seed may not be very random..."),
			randomname);
	    }
	}
	srand(seedbits);
	if (fclose(randomfile)) {
	    if (verbose) putc('\n', stderr);
	    error(0, errno,
		    _("error closing %s"),
		    randomname);
	}
    } else {
	if (readrandom) {
	    if (verbose) putc('\n', stderr);
	    error(0, errno,
		    _("could not open %s for reading,\n"
			"seed will not be very random..."),
		    randomname);
#if !defined(WANT__DEV_URANDOM) && !defined(WANT__DEV_RANDOM)
	} else if (verbose) {
	    putc('\n', stderr);
	    error(0, errno,
		    _("was configured without support for /dev/(u)random,\n"
			"seed will not be very random..."));
#endif
	}
	srand((unsigned int)time(NULL));
    }
    if (verbose) fprintf(stderr, _(" done.\n"));
}

/* Option flags and variables */
static struct option const long_options[] =
{
    {"nosort", no_argument, 0, 'n'},
    {"norandom", no_argument, 0, 'r'},
    {"output", required_argument, 0, 'o'},
    {"help", no_argument, 0, 'h'},
    {"verbose", no_argument, 0, 'v'},
    {"version", no_argument, 0, 'V'},
    {NULL, 0, NULL, 0}
};

/* Set all the option flags according to the switches specified.
   Return the index of the first non-option argument.  */
static int
decode_switches(int argc, char **argv)
{
    int c;

    ofile = stdout;

    while ((c = getopt_long(argc, argv,
		    "n"		/* nosort */
		    "r"		/* norandom */
		    "o:"	/* output */
		    "h"		/* help */
		    "v"		/* verbose */
		    "V",	/* version */
		    long_options, (int *) 0)) != EOF) {
	switch (c) {
	  case 'n':		/* --nosort */
	    nosort = 1;
	    break;
	  case 'r':		/* --norandom */
	    readrandom = 0;
	    break;
	  case 'o':		/* --output */
	    if (strcmp(optarg, "-") == 0)
		break;
	    oname = xstrdup(optarg);
	    ofile = fopen(oname, "w");
	    if (!ofile)
		error(EXIT_FAILURE, errno,
			_("cannot open %s for writing"),
			oname);
	    break;
	  case 'v':		/* --verbose */
	    verbose = 1;
	    break;
	  case 'V':		/* --version */
	    printf("bogosort %s\n", VERSION);
#if defined(WANT__DEV_URANDOM)
	    printf(_("	configured with support for /dev/urandom\n"));
#elif defined(WANT__DEV_RANDOM)
	    printf(_("	configured with support for /dev/random\n"));
#else
	    printf(_("	configured without support for /dev/(u)random\n"));
#endif
	    printf(_("\n"
"Written by Ulrik Haugen\n\n"
"Copyright (C) 2000 Ulrik Haugen\n"
"This is free software; see the source for copying conditions.  There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n")
		  );
	    exit(EXIT_SUCCESS);

	  case 'h':		/* --help */
	    usage(EXIT_SUCCESS);

	  default:
	    usage(EXIT_FAILURE);
	}
    }

    return optind;
}

static void
usage(int status)
{
    printf(_("%s - sorts or doesn't sort files or its standard input\n"),
	    program_name);
    printf(_("Usage: %s [OPTION]... [FILE]...\n"), program_name);
    printf(_("Options:"
"\n		-n, --nosort               do not check if the output is sorted"
"\n		                           this is useful if you do not agree"
"\n		                           with the built in defenition of"
"\n		                           sorted and want to perform this"
"\n		                           check yourself"
"\n		-r, --norandom             if configured with support for"
"\n		                           /dev/(u)random seed rand with time"
"\n		                           anyway, ignored otherwise"
"\n		-o, --output NAME          send output to NAME instead of"
"\n		                           standard output"
"\n		-h, --help                 display this help and exit"
"\n		-v, --verbose              output a '.' to stderr for each"
"\n		                           iteration of the sorting loop"
"\n		-V, --version              output version information and exit"
"\n"));
	exit(status);
}
