/***************************************************************************
                  firebird.cpp  -  Firebird Class Implementation
                             -------------------
    begin                : Mon Aug 18 2003
    copyright            : (C) 2003 by Ken Schenke
    email                : kenschenke at yahoo dot com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include "firebird.h"
#include "browserlist.h"

#include <qfile.h>
#include <qtextstream.h>
#include <qfiledialog.h>

// for access()
#ifdef _WIN32
#include <io.h>
#include <shlobj.h>
#else
#include <unistd.h>
#endif

/***************************************************************************
 *                                                                         *
 *   Firebird::BrowseForBookmarks()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if the user did not select a bookmark file, false otherwise   *
 *   Description:                                                          *
 *      This function is called to present the user with a file dialog box *
 *      allowing them to select the location of their bookmark file.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Firebird::BrowseForBookmarks(const BridgeCfg &cfg, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks.sprintf("%s/Phoenix/Profiles/Default", cfg.appdatadir.c_str());
	bookmarks = QFileDialog::getOpenFileName(
		bookmarks, "Bookmark File (bookmarks.html)",
		parent, "Get Bookmarks File", "Select Your Bookmarks File");

	return (bookmarks == QString::null);
}
#elif defined(Q_WS_X11)
bool Firebird::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = QDir::homeDirPath() + "/.phoenix/default";
	bookmarks = QFileDialog::getOpenFileName(
		bookmarks, "Bookmark File (bookmarks.html)",
		parent, "Get Bookmarks File", "Select Your Bookmarks File");

	return (bookmarks == QString::null);
}
#else
#error "Must define platform-dependent Firebird::BrowseForBookmarks() Function"
#endif

/***************************************************************************
 *                                                                         *
 *   Firebird::classFactory()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      the newly allocated class                                          *
 *   Description:                                                          *
 *      This is a static member function of the Mozilla class.  Its job is *
 *      to allocate an instance of the Mozilla class for the caller.       *
 *                                                                         *
 ***************************************************************************/

BrowserBk *Firebird::classFactory(void)
{
	return new Firebird;
}

/***************************************************************************
 *                                                                         *
 *   Firebird::DetectBrowser()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if an installation of Mozilla was found, false otherwise      *
 *   Description:                                                          *
 *      This function attempts to locate Firebird's bookmark database on   *
 *      system.  It will only work if the user has just one profile.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Firebird::DetectBrowser(const BridgeCfg &cfg, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.phoenix/default
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Phoenix\default

	basePath.sprintf("%s/Phoenix/Profiles/Default", cfg.appdatadir.c_str());

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Firebird obfuscates the
	// profile directory.  It always ends in ".slt", so we'll just search
	// for anything matching this pattern.

	QStringList profileList = base.entryList("*.slt", QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#elif defined(Q_WS_X11)
bool Firebird::DetectBrowser(const BridgeCfg &, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.phoenix/default
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Phoenix\default

	basePath = QDir::homeDirPath() + "/.phoenix/default";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Firebird obfuscates the
	// profile directory.  It always ends in ".slt", so we'll just search
	// for anything matching this pattern.

	QStringList profileList = base.entryList("*.slt", QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#else
#error "Must define platform-dependent Firebird::DetectBrowser() Function"
#endif
