/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * These algorithms are from book:
 * John Viega and Matt Messier - "Secure Programming Cookbook for C and C++"
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "WipeFile.h"
#include "Events.h"
#include "Shared.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <qevent.h>
#include <qapplication.h>

/*------- local constants:
-------------------------------------------------------------------*/
const unsigned char WipeFile::SinglePats[SINGLE_PATS_SIZE] = {
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
    0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
};
const unsigned char WipeFile::TriplePats[TRIPLE_PATS_ROWS][TRIPLE_PATS_COLS] = {
    { 0x92, 0x49, 0x24 },
    { 0x49, 0x24, 0x92 },
    { 0x24, 0x92, 0x49 },
    { 0x6D, 0xB6, 0xDB },
    { 0xB6, 0xDB, 0x6D },
    { 0xDB, 0x6D, 0xB6 }
};
const QString WipeFile::WipeMsg = QT_TR_NOOP( "Wipe file (step #%1/7) : %2" );

//*******************************************************************
// WipeFile                                             CONSTRUCTOR
//*******************************************************************
WipeFile::WipeFile( const QString& in_fpath, QDialog* const in_dialog )
: QObject             ()
, d_fpath             ( in_fpath )
, d_dialog            ( in_dialog )
, d_devrand_fd        ( -1 )
, d_devrand_fd_noblock( -1 )
, d_devurand_fd       ( -1 )
{}
// end of WipeFile

//*******************************************************************
// wipe                                                       PUBLIC
//*******************************************************************
int WipeFile::wipe()
{
    int retval = -1;
	
	if( can_do() ) {
        const int fd = open( d_fpath.ascii(), O_RDWR );
        if( fd != -1 ) {
            retval = fd_wipe( fd );
            close( fd );
	   }
	}
	return retval;
}
// end of wipe


//###################################################################
//#                                                                 #
//#                       P R I V A T E                             #
//#                                                                 #
//###################################################################


//*******************************************************************
// fd_wipe                                                   PRIVATE
//*******************************************************************
int WipeFile::fd_wipe( const int in_fd )
{
    struct stat finfo;

    progress( 0 );
    if( -1 == fstat( in_fd, &finfo )) {
        return -1;
    }
    if(  0 == finfo.st_size ) {
        return  0;
    }

    // czterokrotnie wypelniamy plik liczbami losowymi
    progress( 1 );
    if( !can_do() ) return -1;
    for( int pass = 0; can_do() && ( pass < 4 ); pass++ ) {
        if( -1 == random_pass( in_fd, finfo.st_size )) {
            return -1;
        }
    }

    // jednorkotnie wypelniamy plik szosta liczba
    // (indeks 5, wartosc 0x55) z 'SinglePats'
    progress( 2 );
    if( !can_do() ) return -1;
    memset( d_buffer, SinglePats[5], WIPE_BUFSIZE );
    if( -1 == pattern_pass( in_fd, d_buffer, WIPE_BUFSIZE, finfo.st_size )) {
        return -1;
    }

    // jednokrotnie wypelniamy plik jedenasta liczba
    // (indeks 10, wartosc 0xAA) z 'SinglePats'
    progress( 3 );
    if( !can_do() ) return -1;
    memset( d_buffer, SinglePats[10], WIPE_BUFSIZE );
    if( -1 == pattern_pass( in_fd, d_buffer, WIPE_BUFSIZE, finfo.st_size )) {
        return -1;
    }

    // wypelniamy caly plik trzy razy, uzywajac kolejno
    // trzy trojki z 'TriplePats'
    progress( 4 );
    if( !can_do() ) return -1;
    const int pattern_size = sizeof( TriplePats[0] );
    for( int pass = 0; can_do() && ( pass < 3 ); pass++ ) {
        const unsigned char* const pattern = TriplePats[ pass ];
        const int count = WIPE_BUFSIZE / pattern_size;
        for( int i = 0; can_do() && ( i < count );  i++ ) {
            memcpy( d_buffer + ( i * pattern_size ), pattern, pattern_size );
        }
        if( -1 == pattern_pass( in_fd, d_buffer, pattern_size * count, finfo.st_size )) {
           return -1;
        }
    }

    // Wypelnienie calego pliku po kolei kazda z liczb z 'SinglePats'
    progress( 5 );
    if( !can_do() ) return -1;
    for( unsigned int pass = 0; can_do() && ( pass < SINGLE_PATS_SIZE ); pass++ ) {
        memset( d_buffer, SinglePats[pass], WIPE_BUFSIZE );
        if( -1 == pattern_pass( in_fd, d_buffer, WIPE_BUFSIZE, finfo.st_size )) {
            return -1;
        }
    }
    
    // wypelniamy caly plik uzywajac wszytkich
    // trojek z 'TriplePats'
    progress( 6 );
    if( !can_do() ) return -1;
    for( unsigned int pass = 0; can_do() && ( pass < TRIPLE_PATS_ROWS ); pass++ ) {
        const unsigned char* const pattern = TriplePats[ pass ];
        const int count = WIPE_BUFSIZE / pattern_size;
        for( int i = 0; can_do() && ( i < count ); i++ ) {
            memcpy( d_buffer + ( i * pattern_size ), pattern, pattern_size );
        }
        if( -1 == pattern_pass( in_fd, d_buffer, pattern_size * count, finfo.st_size )) {
            return -1;
        }
    }

    progress( 7 );
    if( !can_do() ) return -1;
    for( int pass = 0; can_do() && ( pass < 4 ); pass++ ) {
        if( -1 == random_pass( in_fd, finfo.st_size )) {
            return -1;
        }
    }

    return can_do() ? 0 : -1;
}
// end of fd_wipe

//*******************************************************************
// pattern_pass                                              PRIVATE
//*******************************************************************
int WipeFile::pattern_pass( int in_fd, unsigned char* in_buffer, size_t in_bufsize, size_t in_filesize )
{
    if( !can_do() ) return -1;
    
    size_t towrite;
    if(( 0 == in_bufsize ) || ( lseek( in_fd, 0, SEEK_SET ) != 0 )) {
        return -1;
    }
    
    while( can_do() && (  in_filesize > 0 )) {
        towrite = ( in_filesize > in_bufsize ) ? in_bufsize : in_filesize;
        if( 0 == write_data( in_fd, in_buffer, towrite )) {
            return -1;
        }
        in_filesize -= towrite;
    }
    fsync( in_fd );
    return 0;
}
// end of pattern_pass

//*******************************************************************
// random_pass                                               PRIVATE
//*******************************************************************
int WipeFile::random_pass( int in_fd, size_t in_nbytes )
{
    if( !can_do() ) return -1;

    size_t towrite;
    unsigned char buffer[WIPE_BUFSIZE];
    size_t wipe_bufsize = static_cast<size_t>( WIPE_BUFSIZE );

    if( lseek( in_fd, 0, SEEK_SET ) != 0 )  return -1;
    while( can_do() && (  in_nbytes > 0 )) {
        towrite = ( in_nbytes > wipe_bufsize ) ? wipe_bufsize : in_nbytes;
        if( -1 == rand( buffer, towrite )) {
            return -1;
        }
        if(  0 == write_data( in_fd, buffer, towrite )) {
            return -1;
        }
        in_nbytes -= towrite;
    }
    fsync( in_fd );
    return 0;
}
// end of random_pass

//*******************************************************************
// write_data                                                PRIVATE
//*******************************************************************
int WipeFile::write_data( const int in_fd, const void* const in_buffer, const size_t in_nbytes )
{
    if( !can_do() ) return 0;

    const char* const buf = static_cast<const char* const>( in_buffer );
    size_t towrite;
    size_t written = 0;
    int result;

    do {
        if( !can_do() ) return 0;
        
        if(( in_nbytes - written ) > SSIZE_MAX ) towrite = SSIZE_MAX;
        else towrite = in_nbytes - written;

        if(( result = write( in_fd, buf + written, towrite )) >= 0 ) {
            written += result;
        }
        else if( errno != EINTR ) {
            return 0;
        }
    } while( written < in_nbytes );

    return 1;
}
// end of write_data

//*******************************************************************
// make_fd_noblocking                                        PRIVATE
//*******************************************************************
int WipeFile::make_fd_noblocking( int in_fd )
{
    int retval = -1;
    int flags  = fcntl( in_fd, F_GETFL );

    if( flags != -1 ) {
        flags |= O_NONBLOCK;
        if( fcntl( in_fd, F_SETFL, flags ) != -1 ) {
            retval = 0;
        }
    }
    return retval;
}
// end of make_fd_noblocking
    
//*******************************************************************
// rand_init                                                 PRIVATE
//*******************************************************************
int WipeFile::rand_init()
{
    d_devrand_fd         = open( "/dev/random" , O_RDONLY );
    d_devrand_fd_noblock = open( "/dev/random" , O_RDONLY );
    d_devurand_fd        = open( "/dev/urandom", O_RDONLY );
        
    if(( -1 == d_devrand_fd ) || ( -1 == d_devrand_fd_noblock )) {
        return -3;
    }
    if( -1 == d_devurand_fd ) {
        return -2;
    }
    return make_fd_noblocking( d_devrand_fd_noblock );
}
// end of rand_init

//*******************************************************************
// rand                                                      PRIVATE
//*******************************************************************
int WipeFile::rand( unsigned char* const out_buffer, size_t in_nbytes )
{
    ssize_t        bytes_read;
    unsigned char* where = out_buffer;
    
    if(( -1 == d_devrand_fd ) || ( -1 == d_devrand_fd_noblock ) || ( -1 == d_devurand_fd )) {
        if( rand_init() != 0 ) {
            return -1;
        }
    }
    
    while( in_nbytes ) {
        if( -1 == ( bytes_read = read( d_devurand_fd, where, in_nbytes ))) {
            if( EINTR == errno ) continue;
            else                 return -1;
        }
        where     += bytes_read;
        in_nbytes -= bytes_read;
    }
    return 0;
}
// end of rand

//*******************************************************************
// progress                                                  PRIVATE
//*******************************************************************
void WipeFile::progress( const int in_progress )
{
    if( can_do() && d_dialog ) {
        BscEvent* event = new BscEvent( WipeFileEvent );
        if( event ) {
            event->m_message = tr(WipeMsg).arg( in_progress ).arg( d_fpath );
            QApplication::postEvent( d_dialog, event );
            Shared::idle();
        }
    }
}
// end of progress

//*******************************************************************
// can_do                                                    PRIVATE
//*******************************************************************
bool WipeFile::can_do()
{
    Shared::idle();
    return ( FALSE == Shared::d_break );
}
// end of can_do
