/**
  \class CFTPClient
  \brief An asynchronous FTP client
  
  This class implements an FTP client, based on RFC 959. Its completely
  asynchronuous, meaning you can create the object, start logging in,
  queue up a 100 commands, file up- and downloads and never have to look
  back until you're done (well, that's the end goal. Currently, 
  connecting && uploading is asynchronous, there's no command queue yet).
  
  It uses a very fine-grained Finite State Machine for the FTP protocol.
  Each state transition is sent through a SIGNAL() to any classes that are
  interested. In addition, there are signals for progress reports and when a
  transfer is completed.

  TODO: 
  * command queueing
  * signal filtering?
  * DIR listing
*/

#include <arpa/inet.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "FTPClient.h"

const char *CFTPClient::StateStr[] = 
{
   "Nop",
   "Connecting",
   "Connected",
   "Login", 
   "Authenticate",
   "Failed",
   "LoggedIn",
   "Ready",
   "SettingType",
   "SendingPort",
   "Storing",
   "Retrieving",
   "WaitData",
   "Transfer",
   "ClosingData",
   "Completed",
   "Closing",
};


CFTPClient::CFTPClient()
{
   CtrlFD = ListenFD = DataFD = LocalFileFD = -1;
   pControlPipe = NULL;
   pListenPipe = NULL;
   pDataPipe = NULL;

   inputbuffer = new char[1024];
   linebuffer = new char[1024];
   responsebuffer = new char[1024];
   LineLen = 0;
   RespLen = 0;
   outputbuffer = new char[1024];
   TBufLen = 32768;
   transferbuffer = new uchar[TBufLen];

   CurrentState = Nop;
}

CFTPClient::~CFTPClient()
{
   // SendQuit();
   CloseAllFD();
   delete inputbuffer;
   delete linebuffer;
   delete responsebuffer;
   delete outputbuffer;
   delete transferbuffer;
}

// private

void CFTPClient::InitBuffers()
{
   LastChar = '\0';
   LineLen = RespLen = 0;
   Response = 0;
   MultiLine = FALSE;
   
   TBufTail = TBufHead = TBufUsed = 0;
}

/**
  \brief Closes all TCP connections immediately
*/

void CFTPClient::CloseAllFD()
{
printf("CFTPClient::CloseAllFD()\n");
   CloseData();
   CloseListen();
   CloseFile();
   CloseControl();
   CurrentState = Nop;
}

/**
  \brief Try to connect to the server's FTP control port.
*/
int CFTPClient::SetupControl()
{
   CtrlFD = socket(PF_INET, SOCK_STREAM, 0);
   if (CtrlFD < 0)
     return -errno;

   // True async operation requires a NO_DELAY on the socket, and some additional code
   // fcntl(CtrlFD, NO_DELAY);
   // CurrentState = Connecting

   if (::connect(CtrlFD, (struct sockaddr *)&ServerAddress, sizeof(ServerAddress)) < 0)
     return -errno;

   pControlPipe = new QSocketNotifier(CtrlFD, QSocketNotifier::Read);
   connect(pControlPipe, SIGNAL(activated(int)), this, SLOT(ControlRead(int)));
   return 0;
}


/**
  \brief Shut down connection to FTP server control port. 
*/
void CFTPClient::CloseControl()
{
   delete pControlPipe;
   pControlPipe = NULL;
   if (CtrlFD >= 0) ::close(CtrlFD);
   CtrlFD = -1;
}

/** 
  \brief Sets up our incoming port where we will listen 
*/
int CFTPClient::SetupListen()
{
   if (ListenFD < 0) {
     ListenFD = socket(PF_INET, SOCK_STREAM, 0);
     if (ListenFD < 0)
       return -errno;

     if (listen(ListenFD, 5) < 0) // Will get assigned a port number
       return -errno;

     pListenPipe = new QSocketNotifier(ListenFD, QSocketNotifier::Read);
     connect(pListenPipe, SIGNAL(activated(int)), this, SLOT(ListenRead(int)));
printf("Set up fd %d for listening.\n", ListenFD);
   }
   return 0;
}

/**
  \brief Close our incoming port
*/
void CFTPClient::CloseListen()
{
printf("Shutting down listen fd %d.\n", ListenFD);
   delete pListenPipe;
   pListenPipe = NULL;
   if (ListenFD >= 0) ::close(ListenFD);
   ListenFD = -1;
}

int CFTPClient::SetupFile(const char *filename)
{
   if (LocalFileFD >= 0) 
     printf("Error: local file fd still open (%d)!\n", LocalFileFD);
   LocalFileFD = ::open(filename, O_RDONLY);
   if (LocalFileFD < 0)
     return -errno;
   return 0;
}

/**
  \brief Close fd of file on disk
*/  
void CFTPClient::CloseFile()
{
printf("Shutting down local file fd.\n");   
   if (LocalFileFD >= 0) ::close(LocalFileFD);
   LocalFileFD = -1;
}
/**
  \brief Close data port
*/  
void CFTPClient::CloseData()
{
printf("CFTPClient::CloseData(): %d\n", DataFD);
   delete pDataPipe;
   pDataPipe = NULL;
   if (DataFD >= 0) ::close(DataFD);
   DataFD = -1;
}

/**
  \brief Set new state en emit signal
*/  
void CFTPClient::SetState(int new_op)
{
   CurrentState = new_op;
   emit StateChange(new_op, Response);
}

/**
  \brief Takes a line from the control port input buffer and unfolds it.
*/  
void CFTPClient::InterpretLine()
{
   int l, resp;
   
//printf("InterpretLine(): %s\n", linebuffer);
   l = strlen(linebuffer);
   if (l == 0) {
     printf("InterpretLine(): short line.\n");
     return;
   }
   
   // See if we have 3 digits at the beginning of the line
   resp = 0;
   if (l >= 3) {
     if (isdigit(linebuffer[0]) && isdigit(linebuffer[1]) && isdigit(linebuffer[2]))
       resp = 100 * (linebuffer[0] - '0') + 10 * (linebuffer[1] - '0') + (linebuffer[2] - '0');
   }
   if (resp)
     Response = resp;

   if (MultiLine) {
     if (resp) {
printf("Multiline: found response %d.\n", Response);
       MultiLine = FALSE;
     }
     else {
printf("Multiline: appending\n");
       if (RespLen < 1023)
         responsebuffer[RespLen++] = '\n';
       if (RespLen + l > 1023) // prevent overflows
         l = 1023 - RespLen;
       if (l) 
         strncpy(responsebuffer + RespLen, linebuffer, l);
       RespLen += l;
     } // ..if resp
   }
   else { // Not multiline
     if (resp == 0) 
       printf("InterpretLine(): line without code and not in multiline mode.\n");
     else {
// printf("Response code %d\n", resp);
       if (l > 3) {
         if (linebuffer[3] == '-')
           MultiLine = TRUE;
         strcpy(responsebuffer, linebuffer + 4); // skip code & space
         RespLen = l - 4;
       }
     }
   }
   if (!MultiLine) {
     responsebuffer[RespLen] = '\0'; // terminate properly
     InterpretResponse();
     RespLen = 0;
   }
}


/**
  \brief Takes a complete, unfolded line from the control port and interprets the response
*/  
void CFTPClient::InterpretResponse()
{
printf("InterpretResponse(%d, \"%s\")\n", Response, responsebuffer);
   switch(Response) {
     case 125: // port is open
       //SetState(Transfer);
       //break;

     case 150: // Transfer is about to start
       /* There is a continuous race condition between the 150 response
          and the actual opening of the data connection. It could be
          the 150 arrives after the datapipe has been opened (happens
          often on a local LAN). 
        */
       if (pDataPipe) {
         printf("Enabling data pipe.\n");
         pDataPipe->setEnabled(TRUE);
       }
       else {
         SetState(WaitData);
         printf("Info: Got 150 but no data pipe yet.\n");
       }
       break;
     
     case 200:
       switch(CurrentState) {
         case SettingType:
           // host accepted type
           SetState(Ready);
           break;
           
         case SendingPort:
           // Great! The other side accepted our port. Now see if we want
           // to transfer anything.
           if (LocalFileFD >= 0) {
             if (Direction)
               StartSending();
             else
               StartReceiving();
           }
           break;
       }    // ..switch CurrentState
       break;
   
     case 220: // Ready for new user
       if (CurrentState == Connected) {
         SetState(Login);
         SendUser();
       }
       break;
     
     case 226: // Transfer completed
       if (CurrentState == ClosingData) {
         CloseListen();
         SetState(Completed);
         SetState(Ready);
       }
       else
         printf("Warning: 226 without transfer?\n");
       break;

     case 230: // Login succeeded
       if (CurrentState == Authenticate) {
         SetState(LoggedIn);
         SetState(Ready);
         // SendTimeOut ??
       }
       break;
       
     case 331: // Password required
       if (CurrentState == Login) {
         SetState(Authenticate);
         SendPass();
       }
       break;
   
     case 530: // Login failed
       SetState(Failed);
       //SetState(Connected); // hmm, retries?
       break;
       
     default:
       printf("Unknown code\n");
       break;
   }

}


/**
  \brief Send content of outputbuffer
*/
void CFTPClient::Send()
{
  int l;
  
printf("CFTPClient::Send(): %s", outputbuffer);  
  l = strlen(outputbuffer);
  if (l > 0 && CtrlFD >= 0)
    write(CtrlFD, outputbuffer, l); // Should be non-blocking as well
}


void CFTPClient::SendUser()
{
   sprintf(outputbuffer, "USER %s\r\n", (const char *)UserName);
   Send();
}

void CFTPClient::SendPass()
{
   sprintf(outputbuffer, "PASS %s\r\n", (const char *)Password);
   Send();
}


void CFTPClient::SendPort(struct sockaddr_in &addr, int port)
{
   unsigned int l;
   
   l = (unsigned int)ntohl(addr.sin_addr.s_addr);
   sprintf(outputbuffer, "PORT %d,%d,%d,%d,%d,%d\r\n",
      (l >> 24) & 0xff, (l >> 16) & 0xff, (l >> 8) & 0xff, l & 0xff,
      (port >> 8) & 0xff, port & 0xff);
   Send();
}

void CFTPClient::SendStore(const char *filename)
{
   sprintf(outputbuffer, "STOR %s\r\n", filename);
   Send();
}



void CFTPClient::StartSending()
{
   /* We are ready to go.. we have our local filedescriptor, the remote
    * side has our port, now send a STOR and see what happens 
    */
   SetState(Storing);
   SendStore((const char *)RemoteFileName);
}

void CFTPClient::StartReceiving()
{

}

// private slots

void CFTPClient::ControlRead(int fd)
{
   int err, i;
   char c;

   /* Read data from control socket. Because FTP responses can consist of 
      multiple lines we need a 3-stage process: 
        inputbuffer contains the raw input data from the socket;
        linebuffer contains a whole line (possibly truncated);
        responsebuffer has the complete command.

      Linebuffer is delimited on the \r\n sequence; the \r\n sequence is
      removed and replaced by a '\0' to make it a proper string.  To prevent
      bad servers from trying to overflow our stack/memory, linebuffer will
      never contain more than 1023 characters; if we receive more than this
      before a \r\n sequence, the rest is ignored.
    */
   err = read(fd, inputbuffer, 1023);
   if (err < 0) {
     // Emit error?
     printf("Error reading from control pipe: %d\n", err);
     return;
   }
   if (err == 0) { // EOF
     printf("Control pipe closed by remote end.\n");
     SetState(Closing);
     CloseAllFD();
     return;
   }
   
   inputbuffer[err] = '\0';
printf("Read from control socket: %s", inputbuffer);

   for (i = 0; i < err; i++) {
      c = inputbuffer[i];
      if (c == '\n') { // End-Of-Line
        // Hmm, We should check if lastchar = \r, but what the heck
        linebuffer[LineLen] = '\0'; // Terminate; LineLen is never larger than 1023
        InterpretLine();
        LineLen = 0;
      }
      else if (isprint(c)) { // Copy character
        if (LineLen < 1023)
          linebuffer[LineLen++] = c;
      }
      // Anything else is ignored
      
      LastChar = c;
   }

}

/**
  \brief Activated when our data port is contacted
*/  
void CFTPClient::ListenRead(int fd)
{
   int i;
   struct sockaddr_in DataAddr;
   socklen_t DLen;

printf("CFTPClient::ListenRead()\n");
   // We got a new connection, hurray
   if (fd != ListenFD)
     abort();

   // Now accept connection
   DLen = sizeof(DataAddr);
   i = accept(ListenFD, (struct sockaddr *)&DataAddr, &DLen);
   if (i < 0) {
     printf("Error: failed to accept connection: %d\n", -errno);
     //SetState(?);
     return;
   }
   
   if (ntohs(DataAddr.sin_port) != 20)
     printf("Warning: got an incoming connection from an unprivilidged port (%d)\n", ntohs(DataAddr.sin_port));

   if (DataFD >= 0) {
     // Hey? That's odd...
     printf("Warning: got a second connection on data port from %s, port %d\n", inet_ntoa(DataAddr.sin_addr), DataAddr.sin_port);
     ::close(i); // drop it on the floor
     return;
   }
   else 
     DataFD = i;
     
   // The listen port has fulfilled its puprose, close it now
   CloseListen();
 
printf("Got connected on data port from %s:%d  fd = %d\n", inet_ntoa(DataAddr.sin_addr), ntohs(DataAddr.sin_port), DataFD);
   TotalTransfered = 0;
   if (Direction) {
     pDataPipe = new QSocketNotifier(DataFD, QSocketNotifier::Write);
     connect(pDataPipe, SIGNAL(activated(int)), this, SLOT(DataWrite(int)));
   }
   else {
     pDataPipe = new QSocketNotifier(DataFD, QSocketNotifier::Read);
     connect(pDataPipe, SIGNAL(activated(int)), this, SLOT(DataRad(int)));
   }
   if (CurrentState != WaitData)
     pDataPipe->setEnabled(FALSE); // wait until 125/150 response
   SetState(Transfer);
}


void CFTPClient::DataRead(int fd)
{

}

/// Activated when the output socket has space
void CFTPClient::DataWrite(int fd)
{
   int sp, wr, rd;

   sp = TBufLen - TBufUsed; // space in buffer;

printf("CFTPClient::DataWrite(Head = %d, Tail = %d, Used = %d, sp = %d, fd = %d)\n", TBufHead, TBufTail, TBufUsed, sp, LocalFileFD);

   if (TBufUsed == 0 && LocalFileFD < 0) { // done!   
     printf("DataWrite done. Transfered %d bytes\n", TotalTransfered);
     SetState(ClosingData);
     CloseData();
     return;
   }

   rd = 0;
   if (sp > 0 && LocalFileFD >= 0) { // there is space and we can read from file
printf("Trying to read %d bytes from file; ", sp);
     if (TBufHead + sp > TBufLen) { // split across buffer end
       rd = read(LocalFileFD, transferbuffer + TBufHead, TBufLen - TBufHead);
       if (rd > 0) {
         sp = read(LocalFileFD, transferbuffer, sp - rd);
         if (sp > 0)
           rd += sp;
       }
     }
     else // single piece
       rd = read(LocalFileFD, transferbuffer + TBufHead, sp);
printf("got %d bytes.\n", rd);
   }

   if (rd <= 0) {
     if (rd < 0)
       printf("Error reading local file: %d\n", -errno);
     CloseFile();
   }
   else {
     TBufHead = (TBufHead + rd) % TBufLen;

     TBufUsed += rd; // Adjust
     sp -= rd;
   }

   wr = 0;   
   if (TBufUsed) {
printf("Trying to write %d bytes to datapipe; ", TBufUsed);
     // try to send data to pipe
     if (TBufTail + TBufUsed > TBufLen) { // split across buffer end
       wr = write(DataFD, transferbuffer + TBufTail, TBufLen - TBufTail);
       if (wr > 0) {
         sp = write(DataFD, transferbuffer, TBufUsed - wr);
         if (sp > 0)
           wr += sp;
       }
     } 
     else // single piece
       wr = write(DataFD, transferbuffer + TBufTail, TBufUsed);
printf("put %d bytes.\n", wr);
   }

   if (wr <= 0) {
     if (wr < 0)
       printf("Erorr writing data pipe: %d\n", -errno);
     
   
   }
   else {
     TBufTail = (TBufTail + wr) % TBufLen;
     TotalTransfered += wr;

     TBufUsed -= wr;
     sp += wr;
   }
}


// public

int CFTPClient::GetState()
{
   return CurrentState;
}

int CFTPClient::Open(const char *user, const char *pass, const char *server, int port = 21)
{
   int err;
   struct hostent *pServerName;

   if (CtrlFD >= 0) {
     // SendAbort();
     CloseAllFD();
   }
   InitBuffers();

   /* Step 1: name resolv of server */
   // Unfortunately, this is a blocking system call. res_mkquery() is not funny.
   pServerName = gethostbyname(server);
   if (pServerName == NULL) {
     switch(h_errno) { // This is tweaking the error codes somewhat :)
       case TRY_AGAIN:      return -EAGAIN; break;
       case HOST_NOT_FOUND: return -ENXIO;  break;
       case NO_ADDRESS:     return -EFAULT; break;
       case NO_RECOVERY:    return -EINVAL; break;
       default:             return -EBUSY;  break;
     }
   }

   if (pServerName->h_length != sizeof(struct in_addr))
     abort(); // Something fishy is going on!
   ServerAddress.sin_family = AF_INET;
   ServerAddress.sin_port = htons(port);
   memcpy(&ServerAddress.sin_addr, pServerName->h_addr_list[0], pServerName->h_length);
printf("Resolved server address: %s (%s)\n", pServerName->h_name, inet_ntoa(ServerAddress.sin_addr));

   err = SetupControl();
   if (err < 0)
     return err;

   UserName = user;
   Password = pass;

   SetState(Connected);

   return 0;
}


/** 
  \brief Upload file to remote site
*/
int CFTPClient::Upload(const char *local_file, const char *remote_file)
{
   struct sockaddr_in DAddr;
   socklen_t DLen;
   int ret;

   if (CurrentState != Ready)
     return -EBUSY;
   if (LocalFileFD >= 0)
     return -EINVAL;

   ret = SetupFile(local_file);
   if (ret < 0)
     return ret;
printf("Local file fd = %d\n", LocalFileFD);     

   if (remote_file == NULL)
     RemoteFileName = local_file;
   else
     RemoteFileName = remote_file;

   if (SetupListen() < 0)
     goto err1;
   
/*   ... SetupData(); */

   // Let's find out our IP address; use CtrlFD, not ListenFD, since ListenFD
   // isn't connected (and never will be)
   DLen = sizeof(MyAddr);
   if (getsockname(CtrlFD, (struct sockaddr *)&MyAddr, &DLen) < 0) {
     ret = -errno;
     goto err2;
   }

   // Find the port we just bounded to. This does use ListenFD (confused yet?)
   DLen = sizeof(DAddr);
   if (getsockname(ListenFD, (struct sockaddr *)&DAddr, &DLen) < 0) {
     ret = -errno;
     goto err2;
   }

printf("Upload(): getsockname: addr = %s, port = %d\n", inet_ntoa(MyAddr.sin_addr), ntohs(DAddr.sin_port));

   Direction = TRUE;
   SetState(SendingPort);
   SendPort(MyAddr, ntohs(DAddr.sin_port));

   // Okay, the async takes over from here...
   return 0;

err2:
   CloseListen();
err1:
   CloseFile();
   return ret;
}


void CFTPClient::SetTypeAscii()
{
   SetState(SettingType);
   strcpy(outputbuffer, "TYPE A\r\n");
   Send();
}

void CFTPClient::SetTypeBinary()
{
   SetState(SettingType);
   strcpy(outputbuffer, "TYPE I\r\n");
   Send();
}

