;;; @file reubase.s
;;; Definitions for the Commodore RAM Expansion Unit
;;; for the Commodore 64 and 128
;;; @author Marko Mkel (msmakela@nic.funet.fi)

	;; flag -> use Action Replay and the REU simultaneously
actionreuplay = 0
		; Action Replay has one write-only I/O register at $de00-$deff.
		; Its bits are as follows.
		; 7 - unused, or extra ROM bank selector (A15)
		; 6 - Resets FREEZE-mode (turns back to normal mode)
		; 5 - 1=enable RAM at ROML ($8000) & I/O2 ($DF00, =$9F00-$9FFF)
		; 4 - ROM bank selector high (A14)
		; 3 - ROM bank selector low (A13)
		; 2 - 1=disable cartridge (turn off the $de00 register)
		; 1 - 1=-EXROM high
		; 0 - 1=-GAME low
#if actionreuplay
	;; map AR ROM at $df00 (normal configuration)
#define ar_rom lda #$0a:sta $de00
	;; map AR RAM at $df00 to avoid interference with the REU registers
#define ar_ram lda #$2a:sta $de00
#define ar_bank1 sta bank
#define ar_bank2 adc bank:sta bank
#else
#define ar_rom
#define ar_ram
#define ar_bank1
#define ar_bank2 adc reu+5
#endif

	;; smallest REU bank number
reubmin = 16
	;; base address of the REU
reu = $df00

	;; slow down to 1 MHz or speed up to 2 MHz
#if allslow
#define reu_slow
#define reu_fast
#else
#define reu_slow dec $d030
#define reu_fast inc $d030
#endif
	;; switch to RAM bank 0, or restore the normal config
#if host = 128
#define mmu_ram0 lda #2:sta $ff00
#define mmu_rest sty $ff00
#else
#define mmu_ram0
#define mmu_rest
#endif
	
	;; auxiliary routines reuinit and incmptreu
#define reu_aux\
	reuinit lda #0:ldx #$40:sta reu+2:stx reu+3:sta reu+7:stx reu+8:\
	sta reu+9:sta reu+10:lda bank:sbc #reubmin:sta reu+6:\
	lda start:sta reu+4:lda start+1:sta reu+5:\
	ar_bank1:mmu_ram0:reu_slow:lda #$f2:sta reu+1:reu_fast:jmp rptrinit:\
	\
	incmptreu inc start:.(:bne nosw:inc start+1:bpl nosw:\
	dex:reu_slow:lda #$f2:sta reu+1:lda #$40:sta start+1:clc:\
	ar_bank2:sta reu+5:lda #$f2:sta reu+1:reu_fast:nosw txa:.):\
	.(:bne ne:lda start:cmp end:bne ne:lda start+1:cmp end+1:ne rts:.)

	;; load a memory block
#define reu_load lda bank:cmp #reubmin:.(:bcc plain:\
	ar_ram:lda end+1:pha:jsr reuinit:LDY0R:\
	.(:loop jsr receive:sta (start),y:jsr incmptreu:bne loop:.):\
	reu_slow:lda #$f2:sta reu+1:pla:sta start+1:sta end+1:\
	ar_rom:jmp eof:\
	plain plain_load:eof:.)

#if s_switch
#define SAVE1R lda (start),y:jsr send_switch:jsr incmptreu:beq eof
#else
#define SAVE1R
#endif		; send != send_switch

	;; save a memory block
#define reu_save lda bank:cmp #reubmin:.(:bcc plain:\
	ar_ram:lda end+1:pha:jsr reuinit:LDY0R:\
	.(:SAVE1R:loop lda (start),y:jsr send:jsr incmptreu:bne loop:\
	eof reu_slow:.):lda #$f2:sta reu+1:pla:sta start+1:sta end+1:\
	ar_rom:mmu_rest:jmp eos:\
	plain plain_save:eos:.)
