/**
 * @file commsel.c
 * Communication protocol selection
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>

#include "comm.h"
#include "info.h"
#include "commsel.h"
#ifdef COMM_SERIAL
# include "c2n232.h"
# include "serial.h"
#endif /* COMM_SERIAL */
#ifdef COMM_PC
# include "kontros.h"
# include "pc64.h"
# include "prlink.h"
# include "x1541.h"
# include "c64net.h"
#endif /* COMM_PC */
#ifdef COMM_AMIGA
# include "emul1541.h"
# include "em1541.h"
# include "prlink48.h"
# include "prlink88.h"
# include "transnib.h"
#endif /* COMM_AMIGA */

/** Communication protocol bindings */
struct commproto
{
  /** name of the protocol */
  const char* name;
  /** connection establishment primitive
   * @param dev		name of the communication device
   * @param hostinfo	(output) the computer model information
   * @return		zero on success, nonzero on failure
   */
  int (*comm_init) (const char* dev, struct hostinfo* hostinfo);
  /** connection termination primitive */
  void (*comm_close) (void);
  /** the communication primitives */
  struct comm comm;
};

/** Dummy replacement function for switching data direction */
static void
dummy (void)
{
}

#ifdef COMM_DEBUG
/** the currently selected communications protocol */
static const struct commproto* currentproto;

/** switch from receive to send */
static void
debug_rs (void)
{
  fprintf (stderr, "%s: receive->send\n", currentproto->name);
  (*currentproto->comm.comm_rs) ();
}

/** switch from receive to send */
static void
debug_sr (void)
{
  fprintf (stderr, "%s: send->receive\n", currentproto->name);
  (*currentproto->comm.comm_sr) ();
}

/** switch from receive to send
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
static int
debug_write (const void* buf, unsigned len)
{
  unsigned i;
  int status;
  fputs (currentproto->name, stderr), fputs (": write:", stderr);
  for (i = 0; i < len; i++)
    fprintf (stderr, " %02x", ((const unsigned char*) buf)[i]);
  fflush (stderr);
  if ((status = (*currentproto->comm.comm_write) (buf, len)))
    fprintf (stderr, ": %d\n", status);
  else
    putc ('\n', stderr);
  return status;
}

/** Receive data
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
static int
debug_read (void* buf, unsigned len)
{
  int status;
  fputs (currentproto->name, stderr), fputs (": read:", stderr);
  fflush (stderr);
  if ((status = (*currentproto->comm.comm_read) (buf, len)))
    fprintf (stderr, "%d\n", status);
  else {
    unsigned i;
    for (i = 0; i < len; i++)
      fprintf (stderr, " %02x", ((const unsigned char*) buf)[i]);
    putc ('\n', stderr);
  }
  return status;
}

static const struct comm debug_comm = {
  debug_rs, debug_sr, debug_write, debug_read
};

#endif /* COMM_DEBUG */

/** Supported communication protocols, starting from the default one */
static const struct commproto protocols[] =
{
#ifdef COMM_SERIAL
  {
    "c2n232", c2n232_init, c2n232_close,
    { dummy, dummy, c2n232_write, c2n232_read }
  },
  {
    "serial", serial_init, serial_close,
    { dummy, dummy, serial_write, serial_read }
  },
#endif /* COMM_SERIAL */
#ifdef COMM_PC
  {
    "kontros", kontros_init, kontros_close,
    { kontros_rs, kontros_sr, kontros_write, kontros_read }
  },
  {
    "pc64", pc64_init, pc64_close,
    { dummy, dummy, pc64_write, pc64_read }
  },
  {
    "prlink", prlink_init, prlink_close,
    { dummy, dummy, prlink_write, prlink_read }
  },
  {
    "x1541", x1541_init, x1541_close,
    { x1541_rs, x1541_sr, x1541_write, x1541_read }
  },
  {
    "c64net", c64net_init, c64net_close,
    { dummy, dummy, c64net_write, c64net_read }
  },
#endif /* COMM_PC */
#ifdef COMM_AMIGA
  {
    "emul1541", emul1541_init, emul1541_close,
    { emul1541_rs, emul1541_sr, emul1541_write, emul1541_read }
  },
  {
    "em1541", em1541_init, em1541_close,
    { em1541_rs, em1541_sr, em1541_write, em1541_read }
  },
  {
    "prlink48", prlink48_init, prlink48_close,
    { dummy, dummy, prlink48_write, prlink48_read }
  },
  {
    "prlink88", prlink88_init, prlink88_close,
    { dummy, dummy, prlink88_write, prlink88_read }
  },
  {
    "transnib", transnib_init, transnib_close,
    { dummy, dummy, transnib_write, transnib_read }
  },
#endif /* COMM_AMIGA */
  /* end of list */
  { 0, 0, 0, { 0, 0, 0, 0 } }
};

/** The connection cleanup function */
static void (*comm_close) (void) = 0;

/** Initialize the communications of a protocol
 * @param protocol	the protocol
 * @param debug		flag: enable debugging
 * @param dev		name of the communication interface device
 * @param comm		(output) pointer to the communication primitives
 * @return		pointer to host information, or NULL on error
 */
static const struct hostinfo*
comm_init (const struct commproto* protocol,
#ifdef COMM_DEBUG
	   int debug,
#endif /* COMM_DEBUG */
	   const char* dev,
	   const struct comm** comm)
{
  /** detected host information */
  static struct hostinfo hostinfo;
  int status;
  comm_close = protocol->comm_close;
  status = (*protocol->comm_init) (dev, &hostinfo);
  if (status) {
    fprintf (stderr, "init for '%s' returned %d\n", protocol->name, status);
    return 0;
  }
#ifdef COMM_DEBUG
  if (debug) {
    *comm = &debug_comm;
    currentproto = protocol;
  }
  else
#endif /* COMM_DEBUG */
  *comm = &protocol->comm;
  return &hostinfo;
}

/** Select a communication protocol by name, and initialize the communications
 * @param name		name of the protocol (NULL=default)
 * @param dev		name of the communication interface device
 * @param comm		(output) pointer to the communication primitives
 * @return		pointer to host information, or NULL on error
 */
const struct hostinfo*
establish (const char* name,
	   const char* dev,
	   const struct comm** comm)
{
  const struct commproto* proto;
#ifdef COMM_DEBUG
  int debug = 0;
#endif /* COMM_DEBUG */
  if (!name)
    return comm_init (protocols,
#ifdef COMM_DEBUG
		      debug,
#endif /* COMM_DEBUG */
		      dev, comm);
#ifdef COMM_DEBUG
  if ((debug = !memcmp (name, "debug_", 6)))
    name += 6;
#endif /* COMM_DEBUG */
  for (proto = protocols; proto->name; proto++)
    if (!strcmp (proto->name, name))
      return comm_init (proto,
#ifdef COMM_DEBUG
			debug,
#endif /* COMM_DEBUG */
			dev, comm);
  fprintf (stderr, "establish: no driver for '%s' found\n"
	   "Try one of the following:\n", name);
  for (proto = protocols; proto->name; proto++)
    fputs (proto->name, stderr), fputc ('\n', stderr);
#ifdef COMM_DEBUG
  fputs ("or one of them prefixed with debug_\n", stderr);
#endif /* COMM_DEBUG */
  return 0;
}

/** Terminate the connection, if it was established */
void
terminate (void)
{
  if (!comm_close)
    return;
  (*comm_close) ();
  comm_close = 0;
}
