/**
 * @file rdfile.c
 * Extension for reading files from the remote host
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001,2002 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "comm.h"
#include "info.h"
#include "ext.h"
#include <stdio.h>
#include <string.h>
#include "rdfile.h"
#include "rdfile-o.h"

/** install the rdfile extension
 * @param comm		the communication primitives
 * @param hostinfo	information on the remote host
 * @param device	device number
 * @param secondary	secondary address
 * @return		zero on success, nonzero on error
 */
int
rdfile_install (const struct comm* comm,
		const struct hostinfo* hostinfo,
		unsigned device,
		unsigned secondary)
{
  switch (hostinfo->host) {
  case PET:
    break;
  case PET3:
    return ext (comm, hostinfo, rdfile_pet3000, sizeof rdfile_pet3000,
		device, secondary) ? 2 : 0;
  case PET4:
    return ext (comm, hostinfo, rdfile_pet4000, sizeof rdfile_pet4000,
		device, secondary) ? 2 : 0;
  case Vic: case C64: case C128: case C264:
    return ext (comm, hostinfo, rdfile_cbm, sizeof rdfile_cbm,
		device, secondary) ? 2 : 0;
  case P500: case B128: case B256:
    return ext (comm, hostinfo, rdfile_cbm2, sizeof rdfile_cbm2,
		device, secondary) ? 2 : 0;
  }

  fprintf (stderr, "rdfile: unsupported server %u\n", hostinfo->host);
  return 1;
}

/** remove the rdfile extension
 * @param comm		the communication primitives
 * @return		zero on success, nonzero on error
 */
int
rdfile_remove (const struct comm* comm)
{
  unsigned char ch = 0xff;
  return (*comm->comm_write) (&ch, 1);
}

/** states of the disk directory parser */
enum dirstate {
  loadaddr1,	/**< reading (and ignoring) first byte of loading address */
  loadaddr2,	/**< reading (and ignoring) second byte of loading address */
  link1,	/**< reading (and ignoring) track link of next block */
  link2,	/**< reading (and ignoring) sector link of next block */
  number1,	/**< reading the least significant byte of line number */
  number2,	/**< reading the most significant byte of line number */
  contents,	/**< reading the contents of a line */
  end		/**< end of directory */
};

/** state of the writedir function */
static enum dirstate writedir_state;
/** current line number of the writedir function */
static unsigned writedir_lineno;

/** display a block of data as a directory
 * @param p		the data buffer
 * @param length	number of bytes in the buffer
 * @param file		output file
 */
static void
writedir (const char* p,
	  unsigned length,
	  FILE* file)
{
  while (length > 0) {
    switch (writedir_state) {
    case loadaddr1:
    case loadaddr2:
      writedir_state++;
      break;
    case link2:
      if (!*p) {
	writedir_state = end;
	break;
      }
      /** fall through */
    case link1:
      writedir_state++;
      break;
    case number1:
      writedir_state++;
      writedir_lineno = (unsigned char) *p;
      break;
    case number2:
      writedir_state++;
      writedir_lineno += 256 * (unsigned) (unsigned char) *p;
      fprintf (file, " %u ", writedir_lineno);
      break;
    case contents:
      if (*p == 0) {
	writedir_state = link1;
	putc ('\n', file);
      }
      else
	putc (*p, file);
      break;
    case end:
      /* ignore any bytes after the end */
      break;
    }
    length--;
    p++;
  }
}

/** read a file either as raw or decoded as a directory listing
 * @param comm		the communication primitives
 * @param file		output file (0=none)
 * @param buf		at least 256-byte transfer buffer
 * @param directory	flag: decode the file as a directory
 * @return		zero on success, nonzero on error
 */
static int
rdfile_do (const struct comm* comm,
	   FILE* file,
	   char* buf,
	   unsigned directory)
{
  /** command for requesting next block */
  static const char cmd_next[2] = { (char) 0x80, 0 };
  /** command for aborting the transfer */
  static const char cmd_abort[2] = { (char) 0x82, 0 };
  if ((*comm->comm_write) (buf, ((unsigned) (unsigned char) *buf) + 2))
    return 3;
  (*comm->comm_sr) ();
  for (;;) {
    /** checksum */
    unsigned check;
    /** EOF flag and status, or length and checksum of available data */
    char reply[2];
    unsigned i;
    if ((*comm->comm_read) (reply, sizeof reply))
      return 3;
    if (!(i = (unsigned char) *reply)) {
      if (reply[1])
	fprintf (stderr, "rdfile failed with %u\n",
		 (unsigned char) reply[1]);
      (*comm->comm_rs) ();
      return reply[1] || 0;
    }

    check = (unsigned char) reply[1];

    if ((*comm->comm_read) (buf, i))
      return 3;
    else {
      register const char* b = buf + i;
      while (b-- > buf)
	check -= (unsigned char) *b;
    }

    (*comm->comm_rs) ();

    if ((check -= i) & 0xff) {
      fputs ("rdfile: checksum mismatch; retrying\n", stderr);
      *buf = (char) 0x81;
      if ((*comm->comm_write) (buf, 1))
	return 3;
      (*comm->comm_sr) ();
      continue;
    }

    if (!file);
    else if (directory)
      writedir (buf, i, file);
    else if (i != fwrite (buf, 1, i, file)) {
      perror ("rdfile: fwrite");
      if ((*comm->comm_write) (cmd_abort, sizeof cmd_abort) ||
	  ((*comm->comm_sr) (), (*comm->comm_read) (buf, 2)))
	return 3;
      if (*buf || buf[1])
	return 4;
      (*comm->comm_rs) ();
      return 1;
    }

    /* request next block */
    if ((*comm->comm_write) (cmd_next, sizeof cmd_next))
      return 3;
    (*comm->comm_sr) ();
  }
}

/** read a file either as raw or decoded as a directory listing
 * @param comm		the communication primitives
 * @param filename	file name search pattern (including drive unit number)
 * @param file		output file (0=none)
 * @param buf		at least 256-byte transfer buffer
 * @param directory	flag: decode the file as a directory
 * @return		zero on success, nonzero on error
 */
static int
rdfile_copy (const struct comm* comm,
	     const char* filename,
	     FILE* file,
	     char* buf,
	     unsigned directory)
{
  unsigned i;
  for (i = 0; filename[i]; i++);
  if (i > (directory ? 126 : 127))
    return fputs ("rdfile: too long file name\n", stderr), 2;
  memcpy (directory ? buf + 2 : buf + 1, filename, i + 1);
  if (directory)
    buf[1] = '$', i++;
  *buf = i, buf[i + 1] = 0;
  return rdfile_do (comm, file, buf, directory);
}

/** display the disk directory
 * @param comm		the communication primitives
 * @param filename	file name search pattern (including drive unit number)
 * @param file		output file
 * @param buf		at least 256-byte transfer buffer
 * @return		zero on success, nonzero on error
 */
int
rdfile_directory (const struct comm* comm,
		  const char* filename,
		  FILE* file,
		  char* buf)
{
  writedir_state = loadaddr1;
  return rdfile_copy (comm, filename, file, buf, 1);
}

/** copy a file
 * @param comm		the communication primitives
 * @param filename	file name pattern (including drive unit number)
 * @param file		output file (0=none)
 * @param buf		at least 256-byte transfer buffer
 * @return		zero on success, nonzero on error
 */
int
rdfile (const struct comm* comm,
	const char* filename,
	FILE* file,
	char* buf)
{
  return rdfile_copy (comm, filename, file, buf, 0);
}

/** copy a file
 * @param comm		the communication primitives
 * @param file		output file (0=none)
 * @param buf		(in) the file name, (out) at most 256 bytes of data
 * @return		zero on success, nonzero on error
 */
int
rdfile_raw (const struct comm* comm,
	    FILE* file,
	    char* buf)
{
  buf[1 + (unsigned) (unsigned char)*buf] = 0;
  return rdfile_do (comm, file, buf, 0);
}
