/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <signal.h>
#include <sys/file.h>
#include <sys/wait.h>

#if defined(linux)
#include        <sys/vt.h>
#include        <sys/kd.h>
#elif defined(__FreeBSD__)
#include <machine/console.h>
#endif

#include <getcap.h>
#include <errors.h>
#include <term.h>

static void     ProcessArgs(int argc, const char *argv[]);

int main(int argc, const char *argv[])
{
    int cfd,mode;
#if defined(linux)
    struct vt_stat vts;
#endif

    fprintf(stderr,"\r\nCCE> Console Chinese Environment " VERSION "\n");
    if (geteuid() != 0)
    {
        fatal("Can't get I/O permissions. CCE needs suid-root.\n");
    }

#if defined(linux)
    cfd = open("/dev/console", O_WRONLY);
    if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0)
        fatal("Can't open /dev/console");
#elif defined(__FreeBSD__)
    cfd = open("/dev/vga", O_WRONLY);
    if (cfd < 0 && (cfd = open("/dev/vga", O_RDONLY)) < 0)
        fatal("Can't open /dev/vga");
#endif

    ioctl(cfd, KDGETMODE, &mode);

#if defined(linux)
    ioctl(cfd, VT_GETSTATE, &vts);
    VtNum = vts.v_active;
#elif defined(__FreeBSD__)
   ioctl(cfd, VT_GETACTIVE, &VtNum);
#endif

    close(cfd);

    /* Even in frame buffer mode, mode is still KD_TEXT(0),not KD_GRAPHICS(1) */

    if (mode != KD_TEXT)
    {
        fatal("You can only start CCE on text console.\n");
    }
   
    ProcessArgs(argc-1, argv+1);

    TermInit();

    if (ReadConfig(CONFIG_NAME) < 0) 
    {
	fprintf(stderr, "CCE> error reading %s\n", CONFIG_NAME);
	exit(EXIT_FAILURE);
    }

    TermStart();

    return 0;
}

static void     ProcessArgs(int argc, const char *argv[])
{
    int i = 0;

    while (i < argc)
    {
        const char      *arg;

        if (argv[i][0] != '-')
        {
            warn("bad arg `%s'; assumed `-%s'\r\n", argv[i]);
            arg = (char *) argv[i];
        } else
            arg = (char *) argv[i] + 1;
        i++;
        if (i >= argc) {
            error("no value for `%s'\r\n", arg);

            break;
        }

        if (!strcasecmp(arg, "e"))
            ConfigExecProg(argv[i]);
        else if (SetCapArg(arg, argv[i]) < 0)
            warn("invalid capability `%s' ignored\r\n", arg);
        i++;
    }
}


/*********************************************************************
 *                ChangeOrigConsole/ChangeNewConsole                 *
 *                                                                   *
 *  kon/cce's original idea:                                         *
 *    If you start cce in text console, just switch to graphics      *
 *    mode; if you start cce in graphics console(X or framebuffer    *
 *    or another cce console), it will find an unused console and    *
 *    start cce there, after cce ends, it will change back to        *
 *    original console.   But I don't think it's necessary here,     *
 *    so I changed it: if it's not text console, just quit           *
 *********************************************************************/

#if 0

static int origVtNum = -1;
void ExitPty(int signum);

/* called when cce exit */
void ChangeOrigConsole()
{
    int cfd;

#if defined(linux)
    cfd = open("/dev/console", O_WRONLY);
    if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0) {
        PerrorExit("/dev/console");
    }
#elif defined(__FreeBSD__)
    cfd = open("/dev/vga", O_WRONLY);
    if (cfd < 0 && (cfd = open("/dev/vga", O_RDONLY)) < 0) {
        PerrorExit("/dev/vga");
    }
#endif
    ioctl(cfd, VT_ACTIVATE, origVtNum);
    close(cfd);
}

/* When cce starts, this function is called in main()
   if current active console is in TEXT mode, simply return,
   if it's in GRAPHICS mode, find another console and open it
*/

void ChangeNewConsole()
{
#if defined(linux)
    struct vt_stat vts;
#endif
    int cfd, vfd, vtNum, child, parent, mode;
    char vtty[12];

#if defined(linux)
    cfd = open("/dev/console", O_WRONLY);

    if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0)
        fatal("can't open /dev/console");
#elif defined(__FreeBSD__)
    cfd = open("/dev/vga", O_WRONLY);
    if (cfd < 0 && (cfd = open("/dev/vga", O_RDONLY)) < 0)
        fatal("can't open /dev/vga");
#endif
    ioctl(cfd, KDGETMODE, &mode);

    if (mode == KD_TEXT)
    {
        close(cfd);
        return;
    }
    else
    {
        fatal("console is in graphics mode!\n");
    }
#if defined(linux)
    ioctl(cfd, VT_GETSTATE, &vts);
    origVtNum = vts.v_active;
#endif
    ioctl(cfd, VT_OPENQRY, &vtNum);
    if (vtNum < 0)
        fatal("can't get free VC");

    parent = getpid();
    if ((child = fork()) == -1)
        PerrorExit("fork child error");

    if (child)
    {
        /* child != 0, then this is the parent process
           SIGHUP 1 Hangup (POSIX).  */

        signal(SIGHUP, ExitPty);
        pause();  /* sleep, waiting for signal, the child end */
    }
    setsid();

   /*
   setsid - creates a session and sets the process group ID
   A process group leader is a process with process group  ID
       equal  to  its  PID.  In order to be sure that setsid will
       succeed, fork and exit, and have the child do setsid().
   */

#if defined(linux)
    sprintf(vtty, "/dev/tty%d", vtNum);
#elif defined(__FreeBSD__)
    sprintf(vtty, "/dev/ttyv%d", vtNum);
#endif
    if ((vfd = open(vtty, O_RDWR)) < 0)
        fatal("can't open %s", vtty);
    if (ioctl(cfd, VT_ACTIVATE, vtNum) != 0)
        fatal("can't activate VC(%d)", vtNum);
    atexit(ChangeOrigConsole);
     /* Register a function to be called at program termination */

    close(cfd);
    dup2(vfd, 0);
    dup2(vfd, 1);
    dup2(vfd, 2);
    kill(parent, SIGHUP);
}


void ExitPty(int signum)
{
    int stat;

#if defined(__FreeBSD__)
    signal(SIGCLD, SIG_DFL);
#endif
    if (wait3(&stat, WNOHANG, 0) != childPid)
    {
        TextMode();
        kill(0, SIGTSTP);
        GraphMode();
        kill(childPid, SIGCONT);
        signal(SIGCLD, ExitPty);
        return;
    }
    if (WEXITSTATUS(stat) & 0x7f)
    {
        if (WIFSIGNALED(stat))
            fatal("child died with signal -- %s\r\n", 
               sys_siglist[WTERMSIG(stat)]);
        else
            fatal("child exited with status %d\r\n", WEXITSTATUS(stat) & 0x7f);
    }
    else if (signum == SIGHUP)
    {
        fprintf(stderr, "CCE> Switched to new VC\r\n");
        exit(EXIT_SUCCESS);
    }
    else
    {
        fprintf(stderr, "CCE> Finished without core dump :)\r\n\r\n");
        exit(EXIT_SUCCESS);
    }
}

#endif
