;;;; unistd.scm - Miscellaneous file- and process-handling routines
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit unistd)
  (uses regex)
  (interrupts-disabled)
  (usual-integrations)
  (no-bound-checks)
  (bound-to-procedure
   ##sys#make-port ##sys#file-info ##sys#update-errno ##sys#fudge ##sys#make-c-string ##sys#check-port 
   ##sys#port-file-resolve ##sys#error ##sys#signal-hook
   ##sys#peek-fixnum ##sys#make-structure ##sys#check-structure ##sys#enable-interrupts)
  (foreign-declare #<<EOF
#include <signal.h>
#include <errno.h>

static int C_not_implemented(void);
int C_not_implemented() { return -1; }

static int C_wait_status;

#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <pwd.h>
#include <sys/mman.h>
#include <time.h>

#ifndef PIPE_BUF
# ifdef __CYGWIN__
#  define PIPE_BUF       _POSIX_PIPE_BUF
# else
#  define PIPE_BUF 1024
# endif
#endif

#ifndef ARG_MAX
# define ARG_MAX 256
#endif

#ifndef MAP_FILE
# define MAP_FILE    0
#endif

#ifndef MAP_ANONYMOUS
# define MAP_ANONYMOUS    0
#endif

extern char **environ;

static char *C_exec_args[ ARG_MAX ];
static struct utsname C_utsname;
static struct flock C_flock;
static DIR *temphandle;
static struct dirent *finddata;
static struct passwd *C_user;
static int C_pipefds[ 2 ];
static time_t C_secs;
static struct tm C_tm;
static fd_set C_fd_sets[ 2 ];
static struct timeval C_timeval;
static char C_hostbuf[ 256 ];

#define C_mkdir(str)        C_fix(mkdir(C_c_string(str), S_IRWXO))
#define C_chdir(str)        C_fix(chdir(C_c_string(str)))
#define C_rmdir(str)        C_fix(rmdir(C_c_string(str)))
#define C_findfirst(str)    (temphandle = opendir(C_c_string(str)), ((temphandle && (finddata = readdir(temphandle))) ? C_fix(temphandle) : C_SCHEME_FALSE))
#define C_findnext(h)       C_mk_bool(finddata = readdir((DIR *)C_unfix(h)))
#define C_foundfile(buf)    (strcpy(C_c_string(buf), finddata->d_name), C_fix(strlen(finddata->d_name)))
#define C_findclose(h)      (closedir((DIR *)C_unfix(h)), C_SCHEME_UNDEFINED)
#define C_curdir(buf)       (getcwd(C_c_string(buf), 256) ? C_fix(strlen(C_c_string(buf))) : C_SCHEME_FALSE)

#define open_binary_input_pipe(a, n, name)   C_mpointer(a, popen(C_c_string(name), "r"))
#define open_text_input_pipe(a, n, name)     open_binary_input_pipe(a, n, name)
#define open_binary_output_pipe(a, n, name)  C_mpointer(a, popen(C_c_string(name), "w"))
#define open_text_output_pipe(a, n, name)    open_binary_output_pipe(a, n, name)
#define close_pipe(p)                        C_fix(pclose(C_port_file(p)))

#define C_fork              fork
#define C_set_exec_arg(i, a) (C_exec_args[ i ] = (a))
#define C_execvp(f)         C_fix(execvp(C_data_pointer(f), C_exec_args))
#define C_waitpid(id, o)    C_fix(waitpid(C_unfix(id), &C_wait_status, C_unfix(o)))
#define C_getpid            getpid
#define C_getppid           getppid
#define C_kill(id, s)       C_fix(kill(C_unfix(id), C_unfix(s)))
#define C_uname             C_fix(uname(&C_utsname))
#define C_chmod(fn, m)      C_fix(chmod(C_data_pointer(fn), C_unfix(m)))
#define C_chown(fn, u, g)   C_fix(chown(C_data_pointer(fn), C_unfix(u), C_unfix(g)))
#define C_getuid            getuid
#define C_getgid            getgid
#define C_geteuid           geteuid
#define C_getegid           getegid
#define C_symlink(o, n)     C_fix(symlink(C_data_pointer(o), C_data_pointer(n)))
#define C_readlink(f, b)    C_fix(readlink(C_data_pointer(f), C_data_pointer(b), FILENAME_MAX))
#define C_fdopen            fdopen
#define C_fileno(p)         C_fix(fileno(C_port_file(p)))
#define C_dup(x)            C_fix(dup(C_unfix(x)))
#define C_dup2(x, y)        C_fix(dup2(C_unfix(x), C_unfix(y)))
#define C_truncate(f, n)    C_fix(truncate((char *)C_data_pointer(f), C_num_to_int(n)))
#define C_ftruncate(f, n)   C_fix(ftruncate(C_unfix(f), C_num_to_int(n)))
#define C_alarm             alarm
#define C_getpwnam(n)       C_mk_bool((C_user = getpwnam((char *)C_data_pointer(n))) != NULL)
#define C_getpwuid(u)       C_mk_bool((C_user = getpwuid(C_unfix(u))) != NULL)
#define C_setvbuf(p, m, s)  C_fix(setvbuf(C_port_file(p), NULL, C_unfix(m), C_unfix(s)))
#define C_access(fn, m)     C_fix(access((char *)C_data_pointer(fn), C_unfix(m)))
#define C_setuid(id)        C_fix(setuid(C_unfix(id)))
#define C_pipe(d)           C_fix(pipe(C_pipefds))
#define C_close(fd)         C_fix(close(C_unfix(fd)))
#define C_getenventry(i)    environ[ i ]
#define C_putenv(s)         C_fix(putenv((char *)C_data_pointer(s)))

#ifdef C_GNU_ENV
# define C_setenv(x, y)     C_fix(setenv((char *)C_data_pointer(x), (char *)C_data_pointer(y), 1))
#else
static C_word C_fcall C_setenv(C_word x, C_word y);
C_word C_fcall C_setenv(C_word x, C_word y) {
  char *sx = C_data_pointer(x),
       *sy = C_data_pointer(y);
  int n1 = C_strlen(sx), n2 = C_strlen(sy);				       
  char *buf = (char *)C_malloc(n1 + n2 + 2);
  if(buf == NULL) return(C_fix(0));
  else {
    C_strcpy(buf, sx);
    buf[ n1 ] = '=';
    C_strcpy(buf + n1 + 1, sy);
    return(C_fix(putenv(buf)));
  }
}
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__)
static int C_uw;
# define C_WIFEXITED(n)      (C_uw = C_unfix(n), C_mk_bool(WIFEXITED(C_uw)))
# define C_WIFSIGNALED(n)    (C_uw = C_unfix(n), C_mk_bool(WIFSIGNALED(C_uw)))
# define C_WIFSTOPPED(n)     (C_uw = C_unfix(n), C_mk_bool(WIFSTOPPED(C_uw)))
# define C_WEXITSTATUS(n)    (C_uw = C_unfix(n), C_fix(WEXITSTATUS(C_uw)))
# define C_WTERMSIG(n)       (C_uw = C_unfix(n), C_fix(WTERMSIG(C_uw)))
# define C_WSTOPSIG(n)       (C_uw = C_unfix(n), C_fix(WSTOPSIG(C_uw)))
#else
# define C_WIFEXITED(n)      C_mk_bool(WIFEXITED(C_unfix(n)))
# define C_WIFSIGNALED(n)    C_mk_bool(WIFSIGNALED(C_unfix(n)))
# define C_WIFSTOPPED(n)     C_mk_bool(WIFSTOPPED(C_unfix(n)))
# define C_WEXITSTATUS(n)    C_fix(WEXITSTATUS(C_unfix(n)))
# define C_WTERMSIG(n)       C_fix(WTERMSIG(C_unfix(n)))
# define C_WSTOPSIG(n)       C_fix(WSTOPSIG(C_unfix(n)))
#endif

#ifdef __CYGWIN__
# define C_mkfifo(fn, m)    C_fix(-1);
#else
# define C_mkfifo(fn, m)    C_fix(mkfifo((char *)C_data_pointer(fn), C_unfix(m)))
#endif

#define C_flock_setup(t, s, n) (C_flock.l_type = C_unfix(t), C_flock.l_start = C_num_to_int(s), C_flock.l_whence = SEEK_SET, C_flock.l_len = C_num_to_int(n), C_SCHEME_UNDEFINED)
#define C_flock_test(p)     (fcntl(fileno(C_port_file(p)), F_GETLK, &C_flock) >= 0 ? (C_flock.l_type == F_UNLCK ? C_fix(0) : C_fix(C_flock.l_pid)) : C_SCHEME_FALSE)
#define C_flock_lock(p)     C_fix(fcntl(fileno(C_port_file(p)), F_SETLK, &C_flock))

#ifndef FILENAME_MAX
# define FILENAME_MAX          1024
#endif

static sigset_t C_sigset;
#define C_sigemptyset(d)    (sigemptyset(&C_sigset), C_SCHEME_UNDEFINED)
#define C_sigaddset(s)      (sigaddset(&C_sigset, C_unfix(s)), C_SCHEME_UNDEFINED)
#define C_sigprocmask(d)    C_fix(sigprocmask(SIG_SETMASK, &C_sigset, NULL))

#define C_open(fn, fl, m)   C_fix(open((char *)C_data_pointer(fn), C_unfix(fl), C_unfix(m)))
#define C_read(fd, b, n)    C_fix(read(C_unfix(fd), C_data_pointer(b), C_unfix(n)))
#define C_write(fd, b, n)   C_fix(write(C_unfix(fd), C_data_pointer(b), C_unfix(n)))

#define C_ftell(p)            C_fix(ftell(C_port_file(p)))
#define C_fseek(p, n, w)      C_mk_nbool(fseek(C_port_file(p), C_unfix(n), C_unfix(w)))
#define C_lseek(fd, o, w)     C_fix(lseek(C_unfix(fd), C_unfix(o), C_unfix(w)))

#define C_zero_fd_set(i)    FD_ZERO(&C_fd_sets[ i ])
#define C_set_fd_set(i, fd) FD_SET(fd, &C_fd_sets[ i ])
#define C_test_fd_set(i, fd) FD_ISSET(fd, &C_fd_sets[ i ])
#define C_select(m)         C_fix(select(C_unfix(m), &C_fd_sets[ 0 ], &C_fd_sets[ 1 ], NULL, NULL))
#define C_select_t(m, t)    (C_timeval.tv_sec = C_unfix(t), C_timeval.tv_usec = 0, C_fix(select(C_unfix(m), &C_fd_sets[ 0 ], &C_fd_sets[ 1 ], NULL, &C_timeval)))

#define C_ctime(n)          (C_secs = (n), ctime(&C_secs))

#if !defined(C_GNU_ENV)
# define C_asctime(v)        (memset(&C_tm, 0, sizeof(struct tm)), C_tm.tm_sec = C_unfix(C_block_item(v, 0)), C_tm.tm_min = C_unfix(C_block_item(v, 1)), C_tm.tm_hour = C_unfix(C_block_item(v, 2)), C_tm.tm_mday = C_unfix(C_block_item(v, 3)), C_tm.tm_mon = C_unfix(C_block_item(v, 4)), C_tm.tm_year = C_unfix(C_block_item(v, 5)), C_tm.tm_wday = C_unfix(C_block_item(v, 6)), C_tm.tm_yday = C_unfix(C_block_item(v, 7)), C_tm.tm_isdst = (C_block_item(v, 8) != C_SCHEME_FALSE), asctime(&C_tm) )
#else
# define C_asctime(v)        (memset(&C_tm, 0, sizeof(struct tm)), C_tm.tm_sec = C_unfix(C_block_item(v, 0)), C_tm.tm_min = C_unfix(C_block_item(v, 1)), C_tm.tm_hour = C_unfix(C_block_item(v, 2)), C_tm.tm_mday = C_unfix(C_block_item(v, 3)), C_tm.tm_mon = C_unfix(C_block_item(v, 4)), C_tm.tm_year = C_unfix(C_block_item(v, 5)), C_tm.tm_wday = C_unfix(C_block_item(v, 6)), C_tm.tm_yday = C_unfix(C_block_item(v, 7)), C_tm.tm_isdst = (C_block_item(v, 8) != C_SCHEME_FALSE), C_tm.tm_gmtoff = C_unfix(C_block_item(v, 9)), asctime(&C_tm) )
#endif
EOF
) )

(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure x y) '(##core#undefined))
    (define-macro (##sys#check-range x y z) '(##core#undefined))
    (define-macro (##sys#check-pair x) '(##core#undefined))
    (define-macro (##sys#check-list x) '(##core#undefined))
    (define-macro (##sys#check-symbol x) '(##core#undefined))
    (define-macro (##sys#check-string x) '(##core#undefined))
    (define-macro (##sys#check-char x) '(##core#undefined))
    (define-macro (##sys#check-exact x) '(##core#undefined))
    (define-macro (##sys#check-port x) '(##core#undefined))
    (define-macro (##sys#check-number x) '(##core#undefined))
    (define-macro (##sys#check-byte-vector x) '(##core#undefined)) ) ]
 [else] )

(register-feature! 'unistd)


;;; Lo-level I/O:

(define-foreign-variable _pipe_buf int "PIPE_BUF")

(define pipe/buf _pipe_buf)

(define-foreign-variable _o_rdonly int "O_RDONLY")
(define-foreign-variable _o_wronly int "O_WRONLY")
(define-foreign-variable _o_rdwr int "O_RDWR")
(define-foreign-variable _o_creat int "O_CREAT")
(define-foreign-variable _o_append int "O_CREAT")
(define-foreign-variable _o_excl int "O_CREAT")
(define-foreign-variable _o_noctty int "O_CREAT")
(define-foreign-variable _o_nonblock int "O_CREAT")
(define-foreign-variable _o_trunc int "O_CREAT")
(define-foreign-variable _o_sync int "O_CREAT")

(define open/rdonly _o_rdonly)
(define open/wronly _o_wronly)
(define open/rdwr _o_rdwr)
(define open/read _o_rdwr)
(define open/write _o_wronly)
(define open/creat _o_creat)
(define open/append _o_append)
(define open/excl _o_excl)
(define open/noctty _o_noctty)
(define open/nonblock _o_nonblock)
(define open/trunc _o_trunc)
(define open/sync _o_sync)

(define-foreign-variable _s_irusr int "S_IRUSR")
(define-foreign-variable _s_iwusr int "S_IWUSR")
(define-foreign-variable _s_ixusr int "S_IXUSR")
(define-foreign-variable _s_irgrp int "S_IRGRP")
(define-foreign-variable _s_iwgrp int "S_IWGRP")
(define-foreign-variable _s_ixgrp int "S_IXGRP")
(define-foreign-variable _s_iroth int "S_IROTH")
(define-foreign-variable _s_iwoth int "S_IWOTH")
(define-foreign-variable _s_ixoth int "S_IXOTH")
(define-foreign-variable _s_irwxu int "S_IRWXU")
(define-foreign-variable _s_irwxg int "S_IRWXG")
(define-foreign-variable _s_irwxo int "S_IRWXO")
(define-foreign-variable _s_isvtx int "S_ISVTX")
(define-foreign-variable _s_isuid int "S_ISUID")
(define-foreign-variable _s_isgid int "S_ISGID")

(define perm/irusr _s_irusr)
(define perm/iwusr _s_iwusr)
(define perm/ixusr _s_ixusr)
(define perm/irgrp _s_irgrp)
(define perm/iwgrp _s_iwgrp)
(define perm/ixgrp _s_ixgrp)
(define perm/iroth _s_iroth)
(define perm/iwoth _s_iwoth)
(define perm/ixoth _s_ixoth)
(define perm/irwxu _s_irwxu)
(define perm/irwxg _s_irwxg)
(define perm/irwxo _s_irwxo)
(define perm/isvtx _s_isvtx)
(define perm/isuid _s_isuid)
(define perm/isgid _s_isgid)

(define file-open
  (let ([defmode (bitwise-ior _s_irwxu (bitwise-ior _s_irgrp _s_iroth))] )
    (lambda (filename flags . mode)
      (let ([mode (if (pair? mode) (car mode) defmode)])
	(##sys#check-string filename)
	(##sys#check-exact flags)
	(##sys#check-exact mode)
	(let ([fd (##core#inline "C_open" (##sys#make-c-string filename) flags mode)])
	  (when (eq? -1 fd)
	    (##sys#update-errno)
	    (##sys#signal-hook #:file-error "can not open file" filename flags mode) )
	  fd) ) ) ) )

(define file-close
    (lambda (fd)
      (##sys#check-exact fd)
      (when (fx< (##core#inline "C_close" fd) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not close file" fd) ) ) )

(define file-read
  (let ([make-string make-string] )
    (lambda (fd size . buffer)
      (##sys#check-exact fd)
      (##sys#check-exact size)
      (let* ([buf (if (pair? buffer) (car buffer) (make-string size))]
	     [n (##core#inline "C_read" fd buf size)] )
	(unless (and (##core#inline "C_blockp" buf) (##core#inline "C_byteblockp" buf))
	  (##sys#signal-hook #:type-error "bad argument type - not a string or byte-vector" buf) )
	(when (eq? -1 n)
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not read from file" fd size) )
	(values buf n) ) ) ) )

(define file-write
    (lambda (fd buffer . size)
      (##sys#check-exact fd)
      (unless (and (##core#inline "C_blockp" buffer) (##core#inline "C_byteblockp" buffer))
	(##sys#signal-hook #:type-error "bad argument type - not a string or byte-vector" buffer) )
      (let ([size (if (pair? size) (car size) (##sys#size buffer))])
	(##sys#check-exact size)
	(let ([n (##core#inline "C_write" fd buffer size)])
	  (when (eq? -1 n)
	    (##sys#update-errno)
	    (##sys#signal-hook #:file-error "can not write to file" fd size) )
	  n) ) ) )


;;; I/O multiplexing:

(define file-select
  (let ([fd_zero (foreign-lambda void "C_zero_fd_set" int)]
	[fd_set (foreign-lambda void "C_set_fd_set" int int)] 
	[fd_test (foreign-lambda bool "C_test_fd_set" int int)] )
    (lambda (fdsr fdsw . timeout)
      (##sys#check-list fdsr)
      (##sys#check-list fdsw)
      (let ([fdmax 0]
	    [tm (if (pair? timeout) (car timeout) #f)] )
	(fd_zero 0)
	(fd_zero 1)
	(for-each
	 (lambda (fd)
	   (##sys#check-exact fd)
	   (set! fdmax (##core#inline "C_i_fixnum_max" fdmax fd))
	   (fd_set 0 fd) )
	 fdsr)
	(for-each
	 (lambda (fd)
	   (##sys#check-exact fd)
	   (set! fdmax (##core#inline "C_i_fixnum_max" fdmax fd))
	   (fd_set 1 fd) )
	 fdsw)
	(let ([n (cond [tm
			(##sys#check-exact tm)
			(##core#inline "C_select_t" (fx+ fdmax 1) tm) ]
		       [else (##core#inline "C_select" (fx+ fdmax 1))] ) ] )
	  (cond [(fx< n 0)
		 (##sys#update-errno)
		 (##sys#signal-hook #:file-error "call to `file-select' failed" fdsr fdsw) ]
		[(eq? n 0) (values '() '())]
		[else
		 (let ([lstr '()]
		       [lstw '()] )
		   (for-each (lambda (fd) (when (fd_test 0 fd) (set! lstr (cons fd lstr)))) fdsr)
		   (for-each (lambda (fd) (when (fd_test 1 fd) (set! lstw (cons fd lstw)))) fdsw)
		   (values lstr lstw) ) ] ) ) ) ) ) )


;;; File attribute access:

(define-foreign-variable _seek_set int "SEEK_SET")
(define-foreign-variable _seek_cur int "SEEK_CUR")
(define-foreign-variable _seek_end int "SEEK_END")

(define seek/set _seek_set)
(define seek/end _seek_end)
(define seek/cur _seek_cur)

(define file-size
    (lambda (filename)
      (##sys#check-string filename)
      (let ([info (##sys#file-info filename)])
	(##sys#update-errno)
	(if info
	    (##sys#slot info 3)
	    (##sys#signal-hook #:file-error "can not access file" filename) ) ) ) )

(define file-modification-time
    (lambda (filename)
      (##sys#check-string filename)
      (let ([info (##sys#file-info filename)])
	(##sys#update-errno)
	(if info
	    (##sys#slot info 2)
	    (##sys#signal-hook #:file-error "can not access file" filename) ) ) ) )

(define file-owner
    (lambda (filename)
      (##sys#check-string filename)
      (let ([info (##sys#file-info filename)])
	(##sys#update-errno)
	(if info
	    (##sys#slot info 6)
	    (##sys#signal-hook #:file-error "can not access file" filename) ) ) ) )

(define file-position
    (lambda (port)
      (let ([pos (cond [(port? port) (##core#inline "C_ftell" (##sys#port-file-resolve port))]
		       [(fixnum? port) (##core#inline "C_lseek" port 0 _seek_cur)]
		       [else (##sys#signal-hook #:type-error "invalid file" port)] ) ] )
	(when (fx< pos 0)
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not retrieve file position of port" port) )
	pos) ) )

(define set-file-position!
    (lambda (port pos . whence)
      (let ([whence (if (pair? whence) (car whence) _seek_set)])
	(##sys#check-exact pos)
	(##sys#check-exact whence)
	(when (fx< pos 0) (##sys#error "invalid negative port position" pos port))
	(unless (cond [(port? port) (##core#inline "C_fseek" (##sys#port-file-resolve port) pos whence)]
		      [(fixnum? port) (##core#inline "C_lseek" port pos whence)]
		      [else (##sys#signal-hook #:type-error "invalid file" port)] )
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not set file position" port pos) ) ) ) )


;;; Directory stuff:

(define create-directory 
    (lambda (name)
      (##sys#check-string name)
      (unless (zero? (##core#inline "C_mkdir" (##sys#make-c-string name)))
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not create directory" name) ) ) )

(define change-directory
    (lambda (name)
      (##sys#check-string name)
      (unless (zero? (##core#inline "C_chdir" (##sys#make-c-string name)))
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not change current directory" name) ) ) )

(define delete-directory
    (lambda (name)
      (##sys#check-string name)
      (unless (zero? (##core#inline "C_rmdir" (##sys#make-c-string name)))
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not delete directory" name) ) ) )
  
(define directory
  (let ([string-append string-append]
	[make-string make-string]
	[string string]
	[substring substring] )
    (lambda (spec)
      (##sys#check-string spec)
      (let ([buffer (make-string 256)]
	    [handle (##core#inline "C_findfirst" (##sys#make-c-string spec))] )
	(if handle
	    (let loop ()
	      (let* ([flen (##core#inline "C_foundfile" buffer)]
		     [file (substring buffer 0 flen)]
		     [next (##core#inline "C_findnext" handle)] )
		(cond [next (cons file (loop))]
		      [else
		       (##core#inline "C_findclose" handle)
		       (cons file '()) ] ) ) )
	    '() ) ) ) ) )

(define (directory? fname)
  (##sys#check-string fname)
  (let ((info (##sys#file-info fname)))
    (and info (fx= 1 (##sys#slot info 4))) ) )

(define current-directory
  (let ([make-string make-string]
	[substring substring] )
    (lambda ()
      (let* ([buffer (make-string 256)]
	     [len (##core#inline "C_curdir" buffer)] )
	(##sys#update-errno)
	(if len
	    (substring buffer 0 len)
	    (##sys#signal-hook #:file-error "can not examine current directory") ) ) ) ) )


;;; Pipes:

(let ()
  (define (mode arg) (if (pair? arg) (##sys#slot arg 0) '###text))
  (define (badmode m) (##sys#error "illegal input/output mode specifier" m))
  (define (check cmd outp r)
    (##sys#update-errno)
    (if (##sys#null-pointer? r)
	(##sys#signal-hook #:file-error "can not open pipe" cmd) 
	(let ([port (##sys#make-port -1 6 #f r)])
	  (##sys#update-errno)
	  (##sys#setslot port 1 outp)
	  (##sys#setslot port 3 "(pipe)")
	  (##sys#setslot port 4 0)
	  (##sys#setslot port 5 0)
	  port) ) )
  (set! open-input-pipe
    (lambda (cmd . m)
      (##sys#check-string cmd)
      (let ([m (mode m)])
	(check
	 cmd #f
	 (case m
	   ((###text) (##core#inline_allocate ("open_text_input_pipe" 2) (##sys#make-c-string cmd)))
	   ((###binary) (##core#inline_allocate ("open_binary_input_pipe" 2) (##sys#make-c-string cmd)))
	   (else (badmode m)) ) ) ) ) )
  (set! open-output-pipe
    (lambda (cmd . m)
      (##sys#check-string cmd)
      (let ((m (mode m)))
	(check
	 cmd #t
	 (case m
	   ((###text) (##core#inline_allocate ("open_text_output_pipe" 2) (##sys#make-c-string cmd)))
	   ((###binary) (##core#inline_allocate ("open_binary_output_pipe" 2) (##sys#make-c-string cmd)))
	   (else (badmode m)) ) ) ) ) ) 
  (set! close-input-pipe
    (lambda (port)
      (##sys#check-port port)
      (let ((r (##core#inline "close_pipe" port)))
	(##sys#update-errno)
	(when (eq? -1 r) (##sys#signal-hook #:file-error "error while closing pipe" port)) ) ) )
  (set! close-output-pipe close-input-pipe) )

(let ([open-input-pipe open-input-pipe]
      [open-output-pipe open-output-pipe]
      [close-input-pipe close-input-pipe]
      [close-output-pipe close-output-pipe] )
  (set! call-with-input-pipe
    (lambda (cmd proc . mode)
      (let ([p (apply open-input-pipe cmd mode)])
	(##sys#call-with-values
	 (lambda () (proc p))
	 (lambda results
	   (close-input-pipe p)
	   (apply values results) ) ) ) ) )
  (set! call-with-output-pipe
    (lambda (cmd proc . mode)
      (let ([p (apply open-output-pipe cmd mode)])
	(##sys#call-with-values
	 (lambda () (proc p))
	 (lambda results
	   (close-output-pipe p)
	   (apply values results) ) ) ) ) )
  (set! with-input-from-pipe
    (lambda (cmd thunk . mode)
      (let ([old ##sys#standard-input]
	    [p (apply open-input-pipe cmd mode)] )
	(set! ##sys#standard-input p)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-input-pipe p)
	    (set! ##sys#standard-input old)
	    (apply values results) ) ) ) ) )
  (set! with-output-to-pipe
    (lambda (cmd thunk . mode)
      (let ([old ##sys#standard-output]
	    [p (apply open-output-pipe cmd mode)] )
	(set! ##sys#standard-output p)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-output-pipe p)
	    (set! ##sys#standard-output old)
	    (apply values results) ) ) ) ) ) )


;;; Pipe primitive:

(define-foreign-variable _pipefd0 int "C_pipefds[ 0 ]")
(define-foreign-variable _pipefd1 int "C_pipefds[ 1 ]")

(define create-pipe
    (lambda ()
      (when (fx< (##core#inline "C_pipe" #f) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not create pipe") )
      (values _pipefd0 _pipefd1) ) )


;;; Pathname operations:

(define absolute-pathname?
  (let ([rx (string-append "([A-Za-z]:)?[/" (string pathname-directory-separator) "].+")]
	[string-match string-match] )
    (lambda (pn)
      (##sys#check-string pn)
      (pair? (string-match rx pn)) ) ) )

(let ([string-append string-append]
      [absolute-pathname? absolute-pathname?]
      [pds (string pathname-directory-separator)] 
      [pes (string pathname-extension-separator)] )
  (define (conc-dirs dirs)
    (##sys#check-list dirs)
    (let loop ([strs dirs])
      (if (null? strs)
	  ""
	  (string-append (car strs) pds (loop (cdr strs))) ) ) )
  (define (_make-pathname dir file . ext)
    (let ([dirs (cond [(or (not dir) (null? dir)) ""]
		      [(string? dir) dir]
		      [else (conc-dirs dir)] ) ]
	  [file (or file "")]
	  [ext (if (pair? ext) (or (car ext) "") "")] )
      (##sys#check-string ext)
      (string-append
       dirs
       (if (and-let* ([dlen (##sys#size dirs)]
		      [(fx> dlen 0)]
		      [c0 (##core#inline "C_subchar" dirs (fx- dlen 1))] 
		      [(not (char=? pathname-directory-separator c0))] )
	     (not (char=? #\/ c0)) )
	   pds
	   "")
       file
       (if (and (fx> (##sys#size ext) 0)
		(not (char=? (##core#inline "C_subchar" ext 0) pathname-extension-separator)) )
	   pes
	   "")
       ext) ) )
  (set! make-pathname _make-pathname)
  (set! make-absolute-pathname
    (lambda (dir file . ext)
      (apply
       _make-pathname
       (let* ([dirs (cond [(or (not dir) (null? dir)) ""]
			  [(string? dir) dir]
			  [else (conc-dirs dir)] ) ]
	      [dlen (##sys#size dirs)] )
	 (if (not (absolute-pathname? dirs))
	     (string-append pds dirs)
	     dirs) )
       file
       ext) ) ) )

(define decompose-pathname
  (let* ([pes (string pathname-extension-separator)]
	 [set (string-append "/" (string pathname-directory-separator))]
	 [rx1 (string-append "^(.*[" set "])?([^" set "]+)(\\" pes "([^" set pes "]+))$")]
	 [rx2 (string-append "^(.*[" set "])?((\\" pes ")?[^" set "]+)$")] 
	 [string-match string-match] )
    (lambda (pn)
      (##sys#check-string pn)
      (let ([m (string-match rx1 pn)])
	(if m
	    (values (cadr m) (caddr m) (car (cddddr m)))
	    (let ([m (string-match rx2 pn)])
	      (if m
		  (values (cadr m) (caddr m) #f)
		  (values pn #f #f) ) ) ) ) ) ) )

(let ([decompose-pathname decompose-pathname])
  (set! pathname-directory 
    (lambda (pn)
      (let-values ([(dir file ext) (decompose-pathname pn)])
	dir) ) )
  (set! pathname-file
    (lambda (pn)
      (let-values ([(dir file ext) (decompose-pathname pn)])
	file) ) )
  (set! pathname-extension 
    (lambda (pn)
      (let-values ([(dir file ext) (decompose-pathname pn)])
	ext) ) )
  (set! pathname-strip-directory 
    (lambda (pn)
      (let-values ([(dir file ext) (decompose-pathname pn)])
	(make-pathname #f file ext) ) ) )
  (set! pathname-strip-extension 
    (lambda (pn)
      (let-values ([(dir file ext) (decompose-pathname pn)])
	(make-pathname dir file) ) ) )
  (set! pathname-replace-directory
    (lambda (pn ext)
      (let-values ([(_ file ext) (decompose-pathname pn)])
	(make-pathname dir file ext) ) ) )
  (set! pathname-replace-file
    (lambda (pn file)
      (let-values ([(dir _ ext) (decompose-pathname pn)])
	(make-pathname dir file ext) ) ) )
  (set! pathname-replace-extension
    (lambda (pn ext)
      (let-values ([(dir file _) (decompose-pathname pn)])
	(make-pathname dir file ext) ) ) ) )

(define create-temporary-file
  (let ([getenv getenv]
	[make-pathname make-pathname] 
	[string-append string-append] 
	[file-exists? file-exists?] 
	[call-with-output-file call-with-output-file] )
    (lambda ext
      (let ([dir (or (getenv "TMPDIR") (getenv "TEMP") (getenv "TMP"))]
	    [ext (if (pair? ext) (car ext) "tmp")])
	(##sys#check-string ext)
	(let loop ()
	  (let* ([n (##sys#fudge 16)]
		 [pn (make-pathname dir (string-append "t" (number->string n 16)) ext)] )
	    (if (file-exists? pn) 
		(loop)
		(call-with-output-file pn (lambda (p) pn)) ) ) ) ) ) ) )


;;; Filename globbing:

(define glob
  (let ([pattern->regexp pattern->regexp]
	[directory directory]
	[make-pathname make-pathname]
 	[decompose-pathname decompose-pathname] )
    (lambda paths
      (let conc ([paths paths])
	(if (null? paths)
	    '()
	    (let ([path (car paths)])
	      (let-values ([(dir file ext) (decompose-pathname path)])
		(let ([rx (pattern->regexp (make-pathname #f (or file "*") ext))])
		  (let loop ([f (directory (or dir "."))])
		    (cond [(null? f) (conc (cdr paths))]
			  [(string-match rx (car f)) 
			   => (lambda (m) (cons (make-pathname dir (car m)) (loop (cdr f)))) ]
			  [else (loop (cdr f))] ) ) ) ) ) ) ) ) ) )


;;; Signal processing:

(define-foreign-variable _nsig int "NSIG")
(define-foreign-variable _sigterm int "SIGTERM")
(define-foreign-variable _sigkill int "SIGKILL")
(define-foreign-variable _sigint int "SIGINT")
(define-foreign-variable _sighup int "SIGHUP")
(define-foreign-variable _sigfpe int "SIGFPE")
(define-foreign-variable _sigill int "SIGILL")
(define-foreign-variable _sigsegv int "SIGSEGV")
(define-foreign-variable _sigabrt int "SIGABRT")
(define-foreign-variable _sigtrap int "SIGTRAP")
(define-foreign-variable _sigquit int "SIGQUIT")
(define-foreign-variable _sigalrm int "SIGALRM")
(define-foreign-variable _sigvtalrm int "SIGVTALRM")
(define-foreign-variable _sigprof int "SIGPROF")
(define-foreign-variable _sigio int "SIGIO")
(define-foreign-variable _sigurg int "SIGURG")
(define-foreign-variable _sigchld int "SIGCHLD")
(define-foreign-variable _sigcont int "SIGCONT")
(define-foreign-variable _sigstop int "SIGSTOP")
(define-foreign-variable _sigtstp int "SIGTSTP")
(define-foreign-variable _sigpipe int "SIGPIPE")
(define-foreign-variable _sigxcpu int "SIGXCPU")
(define-foreign-variable _sigxfsz int "SIGXFSZ")
(define-foreign-variable _sigusr1 int "SIGUSR1")
(define-foreign-variable _sigusr2 int "SIGUSR2")
(define-foreign-variable _sigwinch int "SIGWINCH")

(define signal/term _sigterm)
(define signal/kill _sigkill)
(define signal/int _sigint)
(define signal/hup _sighup)
(define signal/fpe _sigfpe)
(define signal/ill _sigill)
(define signal/segv _sigsegv)
(define signal/abrt _sigabrt)
(define signal/trap _sigtrap)
(define signal/quit _sigquit)
(define signal/alrm _sigalrm)
(define signal/vtalrm _sigvtalrm)
(define signal/prof _sigprof)
(define signal/io _sigio)
(define signal/urg _sigurg)
(define signal/chld _sigchld)
(define signal/cont _sigcont)
(define signal/stop _sigstop)
(define signal/tstp _sigtstp)
(define signal/pipe _sigpipe)
(define signal/xcpu _sigxcpu)
(define signal/xfsz _sigxfsz)
(define signal/usr1 _sigusr1)
(define signal/usr2 _sigusr2)
(define signal/winch _sigwinch)

(let ([oldhook ##sys#interrupt-hook]
      [sigvector (make-vector 256 #f)] )
  (set! set-signal-handler!
    (lambda (sig proc . disabled)
      (##sys#check-exact sig)
      (##core#inline "C_establish_signal_handler" sig (and proc sig))
      (vector-set! 
       sigvector sig
       (if (or (not proc) (and (pair? disabled) (not (car disabled))))
	   proc
	   (lambda (n) (critical-section (proc n))) ) ) ) )
  (set! ##sys#interrupt-hook
    (lambda (reason state)
      (let ([h (##sys#slot sigvector reason)])
	(cond [h (h reason)
		 (##sys#context-switch state) ]
	      [else (oldhook reason state)] ) ) ) ) )

(define set-signal-mask!
    (lambda (sigs)
      (##sys#check-list sigs)
      (##core#inline "C_sigemptyset" 0)
      (for-each
       (lambda (s)
	 (##sys#check-exact s)
	 (##core#inline "C_sigaddset" s) )
       sigs)
      (when (fx< (##core#inline "C_sigprocmask" 0) 0)
	(##sys#update-errno)
	(##sys#error "can not set signal mask") ) ) )


;;; Set SIGINT handler:

(set-signal-handler!
 signal/int
 (lambda (n) (##sys#user-interrupt-hook))
 #t)


;;; Process handling:

(define process-fork
  (let ([fork (foreign-lambda int "C_fork")])
    (lambda thunk
      (let ([pid (fork)])
	(##sys#update-errno)
	(cond [(fx= -1 pid) (##sys#error "can not create child process")]
	      [(and (pair? thunk) (fx= pid 0)) 
	       ((car thunk))
	       ((foreign-lambda void "_exit" int) 0) ] 
	      [else pid] ) ) ) ) )

(define process-execute
  (let ([setarg (foreign-lambda int "C_set_exec_arg" int c-string)] 
	[pathname-strip-directory pathname-strip-directory] )
    (lambda (filename . arglist)
      (##sys#check-string filename)
      (let ([arglist (if (pair? arglist) (car arglist) '())])
	(##sys#check-list arglist)
	(setarg 0 (pathname-strip-directory filename))
	(do ([al arglist (cdr al)]
	     [i 1 (fx+ i 1)] )
	    ((null? al) 
	     (setarg i #f)
	     (let ([r (##core#inline "C_execvp" (##sys#make-c-string filename))])
	       (##sys#update-errno)
	       (when (fx= r -1)
		 (##sys#error "can not execute process" filename) ) ) )
	  (let ([s (car al)])
	    (##sys#check-string s)
	    (setarg i s) ) ) ) ) ) )

(define-foreign-variable _wnohang int "WNOHANG")
(define-foreign-variable _wait-status int "C_wait_status")

(define process-wait
    (lambda args
      (let-optionals* args ([pid #f] [nohang #f])
	(let ([pid (or pid -1)]
	      [options (if nohang _wnohang 0)] )
	  (##sys#check-exact pid)
	  (let* ([r (##core#inline "C_waitpid" pid options)]
		 [x (##core#inline "C_WIFEXITED" _wait-status)] )
	    (##sys#update-errno)
	    (if (fx= r -1)
		(##sys#error "waiting for child process failed" pid)
		(values 
		 r
		 x
		 (cond [x (##core#inline "C_WEXITSTATUS" _wait-status)]
		       [(##core#inline "C_WIFSIGNALED" _wait-status)
			(##core#inline "C_WTERMSIG" _wait-status) ]
		       [else (##core#inline "C_WSTOPSIG" _wait-status)] ) ) ) ) ) ) ) )

(define current-process-id (foreign-lambda int "C_getpid"))
(define parent-process-id (foreign-lambda int "C_getppid"))

(define process-signal
    (lambda (id . sig)
      (let ([sig (if (pair? sig) (car sig) _sigterm)])
	(##sys#check-exact id)
	(##sys#check-exact sig)
	(let ([r (##core#inline "C_kill" id sig)])
	  (##sys#update-errno)
	  (when (fx= r -1) (##sys#error "could not send signal to process" id sig) ) ) ) ) )

(define process-run
  (let ([process-fork process-fork]
	[process-execute process-execute] 
	[getenv getenv] )
    (lambda (f . args)
      (let ([args (if (pair? args) (car args) #f)]
	    [pid (process-fork)] )
	(cond [(not (eq? pid 0)) pid]
	      [args (process-execute f args)]
	      [else
	       (let ([shell (or (getenv "SHELL") "/bin/sh")])
		 (process-execute shell (list "-c" f)) ) ] ) ) ) ) )


;;; Getting system- and user-information:

(define-foreign-variable _uname int "C_uname")
(define-foreign-variable _uname-sysname c-string "C_utsname.sysname")
(define-foreign-variable _uname-nodename c-string "C_utsname.nodename")
(define-foreign-variable _uname-release c-string "C_utsname.release")
(define-foreign-variable _uname-version c-string "C_utsname.version")
(define-foreign-variable _uname-machine c-string "C_utsname.machine")

(define system-information
    (lambda ()
      (when (fx< _uname 0)
	(##sys#update-errno)
	(##sys#error "can not retrieve system information") )
      (values _uname-sysname
	      _uname-nodename
	      _uname-release
	      _uname-version
	      _uname-machine) ) )

(define-foreign-variable _user-name c-string "C_user->pw_name")
(define-foreign-variable _user-passwd c-string "C_user->pw_passwd")
(define-foreign-variable _user-uid int "C_user->pw_uid")
(define-foreign-variable _user-gid int "C_user->pw_gid")
(define-foreign-variable _user-gecos c-string "C_user->pw_gecos")
(define-foreign-variable _user-dir c-string "C_user->pw_dir")
(define-foreign-variable _user-shell c-string "C_user->pw_shell")

(define (user-information user)
  (let ([r (cond [(fixnum? user) (##core#inline "C_getpwuid" user)]
		 [else
		  (##sys#check-string user)
		  (##core#inline "C_getpwnam" (##sys#make-c-string user)) ] ) ] )
    (and r
	 (values _user-name
		 _user-passwd
		 _user-uid
		 _user-gid
		 _user-gecos
		 _user-dir
		 _user-shell) ) ) )


;;; More errno codes:

(define-foreign-variable _eperm int "EPERM")
(define-foreign-variable _enoent int "ENOENT")
(define-foreign-variable _esrch int "ESRCH")
(define-foreign-variable _eintr int "EINTR")
(define-foreign-variable _eio int "EIO")
(define-foreign-variable _enoexec int "ENOEXEC")
(define-foreign-variable _ebadf int "EBADF")
(define-foreign-variable _echild int "ECHILD")
(define-foreign-variable _enomem int "ENOMEM")
(define-foreign-variable _eacces int "EACCES")
(define-foreign-variable _efault int "EFAULT")
(define-foreign-variable _ebusy int "EBUSY")
(define-foreign-variable _eexist int "EEXIST")
(define-foreign-variable _enotdir int "ENOTDIR")
(define-foreign-variable _eisdir int "EISDIR")
(define-foreign-variable _einval int "EINVAL")
(define-foreign-variable _emfile int "EMFILE")
(define-foreign-variable _enospc int "ENOSPC")
(define-foreign-variable _espipe int "ESPIPE")
(define-foreign-variable _epipe int "EPIPE")
(define-foreign-variable _eagain int "EAGAIN")
(define-foreign-variable _erofs int "EROFS")

(define errno/perm _eperm)
(define errno/noent _enoent)
(define errno/srch _esrch)
(define errno/intr _eintr)
(define errno/io _eio)
(define errno/noexec _enoexec)
(define errno/badf _ebadf)
(define errno/child _echild)
(define errno/nomem _enomem)
(define errno/acces _eacces)
(define errno/fault _efault)
(define errno/busy _ebusy)
(define errno/notdir _enotdir)
(define errno/isdir _eisdir)
(define errno/inval _einval)
(define errno/mfile _emfile)
(define errno/nospc _enospc)
(define errno/spipe _espipe)
(define errno/pipe _epipe)
(define errno/again _eagain)
(define errno/rofs _erofs)


;;; Permissions and owners:

(define change-file-mode
    (lambda (fname m)
      (##sys#check-string fname)
      (##sys#check-exact m)
      (when (fx< (##core#inline "C_chmod" (##sys#make-c-string fname) m) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not change file mode" fname m) ) ) )

(define change-file-owner
    (lambda (fn uid gid)
      (##sys#check-string fn)
      (##sys#check-exact uid)
      (##sys#check-exact gid)
      (when (fx< (##core#inline "C_chown" (##sys#make-c-string fn) uid gid) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not change file owner" fn uid gid) ) ) )

(define current-user-id (foreign-lambda int "C_getuid"))
(define current-group-id (foreign-lambda int "C_getgid"))
(define current-effective-user-id (foreign-lambda int "C_geteuid"))
(define current-effective-group-id (foreign-lambda int "C_getegid"))

(define set-user-id!
    (lambda (id)
      (when (fx< (##core#inline "C_setuid" id) 0)
	(##sys#update-errno)
	(##sys#error "can not set user id" id) ) ) )

(define-foreign-variable _r_ok int "R_OK")
(define-foreign-variable _w_ok int "W_OK")
(define-foreign-variable _x_ok int "X_OK")

(let ()
  (define (check filename real acc perm)
    (##sys#check-string filename)
    (if (and (pair? real) (car real))
	(let ([r (fx= 0 (##core#inline "C_access" (##sys#make-c-string filename) acc))])
	  (unless r (##sys#update-errno))
	  r)
	(let ([v (##sys#file-info filename)])
	  (and v (not (fx= 0 (bitwise-and perm (##sys#slot v 5))))) ) ) )
  (set! file-read-access? (lambda (filename . real) (check filename real _r_ok _s_irusr)))
  (set! file-write-access? (lambda (filename . real) (check filename real _w_ok _s_iwusr)))
  (set! file-execute-access? (lambda (filename . real) (check filename real _x_ok _s_ixusr))) )

(define file-permissions
    (lambda (filename)
      (##sys#check-string filename)
      (let ([v (##sys#file-info filename)])
	(##sys#update-errno)
	(if v
	    (##sys#slot v 5)
	    (##sys#signal-hook #:file-error "can not access file permissions" filename) ) ) ) )


;;; Symbolic links:

(define create-symbolic-link
    (lambda (old new)
      (##sys#check-string old)
      (##sys#check-string new)
      (when (fx< (##core#inline "C_symlink" (##sys#make-c-string old) (##sys#make-c-string new)) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not create symbolic link" old new) ) ) )

(define-foreign-variable _filename_max int "FILENAME_MAX")

(define read-symbolic-link
  (let ([substring substring]
	[buf (make-string (fx+ _filename_max 1))] )
    (lambda (fname)
      (##sys#check-string fname)
      (let ([len (##core#inline "C_readlink" (##sys#make-c-string fname) buf)])
	(when (fx< len 0)
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not read symbolic link" fname) )
	(substring buf 0 len) ) ) ) )


;;; Using file-descriptors:

(define-foreign-variable _stdin_fileno int "STDIN_FILENO")
(define-foreign-variable _stdout_fileno int "STDOUT_FILENO")
(define-foreign-variable _stderr_fileno int "STDERR_FILENO")

(define fileno/stdin _stdin_fileno)
(define fileno/stdout _stdout_fileno)
(define fileno/stderr _stderr_fileno)

(let ([fdopen (foreign-lambda c-pointer "C_fdopen" int c-string)] )
  (define (mode outp m)
    (cond [(pair? m)
	   (let ([m (car m)])
	     (case m
	       [(###append) (if outp "a" (##sys#error "invalid mode for input file" m))]
	       [else (##sys#error "invalid mode argument" m)] ) ) ]
	  [outp "w"]
	  [else "r"] ) )
  (define (check fd outp r)
    (##sys#update-errno)
    (if (##sys#null-pointer? r)
	(##sys#signal-hook #:file-error "can not open file" fd) 
	(let ([port (##sys#make-port -1 6 #f r)])
	  (##sys#update-errno)
	  (##sys#setslot port 1 outp)
	  (##sys#setslot port 3 "(fdport)")
	  (##sys#setslot port 4 0)
	  (##sys#setslot port 5 0)
	  port) ) )
  (set! open-input-file*
    (lambda (fd . m)
      (##sys#check-exact fd)
      (check fd #f (fdopen fd (mode #f m)) ) ) )
  (set! open-output-file*
    (lambda (fd . m)
      (##sys#check-exact fd)
      (check fd #t (fdopen fd (mode #t "w")) ) ) ) )

(define port->fileno
    (lambda (port)
      (##sys#check-port port)
      (let ([fd (##core#inline "C_fileno" (##sys#port-file-resolve port))])
	(when (fx< fd 0)
	  (##sys#update-errno)
	  (##sys#error "can not access file-descriptor of port" port) )
	fd) ) )

(define duplicate-fileno
    (lambda (old . new)
      (##sys#check-exact old)
      (let ([fd (if (null? new)
		    (##core#inline "C_dup" old)
		    (let ([n (car new)])
		      (##sys#check-exact n)
		      (##core#inline "C_dup2" old n) ) ) ] )
	(when (fx< fd 0)
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not duplicate file descriptor" old new) )
	fd) ) )


;;; Other file operations:

(define file-truncate
    (lambda (fname off)
      (##sys#check-number off)
      (when (fx< (cond [(string? fname) (##core#inline "C_truncate" (##sys#make-c-string fname) off)]
		       [(fixnum? fname) (##core#inline "C_ftruncate" fname off)]
		       [else (##sys#error "invalid file" fname)] )
		 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not truncate file" fname off) ) ) )


;;; Record locking:

(define-foreign-variable _f_wrlck int "F_WRLCK")
(define-foreign-variable _f_rdlck int "F_RDLCK")
(define-foreign-variable _f_unlck int "F_UNLCK")

(let ()
  (define (setup port args)
    (let-optionals* args ([start 0]
			  [len #t] )
      (##sys#check-port port)
      (##sys#check-number start)
      (if (eq? #t len)
	  (set! len 0)
	  (##sys#check-number len) )
      (##core#inline "C_flock_setup" (if (##sys#slot port 1) _f_wrlck _f_rdlck) start len)
      (##sys#make-structure 'lock port start len) ) )
  (define (err msg lock)
    (##sys#update-errno)
    (##sys#error msg (##sys#slot lock 1) (##sys#slot lock 2) (##sys#slot lock 3)) )
  (set! file-lock
    (lambda (port . args)
      (let ([lock (setup port args)])
	(if (fx< (##core#inline "C_flock_lock" port) 0)
	    (err "can not lock file" lock)
	    lock) ) ) )
  (set! file-test-lock
    (lambda (port . args)
      (let ([lock (setup port args)])
	(cond [(##core#inline "C_flock_test" port) => (lambda (c) (and (not (fx= c 0)) c))]
	      [else (err "can not unlock file" lock)] ) ) ) ) )

(define file-unlock
    (lambda (lock)
      (##sys#check-structure lock 'lock)
      (##core#inline "C_flock_setup" _f_unlck (##sys#slot lock 2) (##sys#slot lock 3))
      (when (fx< (##core#inline "C_flock_lock" (##sys#slot lock 1)) 0)
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not unlock file" lock) ) ) )


;;; FIFOs:

(define create-fifo
    (lambda (fname . mode)
      (##sys#check-string fname)
      (let ([mode (if (pair? mode) (car mode) (bitwise-ior _s_irwxu (bitwise-ior _s_irwxg _s_irwxo)))])
	(##sys#check-exact mode)
	(when (fx< (##core#inline "C_mkfifo" (##sys#make-c-string fname) mode) 0)
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not create FIFO" fname mode) ) ) ) )

(define fifo?
    (lambda (filename)
      (##sys#check-string filename)
      (let ([v (##sys#file-info filename)])
	(##sys#update-errno)
	(if v
	    (fx= 3 (##sys#slot v 4))
	    (##sys#signal-hook #:file-error "file does not exist" filename) ) ) ) )


;;; Environment access:

(define setenv 
  (lambda (var val)
    (##sys#check-string var)
    (##sys#check-string val)
    (##core#inline "C_setenv" var val)
    (##core#undefined) ) )

(define (unsetenv var)
  (##sys#check-string var)
  (##core#inline "C_putenv" (##sys#make-c-string var))
  (##core#undefined) )

(define current-environment
  (let ([get (foreign-lambda c-string "C_getenventry" int)]
	[substring substring] )
    (lambda ()
      (let loop ([i 0])
	(let ([entry (get i)])
	  (if entry
	      (let scan ([j 0])
		(if (char=? #\= (##core#inline "C_subchar" entry j))
		    (cons (cons (substring entry 0 j) (substring entry (fx+ j 1) (##sys#size entry))) (loop (fx+ i 1)))
		    (scan (fx+ j 1)) ) )
	      '() ) ) ) ) ) )


;;; Memory mapped I/O:

(define-foreign-variable _prot_read int "PROT_READ")
(define-foreign-variable _prot_write int "PROT_WRITE")
(define-foreign-variable _prot_exec int "PROT_EXEC")
(define-foreign-variable _prot_none int "PROT_NONE")

(define prot/read _prot_read)
(define prot/write _prot_write)
(define prot/exec _prot_exec)
(define prot/none _prot_none)

(define-foreign-variable _map_fixed int "MAP_FIXED")
(define-foreign-variable _map_shared int "MAP_SHARED")
(define-foreign-variable _map_private int "MAP_PRIVATE")
(define-foreign-variable _map_anonymous int "MAP_ANONYMOUS")
(define-foreign-variable _map_file int "MAP_FILE")

(define map/fixed _map_fixed)
(define map/shared _map_shared)
(define map/private _map_private)
(define map/anonymous _map_anonymous)
(define map/file _map_file)

(define map-file-to-memory
  (let ([mmap (foreign-lambda c-pointer "mmap" c-pointer integer int int int integer)] )
    (lambda (addr len prot flag fd . off)
      (let ([addr (if (not addr) (##sys#null-pointer) addr)]
	    [off (if (pair? off) (car off) 0)] )
	(unless (and (##core#inline "C_blockp" addr) (##core#inline "C_pointerp" addr))
	  (##sys#signal-hook #:type-error "bad argument type - not a foreign pointer" addr) )
	(let ([addr2 (mmap addr len prot flag fd off)])
	  (when (eq? -1 (##sys#pointer->address addr2))
	    (##sys#update-errno)
	    (##sys#signal-hook #:file-error "can not map file to memory" addr len prot flag fd off) )
	  (##sys#make-structure 'mmap addr2 len) ) ) ) ) )

(define ummap-file-from-memory
  (let ([munmap (foreign-lambda int "munmap" c-pointer integer)] )
    (lambda (mmap . len)
      (##sys#check-structure mmap 'mmap)
      (let ([len (if (pair? len) (car len) (##sys#slot mmap 2))])
	(unless (eq? 0 (munmap (##sys#slot mmap 1) len))
	  (##sys#update-errno)
	  (##sys#signal-hook #:file-error "can not unmap file from memory" mmap len) ) ) ) ) )


;;; Time related things:

(define (seconds->local-time secs)
  (##sys#check-exact secs)
  (##sys#decode-seconds secs #f) )

(define (seconds->utc-time secs)
  (##sys#check-exact secs)
  (##sys#decode-seconds secs #t) )

(define seconds->string 
  (let ([ctime (foreign-lambda c-string "C_ctime" int)])
    (lambda (secs) 
      (let ([str (ctime secs)])
	(unless str (##sys#error "can not convert seconds to string" secs))
	str) ) ) )

(define time->string
  (let ([asctime (foreign-lambda c-string "C_asctime" scheme-object)])
    (lambda (tm)
      (##sys#check-vector tm)
      (when (fx< (##sys#size tm) 9) (##sys#error "time vector too short" tm))
      (let ([str (asctime tm)])
	(unless str (##sys#error "can not convert seconds to string" secs))
	str) ) ) )


;;; Other things:

(define _exit
  (let ([ex0 (foreign-lambda void "_exit" int)])
    (lambda code
      (##sys#cleanup-before-exit)
      (ex0 (if (pair? code) (car code) 0)) ) ) )

(define set-alarm! (foreign-lambda int "C_alarm" int))

(define-foreign-variable _iofbf int "_IOFBF")
(define-foreign-variable _iolbf int "_IOLBF")
(define-foreign-variable _ionbf int "_IONBF")
(define-foreign-variable _bufsiz int "BUFSIZ")

(define set-buffering-mode!
    (lambda (port mode . size)
      (##sys#check-port port)
      (let ([size (if (pair? size) (car size) _bufsiz)]
	    [mode (case mode
		    [(###full) _iofbf]
		    [(###line) _iolbf]
		    [(###none) _ionbf]
		    [else (##sys#error "invalid buffering-mode" mode port)] ) ] )
	(##sys#check-exact size)
	(when (fx< (##core#inline "C_setvbuf" (##sys#port-file-resolve port) mode size) 0)
	  (##sys#error "can not set buffering mode" port mode size) ) ) ) )

(define (terminal-port? port)
  (##sys#check-port port)
  (let ([fp (##sys#peek-unsigned-integer port 0)])
    (and (not (eq? fp 0)) (##core#inline "C_tty_portp" port) ) ) )

(define terminal-name
  (let ([ttyname (foreign-lambda c-string "ttyname" int)] )
    (lambda (port)
      (##sys#check-port port)
      (unless (##core#inline "C_tty_portp" (##sys#port-file-resolve port))
	(##sys#error "port is not connected to a terminal" port) )
      (ttyname (##core#inline "C_fileno" fp) ) ) ) )

(define get-host-name
  (let ([getit 
	 (foreign-lambda* c-string () 
	   "if(gethostname(C_hostbuf, 256) == -1) return(NULL);
            else return(C_hostbuf);") ] )
    (lambda ()
      (let ([host (getit)])
	(unless host
	  (##sys#update-errno)
	  (##sys#error "can not retrieve host-name") )
	host) ) ) )
