/*
 * Display an IBM-PC screen dump file on a linux console.
 * Optionally the CHOS map file information can be used
 * to show the boot screen.
 *
 * Usage:
 *    showscreen <screen dump file> [<chos map file>]
 *
 * e.g. showscreen chos.background chos.map
 *
 * V0.1 08/23/97 Bernhard Ehlers <BEhlers@t-online.de>
 *               Initial coding
 * V0.2 11/17/97 Bernhard Ehlers <BEhlers@t-online.de>
 *               Select 1. entry
 *               Show timer value
 *
 * The function 'die' is (c) Tuomo Valkonen 1996-1997.
 * See the README file in the CHOS distribution for details.
 *
 * All other stuff is public domain software, use it any way you like.
 * B U T : A B S O L U T E L Y   N O   W A R R A N T Y   ! ! !
 */
/*
 * Modified by Tuomo Valkonen to work with chos 0.73+
 */
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <fcntl.h>
#include <errno.h>
#include <chos/mapfile.h>

/*
 * Print an error code 'n exit.
 *
 * Stolen from the main.c file of the chos boot loader.
 * (c) Tuomo Valkonen 1996-1997.
 */
static void die (int err, char *parm,...)
{
  va_list args;
  va_start (args, parm);

  if (parm != NULL && err == 0)
    fprintf (stderr, "%s:unknown error!!\n", parm);
  else if (parm != NULL && err == -1)
    vfprintf (stderr, parm, args);
  else if (parm != NULL && err != 0)
    {
      vfprintf (stderr, parm, args);
      fprintf (stderr, ":%s\n", strerror (err));
    }
  else
    fprintf (stderr, "%s\n", strerror (err));

  va_end (args);

  exit (err);

}


static uchar screen[25][80][2];

/*
 * Copy a string into virtuell screen
 */
static void set_string (int x, int y, uchar color, uchar * string)
{
  while (*string != '\0')
    {
      screen[y][x][0] = *string;
      screen[y][x][1] = color;
      x++;
      string++;
    }
}

/*
 * Convert IBM-PC attributes into ansi escape sequence
 */
static void put_attr (uchar attr)
{
  static uchar col_tab[] = {0, 4, 2, 6, 1, 5, 3, 7};
  uchar col;

  putchar ('\033'); putchar ('[');
  col = col_tab[attr & 0x07];
  putchar ('3'); putchar ('0' + col); putchar (';');
  if ((attr & 0x08) == 0) putchar ('2');
  putchar ('1'); putchar (';');
  col = col_tab[(attr & 0x70) >> 4];
  putchar ('4'); putchar ('0' + col); putchar (';');
  if ((attr & 0x80) == 0) putchar ('2');
  putchar ('5');
  putchar ('m');
}

/*
 * The main program
 */
int main (int argc, char *argv[])
{
  uchar tab[256];
  MF_HeaderSector map_hdr;
  MF_ImageSector  images;
  uchar hotstring[4] = "?. ";
  uchar timeleft[8];
  int fd;
  int i, j;
  uchar attr;

  if (argc < 2 || argc > 3)
    die (-1, "usage: showscreen <screen dump file> [<chos map file>]\n");

  for (i = 0; i < 25; i++)
    for (j = 0; j < 80; j++)
      {
	screen[i][j][0] = 0;
	screen[i][j][1] = 0x07;
      }

  if ((fd = open (argv[1], O_RDONLY)) < 0)
    die (errno, "Can't open file %s", argv[1]);
  if (read (fd, screen, sizeof (screen)) < 0)
    die (errno, "Can't read file %s", argv[1]);
  close (fd);

  if (argc == 3)		/* chos map file given */
    {
      if ((fd = open (argv[2], O_RDONLY)) < 0)
	die (errno, "Can't open file %s", argv[2]);
      if ((i = read (fd, &map_hdr, sizeof (map_hdr))) != sizeof (map_hdr) ||
          (i=lseek(fd,512,SEEK_SET))!=512 ||
          (i = read (fd,&images,sizeof(images))) != sizeof(images))
	{
	  if (i >= 0)
	    die (-1, "File %s too small\n", argv[2]);
	  else
	    die (errno, "Can't read file %s", argv[2]);
	}
      close (fd);
      sprintf(timeleft, "%3d", map_hdr.delay);
      set_string (map_hdr.timex, map_hdr.timey, map_hdr.color, timeleft);
      for (i = 0; i < map_hdr.nimages; i++)
	{
	  attr = images.images[i].color;
	  if (i==0)
	    {
	      attr &= 0x0F;
	      if (attr == map_hdr.selcolor) attr |= 0x08;
	      attr |= map_hdr.selcolor << 4;
	    }
	  j = images.images[i].x;
	  if (!(images.images[i].flags & CF_HOT_HIDDEN))
	    {
	      hotstring[0] = images.images[i].hotkey;
	      set_string (j, images.images[i].y, attr, hotstring);
	      j += strlen (hotstring);
	    }
	  set_string (j, images.images[i].y, attr, images.images[i].name);
	}
    }

  /* some IBM char can't be used, e.g. bell, CR, LF, Backspace, ... */
  for (i = 0; i < 256; i++) tab[i] = i;
  tab[0] = ' ';
  for (i = 7; i < 16; i++) tab[i] = ' ';
  tab[24] = '^';
  tab[26] = '<';
  tab[27] = '>';
  tab[127] = 177;
  tab[128 + 27] = 'c';

  fputs ("\033(U\033[0m", stdout);
  for (i = 0; i < 25; i++)
    {
      attr = 0x07; put_attr (attr);
      for (j = 0; j < 80; j++)
	{
	  if (screen[i][j][1] != attr)
	    {
	      attr = screen[i][j][1];
	      put_attr (attr);
	    }
	  putchar (tab[screen[i][j][0]]);
	}
      fputs ("\033[0m", stdout);
      if (i < 24) putchar ('\n');
    }
  fputs ("\033(B", stdout); fflush(stdout);

  if (isatty(0) && isatty(1)) getchar ();

  return 0;
}
