// Copyright (c) 2010 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/http/http_auth_handler_factory.h"

#include "base/stl_util-inl.h"
#include "base/string_util.h"
#include "net/base/net_errors.h"
#include "net/http/http_auth_filter.h"
#include "net/http/http_auth_handler_basic.h"
#include "net/http/http_auth_handler_digest.h"
#include "net/http/http_auth_handler_negotiate.h"
#include "net/http/http_auth_handler_ntlm.h"

namespace net {

int HttpAuthHandlerFactory::CreateAuthHandlerFromString(
    const std::string& challenge,
    HttpAuth::Target target,
    const GURL& origin,
    const BoundNetLog& net_log,
    scoped_ptr<HttpAuthHandler>* handler) {
  HttpAuth::ChallengeTokenizer props(challenge.begin(), challenge.end());
  return CreateAuthHandler(&props, target, origin, CREATE_CHALLENGE, 1,
                           net_log, handler);
}

int HttpAuthHandlerFactory::CreatePreemptiveAuthHandlerFromString(
    const std::string& challenge,
    HttpAuth::Target target,
    const GURL& origin,
    int digest_nonce_count,
    const BoundNetLog& net_log,
    scoped_ptr<HttpAuthHandler>* handler) {
  HttpAuth::ChallengeTokenizer props(challenge.begin(), challenge.end());
  return CreateAuthHandler(&props, target, origin, CREATE_PREEMPTIVE,
                           digest_nonce_count, net_log, handler);
}

// static
HttpAuthHandlerRegistryFactory* HttpAuthHandlerFactory::CreateDefault() {
  HttpAuthHandlerRegistryFactory* registry_factory =
      new HttpAuthHandlerRegistryFactory();
  registry_factory->RegisterSchemeFactory(
      "basic", new HttpAuthHandlerBasic::Factory());
  registry_factory->RegisterSchemeFactory(
      "digest", new HttpAuthHandlerDigest::Factory());
  registry_factory->RegisterSchemeFactory(
      "negotiate", new HttpAuthHandlerNegotiate::Factory());
  registry_factory->RegisterSchemeFactory(
      "ntlm", new HttpAuthHandlerNTLM::Factory());
  return registry_factory;
}

HttpAuthHandlerRegistryFactory::HttpAuthHandlerRegistryFactory() {
}

HttpAuthHandlerRegistryFactory::~HttpAuthHandlerRegistryFactory() {
  STLDeleteContainerPairSecondPointers(factory_map_.begin(),
                                       factory_map_.end());
}

void HttpAuthHandlerRegistryFactory::SetURLSecurityManager(
    const std::string& scheme,
    URLSecurityManager* security_manager) {
  HttpAuthHandlerFactory* factory = GetSchemeFactory(scheme);
  if (factory)
    factory->set_url_security_manager(security_manager);
}

void HttpAuthHandlerRegistryFactory::RegisterSchemeFactory(
    const std::string& scheme,
    HttpAuthHandlerFactory* factory) {
  std::string lower_scheme = StringToLowerASCII(scheme);
  FactoryMap::iterator it = factory_map_.find(lower_scheme);
  if (it != factory_map_.end()) {
    delete it->second;
  }
  if (factory)
    factory_map_[lower_scheme] = factory;
  else
    factory_map_.erase(it);
}

int HttpAuthHandlerRegistryFactory::CreateAuthHandler(
    HttpAuth::ChallengeTokenizer* challenge,
    HttpAuth::Target target,
    const GURL& origin,
    CreateReason reason,
    int digest_nonce_count,
    const BoundNetLog& net_log,
    scoped_ptr<HttpAuthHandler>* handler) {
  if (!challenge->valid()) {
    handler->reset();
    return ERR_INVALID_RESPONSE;
  }
  std::string lower_scheme = StringToLowerASCII(challenge->scheme());
  FactoryMap::iterator it = factory_map_.find(lower_scheme);
  if (it == factory_map_.end()) {
    handler->reset();
    return ERR_UNSUPPORTED_AUTH_SCHEME;
  }
  DCHECK(it->second);
  return it->second->CreateAuthHandler(challenge, target, origin, reason,
                                       digest_nonce_count, net_log, handler);
}

HttpAuthHandlerFactory* HttpAuthHandlerRegistryFactory::GetSchemeFactory(
    const std::string& scheme) const {
  std::string lower_scheme = StringToLowerASCII(scheme);
  FactoryMap::const_iterator it = factory_map_.find(lower_scheme);
  if (it == factory_map_.end()) {
    return NULL;                  // |scheme| is not registered.
  }
  return it->second;
}

}  // namespace net
