/*
	$Id: soundbuffer_session.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Sound"

#ifndef header_soundbuffer_session
#define header_soundbuffer_session

#include <stdlib.h>

class CL_SoundCard;
class CL_SoundPlayBackDesc;
class CL_SoundBuffer_Session
//: CL_SoundBuffer_Session provides control over a playing soundeffect.
// Whenever a soundbuffer is played, it returns a CL_SoundBuffer_Session
// class, which can be used to control the sound (its volume, pitch, 
// pan, position). It can of course also be used to retrigger the sound
// or to stop it. The soundeffect is stored in the layer1 soundmanager
// until all references to the sound played are removed, meaning that
// the soundeffect (session) will be ready to play and remember its
// state and settings (volume, frequency...) until no CL_SoundBuffer_Session
// refers to it, via a soundbuffer session reference system.
{
public:
	CL_SoundBuffer_Session() { handle = NULL; card = NULL; }
	//: Creates a soundbuffer session not attached to any object.

	CL_SoundBuffer_Session(void *handle, CL_SoundCard *_card);

	CL_SoundBuffer_Session(const CL_SoundBuffer_Session &copy);
	//: Copy constructor.

	~CL_SoundBuffer_Session();

	CL_SoundBuffer_Session& operator=(const CL_SoundBuffer_Session &session);

	void *get_handle() const { return handle; }

	CL_SoundCard *get_card() const { return card; }
	//: Returns a pointer to the card on which the soundbuffer controlled
	//: is played/playing.
	//!retval: Pointer to soundcard

	int get_position() const;
	//: Returns the current position of the soundeffect
	//!retval: Current position of soundeffect.
	
	float get_position_relative() const;
	//: Returns the soundeffects relative position, compared
	//: to the total length of the soundeffect.
	//: The value returned will be between 0->1, where 0
	//: means the soundeffect is at the beginning, and 1
	//: means that the soundeffect has reached the end.
	//!retval: Relative position of the soundeffect.

	bool set_position(int new_pos);
	//: Sets the soundeffects position to 'new_pos'.
	//!param: The new position of the soundeffect.
	//!retval: Returns true if operation completed succesfully.
	
	bool set_position_relative(float new_pos);
	//: Sets the relative position of the soundeffect.
	//: Value must be between 0->1, where 0 sets the 
	//: soundeffect to the beginning, and 1 sets it
	//: to the end of the soundbuffer.
	//!param: New relative position
	//!retval: Returns true if operation completed succesfully.

	int get_length() const;
	//: Returns the total length of the soundeffect played.
	//: Value returned will be -1 if the length is unknown (in
	//: case of non-static soundeffects like streamed sound)
	//!retval: Length of the soundeffect.

	int get_frequency() const;
	//: Returns the frequency of the soundeffect played.
	//!retval: Frequency of soundeffect played.
	
	bool set_frequency(int new_freq);
	//: Sets the frequency of the soundeffect played.
	//!param: New frequency of soundeffect.
	//!retval: Returns true, if the operation completed succesfully.

	float get_volume() const;
	//: Returns the linear relative volume of the soundeffect.
	//: 0 means the soundeffect is muted, 1 means the soundeffect
	//: is playing at "max" volume.
	//!retval: Volume of soundeffect played.
	
	bool set_volume(float new_volume);
	//: Sets the volume of the soundeffect in a relative measure (0->1)
	//: A value of 0 will effectively mute the sound (although it will
	//: still be sampled), and a value of 1 will set the volume to "max".
	//!param: new_volume - New volume of soundeffect 
	//!retval: Returns true if the operation completed succesfully.

	float get_pan() const;
	//: Returns the current pan (in a measure from -1 -> 1), where
	//: -1 means the soundeffect is only playing in the left speaker, 
	//: and 1 means the soundeffect is only playing in the right speaker.
	//!retval: Current pan of the soundeffect played.

	bool set_pan(float new_pan);
	//: Sets the panning of the soundeffect played in measures from -1 -> 1
	//: Setting the pan with a value of -1 will pan the soundeffect to the
	//: extreme left (left speaker only), 1 will pan the soundeffect to the
	//: extreme right (right speaker only).
	//!param: new_pan - New pan of the soundeffect played.
	//!retval: Returns true if the operation completed sucecsfully.

	void play();
	//: Starts playback of the session.

	void stop();
	//: Stops playback of the session.

	bool is_playing();
	//: Returns true if the session is playing

	void set_looping(bool loop);
	//: Determines whether this session should loop
	//!param: loop - true if session should loop, false otherwise

private:
	void *handle;
	CL_SoundCard *card;
};

#endif
