/*
	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Network delivery implementation - Unix sockets version.
*/

#ifdef USE_NETWORK

#ifndef header_network_delivery_socket
#define header_network_delivery_socket

#ifdef WIN32
	#include <windows.h>
#else
	#include <sys/time.h>
	#include <sys/types.h>
	#include <sys/socket.h>
	#include <arpa/inet.h>
	#include <netinet/in.h>
	#include <unistd.h>
	#include <errno.h>
#endif

#include <string.h>
#include <list>

#include "Core/Network/Generic/network_delivery_impl.h"

class CL_Connections_Unix;
class CL_UDPConnection;

class CL_UniformSocket : public CL_Connection
{
public:
	virtual int get_socket() { return sock; }

	CL_UniformSocket(CL_ConnectionProvider *provider);
	virtual ~CL_UniformSocket();

	// CL_UniformSocket defined virtual function
	virtual bool try_connect(unsigned long remote_ip_network_format, int port);
	virtual bool init_socket(int init_socket=-1);

	// CL_Connection inherited functions
	virtual bool peek();
	virtual CL_ConnectionPacket receive();
	virtual void send(CL_ConnectionPacket message);
	virtual bool connection_lost();
protected:

#ifdef WIN32
	SOCKET sock;
#else
	int sock;
#endif
	bool is_connection_lost;

	char *cur_message;
	int cur_message_size;

	unsigned long read_int();
	void write_int(unsigned long value);
	inline void write_data(void *data, unsigned int size);

	// read from socket:
	enum
	{
		expect_magic,
		expect_packet_size,
		expect_packet_data,
		packet_finished
	} recv_state;

	bool read_avail();
	bool require_avail(unsigned int size);
	bool get_avail(void *buf, unsigned int size);

	CL_ConnectionProvider *provider;

private:
//	std::deque<char> recv_buffer;
	std::string recv_buffer;
};

class CL_UniformAcceptSocket : public CL_UniformSocket
{
public:
	CL_UniformAcceptSocket(CL_ConnectionProvider *provider);
	virtual ~CL_UniformAcceptSocket();

	bool bind(int port);

	CL_UniformSocket *accept();
	int get_port() { return port; }

	// CL_UniformSocket inherited functions
	virtual bool try_connect(unsigned long remote_ip_network_format, int port);

	// CL_Connection inherited functions
	virtual bool peek();
	virtual CL_ConnectionPacket receive();
	virtual void send(CL_ConnectionPacket message);
	virtual bool connection_lost();

private:
	bool is_inited;
	int port;
};


class CL_UniformUDPConnection : public CL_UDPConnection
{
public:
	virtual int get_socket() { return sock; }

	CL_UniformUDPConnection();
	virtual ~CL_UniformUDPConnection();

	bool bind(unsigned int port);

	// CL_UDPConnection inherited functions
	virtual bool peek();
	virtual CL_UDPConnectionPacket receive();
	virtual void send(CL_UDPConnectionPacket message);
	virtual void broadcast(CL_UDPConnectionPacket message);
	virtual unsigned int get_port();

protected:
	int sock;
	int port;
};

#endif

#endif // USE_NETWORK
