(*                             colllex.ml

    This file is part of COLLATINUS

    COLLATINUS is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    COLLATINUS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with COLLATINUS; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    (c) Yves Ouvrard, Angoulme, 2005 - 2006
*)

let destroy () = GMain.Main.quit ()

(* indice c liste n  renvoie l'indice de la chane c 
   dans la liste liste, de longueur n *)
let rec indice c liste n =
      match liste with
         t::q when t = c -> string_of_int (n - (List.length liste))
        |t::q when t <> c -> indice c q n
        |_ -> "99"

let readfile cin =
    let rec aux acc = try aux ((input_line cin)::acc)
    with End_of_file -> close_in cin; List.rev acc
    in aux [];;       

class tentree l = 
    object (self)
       val mutable k = ""
       val mutable p = 0 
       val mutable rp = ""
       val mutable rs = ""
       val mutable g = ""
       method k = k
       method p = p
       method rp = rp
       method rs = rs
       method g = g

       method set_rp v = rp <- v
       method linea = Printf.sprintf "%s|%d|%s|%s|%s\n" k p rp rs g 
    initializer
       let tabula = Str.split (Str.regexp "|") l in
          k <- List.hd tabula; 
          let tabula = List.tl tabula in
          p <- int_of_string (List.hd tabula);
          let tabula = List.tl tabula in
          rp <- List.hd tabula;
          let tabula = List.tl tabula in
          rs <- List.hd tabula;
          let tabula = List.tl tabula in
          g <- List.hd tabula
    end

let modeles = [|
      "uita"; "amicus"; "puer"; "ager"; "templum"; 
      "miles"; "ciuis"; "corpus"; "mare"; "manus"; 
      "res"; "bonus"; "miser"; "pulcher"; "fortis"; 
      "uetus"; "acer"; "amo"; "moneo"; "lego"; 
      "capio"; "audio"; "sum"; "eo"; "imitor"; 
      "uereor"; "sequor"; "patior"; "potior"; "pronoms"; 
      "invaria" |]

let capsa = open_in Sys.argv.(1)
let incipit_d = "---desinentiae---"
let lexicum = Hashtbl.create 10000

let charge () = 
    let d = ref true in
       while !d do 
           let l = input_line capsa in
           d := (l <> incipit_d);
           if !d then
                let e = new tentree l in
                Hashtbl.add lexicum e#k e;
       done

let lex_scribe f cauda =
    (* try 
    let cauda = readfile capsa in
    with Sys_error -> (); *)
    let noua = open_out f in
    Hashtbl.iter (fun c v ->
       output_string noua v#linea;
       ) lexicum;
    output_string noua "---desinentiae---\n" ;
    List.iter (fun l -> output_string noua (l^"\n")) cauda;
    flush noua;
    close_out noua

let _ = charge ()
let cauda = readfile capsa 

let main () =

   let window = GWindow.window 
   ~title: "Collatini lexicum"
   ~border_width: 10 () in
   
   let to_utf8 msg =
      Glib.Convert.convert msg ~to_codeset:"UTF-8" ~from_codeset:"ISO-8859-1" in

   let to_text msg =
      Glib.Convert.convert msg ~to_codeset:"ISO-8859-1" ~from_codeset:"UTF-8" in

   let box = GPack.vbox ~packing: window#add () in

   let boxquaest = GPack.hbox ~packing: box#add () in

   let _ = GMisc.label
      ~text: "Quod uerbum ?"
      ~packing: boxquaest#add
      () in

   let equaest = GEdit.entry
      ~packing: boxquaest#add
      () in

   let tabula = GPack.table
      ~columns: 5
      ~rows: 2
      ~packing: box#add
      () in

   let luerbum = GMisc.label
      ~text: "uerbi lemma"
      () in
   tabula#attach
       ~left: 0
       ~top: 0
       (luerbum#coerce);
   let euerbum = GEdit.entry
      () in
   tabula#attach
      ~left: 0 
      ~top: 1
      (euerbum#coerce);

   let lrprima = GMisc.label
      ~text: "radix prima"
      () in
   tabula#attach
       ~left: 1
       ~top: 0
       (lrprima#coerce);
   let erprima = GEdit.entry
      () in
   tabula#attach
      ~left: 1 
      ~top: 1
      (erprima#coerce);

   let lrsecunda = GMisc.label
      ~text: "radix secunda"
      () in
   tabula#attach
       ~left: 2
       ~top: 0
       (lrsecunda#coerce);
   let ersecunda = GEdit.entry
      () in
   tabula#attach
      ~left: 2 
      ~top: 1
      (ersecunda#coerce);


   (* paradigmata *)
   let lparadig = GMisc.label
      ~text: "paradigma"
      () in
   tabula#attach
      ~left: 3
      ~top: 0
      (lparadig#coerce);

   let combo_par = GEdit.combo
       ~popdown_strings: (Array.to_list modeles)
   () in
   tabula#attach
      ~left: 3
      ~top: 1
      (combo_par#coerce);
   
   (* uernaculus sermo *)
   let luernac = GMisc.label
      ~text: " morphologia : traductio in uernaculum sermonem"
      () in
   tabula#attach
      ~left: 4
      ~top: 0
      (luernac#coerce);
   let euernac = GEdit.entry
      () in
   tabula#attach
      ~left: 4
      ~top: 1
      (euernac#coerce);

   let boximper = GPack.hbox ~packing: box#add () in

   let bmuta = GButton.button
      ~label: "Muta aut adde"
      ~packing: boximper#add 
      () in

   let bdele = GButton.button
      ~label: "Dele"
      ~packing: boximper#add
      () in

   let bexit = GButton.button 
      ~label: "Exit"
      ~packing: boximper#add 
      () in

   ignore (equaest#connect#changed ~callback: (fun () -> 
      try
        let e = Hashtbl.find lexicum equaest#text in
        euerbum#set_text e#k;
        erprima#set_text e#rp;
        ersecunda#set_text e#rs;
        combo_par#entry#set_text modeles.(e#p);
        euernac#set_text (to_utf8 e#g)
      with Not_found -> (
        euerbum#set_text equaest#text; 
        erprima#set_text "";
        ersecunda#set_text "";
        combo_par#entry#set_text "";
        euernac#set_text "";)
      ));

   ignore (bmuta#connect#clicked ~callback: (fun () -> 
       let nm = indice combo_par#entry#text (Array.to_list modeles) (Array.length modeles) in 
       let lne = String.concat "|" 
             [euerbum#text; nm; erprima#text; ersecunda#text; (to_text euernac#text)] in 
          try
             let ne = new tentree lne in 
             Hashtbl.replace lexicum equaest#text ne;
             lex_scribe "lemmata.noua" cauda
          with Failure "int_of_string" -> 
             Pervasives.prerr_endline lne));

   ignore (bdele#connect#clicked ~callback: (fun () -> 
        let _ = Hashtbl.find lexicum equaest#text in
        try
           Hashtbl.remove lexicum equaest#text;
           lex_scribe "lemmata.noua" cauda
        with Not_found -> () 
        ));

   ignore (bexit#connect#clicked ~callback: window#destroy);
   ignore (window#connect#destroy ~callback: destroy);
   window#show ();
   (* charge (); *)
   GMain.Main.main ()

let _ = main()   
