# smooth.rb : A simple averageing filter
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

module SciYAG

  module Backends

    # A filter that sorts data and then averages the Y values of all the
    # elements which have the same X value.
    class AverageDupFilter < Filter
      include Dobjects

      describe "avgdup", "Average duplicates",
      "Sorts the points according to the X values and average successive "+
        "points with identical X values"

      def initialize
      end

      # There you go: a simple sorting/averaging filter.
      def apply(f)
        h = {}
        for x,y in f
          if h.key? x
            h[x] << y
          else
            h[x] = [y]
          end
        end
        
        nx = Dvector.new
        ny = Dvector.new
        for x in h.keys.sort
          nx << x
          y = 0
          for a in h[x]
            y += a
          end
          y/= h[x].size
          ny << y
        end
        return Function.new(nx,ny)
      end

      # We simply replace the old vectors by the new ones.
      def apply!(f)
        new_f = apply(f)
        f.x.replace(new_f.x)
        f.y.replace(new_f.y)
      end
    end

    class StandardDeviationFilter < Filter
      include Dobjects

      describe "stddev", "Compute standard deviation",
      "Averages successive elements with the same X value and compute standard deviation"

      def initialize
      end

      def apply(f)
        last_x = nil    
        current_y = []
        x_values = Dvector.new
        y_values = []
        f.x.size.times do |i|
          xval = f.x[i]
          if (last_x && xval == last_x)
            current_y << f.y[i]
          else
            if last_x
              y_values << current_y
              x_values << last_x
            end
            current_y = [f.y[i]]
            last_x = f.x[i]
          end
        end
        if last_x
          y_values << current_y
          x_values << last_x
        end

        target_y = Dvector.new(x_values.size)
        ymin = target_y.dup
        ymax = target_y.dup
        x_values.size.times do |i|
          sum = 0.0
          sq_sum = 0.0
          for y in y_values[i]
            sum += y
            sq_sum += y**2
          end
          nb = y_values[i].size
          target_y[i] = sum/nb
          stddev = Math::sqrt((sq_sum/nb) - target_y[i]**2)
          ymin[i] = target_y[i] - stddev
          ymax[i] = target_y[i] + stddev
        end
        
        errors = f.errors.dup.merge!({:ymin => ymin, :ymax => ymax,
                                     :x => x_values, :y => target_y})

        return DataSet2D.new(f.creation_context, 
                             Function.new(x_values, target_y), 
                             errors, f.meta_data) 
      end

      # We simply replace the old vectors by the new ones.
      def apply!(f)
        new_f = apply(f)
        f.x.replace(new_f.x)
        f.y.replace(new_f.y)
        f.errors = new_f.errors
      end
    end

  end
end
