# dataset.rb : The implementation of data sets
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

require 'MetaBuilder/metabuilder'

require 'forwardable'

module SciYAG

  module Backends


    # An abstract class representing a DataSet. You should consider using
    # a subclass, DataSet2D or DataSet3D. A DataSet must be either 2D or 3D,
    # and redefine #is_2D? and #is_3D? accordingly.
    #
    # As DataSets grow more complex, it is likely that it will be less and
    # less easy to tweak directly the data inside. So, please, do use the
    # accessors and tweakers provided in the interface, else you'll expose
    # yourself to some intensive breakages when I get more clever.
    # Meanwhile, the apply(meth,where,*rest) method should be used to keep
    # everything in sync when you apply a method to a Dvector. 
    class DataSet

      extend Forwardable

      # The #creation_context attribute is a hash containing a :backend key
      # indicating the name of the Backend created and filled with the
      # contents of the backend's DescriptionInclude#save_state function.
      attr_accessor :creation_context

      # The data of the set. 
      attr_accessor :data

      # Errors on data, if applicable.
      # For 2D data, this will just be a hash with the following Dvectors:
      #  :xmin, :xmax for errors on the x values + :x, to make sure
      #  we keep it up-to-date.
      #  :ymin, :ymax for errors on y  + :y.
      #
      # Both ?min and ?max have to be specified if any output should be
      # created.
      attr_accessor :errors

      # The metadata given when the set was created.
      attr_accessor :meta_data

      def is_2D?
        return false
      end

      def is_3D?
        return false
      end

      # A few forwarded stuff to make the DataSet behave exactly like
      # Function and it's future 3D counterpart

      def_delegators :@data, :x, :y, :z

      def initialize(backend, data, errors = {}, meta_data = {})
        @creation_context = backend.save_state
        @creation_context[:backend] = backend.description.name
        @data = data
        @errors = errors
        @meta_data = meta_data
      end

      # Apply a Dvector operation to the given dimension, including
      # everything that could be left somewhere in the error bars.
      # It should be very powerful in the end.
      #
      # Beautiful, isn't it ??
      def apply(what, where, *rest)
        self.send(where).send(what, *rest)
        for key,values in @errors
          if key.to_s =~ /^#{where}/  # Very much overkill, but, well...
            values.send(what,*rest)
          end
        end
      end

      # Does a redirection to the underlying @data if that makes sense.
      def method_missing(sym, *args, &b)
        if @data.respond_to?(sym)
          @data.send(sym, *args, &b)
        elsif x.respond_to?(sym)  # We are trying to apply something
          # like data.mul!(:x, factor)
          apply(sym,args.shift,*args,&b)
        else
          super
        end
      end

    end

    # A 2-dimensionnal DataSet
    class DataSet2D < DataSet

      def is_2D?
        return true
      end
    end

    # A 3-dimensionnal DataSet
    class DataSet3D < DataSet
      def is_3D?
        return true
      end
    end


  end
end
