/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 * 
 *  Copyright (C) 2000-2002  Florian Lohoff <flo@rfc822.org>
 *  Copyright (C) 2002,2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 */

#include <ext2fs/ext2_fs.h>
/* ext2fs.h is b0rken, and we don't want to drag everything in. */
#define FILE int
#define _LINUX_TYPES_H
#include <ext2fs/ext2fs.h>
#include <et/com_err.h>

/* Hack! */
#define SIZE_T_DEFINED

#include "delo.h"
#include "stringops.h"

static int partition_start;
static ext2_filsys fs = 0;

static errcode_t delo_ext2_set_blksize(io_channel channel, int blksize);
static errcode_t delo_ext2_open(const char *name, int flags,
				io_channel *channel);
static errcode_t delo_ext2_read_blk(io_channel channel, unsigned long block,
				    int count, void *data);
static errcode_t delo_ext2_close(io_channel channel);
static errcode_t delo_ext2_write_blk(io_channel channel, unsigned long block,
				     int count, const void *data);
static errcode_t delo_ext2_flush(io_channel channel);

static struct struct_io_manager delo_ext2_io_manager = {
	EXT2_ET_MAGIC_IO_MANAGER,
	"delo i/o manager",
	delo_ext2_open,
	delo_ext2_close,
	delo_ext2_set_blksize,
	delo_ext2_read_blk,
	delo_ext2_write_blk,
	delo_ext2_flush,
	NULL,
	{0, }
};

static errcode_t delo_ext2_open(const char *name,
				UNUSED int flags,
				io_channel *channel)
{
	io_channel	io;	
	int		retval;

	dprintf("delo_ext2_open: called\n");
	retval = bootinit(NULL);
	dprintf("delo_ext2_open: bootinit returned %d\n", retval);
	if (retval)
		return EXT2_ET_BAD_DEVICE_NAME;

	/* This cant fail :) */
	io = (io_channel)malloc(sizeof (struct struct_io_channel));

	memset(io, 0, sizeof(struct struct_io_channel));

	io->magic = EXT2_ET_MAGIC_IO_CHANNEL;
	io->manager = &delo_ext2_io_manager;
	io->name = (char *)malloc(strlen(name) + 1);
 	strcpy(io->name, name);
	io->block_size = SECTOR_SIZE;
	io->read_error = 0;
	io->write_error = 0;

	*channel = io;
	return 0;
}

static errcode_t delo_ext2_close(UNUSED io_channel channel)
{
	dprintf("delo_ext2_close: called\n");
	return 0;
}

static errcode_t delo_ext2_set_blksize(io_channel channel, int blksize)
{
	channel->block_size = blksize;
	dprintf("delo_ext2_set_blksize: called %d\n", blksize);
	return 0;
}

static errcode_t delo_ext2_read_blk(io_channel channel, unsigned long block,
				    int count, void *data)
{
	int offset;

	/* FIXME: This is our partition start */
	offset = block * (channel->block_size / SECTOR_SIZE)
		+ partition_start;

	/* If it is negative it means bytes -- positive -> blocks */
	count = (count < 0) ? -count : count * channel->block_size;

	dprintf("delo_ext2_read_blk: called for %d count %d to %p\n",
		offset, count, data);

	if (bootread(offset, data, count) != count)
		return EXT2_ET_SHORT_READ;
	
	return 0;
}

static errcode_t delo_ext2_write_blk(UNUSED io_channel channel,
				     UNUSED unsigned long block,
				     UNUSED int count,
				     UNUSED const void *data)
{
	dprintf("delo_ext2_write_blk: called");
	return 0;
}

static errcode_t delo_ext2_flush(UNUSED io_channel channel)
{
	dprintf("delo_ext2_flush: called");
	return 0;
}

static void ext2fs_error(char *funcname, int errcode)
{
	com_err(funcname, errcode, NULL);
}

/*
 * Get a extent list from an ext2/ext3 filesystem
 */
static blk_t oldblocknr;
static int extent;
static int ext_cnt;

static int ext2_build_extent_list(ext2_filsys fs, blk_t *blocknr,
				  UNUSED int blockcnt,
				  UNUSED void *private)
{
	if (*blocknr != oldblocknr + 1) {
		if (oldblocknr) {
			if ((extentlist_add_tail(extent, ext_cnt))) {
				extentlist_clear();
				return BLOCK_ABORT;
			}
		}
		extent = (*blocknr * (fs->io->block_size / SECTOR_SIZE)
			  + partition_start);
		ext_cnt = fs->io->block_size / SECTOR_SIZE;
	} else
		ext_cnt += fs->io->block_size / SECTOR_SIZE;

	oldblocknr = *blocknr;

	return 0;
}

int readext2blocks(char *partition, char *file)
{
	int retval;
	ext2_ino_t root = EXT2_ROOT_INO;
	ext2_ino_t inode = 0;
	char *block_buf;

	dprintf("Getting partition info\n");
	partition_start = getpartoffset(partition);
	if (!partition_start) {
		printf("Couldn't find partition %s\n", partition);
		return 1;
	}

	retval = ext2fs_open(partition, 0, 0, 0, &delo_ext2_io_manager, &fs);
	if (retval) {
		ext2fs_error("extfs_open", retval);
		return 1;
	}

	retval = ext2fs_namei_follow(fs, root, root, file, &inode);
	if (retval) {
		ext2fs_error("ext2fs_namei_follow", retval);
		return 1;
	}

	/* Build extent list */
	extentlist_clear();
	block_buf = (char *)malloc(3 * fs->io->block_size);
	oldblocknr = 0;
	retval = ext2fs_block_iterate(fs, inode, BLOCK_FLAG_DATA_ONLY,
				      block_buf, ext2_build_extent_list, 0);
	if (retval) {
		ext2fs_error("ext2fs_block_iterate", retval);
		return 1;
	}
	/* Add last extent */
	if (oldblocknr) {
		oldblocknr = 0;
		if ((extentlist_add_tail(extent, ext_cnt))) {
			extentlist_clear();
			return 1;
		}
	}

	return 0;
}
