/* selectops.c
 * operations for selecting, cutting, copying, and pasting music
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001 Matthew Hiller */

#include "calculatepositions.h"
#include "commandfuncs.h"
#include "datastructures.h"
#include "draw.h"
#include "objops.h"
#include "measureops.h"
#include "selectops.h"
#include "staffops.h"
/*For save selection function*/
#include "frogio.h"

/* The copy buffer is a GList of objnode *s -- at first, I was going
 * to use staffnode *s, measurenode *s, and then objnode *s, but I
 * realized that'd be overkill and just complicate the implementation
 * unnecessarily.
 *
 * Each item in the copybuffer list corresponds to the stuff in
 * the buffer on each staff.  */

static GList *copybuffer = NULL;

static gint staffsinbuffer = 0;
static gint measurebreaksinbuffer = 0;

void
clearbuffer ()
{
  g_list_foreach (copybuffer, freeobjlist, NULL);
  g_list_free (copybuffer);
  copybuffer = NULL;
  staffsinbuffer = 0;
  measurebreaksinbuffer = 0;
}

void
saveselection (struct scoreinfo *si)
{
/*  staffnode *curstaff;
  measurenode *curmeasure;
  objnode *curobj;
  objnode *theobjs;
  mudelaobject *clonedobject;
  gint i, j, k;*/

  if (si->markstaffnum == 0)	/* Indicator that there's no selection.  */
    return;

  clearbuffer ();

  staffsinbuffer = si->laststaffmarked - si->firststaffmarked + 1;

  copytobuffer (si);
  si->savebuffer = copybuffer;

  clearbuffer ();

}

void
copytobuffer (struct scoreinfo *si)
{
  staffnode *curstaff;
  measurenode *curmeasure;
  objnode *curobj;
  objnode *theobjs;
  mudelaobject *clonedobject;
  gint i, j, k;

  if (si->markstaffnum == 0)	/* Indicator that there's no selection.  */
    return;

  clearbuffer ();

  staffsinbuffer = si->laststaffmarked - si->firststaffmarked + 1;
  /* Staff loop.  */
  for (i = si->firststaffmarked, curstaff = g_list_nth (si->thescore, i - 1);
       curstaff && i <= si->laststaffmarked; curstaff = curstaff->next, i++)
    {
      /* Initialize first ->data for copybuffer to NULL.  */
      theobjs = NULL;
      /* Measure loop.  */
      for (j = si->firstmeasuremarked, k = si->firstobjmarked,
	   curmeasure = g_list_nth (firstmeasurenode (curstaff), j - 1);
	   curmeasure && j <= si->lastmeasuremarked;
	   curmeasure = curmeasure->next, j++)
	{
	  for (curobj = g_list_nth (curmeasure->data, k);
	       /* cursor_x is 0-indexed */
	       curobj && (j < si->lastmeasuremarked
			  || k <= si->lastobjmarked);
	       curobj = curobj->next, k++)
	    {
	      clonedobject = clone_object (curobj->data);
	      theobjs = g_list_append (theobjs, clonedobject);
	    }			/* End object loop */
	  if (j < si->lastmeasuremarked || k < si->lastobjmarked
	      || staffsinbuffer > 1)
	    {
	      /* That is, there's another measure, the cursor is in appending
	         position, or the selection spans multiple staffs, in which case
	         another measure boundary should be added.  */
	      theobjs = g_list_append (theobjs, newmeasurebreakobject ());
	      if (i == si->firststaffmarked)
		measurebreaksinbuffer++;
	    }
	  k = 0;		/* Set it for next run through object loop */
	}			/* End measure loop */
      copybuffer = g_list_append (copybuffer, theobjs);
    }				/* End staff loop */
}

void
cuttobuffer (struct scoreinfo *si)
{
  staffnode *curstaff;
  measurenode *curmeasure;
  objnode *tempobj;
  gint i, j, max;

  copytobuffer (si);

  if (staffsinbuffer == 1)
    {
      /* Just a single staff is a special case, again.  */
      j = si->firstmeasuremarked;
      curmeasure = g_list_nth (firstmeasurenode (si->currentstaff), j - 1);

      /* Clear the relevant part of the first measure selected */
      if (measurebreaksinbuffer)
	max = G_MAXINT;
      else
	max = si->lastobjmarked;
      for (i = si->firstobjmarked;
	   ((tempobj = g_list_nth (curmeasure->data, si->firstobjmarked))
	    && i <= max); i++)
	{
	  curmeasure->data = g_list_remove_link (curmeasure->data, tempobj);
	  freeobject (tempobj->data);
	  g_list_free_1 (tempobj);
	}
      j++;
      curmeasure = curmeasure->next;

      if (!si->thescore->next)
	{
	  /* That is, the score has only this one staff */
	  curmeasure = removemeasures (si, j - 1, measurebreaksinbuffer - 1);
	  j += measurebreaksinbuffer - 1;
	}
      else
	for (; curmeasure && j < si->lastmeasuremarked;
	     curmeasure = curmeasure->next, j++)
	  {
	    freeobjlist (curmeasure->data, NULL);
	    curmeasure->data = NULL;
	  }
      /* Now clear the relevant part of the last measure selected */
      if (j <= si->lastmeasuremarked)
	{
	  for (i = 0; curmeasure->data && i <= si->lastobjmarked; i++)
	    {
	      tempobj = curmeasure->data;
	      curmeasure->data =
		g_list_remove_link (curmeasure->data, tempobj);
	      freeobject (tempobj->data);
	      g_list_free_1 (tempobj);
	    }
	  /* And delete it, if the measure's been cleared and there's only
	     one staff.  */
	  if (!curmeasure->data && !si->thescore->next)
	    removemeasures (si, j - 1, 1);
	}
      showwhichaccidentalswholestaff (si->currentstaff->data);
      beamsandstemdirswholestaff (si->currentstaff->data);
    }
  else
    {				/* Multiple staff selection */
      if (staffsinbuffer == g_list_length (si->thescore))
	{
	  /* Every staff was part of the selection */
	  removemeasures (si, si->firstmeasuremarked - 1,
			  measurebreaksinbuffer);
	  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
	    {
	      showwhichaccidentalswholestaff (curstaff->data);
	      beamsandstemdirswholestaff (curstaff->data);
	    }
	}
      else
	{
	  /* Staff loop */
	  for (i = si->firststaffmarked,
	       curstaff = g_list_nth (si->thescore, i - 1);
	       curstaff && i <= si->laststaffmarked;
	       curstaff = curstaff->next, i++)
	    {
	      /* Measure loop */
	      for (j = si->firstmeasuremarked,
		   curmeasure = g_list_nth (firstmeasurenode (curstaff),
					    j - 1);
		   curmeasure && j <= si->lastmeasuremarked;
		   curmeasure = curmeasure->next, j++)
		{
		  freeobjlist (curmeasure->data, NULL);
		  curmeasure->data = NULL;
		}
	      showwhichaccidentalswholestaff (curstaff->data);
	      beamsandstemdirswholestaff (curstaff->data);
	    }
	}
    }
  si->markstaffnum = 0;
  /* And set some currents. This would probably be better to split off
   * into a more-generalized version of setcurrents or something;
   * what's here is more-or-less copied from deleteobject in
   * commandfuncs */

  si->currentmeasurenum = si->firstmeasuremarked;
  si->currentmeasure = g_list_nth (firstmeasurenode (si->currentstaff),
				   si->currentmeasurenum - 1);
  si->cursor_x = si->firstobjmarked;
  if (si->cursor_x < g_list_length (si->currentmeasure->data))
    {
      si->currentobject = g_list_nth (si->currentmeasure->data, si->cursor_x);
      si->cursor_appending = FALSE;
    }
  else
    {
      si->currentobject = g_list_last (si->currentmeasure->data);
      si->cursor_appending = TRUE;
    }
  /*   isoffleftside;  */
  find_xes_in_all_measures (si);
  nudgerightward (si);
  gtk_widget_draw (si->scorearea, NULL);
}

/* The updates that are done towards the bottom of this function -
 * beamsandstemdirswholestaff, find_xes_in_all_measures, etc. - are
 * too much gruntwork and are inefficient in terms of everything
 * but additional lines-of-code required for implementation. */

void
pastefrombuffer (struct scoreinfo *si)
{
  staffnode *curstaff;
  measurenode *curmeasure;
  GList *curbuffernode = copybuffer;
  objnode *curbufferobj;
  gint insertat, initialinsertat;
  mudelaobject *clonedobject;
  /*   gboolean clefflag = FALSE;
     gboolean timesigflag = FALSE;
     gboolean keysigflag = FALSE; */
  gint i, initialj, j;
  gint measurestoadd = 0;

  /* First, check to see if the insertion is taking place only in
   * blank measures. If not, insert as many measures as is appropriate. */

  initialj = (staffsinbuffer == 1 && si->cursor_x);
  for (i = 0, curstaff = si->currentstaff;
       curstaff && i < staffsinbuffer; curstaff = curstaff->next, i++)
    for (j = initialj,
	 curmeasure = g_list_nth (firstmeasurenode (curstaff),
				  si->currentmeasurenum - 1 + j);
	 curmeasure && j < measurebreaksinbuffer;
	 curmeasure = curmeasure->next, j++)
      if (curmeasure->data)
	{			/* A-ha! This measure has something in it! */
	  measurestoadd = measurebreaksinbuffer - initialj;
	  goto out;		/* Multilevel break, essentially */
	}
out:
  if (measurestoadd > 0)
    addmeasures (si, si->currentmeasurenum - 1 + initialj, measurestoadd);

  /* All right. Any necessary measures have been inserted - now paste away */

  if (staffsinbuffer == 1)
    initialinsertat = si->cursor_x;
  else
    initialinsertat = 0;

  for (curstaff = si->currentstaff; curstaff && curbuffernode;
       curstaff = curstaff->next, curbuffernode = curbuffernode->next)
    {
      curmeasure = g_list_nth (firstmeasurenode (curstaff),
			       si->currentmeasurenum - 1);
      insertat = initialinsertat;
      for (curbufferobj = curbuffernode->data; curbufferobj && curmeasure;
	   curbufferobj = curbufferobj->next)
	{
	  if (((mudelaobject *) curbufferobj->data)->type == MEASUREBREAK)
	    {
	      curmeasure = curmeasure->next;
	      insertat = 0;
	    }
	  else
	    {
	      clonedobject = clone_object (curbufferobj->data);
	      curmeasure->data = g_list_insert (curmeasure->data,
						clonedobject, insertat);
	      insertat++;
	    }
	}			/* End bufferobj loop */
      showwhichaccidentalswholestaff (curstaff->data);
      beamsandstemdirswholestaff (curstaff->data);
    }				/* End staff loop */
  find_xes_in_all_measures (si);
  nudgerightward (si);
  si->haschanged = TRUE;
  si->currentmeasure = g_list_nth (firstmeasurenode (si->currentstaff),
				   si->currentmeasurenum - 1);
  si->currentobject = g_list_nth (si->currentmeasure->data, initialinsertat);
  if (!si->currentobject)
    /* Yah. There wasn't anything in the buffer after all. */
    si->cursor_appending = TRUE;
  else
    si->cursor_appending = FALSE;
  gtk_widget_draw (si->scorearea, NULL);
}

void
set_mark (struct scoreinfo *si)
{
  si->markstaffnum = si->currentstaffnum;
  si->markmeasurenum = si->currentmeasurenum;
  si->markcursor_x = si->cursor_x;
  calcmarkboundaries (si);
}

void
unset_mark (struct scoreinfo *si)
{
  si->markstaffnum = 0;
  calcmarkboundaries (si);
}

void
copywrapper (gpointer data, guint callback_action, GtkWidget * widget)
{
  copytobuffer (data);
}

void
cutwrapper (gpointer data, guint callback_action, GtkWidget * widget)
{
  cuttobuffer (data);
}

void
pastewrapper (gpointer data, guint callback_action, GtkWidget * widget)
{
  pastefrombuffer (data);
}



void
saveselwrapper (gpointer data, guint callback_action, GtkWidget * widget)
{
  saveselection (data);
}

void
mark_boundaries_helper (struct scoreinfo *si, gint mark_staff,
			gint mark_measure, gint mark_object, gint point_staff,
			gint point_measure, gint point_object,
			enum drag_selection_type type)
{
  if (mark_staff)
    {
      si->firststaffmarked = MIN (mark_staff, point_staff);
      si->laststaffmarked = MAX (mark_staff, point_staff);

      switch (type)
	{
	case NO_DRAG:
	  /* error, really.  */
	  break;
	case NORMAL_SELECT:
	case WHOLE_MEASURES:
	  /* I was thinking of handling these with a fallthrough, but
	     the commonality in setting si->firstmeasuremarked and
	     si->lastmeasuremarked caused it not to work out cleanly.  */
	  si->firstmeasuremarked = MIN (mark_measure, point_measure);
	  si->lastmeasuremarked = MAX (mark_measure, point_measure);
	  if (type == NORMAL_SELECT
	      && si->firststaffmarked == si->laststaffmarked)
	    {
	      if (mark_measure < point_measure)
		{
		  si->firstobjmarked = mark_object;
		  si->lastobjmarked = point_object;
		}
	      else if (mark_measure > point_measure)
		{
		  si->firstobjmarked = point_object;
		  si->lastobjmarked = mark_object;
		}
	      else
		{		/* Same measure */
		  si->firstobjmarked = MIN (mark_object, point_object);
		  si->lastobjmarked = MAX (mark_object, point_object);
		}
	    }
	  else
	    {
	      si->firstobjmarked = 0;
	      si->lastobjmarked = G_MAXINT;
	    }
	  break;
	case WHOLE_STAFFS:
	  si->firstmeasuremarked = 1;
	  si->lastmeasuremarked = g_list_length (si->measurewidths);
	  si->firstobjmarked = 0;
	  si->lastobjmarked = G_MAXINT;
	}
    }
}

void
calcmarkboundaries (struct scoreinfo *si)
{
  mark_boundaries_helper (si, si->markstaffnum, si->markmeasurenum,
			  si->markcursor_x, si->currentstaffnum,
			  si->currentmeasurenum, si->cursor_x, NORMAL_SELECT);
}
