/* dhcp_options.c
 * 
 * Dynamic Host Configuration Options utilities
 *
 * Copyright © 2006 Red Hat, Inc.  All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions of
 * the GNU General Public License v.2.  This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY expressed or
 * implied, including the implied warranties of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.  You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA. Any Red Hat trademarks that are incorporated in the source code or
 * documentation are not subject to the GNU General Public License and may
 * only be used or replicated with the express permission of Red Hat, Inc.
 *
 * Red Hat Author(s):  Jason Vas Dias
 *                     David Cantrell
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "dhcp_options.h"

typedef 
struct dhco_s 
{ /* glibc b-trees */
  /* option b-trees for each universe: */
    void *param;
    void *dhcp;
    void *rai;
    void *nwip;
    void *fqdn;
    void *uu1;  /* uu: user-universe */
    void *uu2;
    void *uu3;
    void *uu4;
    void *uu5;
  /* universe tree: */
    void *universes;
  /* enum tree: */
    void *enums;
    unsigned n_universes;
    void *aliases;
    unsigned n_aliases;
    int  (*eh)(const char *fmt,...);
} DHC; 

typedef
struct dhco_mt_s
{
    uint8_t dhc_code;
    uint8_t map_code;
} DHCP_Option_Type_Map;

typedef
struct dhco_m_s
{
    void *mtree;
    DHCP_Option_Type_Map map[ DHC_N_ATOMIC_TYPES ];
} DHC_Mapping;

static uint8_t dhc_atomic_types[  DHC_N_ATOMIC_TYPES ] =
{
    DHC_T_NONE,
    DHC_T_IP_ADDRESS,
    DHC_T_HEX_STRING,
    DHC_T_TEXT,
    DHC_T_DOMAIN_NAME,
    DHC_T_INT32,
    DHC_T_UINT32,
    DHC_T_INT16,
    DHC_T_UINT16,
    DHC_T_CHAR,
    DHC_T_UCHAR,
    DHC_T_BOOL,
    DHC_T_IMPLICIT_BOOL,
    DHC_T_ENUMERATION
};

static
int universe_name_comparator( const void *p1, const void *p2)
{
    return strcmp((const char*)(((const DHC_Universe*)p1)->name),
                  (const char*)(((const DHC_Universe*)p2)->name)
                 );
}

static
int option_name_comparator( const void *p1, const void *p2)
{
    return strcmp((const char*)(((const DHC_Option*)p1)->name),
                  (const char*)(((const DHC_Option*)p2)->name)
                 );
}


static
int enum_name_comparator( const void *p1, const void *p2)
{
    return strcmp((const char*)(((const DHCP_Enumeration*)p1)->name),
                  (const char*)(((const DHCP_Enumeration*)p2)->name)
                 );
}

static
int enum_value_name_comparator( const void *p1, const void *p2)
{
    return strcmp((const char*)(((const DHCP_EnumValue*)p1)->name),
                  (const char*)(((const DHCP_EnumValue*)p2)->name)
                 );
}

static
int enum_value_num_comparator( const void *p1, const void *p2)
{
    return ( (((const DHCP_EnumValue*)p1)->number == ((const DHCP_EnumValue*)p2)->number)
             ? 0
             : ( (((const DHCP_EnumValue*)p1)->number > ((const DHCP_EnumValue*)p2)->number)
                 ? 1
                 : -1
               )
           );
}

static
int alias_name_comparator( const void *p1, const void *p2)
{
    return strcmp((const char*)(((const DHC_Alias*)p1)->name),
                  (const char*)(((const DHC_Alias*)p2)->name)
                 );
}

static 
int map_comparator( const void *p1, const void *p2)
{
    return ( (*((uint8_t*)p1) == *((uint8_t*)p2))
            ? 0
             : ( (*((uint8_t*)p1) > *((uint8_t*)p2))
                ? 1
                : -1
               )
           );
}

DHC* 
dhco_new( int (*eh)(const char *fmt,...) )
{
    const DHC_Option  *o;
    const DHC_Universe *u;
    const DHCP_Enumeration *ep=&(dhcp_enumerations[0]);
    const DHC_Alias *ap = &(dhcp_aliases[0]);
    void  **tp;
    uint16_t      n, un, en;
    DHCP_EnumValue *ev;

    DHC *d = (DHC *) malloc ( sizeof(DHC) );

    if( d == 0L )
    {
        if( eh ) (*eh)("dhco_new: out of memory\n");
        return( 0L );
    }

    memset( d, '\0', sizeof(DHC) );    

    for( un = 0, u = &(dhcp_universes[0]), tp = &(d->param);
         un < DHC_UNIVERSE_N; 
         un++, u++, tp++
       )
    {
        if ( tsearch( u, &(d->universes), universe_name_comparator ) == 0L )
        {
            if( eh ) (*eh)("dhco_new: out of memory\n");
            dhco_free(d);
            return ( 0L );
        }

        for( n = 0, o = &(u->universe[0]); 
             n < u->n; 
             n++, o++ 
           ) if ( o->name != 0L )
               if ( tsearch( o, tp, option_name_comparator ) == 0L )
               { 
                   if( eh ) (*eh)("dhco_new: out of memory\n");
                   dhco_free(d);
                   return( 0L );
               }
    }
    for( n = 0; n < n_dhcp_enumerations; n++, ep++ )
    {
        if( tsearch( ep, &(d->enums), enum_name_comparator ) == 0L )
        {
            if( eh ) (*eh)("dhco_new: out of memory\n");
            dhco_free( d );
            return( 0L );
        }
        if( ep->n )
        {
            for( ev = &(ep->values[0]), en = 0; en < ep->n; en++, ev++ )
            {           
                if( tsearch( ev, (void*)&(ep->ts), enum_value_name_comparator ) == 0L )
                {
                    if( eh ) (*eh)("dhco_new: out of memory\n");
                    dhco_free( d );
                    return( 0L );
                }       
                if( tsearch( ev, (void*)&(ep->tn), enum_value_num_comparator ) == 0L )
                {
                    if( eh ) (*eh)("dhco_new: out of memory\n");
                    dhco_free( d );
                    return( 0L );
                }
            }
        }
    }
    for( n = 0; n < n_dhcp_aliases; n++, ap++ )
    {
        if( tsearch( ap, &(d->aliases), alias_name_comparator ) == 0L )
        {
            if( eh ) (*eh)("dhco_new: out of memory\n");
            dhco_free( d );
            return( 0L );
        }
    }
    d->n_universes = DHC_UNIVERSE_N;
    d->eh = eh;
    return d;   
}

static
void no_free( void *p ){}

void 
dhco_free( DHC *d )
{
    void   *tp;
    uint16_t un;

    if( d == 0L )
        return;
    if(  d->universes ) tdestroy( d->universes, no_free );
    for( un = 0, tp = &(d->param);
         un < DHC_UNIVERSE_N; 
         un++,  tp++
       ) if(tp) tdestroy( tp, no_free );
    if(d->enums) tdestroy( d->enums, no_free );
    free(d);
}

DHC_Option *
dhco_find_by_name(DHC* d, const char *name )
{
    DHC_Universe *u, **up; 
    DHC_Universe us;
    DHC_Option   *o, **op;
    DHC_Option os;
    DHC_Alias  *ap, **app, as;
    char *un=0L, *opt=0L, *p=0L;
    void **tp=&(d->param);   

    if( (name == 0L) || (d == 0L) || (*name == '\0') )
        return 0L;    

    if((p = strchr( name, DHCP_UNIVERSE_SEPARATOR )) != 0L)
    {
        if( ( un = malloc( (p - name) + 1 ) ) == 0L )
        {
            if( d->eh )
            {
                (*(d->eh))("dhco_find_by_name: out of memory\n");
                return 0L;
            }
        }
        strncpy(un, name, p - name );
        un[ p - name ] = '\0';
        p += 1;
        if( ( opt = malloc( strlen(p) + 1 ) ) == 0L )
        {
            if( d->eh )
            {
                (*(d->eh))("dhco_find_by_name: out of memory\n");
                return 0L;
            }
        }
        strcpy(opt, p);
        name = opt;
    }
          
    if( un == 0L )
    {
        os.name = (char*) name;
        op = tfind( &os, &(d->param), option_name_comparator);
        if((op != 0L) && ((o = *op) != 0L) )
            return( o );
        op = tfind( &os, &(d->dhcp), option_name_comparator);
        if( (op != 0L) && ((o = *op) != 0L) )
            return( o );
        as.name = (char*) name;
        app = tfind(&as, &(d->aliases), alias_name_comparator);
        if( (app != 0L) && ((ap = *app) != 0L) )
            return( ap->opt );
    }else
    {
        us.name = un;
        up = tfind( &us, &(d->universes), universe_name_comparator);
        if( (up == 0L) || ((u = *up) == 0L) )
            return 0L;
        os.name = (char*)opt;
        op = tfind( &os, &(tp[u->code]), option_name_comparator);
        if( (op != 0L) && ((o = *op) != 0L) )
            return( o );
    }
    return 0L;
}

DHC_Option *
dhco_find_by_code(DHC* d, uint8_t o, DHCP_Universe u )
{    
    if ( d==0L )
        return 0L;

    if ( ( u > 0 ) && (u < DHC_UNIVERSE_N) )
    {
        if ( ( o > 0 ) && (o < dhcp_universes[u].n) )
            return &(dhcp_universes[u].universe[o]);
    }else
    if ( ( o > 0 ) && (o < dhcp_universes[DHC_O_Universe].n) )
    {
        return &(dhcp_options[ o ]);
    }
    return 0L;
}

const char *
dhco_type( DHC_Option *o )
{
    if( o == 0L )
        return 0L;
    return o->type;
}

DHCP_Option_Code
dhco_code( DHC_Option *o )
{
    if( o == 0L )
        return 0L;
    return o->code;
}

DHC_Option*
dhco_new_option( DHCO dhc, const char *name, uint8_t code, char *type )
{
    char *tp=type, *endtp, *p, *un, *nn, *oldn;
    uint8_t is_array = 0, un_code=DHC_O_Universe, diff_name, diff_type;
    DHC_Option *opt=0L, *nopt=0L;
    DHC_Universe *u=0L, *const*upp, us;
    void  **utp;

    if( (dhc==0L) || (name == 0L) || (type==0L) || (*name =='\0') || (*type == '\0') )
        return 0L;

    utp=&(dhc->param);    
    endtp = type + strlen(type);

    if((is_array = ( (*(endtp-1) == 'A') || (*(endtp-1) == 'a') )))
        endtp--;
    for(; tp < endtp; tp++)
        switch( *tp )
        {
        case DHC_T_DOMAIN_NAME:
        case DHC_T_TEXT:
        case DHC_T_HEX_STRING:
            if( is_array || (tp != (endtp-1)) )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: Embedded strings are not supported by DHCP.\n");
                return 0L;
            };
            break;

        case DHC_T_ARRAY:
        case DHC_T_LIST:
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: Nested arrays are not supported by DHCP.\n");
            return 0L;

        case DHC_T_OPTIONAL:
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: Optional option values are not supported by DHCP.\n");
            /* actually, DHCP has no means of specifying them as user defined options,
             * but there is at least 1 option with an optional consituent ("slp-service-scope").
             */
            return 0L;

        case DHC_T_OPTION_SPACE:
        case DHC_T_ENUMERATION:
        case DHC_T_ENCAPSULATION:
        case DHC_T_ENCAPSULATION_NO_DIRECTIVE: 
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: New option space encapsulations are not supported.\n");
            /* Because dhclient NEVER writes them to environment - except for
             * vendor_option_space, but that is not "new".
             * We can just ignore them because if new option spaces are in
             * environment, all their options will be separately defined 
             * in the environment.
             */
            return 0L;

        case DHC_T_IMPLICIT_BOOL:
        case DHC_T_IP_ADDRESS:
        case DHC_T_INT32:
        case DHC_T_UINT32:
        case DHC_T_INT16:
        case DHC_T_UINT16:
        case DHC_T_CHAR:
        case DHC_T_UCHAR:
        case DHC_T_BOOL:            
            continue;
       
        default:
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: Unknown option type: %c\n", *tp);
        }
    
    if( ( opt = (DHC_Option*)dhco_find_by_name(dhc, name) ) != 0L )
    {/* replace option with same name - nasty, but allowed */

        if( opt->universe == DHC_P_Universe )
        {
            if(dhc->eh) 
                (*(dhc->eh))("dhco_new_option: cannot create options in the parameter universe\n");
            return 0L;
        }

        if( code == 0 )
            code = opt->code;

        if( code != opt->code )
        {        
            nopt = &(dhcp_universes[ opt->universe ] . universe[ code ]);
            if( (nn = (char *)malloc(strlen(name)+1)) == 0L )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;
            }else
                strcpy(nn, name);
            if( (tp = (char*) malloc(strlen(type)+1)) == 0L )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;
            }else
                strcpy(tp, type);

            diff_name = (( nopt->name == 0L ) || ( strcmp(nopt->name, name) != 0 ) );
            diff_type = (( nopt->type == 0L ) || ( strcmp(nopt->type, type) != 0 ) );

            if( diff_name || diff_type )
            {
                oldn = nopt->name;
                if( diff_name )
                {
                    nopt->name = nn;
                    if( tsearch(nopt, &(utp[ nopt->universe ]), option_name_comparator) == 0L )
                    {
                        nopt->name = oldn;
                        if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                        return 0L;
                    }
                    nopt->name = oldn;
                    tdelete(nopt, &(utp[ nopt->universe ]), option_name_comparator);        
                }

                tdelete(opt, &(utp[ opt->universe ]), option_name_comparator);

                if( nopt->flags != DHC_F_DEFINED )
                {
                    free(nopt->name);
                    free(nopt->type);
                }
                nopt->name = nn;        
                nopt->type = tp;        
                nopt->code = code;
                nopt->flags = DHC_F_NEW;
            }else
            if( !diff_name )
            {
                free(nn);
            }else
            if( !diff_type )
            {
                free(tp);
            }
            return nopt;
        }
        if( strcmp(type, opt->type) != 0)
        {
            tp = (char*) malloc(strlen(type)+1);
            if( tp == 0L )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;
            }
            strcpy(tp,type);        
            if( (nn = (char *)malloc(strlen(name)+1)) == 0L )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;
            }   
            strcpy(nn, name);
            if( opt->flags != DHC_F_DEFINED )       
            {
                free(opt->type);            
                free(opt->name);
            }
            opt->type = tp;
            opt->name = nn;
            opt->flags = DHC_F_NEW;
        } /* same name, same code, same type - WTF!?! */        
        return opt;
    }else
    {     /* new option from scratch */ 
        un=0L;
        opt=0L;
        u=0L;

        if( (tp = (char*) malloc(strlen(type)+1)) == 0L )
        {
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");               
            return 0L;
        }
        strcpy(tp,type);            
        type = tp;

        nn = 0L;
        if((p = strchr( name, DHCP_UNIVERSE_SEPARATOR )) != 0L)
        {
            if( (nn = (char *)malloc(strlen(p+1)+1)) == 0L )
            {
                free(tp);
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;
            }   
            strcpy(nn, p+1);
            
            if( (un = (char *)malloc((p - name)+1)) == 0L )
            {
                free(tp); 
                free(nn);
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");           
                return 0L;              
            }
            strncpy(un, name, (p - name)); 
            un[ p - name ] = '\0';

            us.name = un;

            if( ((upp = tfind(&us, &(dhc->universes), universe_name_comparator)) != 0L)
              &&((u=*upp) != 0L)
              )
            {
                un_code = u->code;
            }else
            {
                free(tp);
                free(nn);
                free(un);
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: Universe %s not found.\n",un);
                return 0L;
            }

            name = nn;

            free(un);
            un = 0L;

        }else
        {
            un_code = DHC_O_Universe;
            u = &(dhcp_universes[un_code]);
        }
        
        if( code == 0 )
        { 
            if( u->n >= (DHCP_OPTION_MAX-1) )
            {
                free(tp);
                if( nn != 0L )
                    free(nn);
                if(dhc->eh) (*(dhc->eh))("dhco_new_option: Universe %s exhausted.\n",u->name);
                return 0L;
            }
            code = u->n;
            u->n = u->n + 1;
        }

        opt = &(u->universe[code]);

        if( opt != 0L )
        {
            if( opt->flags == DHC_F_NEW )
            {
                if( opt->name != 0L )
                    free(opt->name);

                if( opt->type != 0L )
                    free(opt->type);
            }

            if( nn == 0L )
            {
                if((opt->name = (char*)malloc(strlen(name)+1)) == 0L)
                {
                    free(tp);
                    if( nn != 0L )
                        free(nn);
                    if(dhc->eh) (*(dhc->eh))("dhco_new_option: out of memory\n");               
                    return 0L;          
                }               
                strcpy(opt->name, name);
            }else
                opt->name = (char*) name;

            if( tsearch(opt, &(utp[un_code]), option_name_comparator) == 0L)        
                return 0L;

            opt->type = tp;
            opt->code = code;
            opt->universe = un_code;        
            opt->flags = DHC_F_NEW;

        }else
        {
            if(dhc->eh) (*(dhc->eh))("dhco_new_option: Universe not found\n");
            return 0L;
        }
    }
    
    return opt;
}

DHC_Universe *
dhco_find_universe_by_name(DHC *d, const char *un )
{
    DHC_Universe us, *u=0L, *const*upp;

    if(d == 0L) return 0L;
    us.name = (char*)un;
    upp = tfind( &us, &(d->universes), universe_name_comparator);
    if((upp != 0L) && (*upp != 0L))
        u = *upp;
    return u;
}

DHC_Universe *
dhco_find_universe_by_code(DHC *d, uint8_t code )
{
    if(d == 0L) return 0L;
    if( ( code < DHCP_UNIVERSE_MAX ) && ( code < d->n_universes ) )
        return &(dhcp_universes[code]);
    return 0L;
}

DHC_Universe *
dhco_new_universe( DHC *d, const char *name )
{
    DHC_Universe *u = 0L;
/*    printf("\tnew universe: %s\n",name); */
    if((d == 0L) || (name == 0) || (*name == '\0') )
        return 0L;
    if( dhco_find_universe_by_name( d, name ) != 0L )
    {
        if( d->eh ) 
            (*(d->eh))("dhco_new_universe: attempt to create universe %s that already exists.\n",
                       name
                       );
        return 0L;
    }
    if( d->n_universes >= DHCP_UNIVERSE_MAX )
    {   
        if( d->eh ) 
            (*(d->eh))("dhco_new_universe: out of universe space\n");
        return 0L;
    }
    u = &(dhcp_universes[d->n_universes]);
    if( (u->universe = (DHC_Option*)calloc(256, sizeof(DHC_Option))) == 0L )
    {
        if(d->eh) (*(d->eh))("dhco_new_universe: out of memory\n");
        return 0L;
    }
    if( (u->name = (char*) malloc(strlen(name)+1)) == 0L )
    {
        if(d->eh) (*(d->eh))("dhco_new_universe: out of memory\n");
        free(u->universe);
        u->universe = 0L;
        return 0L;
    }
    strcpy(u->name, name);
    if( tsearch(u, &(d->universes), universe_name_comparator) == 0L)
    {
        if(d->eh) (*(d->eh))("dhco_new_universe: out of memory\n");
        free(u->universe);
        u->universe = 0L;
        free(u->name);
        u->name = 0L;
        return 0L;
    }
    u->code = d->n_universes;
    u->n = 0;   
    d->n_universes++;
    return( u );
}

static uint8_t 
n_bits( uint32_t n )
{
    uint8_t b=0;
    for(; (b < 32) && ( (1<<b) <= n); ++b);
    return b;
}

typedef void* (*LookupFunction)(DHC *, const char *);

static void *
lookup_dhclient_name(DHC *dhc, const char *name, char **prefix, LookupFunction lookup)
{
    void *o;
    char *try, *p, *pfx=0;
    unsigned len;
    uint32_t n_un;
    int32_t bits;
    uint8_t b;

    if( (dhc == 0L) || (name == 0L) )
        return 0L;
    
    if( (o = (*lookup)(dhc, name)) != 0L )
        return o;

    if( (try = (char*)malloc(len=strlen(name)+1)) == 0L )
    {
        if(dhc->eh)
            (*(dhc->eh))("dhco_find_by_dhclient_name: out of memory\n");
        return 0L;
    }

    strcpy(try, name);

    pfx=0L;
    if(  (strncmp(name, "new_",4)==0) 
       ||(strncmp(name, "old_",4)==0)
       ||(strncmp(name, "alias_",6)==0)
      )
    {
        p = try + 4;
        pfx = p - 1;
        *pfx='\0';
        if( prefix != 0L )
        {
            if( *prefix == 0L )     
            {
                if( (*prefix = (char *)malloc(4)) == 0L )
                {
                    if(dhc->eh)
                        (*(dhc->eh))("dhco_find_by_dhclient_name: out of memory\n");
                    return 0L;
                }
            }
            strcpy(*prefix, try);
        }              
        try = p;
/*      printf("   find by dhclient name: trying %s\n",try); */
        if( (o = lookup(dhc, try)) != 0L )
            return o;
    }
    
    if( (p=strchr(try,'_')) == 0L )
    {
        if( pfx != 0L )
            free(pfx-3);
        else
            free(try);
        return 0L;
    }

    n_un=0;
    while( p != 0L )
    {
        ++n_un;
        p=strchr(p+1,'_');
    }

    if ( n_un > 31 )
    {
        if( pfx != 0L )
            free(pfx-3);
        else
            free(try);
        return 0L;
    }

    p=try;
    /* since underscores are allowed in option names,
     * we're going to have to try all permutations of
     * replacement of '_' by '-' :(
     */    
    bits = (1 << (n_bits(n_un)+1))-1;    
/*     printf("   find by dhclient name: bits: %d\n",bits); */
    for( ; bits >= 0; bits--)
    {
        for( b = 1, p = try; 
             ((p = strchr(p,'_')) != 0L);
             b<<=1, p++
           )if( bits & b )
                *p='-';         
/*      printf("   find by dhclient name: trying %s\n",try); */
        if ( (o = (*lookup)(dhc, try)) != 0 )
            break;
        if( pfx != 0L )
        {    
            *pfx = '_';
            try = pfx - 3;
/*          printf("   find by dhclient name: trying %s\n",try); */
            if ( (o = (*lookup)(dhc, try)) != 0 )
                break;
            *pfx = '\0';
            try = pfx+1;
        }           
        for(p = strchr(try,'-'); p; p = strchr(p,'-') )
            *p = '_';
    }
    if( pfx != 0L )
        free(pfx-3);
    else
        free(try);
    return o;
}

DHC_Option *
dhco_find_by_dhclient_name(DHC *dhc, const char *name, char **prefix )
{
    return (DHC_Option *)lookup_dhclient_name(dhc, name, prefix, (LookupFunction)dhco_find_by_name);
}

DHC_Universe *
dhco_find_universe_by_dhclient_name(DHC *dhc, const char *name, char **prefix )
{
    return (DHC_Universe*)lookup_dhclient_name(dhc, name, prefix, (LookupFunction)dhco_find_universe_by_name);
}

typedef
struct nor_s
{
    DHC  *dhc;
    char *value;
    unsigned  *n_handled;
    dhco_OptionHandler oh;
    void *oh_param;
} NewOptionRecord;

static DHC_Option*
record_undefined_option
(   DHC *dhc, 
    void **tree, 
    char *po, 
    char *pv, 
    dhco_OptionHandler oh, 
    unsigned *n_handled, 
    void *oh_param
)
{
    DHC_Option *o = (DHC_Option*)calloc(1, sizeof(DHC_Option));
    NewOptionRecord *nor;
    
/*     printf("\trecord_undefined_option %s ( %p ) %s\n",po, po, pv); */
    if( o == 0L )
    {
        if( dhc->eh ) (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
    }else
    {
        if( (o->name = (char*)malloc(strlen(po)+1)) == 0L )
        {
            if( dhc->eh ) (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
            free(o);
            o=0L;
        }else
        {                           
/*          printf("new option name:%p %s\n", o->name, po); */
            strcpy(o->name, po);
            if( (nor = (NewOptionRecord*)malloc(sizeof(NewOptionRecord))) == 0L )
            {
                if( dhc->eh ) (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
                {
                    free(o->name);
                    free(o);
                    o=0L;
                }
            }else
            {
                if( (nor->value = (char*)malloc(strlen(pv)+1)) == 0L )
                {
                    if( dhc->eh ) (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
                    free(nor);
                    free(o->name);
                    free(o);
                    o=0L;               
                }else
                {
                    strcpy(nor->value, pv);
                    nor->dhc=dhc;
                    nor->oh = oh;
                    nor->n_handled = n_handled;             
                    nor->oh_param = oh_param;
                    o->type = (char*)nor;
                    if( tsearch(o, tree, option_name_comparator) == 0L )
                    {
                        if( dhc->eh ) 
                            (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
                        free(o->name);
                        free(o->type);
                        free(o);
                        o = 0L;
                    }
                }
            }
        }
    }           
    return o;
}

static void
process_new_options( const void *p, const VISIT which, const int level )
{
    DHC_Option *const*opp=p;
    DHC_Option *no, *o;
    NewOptionRecord *nor;
    unsigned r;
    void *value=0L;
    char px[4]={'\0'}, *prefix=(char*)&px; 

    if( (opp != 0L) && (*opp != 0L) && ((which == postorder) || (which == leaf)) ) 
    {
        no = *opp;
        nor = (NewOptionRecord*)(no->type);
/*      printf("    process new option: %s = %s\n", no->name, nor->value); */
        if( (o=dhco_find_by_dhclient_name(nor->dhc, no->name, &prefix))==0L )
        {
            if(nor->dhc->eh)
                (*(nor->dhc->eh))("dhco_parse_option_settings: no new option definition for %s\n",no->name);
            return;
        }
        if( ((r = dhco_input_option( nor->dhc, o, nor->value, &value, 0)) == 0 )
          ||(value == 0L)
          )
        {
            if(nor->dhc->eh)
                (*(nor->dhc->eh))("dhco_parse_option_settings: bad option value '%s' for option %s\n", nor->value, no->name);
            return;
        }
        (*(nor->oh))(nor->dhc, o, value, r, prefix, nor->oh_param);
        ++(*(nor->n_handled));
    }
}

void free_new_option_record( void *vop )
{
    DHC_Option *o = vop;
    NewOptionRecord *nor = (NewOptionRecord*)(o->type);
    free(o->name);
    free(nor->value);
    free(o);
}

unsigned
dhco_parse_option_settings(DHC *dhc, char *settings, dhco_OptionHandler oh, void *oh_param)
{
    DHC_Option *o = 0L, os;
    DHC_Universe *u = 0L;
    void *new_option_tree=0L, *value=0L;
    unsigned n=0,r=0;
    char *ps, *pe, *pq, *po, *pu, *pv, *pt, *pn, *endp;
    void **utp;
    uint32_t code;
    char px[4]={'\0'}, *prefix=&(px[0]);
    
    if((dhc == 0L) || (settings == 0L) || (*settings == '\0') || (oh == 0L) )
        return( 0 );

    utp=&(dhc->param);

    for( ps=settings,
         endp = settings + strlen(settings) + 1;
         (ps < endp)
       &&(((pe = strchr(ps, '\n')) != 0L) || ((pe = endp) != 0L))
       &&(ps < pe)
       &&(*ps != '\0');
         ps = pe+1, *prefix='\0'
       )
    {
        *pe = '\0'; 
        if( ( pq = strchr(ps, '=') ) == 0L ) 
        {
            if( dhc->eh )
                (*(dhc->eh))("dhco_parse_option_settings: bad setting statement %p %p '%s'\n", ps, pe, ps);
            if( pe != endp) 
                *pe = '\n';
            continue;
        }
        *pq = '\0';
        pv = pq + 1;
        pu = ps;
/*      printf("Option setting: %s = %s\n",pu,pv); */
        if( (po = strchr(pu, DHCP_UNIVERSE_SEPARATOR)) == 0L)
        {
            po=pu;
/*          printf("\tlookup unqualified %s\n",po); */
            if(( o = dhco_find_by_dhclient_name(dhc, po, &prefix ))==0L)
            {
                o = record_undefined_option(dhc, &new_option_tree, po, pv, oh, &n, oh_param);
            }else
            {
/*              printf("\tinput option value %s\n",pv); */
                value = 0L;
                if( (( r = dhco_input_option(dhc, o, pv, &value, 0)) == 0 )
                  ||( value == 0L )
                  )
                {
                    if( dhc->eh ) 
                        (*(dhc->eh))("dhco_parse_option_settings: bad option setting: %s = %s\n",
                                     po, pv, prefix
                                    );
                }else
                {
                    ++n;
                    (*oh)(dhc, o, value, r, prefix, oh_param);
                }
            }
        }else /* universe qualified option */
        {
            o = 0L;
            *po = '\0';
            prefix[0]='\0';
/*          printf("\tuniverse_qualified_option: %s\n", pu); */
            if( (u = dhco_find_universe_by_dhclient_name(dhc, pu, &prefix)) == 0L )
            {   
/*              printf("pu: %p\n",pu); */
                if( ( strlen(pu) > 4 )
                  &&( (strncmp(pu, "new_", 4)==0)
                    ||(strncmp(pu, "old_", 4)==0)
                    ||(strncmp(pu, "alias_", 6)==0)
                    )
                  )
                {
                    strncpy(prefix,pu,3);
                    prefix[3]='\0';
                    pu += 4;
                }                   
/*              printf("\tcreating new universe: %s\n",pu); */
                if( (u = dhco_new_universe(dhc, pu)) == 0L )
                {
                   if( dhc->eh ) 
                       (*(dhc->eh))("dhco_parse_option_settings: cannot create universe %s\n",pu);
                }
                if( prefix[0] != '\0' )
                {
                    pu -= 4;
/*                  printf("pu: %p\n",pu); */
                }
            }
            *po++ = DHCP_UNIVERSE_SEPARATOR;
            if( u != 0L )
            {
/*              printf("\tgot universe: %p %s\n", u, u->name); */
                if( strncmp(po, "_universe_.", 11 ) == 0 )
                {
                    po += 11;
                    pt =  0L;
/*                  printf("\thandling _universe_ new option format: %s %s\n", po, pv); */
                    if( (sscanf(pv,"%3u:%as", &code, &pt) == 2)
                      &&( code < 255 )
                      &&( pt != 0L )
                      )
                    {
                        if((pn = (char*)malloc(strlen(u->name) + 1 + strlen(po) + 1)) == 0L )
                        {
                            if( dhc->eh ) 
                                (*(dhc->eh))("dhco_parse_option_settings: out of memory\n");
                        }else
                        {
                            sprintf(pn,"%s.%s",u->name,po);
                            if( (o = dhco_new_option(dhc, pn, code, pt)) == 0L )
                                if( dhc->eh )
                                    (*(dhc->eh))("dhco_parse_option_settings: cannot create new option %s\n",po);
                            free(pn);
                        }
                    }else
                        if( dhc->eh )
                                (*(dhc->eh))("dhco_parse_option_settings: bad new option string %s\n",pv);
                }else
                {
                    os.name = po;                   
/*                  printf("\tlookup %s in existing universe %s\n",po,u->name);*/
                    if( (o = dhco_find_by_dhclient_name( dhc, pu, &prefix ) ) != 0L )
                    {
                        value = 0L;
                        if( (( r = dhco_input_option(dhc, o, pv, &value, 0)) == 0 )
                          ||( value == 0L )
                          )
                        {
                            if( dhc->eh ) 
                                (*(dhc->eh))("dhco_parse_option_settings: bad option setting: %s = %s\n",
                                             po, pv
                                            );
                        }else
                        {
                            ++n;
                            (*oh)(dhc, o, value, r, prefix, oh_param);
                        }
                    }else
                    {
                        o = record_undefined_option(dhc, &new_option_tree, pu, pv, oh, &n, oh_param);
                    }                   
                }
            }else
                if( dhc->eh )
                    (*(dhc->eh))("dhco_parse_option_settings: cannot create new universe %s\n",pu);
        }
        *pq = '=';
        if( pe != endp) 
            *pe = '\n';
        if( o == 0L )
        {   
            if( dhc->eh )
                (*(dhc->eh))("dhco_parse_option_settings: bad option statement %s - giving up.\n", ps);
            break;
        }
    }    
    if( new_option_tree != 0L )
    {
        twalk(new_option_tree, process_new_options);
        tdestroy(new_option_tree, free_new_option_record);
    }
    return( n );
}

static int 
name2ip4( char *name, struct in_addr *ip )
{
    struct addrinfo rq_ai, *ai, *aie;
    int r;
    if( (name == 0L) || (ip == 0L) || (*name == '\0') )
        return(-1);
    memset(&rq_ai, '\0', sizeof(struct addrinfo));
    memset(ip, '\0', sizeof(struct in_addr));
    rq_ai.ai_family = AF_INET;
    rq_ai.ai_flags  = AI_CANONNAME;
    if ( (r = getaddrinfo( name, 0L, &rq_ai, &ai )) != 0 )
        return r;
    for(aie=ai; aie != 0L; aie = aie->ai_next)
        if(  (aie->ai_family == AF_INET) 
          && (aie->ai_canonname != 0L)
          && (*(aie->ai_canonname) != '\0')
          && (aie->ai_addr != 0L )
          &&(((struct sockaddr_in*)aie->ai_addr)->sin_addr.s_addr != 0L)
          ) break;
    if ( aie != 0L )
        ip->s_addr = ((struct sockaddr_in*)aie->ai_addr)->sin_addr.s_addr;
    freeaddrinfo(ai);
    return (aie == 0L) ? EAI_NONAME : 0;
}

#define DHC_PAD( type, total_length ) ( ( sizeof(type) - (total_length  & (sizeof(type)-1) ) ) & (sizeof(type)-1) )
/* returns pad to align the address of a value of "type" given total_length preceding bytes 
 * NOTE: only works if sizeof(type) is a power of two, as they all are in our case 
 */

unsigned 
dhco_value_from_text
(   DHC *dhc,
    DHCP_Option_Type type, 
    char *input, 
    void **value, 
    int len, 
    unsigned total_length 
)
{
    unsigned pad=0;
    int r;
    if(  ( input == 0L )  || (value == 0L) 
       ||((*input == '\0') && (type != DHC_T_IMPLICIT_BOOL))
      ) return 0;
    switch( type )    
    {                             
    case DHC_T_IP_ADDRESS:
    {
        uint32_t o[4];  
        if ( sscanf(input, "%3u.%3u.%3u.%3u", &(o[3]),(&o[2]),&(o[1]),&(o[0])) != 4 )
        {
            /* DHCP allows IP addresses to be specified as domain names! */
            if ( name2ip4( input, (struct in_addr*)(void*)&(o[0]) ) == 0 )
            {
                o[0] = ntohl(o[0]);
                o[3] = o[0] >> 24;
                o[2] = (o[0] >> 16) & 0xff;
                o[1] = (o[0] >> 8) & 0xff;
                o[0] &= 0xff;
            }else
                return 0;
        }
        if ( (o[0] > 255) || (o[1] > 255) || (o[2] > 255) || (o[3] > 255) )
            return 0;
        if ( *value == 0L )
        {   
            if((*value = malloc(sizeof(uint32_t)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(uint32_t) )
            return( sizeof(uint32_t) + DHC_PAD(uint32_t, total_length));
        else
        if ( (pad = DHC_PAD(uint32_t, total_length)) > 0 )
        {
            /* printf("PAD: %d %p %p\n", pad, *value, (*value)+pad); */
            *value += pad;
        }
        *((uint32_t*)*value) = htonl((o[3] << 24) | (o[2] << 16) | (o[1] << 8) | o[0]);
        *value -= pad;
        return (sizeof(uint32_t) + pad);
    }   break;

    case DHC_T_HEX_STRING:
    {/* luckily for us, DHCP does NOT support text elements in arrays or structs. */
        int  slen=strlen(input), octs=0;        
        uint8_t *oct, *p=(uint8_t*)input, *endp = (uint8_t*)(input + slen + 1);
        uint32_t v;
        for(; p < endp; p++)
        {
            if( strspn((char*)p, "0123456789abcdefABCDEF") != 2 )
            {
                octs = 0;
                break;
            }
            p += 2;         
            octs += 1;
            if( p < (endp - 1) )
                if( *p != ':' )
                {
                    octs = 0;
                    break;
                }
        }
        if( octs > 0 )
        {
            if( *value == 0 )
            {
                if( (*value = malloc(octs * sizeof(uint8_t))) == 0L)
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                    return 0;
                }
            }else
            if( len < (octs * sizeof(uint8_t)))
                return (octs * sizeof(uint8_t));
            for(p=(uint8_t*)input, oct=(uint8_t*)*value; 
                p < endp;
                p+=3, oct++
                )
            {  
                r=sscanf((char*)p,"%2x", &v);
                *oct = (uint8_t)v;
            }
            return octs;
        }       
        for(p = (uint8_t*)input; p < endp; p++)
        {
            if( (*p < 32) || (*p > 127) )
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: Non ascii char outside valid hex string: %2x\n", *p);
                return 0;
            }
        }
    } /* fall through - hex string is ASCII string */

    case DHC_T_TEXT:
    case DHC_T_DOMAIN_NAME:
    { 
        int slen = strlen(input) + 1;
        char *endp = input + (slen - 1);
        if( *value == 0L )
        {
            if ( ( *value = malloc( slen ) ) == 0L )            
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < slen )
        {
            /* printf("Insufficient length: %d < %d\n", len, slen); */
            return slen;
        }
        if( ( *input == '"' ) && (*endp == '"') )
        {
            input++;
            slen -= 2;
        }
        strncpy( (char *)*value, input, slen );
        return slen;
    }   break;

    case DHC_T_INT32:
    {
        int32_t ui;
        if ( sscanf(input, "%i", &ui) != 1 )
            return 0;
        if ( *value == 0L )
        {   
            if((*value = malloc(sizeof(int32_t)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
            total_length = 0;
        }else
        if ( len < sizeof(int32_t) )
            return(sizeof(int32_t) + DHC_PAD(int32_t,total_length));
        else
        if((pad = DHC_PAD(int32_t,total_length)) > 0)
        {
            /*printf("PAD: %d %p %p\n", pad, *value, (*value)+pad);*/
            *value += pad;
        }
        *((int32_t*)*value)=ui;
        *value -= pad;
        return (sizeof(int32_t) + pad);
    }   break;

    case DHC_T_UINT32:
    {
        int32_t ui;
        if ( (sscanf(input, "%i", &ui) != 1) || (ui < 0))
            return 0;
        if ( *value == 0L )
        {   
            if((*value = malloc(sizeof(uint32_t)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(uint32_t) )
            return ( sizeof(uint32_t) + DHC_PAD(uint32_t,total_length) );
        else
        if ( (pad = DHC_PAD(uint32_t,total_length)) > 0 )
        {
            /*printf("PAD: %d %p %p (ui: %x)\n", pad, *value, (*value)+pad,ui);*/
            *value += pad;
        }
        *((uint32_t*)*value)=ui;
        *value -= pad;
        return( sizeof(uint32_t) + pad);
    }   break;

    case DHC_T_IMPLICIT_BOOL:
        if ( *value == 0L )
        {
            if(( *value = malloc(sizeof(uint8_t)) )==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if( len < sizeof(uint8_t))
            return sizeof(uint8_t);
        *((uint8_t*)*value) = 1;
        return( sizeof(uint8_t) );        
    
    case DHC_T_BOOL:
    {
        uint8_t v;
        if(  (strcasecmp(input, "true") == 0 )
          || (strcasecmp(input, "on") == 0 )
          || (strcasecmp(input, "yes") == 0 )
          || (*input == '1')
          || (*input == 't')
          || (*input == 'T')
          )  v = 1;
        else
        if(  (strcasecmp(input, "false") == 0 )
          || (strcasecmp(input, "off") == 0 )
          || (strcasecmp(input, "no") == 0 )
          || (*input == '0')
          || (*input == 'f')
          || (*input == 'F')
          )  v = 0;
        else
            return 0;
        if ( *value == 0L )
        {
            if(( *value = malloc(sizeof(uint8_t)) )==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if( len < sizeof(uint8_t) )
            return sizeof(uint8_t);
        *((uint8_t*)*value)=v;
        return( sizeof(uint8_t) );
    }   break;  
        
    case DHC_T_INT16:      
    {
        int16_t v;
        if ( sscanf(input, "%hi", &v) != 1 )
            return 0;
        if ( *value == 0L )
        {   
            if((*value = malloc(sizeof(int16_t)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(int16_t) )
            return (sizeof(int16_t) + DHC_PAD( int16_t, total_length));
        else
        if( (pad = DHC_PAD( int16_t, total_length)) > 0)
        {
/*          printf("PAD: %d %p %p\n", pad, *value, (*value)+pad); */
            *value += pad;
        }
        *((int16_t*)*value)=v;
        *value -= pad;
        return (sizeof(int16_t) + pad);
    }   break;
    
    case DHC_T_UINT16:
    {
        int16_t v;
        if ( ( sscanf(input,"%hi", &v) != 1 ) || (v < 0))
            return 0;
        if ( *value == 0L )
        {   
            if((*value = malloc(sizeof(uint16_t)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(uint16_t) )
            return (sizeof(uint16_t) + DHC_PAD(uint16_t, total_length));
        else
        if ((pad = DHC_PAD(uint16_t, total_length) ) > 0)
        {
            /*printf("PAD: %d %p %p\n", pad, *value, (*value)+pad);*/
            *value += pad;
        }
        *((uint16_t*)*value)=v;
        *value -= pad;
        return ( sizeof(uint16_t) + pad );
    }   break;
    
    case DHC_T_CHAR:
    {
        int16_t v;
        if ( (sscanf(input, "%hi", &v)!=1) || (v > 255))
            return 0;
        if ( *value == 0L )
        {
            if(( *value = malloc(sizeof(char)) )==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(char) )
            return sizeof( char );      
        *((char*)*value)=v & 0xff;
        return( sizeof(char) );
    }   break;

    case DHC_T_UCHAR: 
    {
        int16_t v;
        if ( (sscanf(input, "%hi", &v)!=1) || (v > 255) || (v < 0))
            return 0;
        if ( *value == 0L )
        {
            if(( *value = malloc(sizeof(char)) )==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_value_from_text: out of memory\n");
                return 0;
            }
        }else
        if ( len < sizeof(char) )
            return sizeof( char );      
        *((char*)*value)=v & 0xff;
        return( sizeof(char) );
    }   break;

    default:
        break;
    }
    return 0;
}

unsigned 
dhco_text_from_value
(   DHC *dhc,
    DHCP_Option_Type type, 
    void **value, 
    char **output, 
    unsigned len, 
    unsigned vlen, 
    unsigned plen
)
{
    if( (value == 0L) || (*value == 0L) || (output == 0L) )
        return 0;
    switch( type )
    {
    case DHC_T_IP_ADDRESS:
    {
        uint32_t v;
        char c;
        int iplen;

        if( plen > 0 )
            *value += DHC_PAD( uint32_t, plen );
        v = ntohl(*((uint32_t*)*value));
        iplen = snprintf(&c, 0, "%u.%u.%u.%u",
                         ((v >> 24)&0xff), ((v >> 16)&0xff), ((v >> 8) & 0xff), (v & 0xff)
                        ) + 1;       
        if( *output == 0 )
        {
            if((*output = (char*) malloc(iplen))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < iplen )
        {
            *value += sizeof(uint32_t);
            return iplen;
        }
        sprintf((char*)*output,"%u.%u.%u.%u", 
                ((v >> 24)&0xff), ((v >> 16)&0xff), ((v >> 8) & 0xff), (v & 0xff)               
               );
        *value += sizeof(uint32_t);
        return iplen;
    }   break;

    case DHC_T_HEX_STRING:
    {
        uint8_t *p, *o, *endp, is_text=1, is_hex=0;
        uint32_t length;
        if( vlen <= 0 )
            return 0;
        for( p = *value, endp = (*value) + vlen;
             p < endp;
             p++
            )
            if( (*p < 32) || (*p > 127) )
            {
                is_text = 0;
                is_hex  = 1;
            }
        length = (  vlen                     /* n bytes */ 
                  + is_text                  /* '\0' */
                  + (is_text * 2)            /* '"'s */
                  + (vlen  * ( is_hex * 3))  /* nibbles + ':'s or '\0' */
                 );
        if( *output == 0L )
        {
            if((*((uint8_t**)output) = (uint8_t*) malloc( length ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < length )
        {
            *value += vlen;
            return length;
        }
        if( is_text )
        {
            sprintf(*output, "\"%s\"", (char*)*value);
            *value += vlen;
            return length;
        }else
        {
            for(p=*value, o=((uint8_t*)*output);
                p < endp;
                p++
               )
                o += sprintf((char*)o,"%.2x:", *p);
            *(o-1) ='\0';
            *value += vlen;
            return length;
        }               
    }   break;

    case DHC_T_DOMAIN_NAME:
    case DHC_T_TEXT:
    {
        int slen = strnlen((char*)*value,vlen)+1;
        if( *output == 0L )
        {
            if((*output = malloc( slen )) == 0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < slen)
        {
            *value += vlen;
            return slen;
        }
        strcpy(*output, (char*)*value);
        *value += vlen;
        return slen;
    }   break;
    
    case DHC_T_INT32:
    {
        char c;
        int olen;
        if( plen )
            *value += DHC_PAD(int32_t,plen);
        olen = snprintf(&c, 0, "%d", *((int32_t*)*value)) + 1;
        if( *output == 0L )
        {
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < olen )
        {
            *value += sizeof(uint32_t);
            return olen;
        }
        sprintf(*output, "%d", *((int32_t*)*value));
        *value += sizeof(uint32_t);
        return olen;
    }   break;

    case DHC_T_UINT32:
    {
        char c;
        int olen;
        if(plen)
            *value += DHC_PAD(uint32_t, plen);
        olen = snprintf(&c, 0, "%u", *((uint32_t*)*value)) + 1;
        if( *output == 0L )
        {
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < olen )
        {
            *value += sizeof(uint32_t);
            return olen;
        }
        sprintf(*output, "%u", *((uint32_t*)*value));
        *value += sizeof(uint32_t);
        return olen;
    }   break;

    case DHC_T_INT16:
    {
        char c;
        int olen;
        if(plen)
            *value += DHC_PAD(uint16_t, plen);
        olen = snprintf(&c, 0, "%hd", *((int16_t*)*value)) + 1;
        if( *output == 0L )
        {
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < olen )
        {
            *value += sizeof(uint16_t);
            return olen;
        }
        sprintf(*output, "%hd", *((int16_t*)*value));   
        *value += sizeof(uint16_t);
        return olen;
    }   break;

    case DHC_T_UINT16:
    {
        char c;
        int olen;
        if(plen)
            *value += DHC_PAD(int16_t, plen);
        olen = snprintf(&c, 0, "%hu", *((uint16_t*)*value)) + 1;
        if( *output == 0L )
        { 
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if(len < olen )
        {
            *value += sizeof(uint16_t);
            return olen;
        }
        sprintf(*output, "%hu", *((uint16_t*)*value));  
        *value += sizeof(uint16_t);
        return olen;
    }   break;

    case DHC_T_UCHAR:
    {
        char c;
        int olen;
        olen = snprintf(&c, 0, "%hu", *((uint8_t*)*value)) + 1;
        if( *output == 0L )
        {
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if(len < olen )
        {
            *value += sizeof(uint8_t);
            return olen;
        }
        sprintf(*output, "%hu", *((uint8_t*)*value));   
        *value += sizeof(uint8_t);
        return olen;
    }   break;

    case DHC_T_CHAR:
    {
        char c;
        int olen;
        olen = snprintf(&c, 0, "%hd", *((int8_t*)*value)) + 1;
        if( *output == 0L )
        {   
            if((*output = malloc( olen ))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if(len < olen )
        {
            *value += sizeof(uint8_t);
            return olen;
        }
        sprintf(*output, "%hd", *((int8_t*)*value));    
        *value += sizeof(int8_t);
        return olen;
    }   break;
        
    case DHC_T_IMPLICIT_BOOL:
        if( *output == 0L )
        {
            if((*output = malloc(sizeof(char)))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < sizeof(char) )
        {
            *value += sizeof(char);
            return( sizeof(char) );
        }
        *output = '\0';
        *value += sizeof(char);
        return sizeof(char);
        break;

    case DHC_T_BOOL:
    {
        int olen = ( (*((char*)value) == '\0') ? 6 : 5 );
        if( *output == 0L )
        {
            if((*output = malloc(olen))==0L)
            {
                if(dhc->eh) (*(dhc->eh))("dhco_text_from_value: out of memory\n");
                return 0L;
            }
        }else
        if( len < olen )
        {
            *value += sizeof(char);
            return olen;
        }
        sprintf((char*)*output, "%s", (*((char*)value) == '\0')  ? "false" : "true");
        *value += sizeof(char);
        return olen;
    }   break;

    default:
        break;
    }
    return 0;
}

static int 
seek_next_data(char **data_p, char *endp, char* quoted_p)
{
    char *data, *p, c='\0', quoted='\0';
    if( quoted_p != 0 )
        quoted = *quoted_p;
    if(data_p==0L)
        return 0;
    data = *data_p;
    if(data == 0L)
        return 0;
    for(p=data, data=(quoted != '\0') ? data : 0L; (*p != '\0')  && (p  < endp); p++ )
    {
        if( (quoted == '\0') && (c != '\\') && ((*p == '\'') || (*p=='"')) )
        {
            data = p;
            *quoted_p = quoted = *p;
            continue;
        }
        if( (quoted != '\0') && (c != '\\') && (*p == quoted) )
        {
            *quoted_p = quoted = '\0';
            *data_p = data;
            return ((p+1) - data);
        }
        if( quoted  != '\0' )
        {
            c = *p;
            continue;
        }
        if( (c != '\\') && ( ( *p == ' ' )  || ( *p == ' ' ) || (*p =='\t') || (*p=='\n') || (*p == ',') ) )
        {
            if( data != 0L )
            {
                *data_p = data;
                return (p - data);
            }else
            {
                c = *p;
                continue;
            }
        }
        if( data == 0 )     
            data = p;
        c = *p;
    }
    *data_p = data;
    if( data == 0L )
        return 0;
    return (p - data);    
}


unsigned
dhco_input_option
(   DHC *dhc,
    DHC_Option *option, 
    char *input, 
    void **data_value, 
    unsigned data_value_length 
)
{
    const char *type, *tp, *endtp;
    char quoted='\0', c='\0', *data=input, *endp, *value, *p;
    unsigned total_length=0, value_length=0, len=0;
    uint8_t is_array=0, pad;
    void *stored_value=(void*)1;
    DHCP_Enumeration *ep, **epp, es;
    DHCP_EnumValue  *ev, **evp, evs;

    if( ( option == 0L ) || (input == 0L)  || (data_value == 0L) )
        return 0;

    /*printf("data_value: %p\n", *data_value);*/

    type = option->type;
    endtp = type + strlen(type) - 1;
    endp = input + strlen(input);
    
    is_array = (*endtp == DHC_T_ARRAY) || (*endtp == DHC_T_LIST);

    if( !is_array )
        endtp++;
    
    do 
    {
        for(tp=type; tp < endtp; tp++)
        {
            if( (len = seek_next_data( &data, endp, &quoted )) )
            {
                value=0;
                if(quoted != '\0')
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_input_option: unbalanced quote in input: %s\n", input);
                    return 0;
                }

                if((data != 0L) && ((p=(data + len)) <= endp))
                {
                    c = *p;
                    *p ='\0';
                    value=data;
                    switch( *tp )
                    {
                    case DHC_T_ARRAY:
                    case DHC_T_LIST:
                        if(dhc->eh) (*(dhc->eh))("dhco_input_option: nested arrays are not allowed.\n");
                        *p = c;
                        return 0;

                    case DHC_T_ENUMERATION:
                        ++tp;
                        es.name = (char*)tp;
                        if( ((epp = tfind(&es, &(dhc->enums), enum_name_comparator)) == 0L) 
                          ||((ep = *epp) == 0L )
                          )
                        {
                            if(dhc->eh) (*(dhc->eh))("dhco_input_option: unknown enumeration %s.\n", tp);
                            *p = c;
                            return 0;
                        }
                        tp += strlen(tp);
                        evs.name = (char*)value;
                        if( ((evp=tfind( &evs, &(ep->ts), enum_value_name_comparator)) == 0L)
                          ||((ev = *evp)==0L) 
                          )
                        {
                            if(dhc->eh) (*(dhc->eh))("dhco_input_option: unknown enumeration %s value %s.\n", 
                                                     ep->name, value
                                                    );
                            *p = c;
                            return 0;
                        }
                        pad = DHC_PAD( int32_t, total_length);
                        total_length += sizeof(int32_t) + pad;                  
                        if( value_length )
                        {
                            stored_value += pad;
                            *((int32_t*)stored_value)=ev->number;
                        }
                        /* in DHCP, enum value ALWAYS consumes the rest of the data */
                        *p = c;
                        c='\0';
                        p=endp;
                        break;                  

                    case DHC_T_OPTIONAL:
                        continue;

                    case DHC_T_OPTION_SPACE:        
                    case DHC_T_ENCAPSULATION:
                    case DHC_T_ENCAPSULATION_NO_DIRECTIVE:
                        if(dhc->eh) (*(dhc->eh))("dhco_input_option: Unhandled encapsulation type\n");
                        data=p;
                        *p = c;
                        continue;
                
                    case DHC_T_TEXT:
                    case DHC_T_HEX_STRING:
                    case DHC_T_DOMAIN_NAME:
                        if ( is_array || (tp != (endtp-1)) )
                        {
                            if(dhc->eh) (*(dhc->eh))("dhco_input_option: embedded strings are not supported by DHCP.\n");
                            *p = c;
                            return( 0 );
                        }
                        /* in DHCP, a string ALWAYS consumes the rest of the data */
                        *p = c;
                        c='\0';
                        p=endp;

                    default:
                        len = dhco_value_from_text(dhc, *tp, value, &stored_value, value_length, total_length );
                        if( len == 0 ) 
                        {
                            if ( (tp == (endtp-1)) || (*(tp+1) != DHC_T_OPTIONAL) )
                            {
                                if(dhc->eh)
                                    (*(dhc->eh))("dhco_input_option: Value %s cannot be converted to type %c\n",
                                                 value, *tp
                                                );                          
                                *p = c;
                                return 0;
                            }else
                            {
                                data=p;
                                *p = c;
                                continue;
                            }
                        }
                        total_length += len;
                        stored_value += len;
                        if( value_length )
                        {
                            value_length -= len;
                            if( value_length == 0 )
                                value_length = total_length;
                        }
                        /*
                        printf("Value: %p %c %s %d total:%d left:%d\n", 
                               stored_value, *tp, value, len, total_length, value_length
                              );
                        */
                        *p = c;
                    }
                    data = p;
                    if( is_array && (tp == (endtp-1)) )
                        if( seek_next_data( &data, endp, &quoted ) )
                        {
                            data = p;
                            tp = type - 1;
                        }
                }else
                {       
                    if(dhc->eh) (*(dhc->eh))("dhco_input_option: insufficient data in %s for format %s\n",
                                             input, type
                                            );
                    return 0;
                }               
            }
        }
        if( value_length == 0 )
        {/* first "length calculation & parse check" pass 
            printf("first pass complete - %d\n", total_length);
         */
            if( *data_value == 0 )
            {
                if( (*data_value = malloc( total_length )) == 0L )
                    return 0;
            }else
            if( data_value_length < total_length )
            {
                *data_value = 0L;
                return total_length;
            };
            value_length = total_length;
            stored_value = *data_value;
            /*printf("stored_value: %p\n",stored_value);*/
            total_length = 0;
            data=input;
        }else
        {/* data storage pass complete 
            printf("final pass complete\n");
         */
            value_length = 0;
        }           
    }while( value_length > 0 );
    return total_length;
}

unsigned 
dhco_output_option
(   DHC *dhc,
    DHC_Option *option, 
    void *input_value, 
    unsigned input_length,
    char **output_value,     
    unsigned output_length
)
{
    void *input = input_value, *last_input;
    char *output=(char*)1;
    unsigned ilen, vlen=input_length, olen=0, total_olen=0, plen=0;
    char *tp, *type, *endtp;
    uint8_t is_array = 0, pad;
    DHCP_Enumeration *ep, **epp, es;
    DHCP_EnumValue  *ev, **evp, evs;
    
    
    type = option->type;
    endtp = type + strlen(type) - 1;
    
    is_array = (*endtp == DHC_T_ARRAY) || (*endtp == DHC_T_LIST);
    if( !is_array )
        endtp++;
    
    do
    {
        for( tp=type; tp < endtp; tp++ )
        {
            switch( *tp )
            {
            case DHC_T_ARRAY:
            case DHC_T_LIST:
                if(dhc->eh) (*(dhc->eh))("dhco_output_option: nested arrays are not allowed.\n");               
                return 0;

            case DHC_T_OPTIONAL:
                continue;

            case DHC_T_ENUMERATION:         
            {
                int32_t evn;
                ++tp;
                es.name = tp;
                if( ((epp = tfind(&es, &(dhc->enums), enum_name_comparator)) == 0L )
                  ||((ep = *epp)==0L)
                  )
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_output_option: unknown enumeration %s.\n", tp);
                    return 0;
                }
                tp += strlen(tp);
                last_input = input;             
                pad = DHC_PAD(uint32_t, total_olen);
                input += pad;
                evn = *((int32_t *)input);
                evs.number = evn;
                if( ((evp = tfind( &evs, &(ep->tn), enum_value_num_comparator)) == 0L)
                  ||((ev=*evp) == 0L)
                  )
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_output_option: unknown enumeration %s value %d.\n", 
                                                     ep->name, evn
                                            );     
                    return 0;
                }
                input += sizeof(int32_t);               
                total_olen += strlen(ev->name) + 1;
                if( olen )
                {
                    if( olen < (sizeof(int32_t) + pad) )
                    {
                        if(dhc->eh) (*(dhc->eh))("dhco_output_option: insufficient value_length %d for value %d.\n", 
                                                  olen, evn
                                            );     
                        return 0;
                    }
                    sprintf(output,"%s",ev->name);
                    olen = total_olen;
                }
            }   break; /* enum must always be last value! */

            case DHC_T_OPTION_SPACE:        
            case DHC_T_ENCAPSULATION:
            case DHC_T_ENCAPSULATION_NO_DIRECTIVE:
                if(dhc->eh) (*(dhc->eh))("dhco_output_option: Unhandled encapsulation type\n");         
                /*XXX: fixme!*/
                continue;
                
            case DHC_T_TEXT:
            case DHC_T_HEX_STRING:
            case DHC_T_DOMAIN_NAME:
                if ( is_array || (tp != (endtp-1)) )
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_output_option: embedded strings are not supported by DHCP.\n");
                    return( 0 );
                }
                
            default:
                last_input = input;
                /*printf("dhco_out: %c %p %p %d %d %d\n", *tp, input, output, olen, vlen, plen);*/
                ilen = dhco_text_from_value(dhc, *tp, &input, &output, olen, vlen, plen );
                if( ilen == 0 )
                {
                    if ( (tp != (endtp-1)) && (*(tp+1) == DHC_T_OPTIONAL) )
                        /*XXX: fixme! this probably won't work ... */
                        continue;
                    if(dhc->eh) (*(dhc->eh))("dhco_output_option: Output conversion failed for %c at %p\n",
                                             *tp, input
                                            );
                    return 0;
                }
                /*printf("out: %d\n",ilen);*/
                total_olen += ilen;
                plen += input - last_input;
                vlen -= input - last_input;
                /*printf("Output: %c %d '%s' %p %p %d olen:%d vlen:%d plen:%d tot:%d\n",
                       *tp, ilen, (olen ? output : ""), output, input, (input - last_input), olen, vlen, plen, total_olen
                      );
                */
                if( olen )
                {
                    olen -= ilen;
                    if( olen == 0 )
                        olen = total_olen;
                }
                if( ( tp < (endtp - 1) ) || ( is_array && (vlen > 0) ) )
                {
                    if( vlen == 0 ) 
                    {
                        if(dhc->eh) (*(dhc->eh))("dhco_output_option: Insufficient data for format %c at %p\n",
                                                 *tp, input
                                                );
                        return 0;
                    }else
                    {
                        if( olen > 0 )
                            *(output + (ilen-1))=' ';                                               
                    }
                }
                output += ilen;
            }
            if( (tp == (endtp-1)) && (vlen > 0) )
            {
                if( is_array ) 
                {
                    /*printf("Array - continuing\n");*/
                    tp = type - 1;
                    continue;
                }else
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_output_option: %d bytes left over data for type %s at %p\n",
                                             vlen, type, input
                                            );
                    return 0;
                }
            }
        }
                    
        if( ( olen == 0 ) && (total_olen > 0) )
        {
            /*printf("output: first pass complete: %d bytes required\n", total_olen );*/
            if( *output_value == 0L )
            {
                if((*output_value = malloc( total_olen ))==0)
                    return 0;
            }else
            if( output_length < total_olen )
                return total_olen;
            plen=0;
            vlen=input_length;
            olen=total_olen;        
            input = input_value;
            output= *output_value;
            total_olen = 0;
        }else
        {
            /*printf("output: final pass complete: %d\n", total_olen );*/
            olen = 0;
        }
    } while( olen );
    return total_olen;
}

DHCP_Option_Type_Mapping
dhco_new_mapping(DHC *dhc, DHCP_Option_Type_Map_Table mt )
{
    DHC_Mapping *m = (DHC_Mapping*) malloc( sizeof(DHC_Mapping) );
    uint8_t t;
    
    if( dhc==0L)
        return 0L;
    
    if( m == 0L) 
    {
        if(dhc->eh) (*(dhc->eh))("dhco_new_mapping: out of memory\n");
        return 0L;
    }

    m->mtree = 0L;

    for(t=0; t <  DHC_N_ATOMIC_TYPES; t++)
    {
        m->map[t].map_code = mt[t];
        m->map[t].dhc_code = dhc_atomic_types[t];
        tsearch( &(m->map[t]), &(m->mtree), map_comparator);
    }
    return m;
}

unsigned
dhco_map_option(DHC* dhc, DHC_Option *opt, void *value, unsigned total_length,
                DHCP_Option_Type_Mapping mapping, 
                void (*map)(uint8_t, uint8_t, void*, unsigned, void*),
                void *arg
               )
{
    uint8_t is_array=0;
    char *tp, *type, *endtp;
    unsigned length = 0, n_mapped=0;
    DHCP_Option_Type_Map *m, *const*mpp;

    if( ( opt == 0L ) || (value == 0L)  || (total_length == 0) || (map==0L)  )
        return 0;

    type = opt->type;
    endtp = type + strlen(type) - 1;
    
    is_array = (*endtp == DHC_T_ARRAY) || (*endtp == DHC_T_LIST);

    if( !is_array )
        endtp++;
    
    length=0;
    do
    {
        for( tp=type; tp < endtp; tp++ )
        {
            mpp = tfind(tp, &(mapping->mtree), map_comparator);
            if( (mpp==0L) || ((m=*mpp)==0L))
            {
                if(dhc->eh) (*(dhc->eh))("dhco_map_option: can't find mapping for type code %c\n", *tp);
                return n_mapped;
            }
            if( length >= total_length )
            {
                if( (!is_array) && (length > total_length) )
                {
                    if(dhc->eh) (*(dhc->eh))("dhco_map_option: required length %d exceeds value length %d\n",
                                             length, total_length
                                            );              
                }
                return n_mapped;
            }

            switch( *tp )
            {
            case DHC_T_ENUMERATION:         
                tp = endtp;
            case DHC_T_INT32:
            case DHC_T_UINT32:
            case DHC_T_IP_ADDRESS:
                length += DHC_PAD(uint32_t, length);
                (*map)( m->map_code,  m->dhc_code,  value + length, sizeof(uint32_t), arg );
                length += sizeof(uint32_t);
                ++n_mapped;
                break;

            case DHC_T_TEXT:
            case DHC_T_HEX_STRING:
            case DHC_T_DOMAIN_NAME:
                (*map)( m->map_code, m->dhc_code, value + length, total_length - length,  arg );
                ++n_mapped;
                return n_mapped; /* strings always consume rest of data! */

            case DHC_T_INT16:
            case DHC_T_UINT16:
                length += DHC_PAD(uint16_t, length);
                (*map)( m->map_code, m->dhc_code, value + length, sizeof(uint16_t), arg );
                length += sizeof(uint16_t);
                ++n_mapped;
                break;

            case DHC_T_BOOL:
            case DHC_T_IMPLICIT_BOOL:
            case DHC_T_CHAR:
            case DHC_T_UCHAR:
                (*map)( m->map_code, m->dhc_code,  value + length, sizeof(uint8_t), arg );
                length += sizeof(uint8_t);
                ++n_mapped;
                break;

            case DHC_T_ARRAY:
            case DHC_T_LIST:
                if(dhc->eh) (*(dhc->eh))("dhco_map_option: nested arrays are not allowed.\n");          
                return 0;

            case DHC_T_OPTIONAL:
                continue;
            case DHC_T_OPTION_SPACE:        
            case DHC_T_ENCAPSULATION:
            case DHC_T_ENCAPSULATION_NO_DIRECTIVE:
                if(dhc->eh) (*(dhc->eh))("dhco_map_option: Unhandled encapsulation type\n");            
                /*XXX: fixme!*/
                continue;
                
            default:
                if(dhc->eh) (*(dhc->eh))("dhco_map_option: Unhandled type code: %s\n");         
                /*XXX: fixme!*/
                continue;
            }
        }
    } while(length < total_length);
    return n_mapped;
}
