/*
 *   This file is part of Dianara
 *   Copyright 2012-2017  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "downloadwidget.h"


DownloadWidget::DownloadWidget(QString fileUrl,
                               QString suggestedFN,
                               PumpController *pumpController,
                               QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;
    this->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    this->downloading = false;

    this->fileUrl = fileUrl;
    this->suggestedFilename = suggestedFN;

    QFont infoFont;
    infoFont.setPointSize(infoFont.pointSize() - 1);

    infoLabel = new QLabel(this);
    infoLabel->setAlignment(Qt::AlignCenter);
    infoLabel->setFont(infoFont);
    infoLabel->hide();


    openButton = new QPushButton(QIcon::fromTheme("document-open",
                                                 QIcon(":/images/button-open.png")),
                                tr("Open", "Verb, as in: Open the downloaded file"),
                                this);
    openButton->setFlat(true);
    openButton->hide();
    connect(openButton, SIGNAL(clicked()),
            this, SLOT(openAttachment()));


    downloadButton = new QPushButton(QIcon::fromTheme("download",
                                                      QIcon(":/images/button-download.png")),
                                     tr("Download"),
                                     this);
    downloadButton->setToolTip("<b></b>"
                               + tr("Save the attached file to your folders"));
    downloadButton->setFlat(true);
    connect(downloadButton, SIGNAL(clicked()),
            this, SLOT(downloadAttachment()));


    progressBar = new QProgressBar(this);
    progressBar->setValue(0);
    progressBar->hide();       // Initially hidden


    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel",
                                                    QIcon(":/images/button-cancel.png")),
                                   tr("Cancel"),
                                   this);
    cancelButton->hide();
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(cancelDownload()));


    mainLayout = new QHBoxLayout();
    mainLayout->addWidget(infoLabel);
    mainLayout->addWidget(openButton);
    mainLayout->addWidget(downloadButton);
    mainLayout->addWidget(progressBar);
    mainLayout->addWidget(cancelButton);
    this->setLayout(mainLayout);

    qDebug() << "DownloadWidget created";
}

DownloadWidget::~DownloadWidget()
{
    qDebug() << "DownloadWidget destroyed";
}


void DownloadWidget::resetWidget()
{
    this->downloadButton->show();

    this->progressBar->hide();
    this->openButton->setToolTip("");
    this->openButton->hide();
    this->cancelButton->hide();

    networkReply->disconnect();
    networkReply->deleteLater();

    disconnect(pController, SIGNAL(downloadCompleted(QString)),
               this, SLOT(completeDownload(QString)));
    disconnect(pController, SIGNAL(downloadFailed(QString)),
               this, SLOT(onDownloadFailed(QString)));

    downloadedFile.close();

    this->downloading = false;
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////// SLOTS ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////


void DownloadWidget::downloadAttachment()
{
    if (downloading)
    {
        qDebug() << "Already downloading!!";
        return;
    }

    QString filename;
    filename = QFileDialog::getSaveFileName(this, tr("Save File As..."),
                                            QDir::homePath() + "/"
                                            + suggestedFilename,
                                            tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        this->downloading = true;

        this->infoLabel->setText("...");
        this->infoLabel->setToolTip("");
        this->infoLabel->show();
        this->downloadButton->hide();
        this->openButton->hide();

        this->progressBar->show();
        this->progressBar->setValue(0);
        this->progressBar->setToolTip("");

        this->cancelButton->show();

        this->downloadedFile.setFileName(filename);
        downloadedFile.open(QIODevice::WriteOnly);

        networkReply = this->pController->getMedia(fileUrl);
        connect(networkReply, SIGNAL(readyRead()),
                this, SLOT(storeFileData()));
        connect(networkReply, SIGNAL(downloadProgress(qint64,qint64)),
                this, SLOT(updateProgressBar(qint64,qint64)));

        connect(pController, SIGNAL(downloadCompleted(QString)),
                this, SLOT(completeDownload(QString)));
        connect(pController, SIGNAL(downloadFailed(QString)),
                this, SLOT(onDownloadFailed(QString)));
    }
}


void DownloadWidget::openAttachment()
{
    /*
     * TODO: Open button could be visible from the start, and actually
     *       download the file to temporary storage before opening it.
     *       This would hide the Download button.
     *       When download completes, a new "Save" button would appear, to
     *       let the user save the temporary file to regular storage.
     */
    if (downloadedFile.exists())
    {
        QDesktopServices::openUrl(QUrl::fromLocalFile(downloadedFile.fileName()));
    }
    else
    {
        this->infoLabel->setText(tr("File not found!"));
        this->infoLabel->setToolTip(downloadedFile.fileName());
        this->openButton->hide();
        this->downloadButton->show(); // Chance to download again.
    }
}


void DownloadWidget::cancelDownload()
{
    int confirmation = QMessageBox::question(this, tr("Abort download?"),
                                             tr("Do you want to stop "
                                                "downloading the attached "
                                                "file?"),
                                             tr("&Yes, stop"),
                                             tr("&No, continue"),
                                             "", 1, 1);

    if (confirmation == 0)
    {
        this->networkReply->abort();

        infoLabel->setText(tr("Download aborted"));
        resetWidget();
    }
    else
    {
        qDebug() << "Confirmation cancelled, NOT stopping download";
    }
}


void DownloadWidget::completeDownload(QString url)
{
    if (url == this->fileUrl) // Ensure completed download is this download
    {
        infoLabel->setText(tr("Download completed"));
        resetWidget(); // This also closes the file

        // FIXME: pass this through GlobalObject
        this->pController->showStatusMessageAndLogIt(tr("Attachment downloaded "
                                                        "successfully to %1",
                                                        "%1 = filename")
                                                     .arg(this->downloadedFile.fileName()));

        this->openButton->setToolTip("<b></b>"
                                     + tr("Open the downloaded attachment with "
                                          "your system's default program for "
                                          "this type of file.")
                                     + "<br /><br />"
                                     + this->downloadedFile.fileName());
        this->openButton->show();
        this->downloadButton->hide();
    }
}


void DownloadWidget::onDownloadFailed(QString url)
{
    if (url == this->fileUrl) // Ensure failed download is this download
    {
        infoLabel->setText(tr("Download failed"));
        infoLabel->setToolTip(url);
        qDebug() << "Download FAILED!" << url;

        resetWidget();

        // FIXME: pass this through GlobalObject
        this->pController->showStatusMessageAndLogIt(tr("Downloading attachment "
                                                        "failed: %1",
                                                        "%1 = filename")
                                                     .arg(this->downloadedFile.fileName()),
                                                     url);
    }
}


void DownloadWidget::storeFileData()
{
    QByteArray data = networkReply->readAll();
    this->downloadedFile.write(data);
}


void DownloadWidget::updateProgressBar(qint64 received, qint64 total)
{
    this->progressBar->setRange(0, total);
    this->progressBar->setValue(received);

    this->infoLabel->setText(tr("Downloading %1 KiB...")
                             .arg(QLocale::system().toString(total / 1024)));

    QString downloadedTooltip = tr("%1 KiB downloaded")
                                .arg(QLocale::system().toString(received / 1024));
    this->infoLabel->setToolTip(downloadedTooltip);
    this->progressBar->setToolTip(downloadedTooltip);
}
