/* idlist.c -- routines for manipulating lists of id-based item
 *
 * Copyright (c) 1998-2002  Mike Oliphant <oliphant@gtk.org>
 *
 *   http://www.nostatic.org/grip
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "idlist.h"
#include "rrand.h"

static int NumCompare(const void *ptr1,const void *ptr2);

/* Free up the contents of an idlist */
void FreeIDList(IDList *idlist)
{
  if(idlist->ids) g_free(idlist->ids);

  idlist->ids=NULL;
  idlist->num_ids=0;
  idlist->current_id=0;
}
/* Copy one idlist to another */
void CopyIDList(IDList *src_idlist,IDList *dest_idlist)
{
  dest_idlist->ids=g_new(int,src_idlist->num_ids);

  memcpy(dest_idlist->ids,src_idlist->ids,sizeof(int)*src_idlist->num_ids);

  dest_idlist->num_ids=src_idlist->num_ids;
  dest_idlist->current_id=0;
}

void TrimIDList(IDList *idlist,int size)
{
  int *new_ids;

  if(idlist->num_ids>size) {
    new_ids=g_new(int,size);
    memcpy(new_ids,idlist->ids,sizeof(int)*size);

    g_free(idlist->ids);

    idlist->ids=new_ids;
    idlist->num_ids=size;

    if(idlist->current_id>(idlist->num_ids-1)) 
      idlist->current_id=idlist->num_ids-1;
  }
}

/* Remove 'num' items from an idlist, beginning at position 'pos' */
void RemoveIDs(IDList *idlist,int pos,int num)
{
  /* Can't remove past the end */
  if(pos+num>idlist->num_ids) return;

  if((pos+num)<idlist->num_ids){
    memmove(idlist->ids+pos,idlist->ids+pos+num,
	    sizeof(int)*(idlist->num_ids-(pos+num)));
  }

  idlist->num_ids-=num;

  if(idlist->current_id>(idlist->num_ids-1))
    idlist->current_id=idlist->num_ids-1;
}

void InsertIDs(IDList *idlist,IDList *insert_list,int pos)
{
  int *new_ids;

  new_ids=(int *)g_new(int,idlist->num_ids+insert_list->num_ids);

  if(pos)
    memmove(new_ids,idlist->ids,sizeof(int)*pos);

  memmove(new_ids+pos,insert_list->ids,sizeof(int)*insert_list->num_ids);

  if(pos!=idlist->num_ids)
    memmove(new_ids+pos+insert_list->num_ids,idlist->ids+pos,
	    sizeof(int)*(idlist->num_ids-pos));

  g_free(idlist->ids);

  idlist->ids=new_ids;
  idlist->num_ids+=insert_list->num_ids;
}

void InsertID(IDList *idlist,int id,int pos)
{
  int *new_ids;

  new_ids=(int *)g_new(int,idlist->num_ids+1);

  if(pos)
    memmove(new_ids,idlist->ids,sizeof(int)*pos);

  new_ids[pos]=id;

  if(pos!=idlist->num_ids)
    memmove(new_ids+pos+1,idlist->ids+pos,
	    sizeof(int)*(idlist->num_ids-pos));

  g_free(idlist->ids);

  idlist->ids=new_ids;
  idlist->num_ids+=1;
}

static int NumCompare(const void *ptr1,const void *ptr2)
{
  int *val1,*val2;

  val1=(int *)ptr1;
  val2=(int *)ptr2;

  return *val1-*val2;
}

/* Sort an idlist by id */
void SortIDList(IDList *idlist)
{
  qsort(idlist->ids,idlist->num_ids,sizeof(int),NumCompare);
}

/* Ensure that an idlist has no duplicate ids */
void UniqeIDList(IDList *idlist)
{
  int pos;
  int newpos;

  /* First, sort it */
  SortIDList(idlist);

  /* Now, remove any dups */
  for(pos=newpos=0;pos<(idlist->num_ids-1);pos++) {
    if(idlist->ids[pos]!=idlist->ids[pos+1])
      idlist->ids[newpos++]=idlist->ids[pos];
  }

  /* Make sure we get the last one */
  idlist->ids[newpos++]=idlist->ids[pos];

  idlist->num_ids=newpos;
}

/* Randomize an array of integers */
void ScrambleInts(int *list,int size)
{
  int p1,p2,tmp;

  for(p1=0;p1<size;p1++) {
    p2=RRand(size);

    tmp=list[p1];
    list[p1]=list[p2];
    list[p2]=tmp;
  }
}

/* Create a new id stack */
IDStack *NewIDStack(int size)
{
  IDStack *idstack;
  int pos;

  idstack=g_new(IDStack,1);
  if(!idstack) return NULL;

  idstack->id_lists=g_new(IDList,size);
  if(!idstack->id_lists) {
    g_free(idstack);

    return NULL;
  }

  for(pos=0;pos<size;pos++) {
    idstack->id_lists[pos].ids=NULL;
    idstack->id_lists[pos].num_ids=0;
    idstack->id_lists[pos].current_id=0;
  }

  idstack->stack_pos=0;
  idstack->stack_size=size;
  idstack->current_idlist=&idstack->id_lists[0];

  return idstack;
}

/* Free an ID stack and it's song lists*/
void FreeIDStack(IDStack *idstack)
{
  int pos;

  for(pos=0;pos<idstack->stack_size;pos++) {
    FreeIDList(&idstack->id_lists[pos]);
  }

  g_free(idstack->id_lists);
  g_free(idstack);
}

/* Push the idlist stack, giving us a fresh one at the top */
void PushIDList(IDStack *idstack)
{
  int list_num;
  IDList *idlists;

  idlists=idstack->id_lists;

  if(idstack->stack_pos==(idstack->stack_size-1)) {
    /* Need to throw one away */
    FreeIDList(&idlists[0]);

    /* Shift everything down one position in the stack */
    for(list_num=1;list_num<idstack->stack_size;list_num++) {
      idlists[list_num-1]=idlists[list_num];
    }
  }
  else {
    /* Increment to the next position in the stack */
    idstack->stack_pos++;
    idstack->current_idlist=&idlists[idstack->stack_pos];
  }
}

/* Pop off the top of an ID stack */
gboolean PopIDList(IDStack *idstack)
{
  if(!idstack->stack_pos) return FALSE;

  FreeIDList(idstack->current_idlist);
  idstack->stack_pos--;
  idstack->current_idlist=&idstack->id_lists[idstack->stack_pos];

  return TRUE;
}
