#!/usr/local/bin/perl

# GEBRUIK: Verifieer EGB96 OWW LIJST
#
# DOEL:   programma om woordenlijst (LIJST) te corrigeren aan de hand van
#         een woordenboek (DICT) en een lijst van onregelmatige 
#	  werkwoorden (OWW). Het resultaat zijn de woorden uit LIJST
#         voorafgegaan door een extra teken om aan te geven hoe de relatie
#	  is met DICT. De volgende codes komen voor:
#	     = identiek aan EGB96
#	     # ontbrekende woordvorm in EGB96
#	     + komt niet voor in EGB96
#	     - spelling vervallen vervalt: alternatief
#	     ? twijfelachtig woord
#
#         Indien op grond van eenvoudige heurisieken blijkt dat een 
#	  woord losgeschreven dient te worden of een ongeldige werkwoords-
#	  vorm is (uitgevallenen) dan komt aan het einde van de regel de
#	  kode `>' (eruit). Voor woordvormen die ten onrechte ontbreken
#	  omdat ze niet voorkomen in de DICT worden voorzien van `<' op 
#	  het einde (erin). Dit om het aantal woorden dat anders met de hand 
#	  nagelopen moet worden te beperken.
#	  Het programma werkt niet op delen van woordenbestanden tenzij
#	  alle ge-werkwoordsvormen aanwezig zijn. Woorden zullen na afloop
#	  handmatig moeten worden gekontroleerd!!
#	  Het programma vreet erg veel geheugen omdat alle woorden uit
#	  DICT in het geugen worden ingelezen.

# AUTEUR: Piet Tutelaers
# VERSIE: 1.0 (augustus 1996)

$debug = 0;

die "Gebruik: verifieer groenboekje oww lijst\n" unless @ARGV == 3;

open(DICT, "< $ARGV[0]") || die "Can't open $ARGV[0]\n";
open(OWW,  "< $ARGV[1]") || die "Can't open $ARGV[1]\n";
open(LIJST,"< $ARGV[2]") || die "Can't open $ARGV[2]\n";

%dictionary = ();       # uitspraak per woord
%anders = ();           # spelling inclusief hoofdletters
%uitspraken = ();       # woorden met dezelfde uitspraak

&voorzetsels; 		# initialiseer

# lees onregematige werkwoordsvormen
while (<OWW>) {
   chop;
   s/#.*$//;
   next if /^$/;
   (@woord)  = split(/\s+/);
   warn "$_: overgeslagen\n" unless @woord == 6;
   next unless @woord == 6;
   $oww{$woord[0]} = 1; # onregelmatig werkwoord
   $ovt{$woord[1]} = 1; # onregematige verledentijd
   $ovt{$woord[2]} = 1; # onregematige verledentijd
   $ott{$woord[3]} = 1; # onregematige tegenwoordigetijd
   $ott{$woord[4]} = 1; # onregematige tegenwoordigetijd
   $ovd{$woord[5]} = 1; # onregelmatig voltooid deelwoord
}

while (<DICT>) {
   chop;
   s/\255//g; # verwijder afbreekstreepjes 
   $spelling = $_;
   tr/A-Z/aacenoua-z/;
   s/[' -]//g;
   $anders{$_} = $spelling if $spelling ne $_;
   $uitspraak =  &klinktals($_);
   $dictionary{$_} = $uitspraak;
   $uitspraken{$uitspraak} .= $uitspraken{$uitspraak} ? "|$_" : "$_";
}

while (<LIJST>) {
   chop;
   s/\255//g; # verwijder afbreekstreepjes 
   $lijst++;
   if ($anders{$_}) {
      print "-$_:$anders{$_}\n";		# andere spelling
      next;
   }
   if ($dictionary{$_}) {
      print "=$_\n";				# identiek aan DICT
      next;
   }
   $uitspraak =  &klinktals($_);
   if ($uitspraken{$uitspraak}) {
      print "-$_:$uitspraken{$uitspraak}\n";	# vervallen spelling?
      next;
   }
   if (/er$/) {			# ontbrekende comparatief?
      ($grondwoord) = /(.*)er$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
      # probeer: iler, [gz]amer, maner, lozer, euzer
      $grondwoord =~ s/l$/eel/	if $grondwoord =~ /il$/;
      $grondwoord =~ s/am$/aam/	if $grondwoord =~ /[gz]am$/;
      $grondwoord =~ s/an$/aan/	if $grondwoord =~ /[m]an$/;
      $grondwoord =~ s/oz$/oos/	if $grondwoord =~ /[l]oz$/;
      $grondwoord =~ s/z$/s/	if $grondwoord =~ /euz$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
   }
   if (/st$/) {			# ontbrekende superlatief?
      ($grondwoord) = /(.*)st$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
      # probeer: eust
      $grondwoord =~ s/eu$/eus/	if $grondwoord =~ /eu$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
   }
   if (/je[s]?$/) { 		# ontbrekende -je of -jes vorm?
      ($grondwoord) = /(.*)je[s]?$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
      if ($grondwoord =~ /t$/) {
         # probeer: grondwoord - t
	 chop($grondwoord);
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
   	    next;
         }
         # probeer: mmetje, nnetje, ngetje
         $grondwoord =~ s/..$//	if $grondwoord =~ /(mm|nn)e$/;
         $grondwoord =~ s/.$//	if $grondwoord =~ /nge$/;
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
   	    next;
         }
      }
   }
   # verwijder foutieve woorden als uitgevallene/uitgevallenen
   # we gaan ervan uit dat woorden als veroordeelde(n) in GB staan
   if (/ne[n]?$/) {
      ($vz, $grondwoord) = &splitswerkwoord($_);
      $grondwoord =~ s/e[n]?$//;
      if ($ovd{$grondwoord}) {
         print "-$_>\n";		# fout woord
         next;
      }
   }
   # verwijder foutieve woorden als uitgedoofden
   if (/[dt]en$/) {
      ($vz, $grondwoord) = &splitswerkwoord($_);
      $grondwoord =~ s/en$//;
      if ($vz && $dictionary{$grondwoord}) {
         print "-$_>\n";		# fout woord
         next;
      }
   }
   # ontbrekend meervoud/bijv. naamwoord
   if (/e[n]?$/ && !/(dd|tt)e[n]?$/) {
      ($grondwoord) = /(.*)e[n]?$/;
      &show($grondwoord);
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteren?
	 next;
      }
      # probeer: golven, elzen 
      if (/[^aeiou][vz]en$/) {
         ($grondwoord) = /(.*)en$/;
	 $grondwoord =~ s/v$/f/;
	 $grondwoord =~ s/z$/s/;
         &show($grondwoord);
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
      }
      # probeer: isme+n, nome+n, ine+n, dove+n
      if (/en$/) {
         ($grondwoord) = /(.*)n$/;
         &show($grondwoord);            
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
	 if (/omenen$/) { # phenomena
	    ($grondwoord = $_) =~ s/en$/a/;
            &show($grondwoord);             
            if ($dictionary{$grondwoord}) {
               print "#$_<\n";	# accepteren?
               next;
            }
	 }
	 if (/smeden$/) { # smeden
	    ($grondwoord = $_) =~ s/eden$/id/;
            &show($grondwoord);
            if ($dictionary{$grondwoord}) {
               print "#$_<\n";		# accepteren?
               next;
            }
	 }
      }
      # probeer: mme[n], nne[n]
      if (/(bb|cc|dd|ff|gg|kk|ll|mm|nn|pp|rr|ss)e[n]?$/) {
         ($grondwoord) = /(.*).e[n]?$/;
         &show($grondwoord);             
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
      }
      # probeer: fone[n], ane[n], ianen, etc.
      if (/([^aeiou][aeiou][bcdfgklmnprstvwz]|ian)e[n]?$/) {
         ($grondwoord) = /(.*)e[n]?$/;
	 $grondwoord =~ s/z$/s/; $grondwoord =~ s/v$/f/;
	 $grondwoord =~ s/(.)(.)$/\1\1\2/;
         &show($grondwoord);
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
      }
      # probeer: heden (heid)
      if (/heden$/) {
	 ($grondwoord = $_) =~ s/heden$/heid/;
         &show($grondwoord);
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
      }
      # probeer: ile (ieel) en euze (eus)
      if (/ile$/ || /euze$/) {
	 ($grondwoord = $_) =~ s/le$/eel/;
	 $grondwoord =~ s/euze$/eus/;
         &show($grondwoord);
         if ($dictionary{$grondwoord}) {
            print "#$_<\n";	# accepteren?
	    next;
         }
      }
   }
   if (/[^s]s$/) {
      ($grondwoord) = /(.*)s$/;
      if ($dictionary{$grondwoord}) {
         print "#$_<\n";	# accepteer meervoudsvorm
	 next;
      }
   }
   if (/ss$/) {
      ($grondwoord) = /(.*)s$/;
      if ($dictionary{$grondwoord}) {
         print "-$_:$grondwoord'\n";	# eindapostroph (verouderd?)
	 next;
      }
   }
   # ontbrekende werkwoordsvorm?
   $wwvorm = &werkwoord($_);
   if ($wwvorm ne "") {
      if ($wwvorm =~ / /) {
         print "-$_:$wwvorm>\n";# losgeschreven!
      	 next;
      }
      print "#$_<\n";		# ontbrekend ww
      next;
   }
   if (/(ing|ingen)$/) {
      ($grondwoord) = /(.*)(ing|ingen)$/;
      if (&werkwoord($grondvorm) ne "") {
         print "#$_<\n";	# ontbrekend -ing/ingen vorm?
         next;
      }
   }
   print "+$_\n";		# extra woord
}

sub klinktals {
   local($_) = @_;
   
   tr//aaaaceeeiiinooouuu/;
   s/ch([aeiou])/CH\1/g;	# kachel wordt kaCHel
   s/c/K/g;		        # carel wordt Karel
   s/e([])[n]?/E\1/g;	# kippe[n]i wordt kippEi
   s//-a/g;		        # 
   s//-e/g;		        # 
   s//-i/g;		        # 
   s//-o/g;		        # 
   s//-u/g;		        # 
   s/([aeiou])ng/\1NG/g;	# ring wordt riNG
   s/([^aeiou])e[n]?([^aeiou])/\1E\2/g;	# note[n]boom wordt notEboom
   s/ss/S/g;			# dubbel ss
   $_;
}

# ga na of de voltooid deelwoord bestaat en zo ja retourneer
# deze, anders de lege string. Alleen regelmatige werkwoorden.
sub voltooiddeelwoord {
   local($_, $uitgang) = @_; # stam plus optionele uitgang

   s/$uitgang$// if $uitgang;
   # de kofschip-regel
   if (/(k|f|s|ch|p)$/ && !/(aa|ee|oo|uu|ie|ei|eu|ui|ij|ou|au)[fs]$/) {
      $_ .= "t";
   }
   elsif (/t$/) { # woord op -t
      $_ .= "t";
   }
   elsif (/d$/) { # woord op -d
      $_ .= "d";
   }
   else { # woorden die niet voldoen aan kofschip
      $_ .= "d";
   }
   return "ge" . $_ if $dictionary{"ge" . $_};
   return $_ if $dictionary{$_} && &vtenkelvoud($_, "e");
   "";
}

# ga na of de verleden tijd enkelvoud bestaat en zo ja retourneer
# deze, anders de lege string. Alleen regelmatige werkwoorden.
sub vtenkelvoud {
   local($_, $uitgang) = @_; # stam plus optionele uitgang

   s/$uitgang$// if $uitgang;
   # de kofschip-regel
   if (/(k|f|s|ch|p)$/ && !/(aa|ee|oo|uu|ie|ei|eu|ui|ij|ou|au)[fs]$/) {
      $_ .= "te";
   }
   elsif (/t$/) { # woord op -t
      $_ .= "te";
   }
   elsif (/d$/) { # woord op -d
      $_ .= "de";
   }
   else { # woorden die niet voldoen aan kofschip
      $_ .= "de";
   }
   defined $dictionary{$_} ? $_ : "";
}

# Bepaal langste voorvoegsel in werkwoord

sub splitswerkwoord {
   local($_) = @_;
   local($i, $l, $prefix, $voorvoegsel);

   $l = length($_);
   for ($i = $vzmin; $i < $vzmax; $i++) {
      last if $i > $l;
      $prefix = substr($_, 0, $i);
      $voorvoegsel = $prefix if defined $vz{$prefix};
   }
   if (defined $voorvoegsel) {
      $i = length($voorvoegsel);
      ($voorvoegsel, substr($_, $i, $l-$i));
   }
   else {
      ('', $_);
   }
}

# Ga na of argument een ww vorm is. En zo ja retourneer de korrekte
# schrijfwijze (`weg fietst' of `overlegt').
sub werkwoord {
   local($ww) = @_;
   local($_, $vz, $stam);

   # splits werkwoord in voorvoegsel en de `ruwe' stam
   # maar voorkom dat zoekt gesplitst wordt in `zoek' en `t'
   ($vz, $stam) = &splitswerkwoord($ww);
   if (length($stam) < 3 && $stam ne "at") {
      $stam = $vz . $stam;  
      $vz = '';
   }

   # een onregelmatig werkwoord?
   return $ww if $ott{$ww} || $ovt{$ww};
   return "$vz $stam" if $ott{$stam} || $ovt{$stam};

   # De verleden tijd enkelvoud? Het meest eenvoudige geval, immers
   # deze staan in DICT!
   if ($stam =~ /[dt]e$/) { # stam+[dt]e
      return $ww if $dictionary{$ww};
      return "$vz $stam" if $dictionary{$stam};
      return "";
   }

   # de verleden tijd meervoud?
   if ($stam =~ /[dt]en$/) { # stam+[dt]en
      return $ww  if $dictionary{$ww}; # werkwoord als `braden'
      return "$vz $stam" if $dictionary{$stam}; # `aan braden'
      chop($stam);
      return $ww if $dictionary{"$vz$stam"}; # braadden
      return "$vz ${stam}n" if $dictionary{$stam}; # aan braadden
      return "";
   }

   # regelmatige stam+t ?
   if ($stam =~ /t$/) {
      return $ww if &vtenkelvoud($ww, "t");
      return "$vz $stam" if &vtenkelvoud($stam, "t");
   }

   # regelmatige stam ?
   return $ww if &vtenkelvoud($ww);
   return "$vz $stam" if &vtenkelvoud($stam);
   return "";

   # stam?
   # 1: verdubbel laatste medeklinker (behalve bij drentel-en)
   $_ = $stam;
   if (/[^aeiou][aeiou][ptkcbdfsglrmn]$/) {
      s/(.)$/\1\1/ unless defined $dictionary{$_."en"};
   }
   # 2: verenkel dubbele klinker; vervang f door v en s door z
   elsif (/(aa|ee|oo|uu|ie|ei|eu|ui|ij|ou|au)[ptkcbdfsglrmn]$/) {
      s/(.)(.)(.)$/\1\3/ if /(aa|ee|oo|uu).$/;
      s/f$/v/; s/s$/z/;
   }
   return $ww  if $dictionary{"$vz$_de"}; # werkwoord als `braadde'
   return "$vz $stam"  if $dictionary{$_ . "de"}; # `aan braadde'
}

sub show{
   local($_) = @_;
   print "probeer: $_\n" if $debug;
}

sub showonly{
   local($_, $tekst) = @_;
   $tekst = "probeer" unless $tekst;
   print "$tekst: $_\n";
}

# initilaiseer %vz met voorzetsels van werkwoorden die losgeschreven 
# moeten te worden.
sub voorzetsels {
   $vz{'aan'} = 1;	   $vz{'gewaar'} = 1;	   $vz{'rond'} = 1;
   $vz{'aaneen'} = 1;	   $vz{'glad'} = 1;	   $vz{'samen'} = 1;
   $vz{'achter'} = 1;	   $vz{'goed'} = 1;	   $vz{'schaatsen'} = 1;
   $vz{'achteraan'} = 1;   $vz{'groen'} = 1;	   $vz{'school'} = 1;
   $vz{'achterna'} = 1;	   $vz{'groot'} = 1;	   $vz{'schoon'} = 1;
   $vz{'achterom'} = 1;	   $vz{'heen'} = 1;	   $vz{'schoot'} = 1;
   $vz{'achterop'} = 1;	   $vz{'heet'} = 1;	   $vz{'schuil'} = 1;
   $vz{'achterover'} = 1;  $vz{'hoog'} = 1;	   $vz{'stand'} = 1;
   $vz{'achteruit'} = 1;   $vz{'huis'} = 1;	   $vz{'stil'} = 1;
   $vz{'adem'} = 1;	   $vz{'in'} = 1;	   $vz{'stop'} = 1;
   $vz{'af'} = 1;	   $vz{'ineen'} = 1;	   $vz{'storm'} = 1;
   $vz{'auto'} = 1;	   $vz{'kaart'} = 1;	   $vz{'stuk'} = 1;
   $vz{'bakzeil'} = 1;	   $vz{'kapot'} = 1;	   $vz{'tegen'} = 1;
   $vz{'beet'} = 1;	   $vz{'klaar'} = 1;	   $vz{'tekeer'} = 1;
   $vz{'bekend'} = 1;	   $vz{'koffie'} = 1;	   $vz{'teleur'} = 1;
   $vz{'bezig'} = 1;	   $vz{'krom'} = 1;	   $vz{'teloor'} = 1;
   $vz{'bij'} = 1;	   $vz{'kwaad'} = 1;	   $vz{'tentoon'} = 1;
   $vz{'bijeen'} = 1;	   $vz{'kwijt'} = 1;	   $vz{'terecht'} = 1;
   $vz{'binnen'} = 1;	   $vz{'lam'} = 1;	   $vz{'terug'} = 1;
   $vz{'blind'} = 1;	   $vz{'leeg'} = 1;	   $vz{'teweeg'} = 1;
   $vz{'bloot'} = 1;	   $vz{'lief'} = 1;	   $vz{'thuis'} = 1;
   $vz{'boek'} = 1;	   $vz{'los'} = 1;	   $vz{'toe'} = 1;
   $vz{'bot'} = 1;	   $vz{'maat'} = 1;	   $vz{'uit'} = 1;
   $vz{'boven'} = 1;	   $vz{'mede'} = 1;	   $vz{'uiteen'} = 1;
   $vz{'buit'} = 1;	   $vz{'mee'} = 1;	   $vz{'vaneen'} = 1;
   $vz{'buiten'} = 1;	   $vz{'meer'} = 1;	   $vz{'vast'} = 1;
   $vz{'cologne'} = 1;	   $vz{'meest'} = 1;	   $vz{'vet'} = 1;
   $vz{'daar'} = 1;	   $vz{'mis'} = 1;	   $vz{'vol'} = 1;
   $vz{'dank'} = 1;	   $vz{'mooi'} = 1;	   $vz{'voor'} = 1;
   $vz{'deel'} = 1;	   $vz{'na'} = 1;	   $vz{'vooraf'} = 1;
   $vz{'dicht'} = 1;	   $vz{'nabij'} = 1;	   $vz{'voorbij'} = 1;
   $vz{'dood'} = 1;	   $vz{'neer'} = 1;	   $vz{'voorop'} = 1;
   $vz{'door'} = 1;	   $vz{'om'} = 1;	   $vz{'voort'} = 1;
   $vz{'dooreen'} = 1;	   $vz{'omver'} = 1;	   $vz{'vooruit'} = 1;
   $vz{'droog'} = 1;	   $vz{'onder'} = 1;	   $vz{'vrij'} = 1;
   $vz{'feest'} = 1;	   $vz{'op'} = 1;	   $vz{'vuil'} = 1;
   $vz{'fijn'} = 1;	   $vz{'opeen'} = 1;	   $vz{'waar'} = 1;
   $vz{'flauw'} = 1;	   $vz{'open'} = 1;	   $vz{'weder'} = 1;
   $vz{'ga'} = 1;	   $vz{'over'} = 1;	   $vz{'weer'} = 1;
   $vz{'gade'} = 1;	   $vz{'overeen'} = 1;	   $vz{'weerom'} = 1;
   $vz{'gelijk'} = 1;	   $vz{'paard'} = 1;	   $vz{'weg'} = 1;
   $vz{'geluk'} = 1;	   $vz{'plaats'} = 1;	   $vz{'wel'} = 1;
   $vz{'gereed'} = 1;	   $vz{'prijs'} = 1;	   $vz{'wijs'} = 1;
   $vz{'gering'} = 1;	   $vz{'recht'} = 1;	   $vz{'zoek'} = 1;
   $vz{'gerust'} = 1;		
   $vzmin=20; $vzmax=0;
   for (keys(%vz)) {
      $l = length($_);
      $vzmin = $l if $l < $vzmin;
      $vzmax = $l if $l > $vzmax;
   }
}
