# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import os.path
import copy
import xml.dom.minidom

import gtk
import gobject

from gazpacho import util
from gazpacho.loader import tags
from gazpacho.propertyclass import PropertyClass, PropertyException
from gazpacho.propertyclass import SubPropertyClass
from gazpacho.propertyclass import type2str
from gazpacho.path import pixmaps_dir
from gazpacho.widgetregistry import widget_registry
from gazpacho import function

class WidgetClass(object):
    def __init__(self, node, library, resource_path=None,
                 register_type_function=None):
        name = node.getAttribute(tags.NAME)
        adaptor_class_name = node.getAttribute(tags.ADAPTOR_CLASS)

        if widget_registry.has_name(name):
            print _('The widget class %s has at least two different definitions') % name
            return

        self.name = name
        self.generic_name = node.getAttribute(tags.GENERIC_NAME)
        self.palette_name = node.getAttribute(tags.TITLE)
        self.library = library

        self.resource_path = resource_path
        
        self.in_palette = self.generic_name and True or False

        self.library.register_type(name, register_type_function)
        
        # now we can get the type safely
        try:
            self.type = gobject.type_from_name(self.name)
        except RuntimeError:
            raise AttributeError, "There is no registered widget called %s" % \
                  self.name

        # get the adaptor class
        self.adaptor = self.library.get_widget_adaptor_class(self.name,
                                                             adaptor_class_name)
                
        self.properties = self._list_properties()
        self.child_properties = self._list_child_properties()
        self.signals = self._list_signals()
        
        # Is property_class of ancestor applicable to the widget? Usually
        # property_class only applies to direct children of a given ancestor
        self.child_property_applies = _direct_children
        
        self.icon = self._create_icon()

        # This method replaces a child widget with another one: it's used to
        # replace a placeholder with a widget and viceversa.
        self.replace_child = None

        # Executed after widget creation: it takes care of creating the
        # GladeWidgets associated with internal children. It's also the place
        # to set sane defaults, e.g. set the size of a window.
        self.post_create_function = None

        # Executed before setting the properties of the widget to its initial
        # value as specified in the xml file
        self.pre_create_function = None

        # If the widget is a container, this method takes care of adding the
        # needed placeholders.
        self.fill_empty_function = None

        # Retrieves the internal child of the given name or if the name
        # is None return a tuple with the possible names for its internal
        # children
        self.get_internal_child = None

        # this virtual method is used to serialize the widget (e.g. get
        # a XML node of its contents)
        self.save_function = None

        # this virtual method is used to build a GWidget from a GtkWidget
        # The loading is a two step process: first we get the widget tree from
        # gazpacho.loader or libglade and then we generate the wrappers
        # (GWidgets) from that widget tree. This function is responsable of
        # the second step of this loading process
        self.load_function = None
        
        parent = gobject.type_parent(self.type)
        while parent:
            parent_class = widget_registry.get_by_type(parent, None)
            if parent_class:
                self._merge(parent_class)
            try:
                parent = gobject.type_parent(parent)
            except RuntimeError:
                # no more parents
                parent = None

        # store the GladeWidgetClass on the cache
        widget_registry.add(gobject.type_name(self.type), self)

        self._extend_with_node(node)


    def is_toplevel(self):
        window_gtype = gobject.type_from_name('GtkWindow')
        return gobject.type_is_a(self.type, window_gtype)

    def _list_properties(self):
        result = []
        for prop in gobject.list_properties(self.type):
            if prop.name == 'user-data':
                continue
            # we only use writable properties
            if (prop.flags & gobject.PARAM_WRITABLE != 0) and \
               (prop.flags & gobject.PARAM_CONSTRUCT_ONLY == 0):
                try:
                    if prop.name == 'ellipsize':
                        continue
                    prop_class = PropertyClass(prop)
                    prop_class.optional = False
                    if gobject.type_name(prop.owner_type) == 'GtkWidget' and \
                           prop.name != 'name':
                        prop_class.common = True
                    else:
                        prop_class.common = False

                    result.append(prop_class)
                except PropertyException:
                    pass

            
        return result

    def _list_child_properties(self):
        result = []

        # only containers has child properties
        container_gtype = gobject.type_from_name('GtkContainer')
        if gobject.type_is_a(self.type, container_gtype):
            for prop in gtk.container_class_list_child_properties(self.type):
                # we only use writable properties
                if (prop.flags & gobject.PARAM_WRITABLE) != 0:
                    prop_class = PropertyClass(prop)
                    prop_class.optional = False
                    prop_class.packing = True
                    result.append(prop_class)

        return result

    def _list_signals(self):
        result = []

        gobject_gtype = gobject.type_from_name('GObject')
        gtype = self.type
        while True:
            kn = gobject.type_name(gtype)
            signal_names = list(gobject.signal_list_names(gtype))
            signal_names.sort()
            result += [(name, kn) for name in signal_names]
            if gtype != gobject_gtype:
                gtype = gobject.type_parent(gtype)
            else:
                break

        return result

    def _create_icon(self):
        icon = None
        if self.generic_name:
            if self.resource_path is not None:
                path = os.path.join(self.resource_path, 'pixmaps',
                                    '%s.png' % self.generic_name)
            else:
                path = os.path.join(pixmaps_dir, '%s.png' % self.generic_name)

            icon = gtk.Image()
            icon.set_from_file(path)

        return icon
            
    def _extend_with_node(self, rootNode):
        functions = util.xml_get_elements_one_level (rootNode, tags.FUNCTION)

        for func in functions:
            func_name = func.getAttribute(tags.NAME)

            if func_name == tags.FILL_EMPTY:
                function.override(func, self, 'fill_empty_function')
            elif func_name == tags.PRE_CREATE:
                function.override(func, self, 'pre_create_function')
            elif func_name == tags.POST_CREATE:
                function.override(func, self, 'post_create_function')
            elif func_name == tags.REPLACE_CHILD:
                function.override(func, self, 'replace_child')
            elif func_name == tags.GET_INTERNAL_CHILD:
                function.override(func, self, 'get_internal_child')
            elif func_name == tags.CHILD_PROPERTY_APPLIES:
                function.override(func, self, 'child_propery_applies')
            elif func_name == tags.SAVE:
                function.override(func, self, 'save_function')
            elif func_name == tags.LOAD:
                function.override(func, self, 'load_function')
            else:
                print "unknown function type", func_name
                
        tmp = rootNode.getElementsByTagName(tags.PROPERTIES)
        if tmp:
            self._update_properties_from_node(tmp[0], self.properties)

        tmp = rootNode.getElementsByTagName(tags.CHILD_PROPERTIES)
        if tmp:
            self._update_properties_from_node(tmp[0], self.child_properties)

        tmp = rootNode.getElementsByTagName(tags.PACKING_DEFAULTS)
        if tmp:
            self._set_packing_defaults_from_node(tmp[0])

    def _update_properties_from_node(self, xmlNode, properties):
        """Given a XML Node update the properties argument by changing existing
        properties or adding new ones.
        """
        for node in xmlNode.childNodes:
            if (node.nodeType != node.ELEMENT_NODE or
                node.nodeName != tags.PROPERTY):
                continue
            
            node_id = node.getAttribute(tags.ID)
            if not node_id:
                continue

            # find the property in our list, if not found append a new prop
            for p in properties:
                if p.id == node_id:
                    prop = p
                    break
            else:
                # Check if we have a sub property
                if '.' in node_id:
                    prop = SubPropertyClass()
                else:
                    prop = PropertyClass()
                prop.id = node_id
                properties.append(prop)
                
            if not prop.update_from_node(node, self):
                print _('Failed to update %s property of %s from xml') % \
                      (node_id, self.name)
            
            if prop.disabled:
                properties.remove(prop)

    def _set_packing_defaults_from_node(self, xmlNode):
        self._child_defaults = {}
        
        for node in xmlNode.childNodes:
            if (node.nodeType != node.ELEMENT_NODE or
                node.nodeName != tags.PARENT_CLASS):
                continue

            parent_name = node.getAttribute(tags.NAME)

            child_defaults = {}
            for childNode in node.childNodes:
                if (childNode.nodeType != node.ELEMENT_NODE or
                    childNode.nodeName != tags.CHILD_PROPERTY):
                    continue

                prop_id = childNode.getAttribute(tags.ID)
                if not prop_id:
                    continue

                value = childNode.getAttribute(tags.DEFAULT)
                if not value:
                    continue

                child_defaults[prop_id] = value
                
            self._child_defaults[parent_name] = child_defaults
            
    
    def _merge(self, parent_class):
        if not self.replace_child:
            self.replace_child = parent_class.replace_child

        if not self.pre_create_function:
            self.pre_create_function = parent_class.pre_create_function

        if not self.post_create_function:
            self.post_create_function = parent_class.post_create_function

        if not self.fill_empty_function:
            self.fill_empty_function = parent_class.fill_empty_function

        if not self.child_property_applies:
            self.child_property_applies = parent_class.fill_property_applies

        if not self.get_internal_child:
            self.get_internal_child = parent_class.get_internal_child

        if not self.save_function:
            self.save_function = parent_class.save_function

        if not self.load_function:
            self.load_function = parent_class.load_function
            
        self._merge_properties(self.properties, parent_class.properties)
        
        self._merge_properties(self.child_properties,
                               parent_class.child_properties)

    def _merge_properties(self, widget_properties, parent_properties):
        for parentprop in parent_properties:
            # search the child's properties for one with the same id
            pc = [childprop for childprop in widget_properties
                                if childprop.id == parentprop.id]

            # if not found, prepend a clone of the parent's one
            if not pc:
                # note that Python copy.copy is equivalent to C memcpy :)
                prop_class = copy.copy(parentprop)
                widget_properties.insert(0, prop_class)
                
            # if found but the parent one was modified substitute it
            elif parentprop.is_modified:
                prop_class = copy.copy(parentprop)
                index = widget_properties.index(pc[0])
                widget_properties[index] = prop_class
        
    def has_property(self, name):
        if [pclass for pclass in self.properties
                       if pclass.id == name]:
            return True
        return False
        
    def dump_param_specs(self):
        print _('Dumping ParamSpec for %s') % self.name
        for i, spec in enumerate(gobject.list_properties(self.type)):
            f =spec.flags & gobject.PARAM_WRITABLE and _('Writable') \
                or _('ReadOnly')
            print '%02d - %-25s %-25s (%s)' % (i+1, spec.name, spec.value_type,
                                               f)
    def dump_properties(self):
        for i, prop in enumerate(self.properties):
            print '%02d - %-25s %-25s %02d %s:%s' % \
                  (i+1, prop._id, prop._name, prop._type, _('Disabled'),
                   prop._disabled)
            if prop._tooltip:
                print '     - %s:%s' % (_('Tooltip'), prop._tooltip[:50])
            print '     - %s:%s %s:%s %s:%s' % \
                  (_('Default'), prop._default,
                   _('Optional'), prop._optional,
                   _('Optional Default'), prop._optional_default)
            if prop._visibility_function:
                print '     - %s:%s' % (_('VisibilityFunction'),
                                        prop._visibility_function)
            if prop._child:
                print '     - %s:%s' % (_('Child'), prop._child)

            print '     - %s:%s %s:%s %s:%s' % \
                  (_('Common'), prop._common,
                   _('Packing'), prop._packing,
                   _('Modified'), prop._is_modified)
            if prop._set_function:
                print '     - %s:%s' % (_('Setter'), prop._set_function)
            if prop._get_function:
                print '     - %s:%s' % (_('Getter'), prop._get_function)



    def get_packing_default (self, container_class, id):
        try:
            child_defaults = self._child_defaults[container_class.name]
            ret_val = child_defaults[id]
        except:
            ret_val = None

        if ret_val:
            return ret_val

        # Look for packing defaults in containers base class
        # This makes it possible to define a packing default for a GtkBox and
        # have it used in both GtkVBox and GtkHBox.
        gobject_gtype = gobject.type_from_name('GObject')
        if container_class.type == gobject_gtype:
            return None
        
        base = gobject.type_parent(container_class.type)

        base_class = widget_registry.get_by_type(base)
        container_gtype = gobject.type_from_name('GtkContainer')
        if base_class and gobject.type_is_a(base_class.type, container_gtype):
            return self.get_packing_default(base_class, id)

        return None

def _direct_children(ancestor, widget, property_id):
    if ancestor and widget.parent == ancestor:
        return True

    return False
