/*
 * http_digest.c -- calculates response for Digest Authentication,
 * compliant to RFC2617. Much of the source is taken from RFC itself.
 *
 * Created: Espeleta Tomas <espeleta@libero.it>, 12-Apr-2001
 * libghttp by: Christopher Blizzard <blizzard@appliedtheory.com>
 *
 * Copyright (C) 1998 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "ghttp.h"
#include "ghttp_constants.h"
#include "http_digest.h"
#include "md5.h"

void CvtHex(IN HASH Bin, OUT HASHHEX Hex)
{
    unsigned short i;
    unsigned char j;

    for (i = 0; i < HASHLEN; i++) {
        j = (Bin[i] >> 4) & 0xf;
        if (j <= 9)
            Hex[i*2] = (j + '0');
         else
            Hex[i*2] = (j + 'a' - 10);
        j = Bin[i] & 0xf;
        if (j <= 9)
            Hex[i*2+1] = (j + '0');
         else
            Hex[i*2+1] = (j + 'a' - 10);
    }
    Hex[HASHHEXLEN] = '\0';
}

/* calculate H(A1) as per spec */
void DigestCalcHA1(
    IN DIGEST_STRUCT *mystruct, OUT HASHHEX SessionKey
)
{
      MD5_CTX Md5Ctx;
      HASH HA1;

      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, mystruct->pszUserName, strlen(mystruct->pszUserName));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, mystruct->pszRealm, strlen(mystruct->pszRealm));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, mystruct->pszPassword, strlen(mystruct->pszPassword));
      MD5Final(HA1, &Md5Ctx);
      if (strcasecmp(mystruct->pszAlg, "md5-sess") == 0) {
            MD5Init(&Md5Ctx);
            MD5Update(&Md5Ctx, HA1, HASHLEN);
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, mystruct->pszNonce, strlen(mystruct->pszNonce));
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, mystruct->pszCNonce, strlen(mystruct->pszCNonce));
            MD5Final(HA1, &Md5Ctx);
      }
      CvtHex(HA1, SessionKey);
}

/* calculate request-digest/response-digest as per HTTP Digest spec */
void DigestCalcResponse(
	IN DIGEST_STRUCT *mystruct, OUT HASHHEX Response      /* request-digest or response-digest */
) {
      MD5_CTX Md5Ctx;
      HASH HA2;
      HASH RespHash;
      HASHHEX HA1, HA2Hex;

      // calculate H(A1)
      DigestCalcHA1(mystruct, HA1);

      // calculate H(A2)
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, mystruct->pszMethod, strlen(mystruct->pszMethod));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, mystruct->pszDigestUri, strlen(mystruct->pszDigestUri));
      if (strcasecmp(mystruct->pszQop, "auth-int") == 0) {
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, mystruct->HEntity, HASHHEXLEN);
      };
      MD5Final(HA2, &Md5Ctx);
      CvtHex(HA2, HA2Hex);

      // calculate response
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, HA1, HASHHEXLEN);
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, mystruct->pszNonce, strlen(mystruct->pszNonce));
      MD5Update(&Md5Ctx, ":", 1);
      if (*(mystruct->pszQop)) {
          MD5Update(&Md5Ctx, mystruct->pszNonceCount, strlen(mystruct->pszNonceCount));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, mystruct->pszCNonce, strlen(mystruct->pszCNonce));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, mystruct->pszQop, strlen(mystruct->pszQop));
          MD5Update(&Md5Ctx, ":", 1);
      };
      MD5Update(&Md5Ctx, HA2Hex, HASHHEXLEN);
      MD5Final(RespHash, &Md5Ctx);
      CvtHex(RespHash, Response);
}

/* --------------------------------------------------------------------
   Function http_digest_header, written by Espeleta Tomas (c) 2001
*/

auth_data *
http_digest_header(ghttp_request *request,
			auth_data *authdata,
			int mantainlogged)
{
	char *WWWAuthenticate = NULL;
	char my_header_format[] = "Digest username=\"%s\", realm=\"%s\", qop=\"%s\", algorithm=\"%s\", uri=\"%s\", nonce=\"%s\", nc=%s, cnonce=\"%s\", response=\"%s\"";
	char *tok, *which, *k;
	char *pair, *key, *value;
	char buffer[32];

	DIGEST_STRUCT *auth;
	HASHHEX response;
	
	/* Set data for authorization-header */
	auth = (DIGEST_STRUCT *) malloc (sizeof(DIGEST_STRUCT));
	memset(auth, '\0', sizeof(DIGEST_STRUCT));

	auth->pszCNonce     = (char *) malloc(33);  /* 32 chars + '\0' */
	auth->pszNonceCount = (char *) malloc(9);   /*  8 chars + '\0' */
	auth->pszUserName   = authdata->username;
	auth->pszPassword   = authdata->password;
	auth->pszDigestUri  = authdata->path;
	auth->pszMethod     = authdata->method;
	auth->pszAlg        = "MD5";

	/* if "mantainlogged" is true, let's try to use old data for logging */
	if (!mantainlogged) {
		auth->NonceCount = authdata->old_nc = 1;
		WWWAuthenticate = (char *) authdata->hdr_wwwauth;
	} else {
		auth->NonceCount = ++(authdata->old_nc);
		auth->pszRealm = authdata->old_realm;
		auth->pszNonce = authdata->old_nonce;
		auth->pszQop = authdata->old_qop;
	}

 	/* Set a random Client-nonce (cnonce) */
	srand((int)clock() + (int)time(NULL));
	sprintf(buffer, "%d", rand() );
	md5(buffer, auth->pszCNonce);

	/* Parse WWW-Authenticate header */
	if (WWWAuthenticate != NULL && strstr(WWWAuthenticate, "Digest") != NULL) {
		pair = (char *) malloc( (size_t) strlen(WWWAuthenticate) );
		for (which = WWWAuthenticate + 7; (tok = strtok(which, ",")) != NULL; which = NULL) {
			/* Now tok should contain a pair <key="value"> */
			strcpy(pair, tok);
			if ((k = index(pair, '=')) != NULL) {
				*k = '\0';
				value = k + 1;

			/* Trim-off leading spaces & quotations */
				for (key = pair; *key == ' '; ) key++;
				if (value[strlen(value)-1] == '\"') value[strlen(value)-1] = '\0';
				if (value[0] == '\"') value++;

 				if (!strcasecmp(key, "realm"))
					authdata->old_realm = auth->pszRealm = strdup(value);
				if (!strcasecmp(key, "nonce"))
					authdata->old_nonce = auth->pszNonce = strdup(value);
				if (!strcasecmp(key, "qop"))
					authdata->old_qop = auth->pszQop = strdup(value);
			}
		}
		free(pair);
	}

	/* Let's prepare & return header */
	if (auth->pszUserName && auth->pszRealm && auth->pszQop && auth->pszAlg
	    && auth->pszDigestUri && auth->pszNonce && auth->pszNonceCount
	    && auth->pszCNonce && auth->pszMethod)     /* everything OK? */
	{
		sprintf(auth->pszNonceCount, "%08d", auth->NonceCount);
		DigestCalcResponse(auth, response);

		/* let's allocate new created header:
		 * just some more bytes than what we really need... */
		authdata->header = (char *) malloc(
			strlen(my_header_format) +
			strlen(auth->pszUserName) +
			strlen(auth->pszRealm) +
			strlen(auth->pszQop) +
			strlen(auth->pszAlg) +
			strlen(auth->pszDigestUri) +
			strlen(auth->pszNonce) +
			strlen(auth->pszNonceCount) +
			strlen(auth->pszCNonce) +
			strlen(response) + 1 );
		sprintf(authdata->header,
			my_header_format,
			auth->pszUserName,
			auth->pszRealm,
			auth->pszQop,
			auth->pszAlg,
			auth->pszDigestUri,
			auth->pszNonce,
			auth->pszNonceCount,
			auth->pszCNonce,
			response );
		free(auth->pszCNonce);
		free(auth->pszNonceCount);
		free(auth);
		return authdata;
	}
	else {
		free(auth->pszCNonce);
		free(auth->pszNonceCount);
		free(auth);
		return NULL;
	}
	/* Data that we don't free "should" be still useful! ;-) */
}
