/*
 *  Hubcot (USB-HUB MASCOT) driver for Linux
 *
 *  Copyright (C) 2001 Tomoaki MITSUYOSHI <micchan@geocities.co.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  This source made use of the result of uhubcot driver for NetBSD
 *  by Takuya SHIOZAKI <tshiozak@netbsd.org>.
 *
 *  "Hubcot" is trademarked by Dreams come true co.,Ltd.
 */

/* $Id: hubcot.c,v 1.2 2001/06/30 07:46:50 mitsu Exp mitsu $ */

#include <linux/kernel.h>
#include <linux/poll.h>
#include <linux/malloc.h>
#include <linux/usb.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/module.h>

#include "hubcot.h"


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
typedef struct {
  struct usb_device  *dev;
  wait_queue_head_t   remove_ok;   /* for disconnecting */
  int                 isopen;
} hubcot_usb_data_t;

static hubcot_usb_data_t* hubcot_usb_data;


/*---------------------------------------------------------------------------*
 * Control Hubcot
 *---------------------------------------------------------------------------*/
static int hubcot_do_something (hubcot_usb_data_t* pdata,
				unsigned char reqtype, short speed)
{
  int r;
  unsigned int pipe;

  if (pdata->dev == NULL) {
    return -1;
  }

  /* create pipe for control tranfer */
  pipe = usb_sndctrlpipe (pdata->dev, 0);

  /* issue a control transfer message */
  r = usb_control_msg (pdata->dev,
		       pipe,
		       reqtype,
		       USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_OUT,
		       speed,
		       0,
		       NULL,
		       0,
		       HZ);

  if (r < 0) {
    err ("error occured after calling usb_control_msg!!");
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * driver entry functions
 *****************************************************************************/
/*---------------------------------------------------------------------------*
 * hubcot_open:
 *---------------------------------------------------------------------------*/
static int hubcot_open (struct inode *inode, struct file *file)
{
  hubcot_usb_data_t* pdata;

  pdata = hubcot_usb_data;

  if (pdata == NULL) {
    /* Device isn't connected */
    return -ENODEV;
  }

  if (pdata->isopen) {
    /* Device is already opened */
    return -EBUSY;
  }

  /* init waitqueue for disconnecting */
  init_waitqueue_head (&pdata->remove_ok);

  file->private_data = pdata;

  pdata->isopen = 1;

  /* info ("Hubcot opened."); */

  return 0;
}

/*---------------------------------------------------------------------------*
 * hubcot_close:
 *---------------------------------------------------------------------------*/
static int hubcot_close (struct inode *inode, struct file *file)
{
  hubcot_usb_data_t* pdata;

  pdata = hubcot_usb_data;

  if (pdata->isopen) {
    pdata->isopen = 0;
    wake_up (&pdata->remove_ok);
  }

  /* info ("Hubcot closed."); */

  return 0;
}

/*---------------------------------------------------------------------------*
 * hubcot_ioctl:
 *---------------------------------------------------------------------------*/
static int hubcot_ioctl (struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
  int speed;
  hubcot_usb_data_t* pdata;

  pdata = file->private_data;

  /* check against unexpected events at Hubcot */
  if (pdata == NULL ||
      pdata->dev == NULL ||
      pdata->isopen != 1 ) {
    return -1;
  }

  /* info ("command = %d", cmd); */

  switch(cmd) {
  /* should be more elegant ^^;; */
  case HUBCOT_REQ_NOOP:
  case HUBCOT_REQ_RIGHT:
  case HUBCOT_REQ_LEFT:
  case HUBCOT_REQ_BOTH:
  case HUBCOT_REQ_BOTH_QUAD:
  case HUBCOT_REQ_LRLRLR:
  case HUBCOT_REQ_RIGHT_TRIPLE:
  case HUBCOT_REQ_BOTH_TRIPLE:
  case HUBCOT_REQ_RL:
  case HUBCOT_REQ_RLRLRL:
    speed = (short)arg;
    hubcot_do_something (pdata, cmd, speed);
    break;
      
  default:
    return -ENOIOCTLCMD;
  }

  return 0;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 * hubcot_probe:
 *   probing device function
 *---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE  >= KERNEL_VERSION(2,4,0)
static void* hubcot_probe (struct usb_device *dev, unsigned int ifnum,
			   const struct usb_device_id *id)
#else

static void* hubcot_probe (struct usb_device *dev, unsigned int ifnum)
#endif
{
  info ("Hubcot found at address %d", dev->devnum);

  if (hubcot_usb_data) {
    info ("no more probe.");
    return NULL;
  }

  hubcot_usb_data = kmalloc (sizeof(hubcot_usb_data_t), GFP_KERNEL);

  if (hubcot_usb_data == NULL) {
    info ("no memory");
    return NULL;
  }

  hubcot_usb_data->dev    = dev;
  hubcot_usb_data->isopen = 0;

  return hubcot_usb_data;
}

/*---------------------------------------------------------------------------*
 * hubcot_disconnect:
 *   disconnecting a device function
 *---------------------------------------------------------------------------*/
static void hubcot_disconnect (struct usb_device *dev, void *ptr)
{
  if (hubcot_usb_data) {

    if (hubcot_usb_data->isopen) {
      hubcot_usb_data->isopen = 0;

      /* to avoid accessing to the device */
      hubcot_usb_data->dev = NULL;

      /* wait until the device resource is releasable */
      sleep_on (&hubcot_usb_data->remove_ok);
    }

    kfree (hubcot_usb_data);
    hubcot_usb_data = NULL;

    info ("Hubcot disconnected.");
  }
}


/*****************************************************************************
 * registering data pool
 *****************************************************************************/

static struct file_operations hubcot_fileopes = {
  owner:          THIS_MODULE,
  ioctl:          hubcot_ioctl,
  open:           hubcot_open,
  release:        hubcot_close,
};

static struct usb_device_id hubcot_table [] = {
  { USB_DEVICE (USB_VENDOR_HUBCOT_TORO, USB_PRODUCT_HUBCOT_TORO) },
                                                           /* Hubcot toro */
  { USB_DEVICE (USB_VENDOR_HUBCOT_KITTY, USB_PRODUCT_HUBCOT_KITTY) },
                                                          /* Hubcot kitty */
  { }                                                /* Terminating entry */
};

MODULE_DEVICE_TABLE (usb, hubcot_table);

static struct usb_driver hubcot_driver = {
  name:        "hubcot",
  probe:       hubcot_probe,
  disconnect:  hubcot_disconnect,
  fops:        &hubcot_fileopes,
  minor:       HUBCOT_MINOR,
#if LINUX_VERSION_CODE  >= KERNEL_VERSION(2,4,0)
  id_table:    hubcot_table,
#endif
};


/*****************************************************************************
 * module functions
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * hubcot_init:
 *---------------------------------------------------------------------------*/
int __init hubcot_init (void)
{
  if (usb_register (&hubcot_driver) < 0) {
    return -1;
  }

  info ("Hubcot driver registered.");

  return 0;
}

/*---------------------------------------------------------------------------*
 * hubcot_exit:
 *---------------------------------------------------------------------------*/
void __exit hubcot_exit(void)
{
  usb_deregister(&hubcot_driver);

}

module_init(hubcot_init);
module_exit(hubcot_exit);

MODULE_AUTHOR("Tomoaki MITSUYOSHI <micchan@geocities.co.jp>");
MODULE_DESCRIPTION("USB Hubcot driver");
