/*
JuffEd - An advanced text editor
Copyright 2007-2009 Mikhail Murzin

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License 
version 2 as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "PluginManager.h"

#include <QtCore/QDir>
#include <QtCore/QMap>
#include <QtCore/QPluginLoader>
#include <QtGui/QAction>
#include <QtGui/QDockWidget>
#include <QtGui/QMainWindow>
#include <QtGui/QToolBar>

#include "AppInfo.h"
#include "gui/GUI.h"
#include "JuffPlugin.h"
#include "Log.h"
#include "PluginSettings.h"

namespace Juff {

class PluginManager::Interior {
public:
	Interior() {
		managerInt_ = 0;
		gui_ = 0;
		curEngine_ = "";
	}
	~Interior() {
	}
	
	bool addPlugin(JuffPlugin* plugin) {
		QString engine = plugin->targetEngine();

		Log::debug(engine);
		
		if ( engines_.contains(engine) ) {
			//	plugin has specified its type and this type
			//	is allowed by PluginManager
			plugins_[engine].append(plugin);
			return true;
		}
		else {
			Log::printToLog(QString("Engine type %1 is not acceptable. Plugin %2 was not loaded").arg(engine).arg(plugin->name()));
			return false;
		}
	}
	
	ManagerInterface* managerInt_;
	GUI::GUI* gui_;
	QMap<QString, PluginList> plugins_;

	QStringList engines_;
	QString curEngine_;
	
	//	stores and manages dock windows (shows and hides when active engine is changed
	QMap<QString, QWidgetList> docks_;
	
	//	stores all plugins' context menu actions
	QMap<QString, ActionList> contextMenuActions_;
};


PluginManager::PluginManager(const QStringList& engines, ManagerInterface* m, GUI::GUI* gui) : QObject() {
	pmInt_ = new Interior();
	pmInt_->engines_ = engines;
	pmInt_->engines_ << "all";
	pmInt_->gui_ = gui;
	pmInt_->managerInt_ = m;
}

PluginManager::~PluginManager() {
	delete pmInt_;
}

////////////////////////////////////////////////////////////
//	Plugins Events

void PluginManager::notifyDocCreated(const QString& fileName) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocCreated(fileName);
			}
		}
	}
}

void PluginManager::notifyDocActivated(const QString& fileName) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocActivated(fileName);
			}
		}
	}
}

void PluginManager::notifyDocModified(const QString& fileName, bool modified) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocModified(fileName, modified);
			}
		}
	}
}

void PluginManager::notifyDocClosed(const QString& fileName) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocClosed(fileName);
			}
		}
	}
}

void PluginManager::notifyDocRenamed(const QString& oldFileName, const QString& newFileName) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocRenamed(oldFileName, newFileName);
			}
		}
	}
}

void PluginManager::notifyDocSaved(const QString& fileName) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onDocSaved(fileName);
			}
		}
	}
}

void PluginManager::notifyContextMenuCalled(int line, int col) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onContextMenuCalled(line, col);
			}
		}
	}
}

#if QT_VERSION >= 0x040500
void PluginManager::notifyTabMoved(int from, int to) {
	JUFFENTRY;
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin ) {
				plugin->onTabMoved(from, to);
			}
		}
	}
}
#endif

////////////////////////////////////////////////////////////

bool PluginManager::pluginExists(const QString& name) {
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			if ( plugin && plugin->name() == name ) {
				return true;
			}
		}
	}
	return false;
}

void PluginManager::loadPlugin(const QString& path) {
	JUFFENTRY;

	QPluginLoader loader(path);
	if ( !loader.load() ) {
		Log::warning(QString("Plugin '%1' was NOT loaded: %2").arg(path).arg(loader.errorString()));
		return;
	}
	
	QObject *obj = loader.instance();
	if ( obj ) {
		JuffPlugin* plugin = qobject_cast<JuffPlugin*>(obj);
		if ( plugin ) {

			//	Check if we need to load it
			if ( !PluginSettings::pluginEnabled(plugin->name()) ) {
				pmInt_->gui_->addPluginSettingsPage(plugin->name(), 0);
				Log::debug("Plugin is disabled in Settings");
				return;
			}

			//	Check if plugin with the same name was already loaded.
			//	If is was then exit.
			if ( pluginExists(plugin->name()) )
				return;

			plugin->setManager(pmInt_->managerInt_);
			if ( pmInt_->addPlugin(plugin) ) {

				Log::debug(QString("-----=====((((( Plugin '%1' was loaded successfully! )))))=====-----").arg(plugin->name()));

				//	context menu actions
				QString type = plugin->targetEngine();
				pmInt_->contextMenuActions_[type] << plugin->contextMenuActions();
				
				//	settings page
				pmInt_->gui_->addPluginSettingsPage(plugin->name(), plugin->settingsPage());
			}
			else {
				loader.unload();
			}
		}
		else {
			Log::warning("Error while casting to JuffPlugin type");
		}
	}
	else {
		Log::debug("Empty plugin instance");
	}
}

void PluginManager::loadPlugins() {
	JUFFENTRY;
	//	user's plugins
	QDir pluginDir(AppInfo::configDirPath() + "/plugins");
	foreach (QString fileName, pluginDir.entryList(QDir::Files)) {
		QString path = pluginDir.absoluteFilePath(fileName);
		loadPlugin(path);
	}
	
	//	global plugins
	QDir gPluginDir(AppInfo::appDirPath() + "/plugins");
	foreach (QString fileName, gPluginDir.entryList(QDir::Files)) {
		QString path = gPluginDir.absoluteFilePath(fileName);
		loadPlugin(path);
	}
	
	foreach (QString type, pmInt_->docks_.keys()) {
		pmInt_->gui_->addDocks(type, pmInt_->docks_[type]);
	}
}

void PluginManager::applySettings() {
	foreach (QString engine, pmInt_->engines_) {
		foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
			plugin->applySettings();
		}
	}
}



////////////////////////////////////////////////////////////
//	GUI controls

MenuList PluginManager::getMenus(const QString& engine) {
	JUFFENTRY2;
	MenuList menus;
	foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
		if ( plugin->menu() )
			menus << plugin->menu();
	}
	return menus;
}

ActionList PluginManager::getMainMenuActions(const QString& engine, MenuID id) {
	JUFFENTRY2;
	ActionList list;
	foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
		list << plugin->mainMenuActions(id);
	}
	return list;
}

ToolBarList PluginManager::getToolBars(const QString& engine) {
	JUFFENTRY2;
	ToolBarList list;
	foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
		if ( plugin->toolBar() )
			list << plugin->toolBar();
	}
	return list;
}

QWidgetList PluginManager::getDocks(const QString& engine) {
	JUFFENTRY2;
	QWidgetList list;
	foreach (JuffPlugin* plugin, pmInt_->plugins_[engine]) {
		list << plugin->dockList();
	}
	return list;
}

ActionList PluginManager::getContextMenuActions(const QString& engine) {
	if ( pmInt_->contextMenuActions_.contains(engine) )
		return pmInt_->contextMenuActions_[engine];
	else
		return ActionList();
}


}	//	namespace Juff
