#include "guibrowser.h"
#include "browser.h"
#include "createitemdialog.h"

#include <QMessageBox>
#include <QStatusBar>
#include <QDebug>
#include <QApplication>
#include <QItemSelectionModel>

#include "ui_newconnectionwidget.h"
#include "guibrowser.moc"

NewConnectionDialog::NewConnectionDialog( QWidget *parent )
	: QDialog( parent )
	, ui( new Ui_newConnectionWidget )
{
	ui->setupUi( this );
	connect( ui->okPushButton, SIGNAL(clicked(bool)),
		this, SLOT(slotOkClicked(bool)) );
	connect( ui->cancelPushButton, SIGNAL(clicked(bool)),
		this, SLOT(slotCancelClicked(bool)) );
}

void NewConnectionDialog::slotOkClicked( bool checked )
{
	if( !ui->hostnameLineEdit->text().isEmpty() &&
		!ui->portLineEdit->text().isEmpty() )
	{
		emit(createConnection( ui->hostnameLineEdit->text(),
			ui->portLineEdit->text().toUInt() ));
	}
	else
		qDebug() << "invalid input.";
	done(0);
}

void NewConnectionDialog::slotCancelClicked( bool checked )
{
	done(0);
}

BrowserMainWindow::BrowserMainWindow( QWidget *parent )
	: QMainWindow( parent )
	, fileModel( new QInfinity::BrowserModel( this ) )
	, plugin( new QInfinity::DefaultTextPlugin( this ) )
{
	qDebug() << "Plugin is type " << plugin->infPlugin()->note_type;
	fileModel->addPlugin( *plugin );
	setupActions();
	setupUi();
}

void BrowserMainWindow::slotNewConnection()
{
	NewConnectionDialog *dialog = new NewConnectionDialog( this );
	connect( dialog, SIGNAL(createConnection( const QString&, unsigned int )),
		this, SLOT(slotCreateConnection( const QString&, unsigned int )) );
	dialog->exec();
	delete dialog;
}

void BrowserMainWindow::slotCreateConnection( const QString &hostname,
	unsigned int port )
{
	Connection *conn = new Connection( hostname, port, this );
	connect( conn, SIGNAL(connecting(Connection*)),
		this, SLOT(slotConnectionConnecting(Connection*)) );
	connect( conn, SIGNAL(disconnecting(Connection*)),
		this, SLOT(slotConnectionDisconnecting(Connection*)) );
	connect( conn, SIGNAL(connected(Connection*)),
		this, SLOT(slotConnectionConnected(Connection*)) );
	connect( conn, SIGNAL(error(Connection*, QString)),
		this, SLOT(slotConnectionError(Connection*, QString)) );
	connect( conn, SIGNAL(disconnected(Connection*)),
		this, SLOT(slotConnectionDisconnected(Connection*)) );
	conn->open();
}

void BrowserMainWindow::slotConnectionConnecting( Connection *conn )
{
	QString msg = "Connecting to " + conn->name() + ".";
	statusLabel->setText( msg );
}

void BrowserMainWindow::slotConnectionDisconnecting( Connection *conn )
{
	QString msg = "Disconnecting from " + conn->name() + ".";
	statusLabel->setText( msg );
}

void BrowserMainWindow::slotConnectionConnected( Connection *conn )
{
	qDebug() << "Adding " << conn->name();
	statusLabel->setText( "Connected." );
	fileModel->addConnection( conn->xmppConnection(), conn->name() );
}

void BrowserMainWindow::slotConnectionDisconnected( Connection *conn )
{
	statusLabel->setText( "Disconnected." );
}

void BrowserMainWindow::slotConnectionError( Connection *conn,
	QString message )
{
	QString full_msg = "Error with connection ";
	full_msg += conn->name();
	full_msg += ": ";
	full_msg += message;
	QMessageBox::critical( this, "Connection error", full_msg );
}

void BrowserMainWindow::slotSelectionChanged( const QItemSelection &selected,
	const QItemSelection &deselected )
{
	Q_UNUSED(deselected);
	newConnectionAction->setEnabled( canCreateConnection( selected ) );
	newFolderAction->setEnabled( canCreateFolder( selected ) );
	newNoteAction->setEnabled( canCreateNote( selected ) );
	deleteAction->setEnabled( canDeleteItem( selected ) );
	openAction->setEnabled( canOpenItem( selected ) );
}

void BrowserMainWindow::slotQuit()
{
	exit(0);
}

void BrowserMainWindow::slotCreateFolder()
{
	QModelIndexList indexes = selectedIndexes();
	if( indexes.size() != 1 )
	{
		qDebug() << "Cant create folder without parent.";
		return;
	}
	CreateItemDialog *dialog = new CreateItemDialog( "Create folder named: ",
		this );
	dialog->exec();
	fileModel->createDirectory( indexes[0], dialog->itemName() );
	delete dialog;
}

void BrowserMainWindow::slotCreateNote()
{
	QModelIndexList indexes = selectedIndexes();
	if( indexes.size() != 1 )
	{
		qDebug() << "Can only create notes with one parent.";
		return;
	}
	CreateItemDialog *dialog = new CreateItemDialog( "Create note named: ",
		this );
	dialog->exec();
	fileModel->createNote( indexes[0], *plugin, dialog->itemName() );
	delete dialog;
}

void BrowserMainWindow::slotDelete()
{
	QModelIndexList selected = selectedIndexes();
	QModelIndex index;

	foreach( index, selected )
		fileModel->removeRow( index.row(), index.parent() );
}

void BrowserMainWindow::slotOpen()
{
	QModelIndexList selected = selectedIndexes();
	QModelIndex index;
	QStandardItem *stdItem;
	QInfinity::NodeItem *nodeItem;

	foreach( index, selected )
	{
		stdItem = fileModel->itemFromIndex( index );
		if( stdItem->type() == QInfinity::BrowserItemFactory::NodeItem )
		{
			nodeItem = dynamic_cast<QInfinity::NodeItem*>(stdItem);
			QInfinity::BrowserIter itr = nodeItem->iter();
			itr.browser()->subscribeSession( itr );
		}
	}
}

void BrowserMainWindow::contextMenuEvent( QContextMenuEvent *event )
{
	if( !event )
		return;

	showContextMenu( event->globalPos() );
}

void BrowserMainWindow::setupUi()
{
	treeView = new QTreeView( this );
	treeView->setModel( fileModel );
	connect( treeView, SIGNAL(expanded(const QModelIndex&)),
		fileModel, SLOT(itemActivated(const QModelIndex&)) );
	connect( treeView->selectionModel(), SIGNAL(
			selectionChanged(const QItemSelection&, const QItemSelection&)),
		this, SLOT(slotSelectionChanged(
			const QItemSelection&, const QItemSelection&)) );
	setStatusBar( new QStatusBar( this ) );
	statusLabel = new QLabel();
	statusBar()->addWidget( statusLabel );
	setCentralWidget( treeView );
}

void BrowserMainWindow::setupActions()
{
	quitAction = new QAction( tr("&Quit"), this );
	newConnectionAction = new QAction( tr("&New Connection..."), this );
	newFolderAction = new QAction( tr("Create &Folder..."), this );
	newFolderAction->setEnabled( false );
	newNoteAction = new QAction( tr("Create &Note..."), this );
	newNoteAction->setEnabled( false );
	deleteAction = new QAction( tr("Delete"), this );
	deleteAction->setEnabled( false );
	openAction = new QAction( tr("Open"), this );
	openAction->setEnabled( false );

	connect( quitAction, SIGNAL(triggered(bool)),
		this, SLOT(slotQuit()) );
	connect( newConnectionAction, SIGNAL(triggered(bool)),
		this, SLOT(slotNewConnection()) );
	connect( newFolderAction, SIGNAL(triggered(bool)),
		this, SLOT(slotCreateFolder()) );
	connect( newNoteAction, SIGNAL(triggered(bool)),
		this, SLOT(slotCreateNote()) );
	connect( deleteAction, SIGNAL(triggered(bool)),
		this, SLOT(slotDelete()) );
	connect( openAction, SIGNAL(triggered(bool)),
		this, SLOT(slotOpen()) );

	QMenu *fileMenu = new QMenu( tr("&File"), this );
	fileMenu->addAction( newConnectionAction );
	fileMenu->addAction( quitAction );

	menuBar()->addMenu( fileMenu );
}

void BrowserMainWindow::showContextMenu( const QPoint &globalPos )
{
	static QMenu *contextMenu = 0;
	if( !contextMenu )
	{
		contextMenu = new QMenu( this );
		contextMenu->addAction( newConnectionAction );
		contextMenu->addAction( newFolderAction );
		contextMenu->addAction( newNoteAction );
		contextMenu->addAction( deleteAction );
		contextMenu->addAction( openAction );
	}
	contextMenu->popup( globalPos );
}

bool BrowserMainWindow::canCreateConnection( const QItemSelection &selected )
{
	return true;
}

bool BrowserMainWindow::canCreateFolder( const QItemSelection &selected )
{
	QList<QModelIndex> indexes = selected.indexes();
	if( indexes.size() != 1 )
		return false;
	QStandardItem *item = fileModel->itemFromIndex( indexes[0] );
	if( item->type() != QInfinity::BrowserItemFactory::NodeItem )
		return false;
	return dynamic_cast<QInfinity::NodeItem*>(item)->isDirectory();
}

bool BrowserMainWindow::canCreateNote( const QItemSelection &selected )
{
	return canCreateFolder( selected );
}

bool BrowserMainWindow::canDeleteItem( const QItemSelection &selected )
{
	QList<QModelIndex> indexes = selected.indexes();
	return indexes.size() > 0;
}

bool BrowserMainWindow::canOpenItem( const QItemSelection &selected )
{
	QList<QModelIndex> indexes = selected.indexes();
	return indexes.size() > 0;
}

QModelIndexList BrowserMainWindow::selectedIndexes() const
{
	return treeView->selectionModel()->selection().indexes();
}

QInfinity::NodeItem *BrowserMainWindow::nodeItemFromIndex( const QModelIndex &index ) const
{
	QStandardItem *item = fileModel->itemFromIndex( index );
	if( item->type() != QInfinity::BrowserItemFactory::NodeItem )
	{
		qDebug() << "Cannot convert item of non NodeItem type to NodeItem.";
		return 0;
	}
	return dynamic_cast<QInfinity::NodeItem*>(item);
}

int main( int argc, char **argv )
{
	QInfinity::init();
	QApplication app( argc, argv );
	BrowserMainWindow mainWindow;
	mainWindow.setVisible( true );

	app.exec();
	QInfinity::deinit();
	return 0;
}

