/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: write.cpp,v 1.59 2004/03/25 10:12:43 robert Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#  if _MSC_VER == 1200
#       pragma warning(disable:4786)
#   endif
#  include <io.h>
extern "C"
{
   // For some reason MSVC++ can't find the prototype for this function
   int access(const char *, int);
}
#  include <direct.h>
#else
#  include <unistd.h>
#  include <sys/stat.h>
#  include <sys/types.h>
#  include <fcntl.h>
#  if defined(__APPLE__) || defined(__NetBSD__) || defined(__FreeBSD__)
#    include <sys/param.h>
#    include <sys/mount.h>
#  else
#    include <sys/vfs.h>
#  endif
#endif


#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#include "../config.h"
#include "write.h"
#include "tunepimp.h"
#include "file_meta.h"
#include "id3_meta.h"
#include "vorbis_meta.h"
#include "flac_meta.h"
#include "ape_meta.h"

#ifdef WIN32
const char *dirSep = "\\";
const char dirSepChar = '\\';
const char *disallowedFileNameChars = ".\"*/:<>?|";
#else
const char *dirSep = "/";
const char dirSepChar = '/';
#endif

#define DB printf("%s:%d\n", __FILE__, __LINE__);

const int   numRepVars = 16;
const char *repVars[numRepVars] = { "%artist", "%sortname", "%abc2", "%abc3", "%abc",
                                    "%album", "%0num", "%num", "%track", "%format",
                                    "%type", "%status", "%month", "%day", "%year", "%country" };

//---------------------------------------------------------------------------

void FileNameMaker::makeNewFileName(const Metadata &dataArg,
                                    string         &fileName,
                                    int             index)
{
    Metadata          data = dataArg;
    string            name, origPath, origFile, ext, allowedFileChars, rep;
    string::size_type i;
    int               numCharsLeft, varCount = 0, varLen;

    origPath = extractFilePath(fileName);
    origFile = extractFileName(fileName);
    ext = extractFileExt(fileName);
    if (data.variousArtist)
        name = context->getVariousFileMask();
    else
        name = context->getFileMask();

    allowedFileChars = context->getAllowedFileCharacters();

    if (data.sortName.empty())
        data.sortName = data.artist;

    numCharsLeft = context->getMaxFileNameLen() - ext.length();
    if (numCharsLeft > 0)
    {
        if (context->getMoveFiles())
            numCharsLeft -= context->getDestDir().length() + 1;
        else
            numCharsLeft -= origPath.length() + 1;

        // Preparse the file spec and see how to allocate space to ensure everything
        // fits into the space we have for a filename.
        for(i = 0; i < name.length(); i++)
        {
            if (name[i] != '%')
            {
                numCharsLeft--;
                continue;
            }

            for(int j = 0; j != numRepVars; j++)
            {
                char num[10];

                if (strncmp(name.c_str() + i, repVars[j], strlen(repVars[j])) == 0)
                {
                    switch(j)
                    {
                        case 0: // artist
                            varCount++;
                            break;
                        case 1: // sortname
                            varCount++;
                            break;
                        case 2: // abc2
                            numCharsLeft -= 2;
                            break;
                        case 3: // abc3
                            numCharsLeft -= 3;
                            break;
                        case 4: // abc
                            numCharsLeft -= 1;
                            break;
                        case 5: // album
                            varCount++;
                            break;
                        case 6: // 0num
                            numCharsLeft -= 2;
                            break;
                        case 7: // num
                            sprintf(num, "%d", data.trackNum);
                            numCharsLeft -= strlen(num);
                            break;
                        case 8: // track
                            varCount++;
                            break;
                        case 9: // format
                            numCharsLeft -= data.fileFormat.length();
                            break;
                        case 10: // type
                        {
                            string type;
                            convertFromAlbumType(data.albumType, type);
                            numCharsLeft -= type.length();
                            break;
                        }
                        case 11: // status
                        {
                            string status;
                            convertFromAlbumStatus(data.albumStatus, status);
                            numCharsLeft -= status.length();
                            break;
                        }
                        case 12: // month
                        case 13: // day
                        case 15: // country
                        {
                            numCharsLeft -= 2;
                            break;
                        }
                        case 14: // year
                        {
                            numCharsLeft -= 4;
                            break;
                        }
                    }
                    i += strlen(repVars[j]);
                }
            }
        }
    }
    else
    {
        numCharsLeft = 999999999;
        varCount = 100;
    }

    for(i = 0; i < name.length(); i++)
    {
        if (name[i] != '%')
        {  
            numCharsLeft--;
            continue;
        }

        for(int j = 0; j != numRepVars; j++)
        {
            char num[10];

            if (strncmp(name.c_str() + i, repVars[j], strlen(repVars[j])) == 0)
            {
                switch(j)
                {
                    case 0: // artist
                        varLen = numCharsLeft / varCount;
                        varCount--;
                        rep = shortenString(data.artist, varLen);
                        numCharsLeft -= varLen;
                        break;
                    case 1: // sortname
                        varLen = numCharsLeft / varCount;
                        varCount--;
                        if (data.sortName.length())
                            rep = shortenString(data.sortName, varLen);
                        else
                            rep = shortenString(data.artist, varLen);
                        numCharsLeft -= varLen;
                        break;
                    case 2: // abc2
                        if (data.variousArtist)
                            rep = data.album.substr(0, 2);
                        else
                            rep = data.sortName.substr(0, 2);
                        break;
                    case 3: // abc3
                        if (data.variousArtist)
                            rep = data.album.substr(0, 3);
                        else
                            rep = data.sortName.substr(0, 3);
                        break;
                    case 4: // abc
                        if (data.variousArtist)
                            rep = data.album.substr(0, 1);
                        else
                            rep = data.sortName.substr(0, 1);
                        break;
                    case 5: // album
                        varLen = numCharsLeft / varCount;
                        varCount--;
                        rep = shortenString(data.album, varLen);
                        numCharsLeft -= varLen;
                        break;
                    case 6: // 0num
                        sprintf(num, "%02d", data.trackNum);
                        rep = string(num);
                        break;
                    case 7: // num
                        sprintf(num, "%d", data.trackNum);
                        rep = string(num);
                        break;
                    case 8: // track
                        varLen = numCharsLeft / varCount;
                        varCount--;
                        rep = shortenString(data.track, varLen);
                        numCharsLeft -= varLen;
                        break;
                    case 9: // format
                        rep = data.fileFormat;
                        break;
                    case 10: // type
                        convertFromAlbumType(data.albumType, rep);
                        break;
                    case 11: // status
                        convertFromAlbumStatus(data.albumStatus, rep);
                        break;
                    case 12: // month
                    {
                        char temp[10];

                        sprintf(temp, "%02d", data.releaseMonth);
                        rep = string(temp);
                        break;
                    }
                    case 13: // day
                    {
                        char temp[10];

                        sprintf(temp, "%02d", data.releaseDay);
                        rep = string(temp);
                        break;
                    }
                    case 14: // year
                    {
                        char temp[10];

                        sprintf(temp, "%04d", data.releaseYear);
                        rep = string(temp);
                        break;
                    }
                    case 15: // country
                    {
                        rep = data.releaseCountry;
                        if (rep.length() == 0)
                            rep = "__";
                        break;
                    }
                }
                rep = sanitize(rep);
                name.erase(i, strlen(repVars[j]));
                name.insert(i, rep);
                i += rep.length() - 1;
                rep = "";
            }
        }
    }

#ifdef WIN32

    string::size_type pos = 0;
    // Some windows systems can't handle three periods in a row. Fucking lame!
    for(;;)
    {
        pos = name.find(" ...");
        if (pos != string::npos)
            name.erase(pos, 4);
		else
		{
			pos = name.find("...");
            if (pos != string::npos)
                name.erase(pos, 3);
			else
			    break;
		}
	}

	// Now remove any characters that the filesystem might bitch about
    for(unsigned i = 0; i < name.size(); i++)
    {
        if (strchr(disallowedFileNameChars, name[i]))
        {
            name.erase(i, 1);
            i--;
        }
    }

    for(;;)
    {
        pos = name.find("  ");
        if (pos != string::npos)
            name.erase(pos, 1);
		else
			break;
	}

    for(;;)
    {
        pos = name.find(" \\");
        if (pos != string::npos)
            name.erase(pos, 1);
		else
			break;
	}

#endif

    // Rewrite spaces to underscores if allowedFileNameChars doesn't contain
    // a space character.
    if (!allowedFileChars.empty() && 
        strchr(allowedFileChars.c_str(), ' ') == NULL )
    {
        for (unsigned i = 0; name[i] != '\0'; i++)
        {
            if ( name[i] == ' ' )
                name[i] = '_';
        }
    }

    if (context->getMoveFiles())
    {
        if (context->getRenameFiles())
            name = context->getDestDir() + string(dirSep) + name;
        else
            name = context->getDestDir() + string(dirSep) + extractFilePath(name) +
                       string(dirSep) + extractFileBase(origFile);
    }
    else
    {
        if (context->getRenameFiles())
            name = string(origPath) + string(dirSep) + extractFileBase(name);
        else
            name = string(origPath) + string(dirSep) + extractFileBase(origFile);
    }

    // Remove everything from name that isn't in allowedFileChars.
    // However, never remove the directory separator char:
    if (!allowedFileChars.empty())
        for(unsigned i = 0; i < name.size(); i++)
        {
            if (name[i] != dirSepChar && ! strchr(allowedFileChars.c_str(), name[i]) )
            {
                name.erase(i, 1);
                i--;
            }
        }

    // Check to see if the partition we're writing to is FAT, and if so, convert the filename
    // to OEM encoding. This should fix problems with funky chars crashing the tagger
#ifdef WIN32
    string drive = extractVolume(name);
    char   fsName[100];
    if (GetVolumeInformation(drive.c_str(), NULL, 0, NULL, NULL, NULL, fsName, 100))
    {
        if (strncmp(fsName, "FAT", 3) == 0)
        {
            char *temp = (char *)malloc(name.length() + 1);
            CharToOem(name.c_str(), temp);
            name = string(temp);
        }
    }
#endif

    if (index > 0)
    {
        char temp[10];

        sprintf(temp, " (%d)", index);
        fileName = name + string(temp) + string(ext);
    }
    else
        fileName = name + string(ext);



}

//---------------------------------------------------------------------------

string FileNameMaker::shortenString(const string &in, int &len)
{
     int front, back;
     string out;

     if ((int)in.length() <= len)
     {
         len = in.length();
         return in;
     }

     if (len < 3 || in.length() < 3)
     {
         len = 0;
         return string();
     }

     len -= 3;
     front = len / 2;
     back = len / 2 + len % 2;

     out = in.substr(0, front) + string("___") +
           in.substr(in.length() - back, back);
     len = out.length();

     return out;
}

//---------------------------------------------------------------------------

string FileNameMaker::extractFilePath(const string &file)
{
    string::size_type pos;
    
    pos = file.rfind(dirSep, file.size() - 1);
    if (pos == string::npos)
        return string(".");

    return file.substr(0, pos);
}

//---------------------------------------------------------------------------

string FileNameMaker::extractFileName(const string &file)
{
    string::size_type pos;
    
    pos = file.rfind(dirSep, file.size() - 1);
    if (pos == string::npos)
        return file;

    return file.substr(pos + 1, file.size());
}

//---------------------------------------------------------------------------

string FileNameMaker::extractFileBase(const string &fileArg)
{
    string            file = fileArg;
    string::size_type pos;

    file = extractFileName(file);
    pos = file.rfind(".", file.size() - 1);
    if (pos == string::npos)
        return file;

    return file.substr(0, pos);
}

//---------------------------------------------------------------------------

string FileNameMaker::extractFileExt(const string &file)
{
    string::size_type pos;
    
    pos = file.rfind(".", file.size() - 1);
    if (pos == string::npos)
        return file;

    return file.substr(pos, file.size());
}

//---------------------------------------------------------------------------

string FileNameMaker::extractVolume(const string &file)
{
#ifdef WIN32
    string::size_type pos;

    if (file.size() > 2 && file[0] == '\\' && file[1] == '\\')
    {
        pos = file.find("\\", 2);
        if (pos == string::npos)
            return "";

        return file.substr(0, pos);
    }

    if (file.size() > 2 && isalpha(file[0]) && file[1] == ':')
    {
        pos = file.find("\\", 1);
        if (pos == string::npos)
            return "";

        return file.substr(0, pos + 1);
    }
#endif
   
    return "";
}

//---------------------------------------------------------------------------

const string FileNameMaker::sanitize(const string &str)
{
    string data;

    data = str;
    for(unsigned i = 0; i < str.size(); i++)
       if (str[i] == dirSepChar)
           data.erase(i, 1);

    return data;
}

//---------------------------------------------------------------------------

WriteThread::WriteThread(TunePimp  *tunePimpArg, FileCache *cacheArg) 
            :Thread(), FileNameMaker(&tunePimpArg->context)
{
    tunePimp = tunePimpArg;
    cache = cacheArg;

    exitThread = false;
    sem = new Semaphore();
}

//---------------------------------------------------------------------------

WriteThread::~WriteThread(void)
{
    exitThread = true;
    sem->signal();
    join();
    delete sem;
}

//---------------------------------------------------------------------------

void WriteThread::wake(void)
{
    sem->signal();
}

//---------------------------------------------------------------------------

void WriteThread::threadMain(void)
{
    Metadata  server;
    string    fileName, status, trm, trackId;
    Track    *track;
    bool      checkedTrack = false, writeError = false;

    for(; !exitThread;)
    {
        track = cache->getNextItem(eVerified);
        if (track == NULL)
        {
            if (checkedTrack)
            {
                checkedTrack = false;
                tunePimp->writeTagsComplete(!writeError);
                writeError = false;
            }
            sem->wait();
            continue;
        }

          checkedTrack = true;

        track->lock();
        track->getServerMetadata(server);
        track->getTRM(server.fileTrm);

        if (track->hasChanged())
        {
            track->unlock();
            if (writeTrack(track, server))
            {
                track->lock();
                if (track->getStatus() == eVerified)
                {
                    if (context->getAutoRemoveSavedFiles())
                        track->setStatus(eDeleted);
                    else
					{
						track->setLocalMetadata(server);
                        track->setStatus(eSaved);
					}
                    track->setError("Track saved.");
                }
            }
            else
            {
                track->lock();
                track->setStatus(eError);
                writeError = true;
            }
            tunePimp->wake(track);
        }
        else
        {
            track->getFileName(fileName);
            if (context->getAutoRemoveSavedFiles())
                track->setStatus(eDeleted);
            else
                track->setStatus(eSaved);
        }

        track->unlock();

        tunePimp->wake(track);
        cache->release(track);
    }
}

//---------------------------------------------------------------------------

bool WriteThread::writeTrack(Track *track, const Metadata &server)
{
    string           ext, fileName;
    FileMetadata    *fileMeta = NULL;
    unsigned long    fileSize;

    track->lock();
    track->getFileName(fileName);
    ext = extractFileExt(fileName);

    track->unlock();
    fileSize = fileOpenTest(fileName);
    track->lock();

    if (fileSize == 0)
    {
        track->setError("Cannot remove existing file -- access denied.");
        track->unlock();
        return false;
    }
    
    track->unlock();
    if (!diskSpaceTest(fileName, fileSize))
    {
        track->lock();
        track->setError("Not enough available diskspace for writing tags to the existing file.");
        track->unlock();
        return false;
    }

#ifdef HAVE_LIBMAD
    if (strcasecmp(ext.c_str(), ".mp3") == 0)
        fileMeta = new ID3(tunePimp->context.getWriteID3v1());
    else
#endif
#ifdef HAVE_OGGVORBIS
    if (strcasecmp(ext.c_str(), ".ogg") == 0)
        fileMeta = new Vorbis();
    else
#endif
#ifdef HAVE_FLAC
    if (strcasecmp(ext.c_str(), ".flac") == 0)
        fileMeta = new FLAC();
    else
#endif
#ifdef HAVE_LIBAPE
    if (strcasecmp(ext.c_str(), ".ape") == 0)
        fileMeta = new Ape();
    else
#endif
        fileMeta = NULL;

    if (fileMeta)
    {
        bool   ret;
        string err;

        try
        {
            ret = fileMeta->write(fileName, server, tunePimp->context.getClearTags());
        }
        catch(...)
        {
            ret = false;
        }
        fileMeta->getError(err);
        delete fileMeta;

        if (!ret)
        {
            track->lock();
            track->setError(string("Could not write metadata to track: ") + err);
            track->unlock();
            return false;
        }
    }

    if (tunePimp->context.getRenameFiles() || tunePimp->context.getMoveFiles())
    {
        string newName, err;
        int    ret;

        for(int j = 0;; j++)
        {
            newName = fileName;
            makeNewFileName(server, newName, j);
            if (!tunePimp->context.getMoveFiles() || createPath(newName))
            {
                if (newName == fileName)
                   break;

				if (access(newName.c_str(), 0) == 0)
					continue;

                fileSize = fileOpenTest(fileName);
                if (fileSize == 0)
                {
                    track->lock();
                    track->setError("Cannot write to new file -- access denied.");
                    track->unlock();
                    return false;
                }
    
                if (!diskSpaceTest(newName, fileSize))
                {
                    track->lock();
                    track->setError("Not enough available diskspace for writing a new file.");
                    track->unlock();
                    return false;
                }

#ifdef WIN32
                ret = (int)!MoveFile(fileName.c_str(), newName.c_str());
#else
                ret = rename(fileName.c_str(), newName.c_str());
#endif
				if (ret != 0 && errno == EEXIST)
                    continue;

                if (ret != 0)
                {
                    track->lock();
                    track->setError("Could not rename file.");
                    track->unlock();
                    return false;
                }

                if (tunePimp->context.getMoveFiles())
                    cleanPath(fileName);
            }
            else
            {
                string path = extractFilePath(newName);

                err = string("Could not create destination directory: ") + path;
                track->lock();
                track->setError(err);
                track->unlock();
                return false;
            }

            break;
        }
        track->lock();
        track->setFileName(newName);
        track->unlock();
    }

    return true;
}

//---------------------------------------------------------------------------

#ifdef WIN32
unsigned long WriteThread::fileOpenTest(const string &fileName)
{
    HANDLE        openTest;
    unsigned long fileSize;
    
    openTest = CreateFile(fileName.c_str(), GENERIC_READ | GENERIC_WRITE,
                          0, NULL, OPEN_EXISTING, 0, NULL);
    if (openTest == INVALID_HANDLE_VALUE)
        return 0;

    fileSize = SetFilePointer(openTest, 0, NULL, FILE_END);
    CloseHandle(openTest);

    return fileSize;
}

#else
//---------------------------------------------------------------------------

unsigned long WriteThread::fileOpenTest(const string &fileName)
{
    int           openTest;
    unsigned long fileSize;
   
    // This test is probably not NFS 
    openTest = open(fileName.c_str(), O_EXCL | O_RDWR);
    if (openTest < 0)
        return 0;

    fileSize = lseek(openTest, 0, SEEK_END);
    close(openTest);

    return fileSize;
}
#endif

//---------------------------------------------------------------------------

#ifdef WIN32
bool WriteThread::diskSpaceTest(const string &fileName, unsigned long fileSize)
{
    ULARGE_INTEGER temp; 
    __int64        diskSpace;
    bool           ret;

    string path = extractFilePath(fileName);

    // GetDiskFreeSpaceEx says that if a path is a UNC then it needs
    // to end with a backslash. 
    if (path.size() >= 2 && path[0] == '\\' && path[1] == '\\')
        path += "\\";

    ret = GetDiskFreeSpaceEx(path.c_str(), &temp, NULL, NULL);
    if (!ret && GetLastError() == ERROR_INVALID_NAME)
    {
        path.erase(3, path.size());
        ret = GetDiskFreeSpaceEx(path.c_str(), &temp, NULL, NULL);
    }

    if (ret)
    {
        diskSpace = *(__int64 *)&temp;

        // Increase the size a bit in case the file grows and what not.
        fileSize += fileSize / 10;

        return diskSpace > (__int64)fileSize;
    }

    // If we can't determine if the diskspace is ok, just assume it is. There is
    // probably some bigger bug causing havoc...
    return true;
}

#else
//---------------------------------------------------------------------------

bool WriteThread::diskSpaceTest(const string &fileName, unsigned long fileSize)
{
    struct statfs stat;

    string path = extractFilePath(fileName);
    if (statfs(path.c_str(), &stat) == 0)
    {
        fileSize += fileSize / 10;
        fileSize /= stat.f_bsize;

        return fileSize < (unsigned long)stat.f_bavail;
    }
    else
        return false;
}
#endif

bool WriteThread::createPath(const string &pathArg)
{
    string            path = string(extractFilePath(pathArg).c_str());
    string            volume = string(extractVolume(pathArg).c_str());
    string            partial;
    string::size_type pos;


    if (volume.size() > 0)
        path.erase(0, volume.size());

    if (path[path.size() - 1] != dirSepChar)
        path += dirSep;

    for(pos = 1;;)
    {
        pos = path.find(dirSep, pos);
        if (pos == string::npos)
            break;

        partial = volume + path.substr(0, pos);
        if (access(partial.c_str(), 0))
        {
#ifdef WIN32
            if (CreateDirectory(partial.c_str(), NULL) < 0)
#else
            if (mkdir(partial.c_str(), 0755) < 0)
#endif
                return false;
        }

        pos++;
    }

    return true;
}

void WriteThread::cleanPath(const string &pathArg)
{
    string      path = string(extractFilePath(pathArg).c_str());
    string      volume = string(extractVolume(pathArg).c_str());
    string      srcDir, complete;
    unsigned    pos;
    bool        ret;

    srcDir = tunePimp->context.getTopSrcDir();
    if (volume.size() > 0)
        path.erase(0, volume.size());

    if (path[path.size() - 1] == dirSepChar)
        path.erase(path.size() - 1);

    if (srcDir[srcDir.size() - 1] == dirSepChar)
        srcDir.erase(srcDir.size() - 1);

    for(;;)
    {
        complete = volume + path;
        if (strcasecmp(srcDir.c_str(), complete.c_str()) == 0)
        {
            break;
        }
           
#ifdef WIN32
        ret = RemoveDirectory(complete.c_str());
#else
        ret = (bool)!rmdir(complete.c_str());
#endif
        if (!ret)
            break;

        pos = path.rfind(dirSep);
        if (pos == string::npos)
            break;

        path.erase(pos);
    }
}
