/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module lists all packages, sorted by installed size.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

#include "parser.h"

#define OUTPUT_FILE	"isizes.html"


/*+
Comparison function for sorting by installed size.
+*/
static int cmp_size (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);

	if (!pp->installed_size && !qq->installed_size)
		return 0;
	else if (!pp->installed_size)
		return -1;
	else if (!qq->installed_size)
		return -1;

	return -(pp->installed_size - qq->installed_size);
}
	

/*+
+*/
static int isize_init (void)
{
	int i;
	FILE *fp;
	struct package_info *p;
	long total = 0;

#	define MAX_VALUES	100
	int histogram[MAX_VALUES];

	memset (histogram, 0, sizeof histogram);

	qsort (cache, cachecnt, sizeof (cache[0]), cmp_size);

	fp = fopen (OUTPUT_FILE, "w");
	mp_doc_open (fp, "Installed Packages: Ordered by Size");

	mp_abstract (fp, "\
This is a list of packages ordered by installed size.  It could be \n\
useful to someone trying to select packages a package to purge\n\
(temporarily?) to free some disk space.");

	total = 0;
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->installed || p->installed_size == 0)
			continue;

		total += p->installed_size;
		if (p->installed_size/10 < MAX_VALUES)
			histogram[p->installed_size/10]++;
	}

	fprintf (fp, "Total space required: %ldk\n", total);

	mp_list_open (fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->installed && !p->unpacked)
			continue;
		if (p->installed_size == 0)
			continue;

		mp_package (fp, p, 2, T_INSTALLED_SIZE);
	}
	mp_list_close (fp);

	mp_break (fp);
	mp_title (fp, 2, "Histogram");
	fprintf (fp, "\
This is a histogram showing the number of packages with the indicated\n\
installed package size.  Each tick is 20k.  Packages over %dk are ignored.\n",
		MAX_VALUES * 20);

	mp_histogram (fp, histogram, MAX_VALUES);

	mp_doc_close (fp);
	fclose (fp);

	gzip (OUTPUT_FILE);

	return 0;
}


/*+
List all packages ordered by installed size.
+*/
static int isize_index (FILE *fp, int type)
{
	if (type != MAGPIE_INSTALLED_PACKAGES)
		return 0;

	mp_item_open (fp); 
	mp_url (fp, "%s.gz", "Ordered by actual size", 
		OUTPUT_FILE);
	mp_item_close (fp); 

	return 0;
}


struct magpie_module mod_installed_sizes = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by installation size",
	init              : isize_init,
	unannotated_index : isize_index
};
