/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module lists all packages, sorted by package source.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

#define OUTPUT_FILE	"sources.html"


/*+
Comparison function for sorting by source
We provide one list sorted by source since that can suggest
other packages we may be interested in.  Unfortunately, only
about a third of all packages provide a source value.
+*/
static int cmp_source (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	if (!pp->source && qq->source)
		return -1;
	if (!qq->source && pp->source)
		return 1;

	if (pp->source == 0 && qq->source == 0)
		return strcoll (pp->name, qq->name);

	r = strcoll (pp->source->name, qq->source->name);
	if (r)
		return r;

	if (pp->source->restriction == 0 && 
		qq->source->restriction == 0)
		return 0;
	if (!pp->source->restriction)
		return -1;
	if (!qq->source->restriction)
		return 1;

	r = strcoll (pp->source->restriction, qq->source->restriction);

	if (r)
		return r;

	return strcoll (pp->name, qq->name);
}


/*+
Print the list of packages, grouped by source.  We break our HTML
encapsulation by using <ol></ol> to allow a quick count of the
number of packages.
+*/
static int sources_init (void)
{
	int i, cnt;
	FILE *fp;
	struct package_info *p, *q;
	char short_category[30];
	char *s;

#	define MAX_VALUES 100
	int histogram[MAX_VALUES];

	qsort (cache, cachecnt, sizeof (cache[0]), cmp_source);

	fp = fopen (OUTPUT_FILE, "w");
	mp_doc_open (fp, "Grouped by Package Source");

	mp_abstract (fp, "\
This page lists all packages, grouped by their underlying source.\n\
It can be useful if you want to make sure you've installed all related\n\
packages.");

	mp_title (fp, 2, "Packages with explicit source");

	mp_list_open (fp);
	for (i = 0; i < cachecnt; ) {
		p = cache[i];
		if (!p->source)
			i++;
		else {

			/* count the number of packages with this source. */
			for (cnt = 0; i + cnt < cachecnt; cnt++) {
				q = cache[i+cnt];
				if (!q->source || 
					strcoll (p->source->name, q->source->name) != 0)
					break;
			}
			if (cnt < MAX_VALUES)
				histogram[cnt]++;

			s = p->source->restriction;
			if (s == NULL)
				s = "";

			mp_name (fp, p->source->name);
			mp_item_open (fp);
			fprintf (fp, "%s %s (%d)", p->source->name, s, cnt);
			mp_item_close (fp);

			mp_list_open (fp);
			while (cnt--) {

				strcpy (short_category, categories[p->category]);
				s = strchr (short_category, '/');
				if (s)
					*s = '\0';

				mp_item_open (fp);
				mp_url (fp, "details/%1$1.1s/%1$s.html.gz", "%1$s", 
					p->name);
				mp_nbsp (fp);
				mp_text (fp, p->summary);
				mp_item_close (fp);

				p = cache[++i];
			}

			mp_list_close (fp);
		}
	}
	mp_list_close (fp);

	mp_break (fp);
	mp_title (fp, 2, "Histogram");

	fprintf (fp, "\
This histogram shows the number of sources with the indicated\n\
number of packages.");

	mp_histogram (fp, histogram, MAX_VALUES);

	mp_break (fp);

	mp_title (fp, 2, "Packages without explicit source");
	fprintf (fp, "<ol>\n"); /* HTML */
	q = NULL;
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->source) {
			mp_item_open (fp);
			mp_nbsp (fp);
			mp_url (fp, "details/%1$1.1s/%1$s.html.gz", "%1$s", p->name);
			mp_text (fp, p->summary);
			mp_item_close (fp);
		}
	}
	fprintf (fp, "</ol>\n"); /* HTML */

	mp_doc_close (fp);
	fclose (fp);

	gzip (OUTPUT_FILE);

	return 0;
}


/*+
Index the packages by name.
+*/
static int sources_index (FILE *fp, int type)
{
	if (type != MAGPIE_ALL_PACKAGES)
		return 0;

	mp_item_open (fp);
	mp_url (fp, "%s.gz", "Grouped by package source", 
		OUTPUT_FILE);
	mp_item_close (fp);

	return 0;
}


struct magpie_module mod_sources = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by source package",
	init              : sources_init,
	unannotated_index : sources_index
};
